<?php
/**
 * Tabbed Admin Interface for Delcampe Plugin
 * File: class-admin-tabbed-page.php
 * Version: 1.8.0.0
 *
 * Creates a professional tabbed interface for all plugin settings
 * Similar to WP Lister and other professional plugins
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Admin
 * @since 1.0.9.0
 * @version 1.8.0.0
 * 
 * Changelog:
 * 1.8.0.0 - Updated changelog to include all versions up to production release
 * 1.3.0.0 - Added shipping tab for managing shipping models and mappings
 * 1.2.0.0 - Added sync management tab for queue monitoring and manual sync
 * 1.1.2.1 - Fixed translation loading by using static tab keys
 * 1.1.1.0 - Added get_tabs() method for lazy initialization
 * 1.1.0.0 - Enhanced tab structure and version display
 * 1.0.9.0 - Initial release with tabbed interface
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_Admin_Tabbed_Page
 * 
 * Manages the tabbed settings interface for the plugin
 * Consolidates all settings into one professional interface
 * 
 * @version 1.8.0.0
 */
class Delcampe_Admin_Tabbed_Page {

    /**
     * Available tabs for the settings page
     * Using static keys to avoid early translation loading
     * @var array
     * @since 1.1.2.1
     * @version 1.3.0.0 - Added shipping tab
     */
    private static $tab_keys = array(
        'general',
        'categories', 
        'sync',
        'shipping',
        'webhooks',
        'advanced',
        'developer',
        'license'
    );

    /**
     * Initialize the tabbed interface
     * Enhanced in version 1.1.2.1 to avoid early translation loading
     * NOTE: This method should be called AFTER WordPress has loaded translations
     */
    public static function init() {
        // Register settings
        add_action('admin_init', array(__CLASS__, 'register_settings'));
        
        // Handle form submissions
        add_action('admin_post_save_delcampe_categories', array(__CLASS__, 'handle_category_save'));
        
        // AJAX handlers (v1.10.13.0)
        add_action('wp_ajax_test_delcampe_connection', array(__CLASS__, 'ajax_test_connection'));
        add_action('admin_post_import_delcampe_branch', array(__CLASS__, 'handle_branch_import'));
        
        // AJAX handlers
        add_action('wp_ajax_delcampe_count_subcategories', array(__CLASS__, 'ajax_count_subcategories'));
        
        // Enqueue admin styles and scripts
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_admin_assets'));
    }

    /**
     * Get tab labels with translations
     * This method is only called when rendering, avoiding early translation loading
     * 
     * @since 1.1.2.1
     * @version 1.3.0.0 - Added shipping tab
     * @return array Tab keys with translated labels
     */
    private static function get_tab_labels() {
        return array(
            'general' => __('General Settings', 'wc-delcampe-integration'),
            'categories' => __('Categories', 'wc-delcampe-integration'),
            'sync' => __('Sync Management', 'wc-delcampe-integration'),
            'shipping' => __('Shipping', 'wc-delcampe-integration'),
            'webhooks' => __('Webhooks', 'wc-delcampe-integration'),
            'advanced' => __('Advanced', 'wc-delcampe-integration'),
            'developer' => __('Developer', 'wc-delcampe-integration'),
            'license' => __('License', 'wc-delcampe-integration'),
        );
    }

    /**
     * Enqueue admin CSS and JavaScript
     * Enhanced in version 1.2.0.0 with sync management scripts
     */
    public static function enqueue_admin_assets($hook) {
        // Only load on our settings page
        if ($hook !== 'delcampe_page_delcampe-settings') {
            return;
        }

        // Add inline CSS for tabs
        wp_add_inline_style('wp-admin', '
            .delcampe-tabs {
                margin: 20px 0;
                border-bottom: 1px solid #ccc;
            }
            .delcampe-tabs .nav-tab {
                margin-left: 0;
                margin-right: 0.5em;
            }
            .delcampe-tab-content {
                display: none;
                padding: 20px 0;
            }
            .delcampe-tab-content.active {
                display: block;
            }
            .delcampe-section {
                background: #fff;
                border: 1px solid #e5e5e5;
                padding: 20px;
                margin-bottom: 20px;
                box-shadow: 0 1px 1px rgba(0,0,0,.04);
            }
            .delcampe-section h3 {
                margin-top: 0;
                padding-bottom: 10px;
                border-bottom: 1px solid #eee;
            }
            /* Added in v1.1.0.0 for better version display */
            .delcampe-version {
                color: #666;
                font-size: 0.9em;
                margin-left: 10px;
            }
            /* Added in v1.2.0.0 for sync status */
            .sync-status-indicator {
                display: inline-block;
                width: 10px;
                height: 10px;
                border-radius: 50%;
                margin-right: 5px;
            }
            .sync-status-indicator.active { background: #46b450; }
            .sync-status-indicator.pending { background: #f0ad4e; }
            .sync-status-indicator.error { background: #dc3232; }
            .sync-status-indicator.ended { background: #999; }
            .sync-status-indicator.deleted { background: #666; }
            .sync-queue-item {
                padding: 10px;
                border-bottom: 1px solid #eee;
            }
            .sync-queue-item:last-child {
                border-bottom: none;
            }
            /* Added in v1.8.0.0 for changelog */
            .delcampe-changelog {
                max-height: 600px;
                overflow-y: auto;
                padding-right: 15px;
            }
            .delcampe-changelog h4 {
                margin-top: 20px;
                margin-bottom: 10px;
                color: #23282d;
                font-size: 14px;
                font-weight: 600;
            }
            .delcampe-changelog ul {
                margin-left: 20px;
                margin-bottom: 15px;
            }
            .delcampe-changelog li {
                margin-bottom: 5px;
            }
        ');

        // Add inline CSS for auto-checking animations
        wp_add_inline_style('delcampe-admin', '
            .check-delcampe-status.checking-auto {
                background: #f0ad4e !important;
                color: #fff !important;
                border-color: #eea236 !important;
                position: relative;
                overflow: hidden;
            }
            .check-delcampe-status.checking-auto:before {
                content: "";
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
                animation: autoCheckPulse 1.5s infinite;
            }
            @keyframes autoCheckPulse {
                0% { left: -100%; }
                100% { left: 100%; }
            }
            .sync-auto-status {
                display: inline-block;
                margin-left: 10px;
                padding: 3px 8px;
                background: #f0f0f1;
                border-radius: 3px;
                font-size: 11px;
                color: #666;
            }
            .sync-auto-status.active {
                background: #d4edda;
                color: #155724;
            }
            
            /* v1.10.35.4: Real-time animation styles */
            @keyframes pulse {
                0% { opacity: 1; }
                50% { opacity: 0.8; }
                100% { opacity: 1; }
            }
            
            @keyframes blink {
                0%, 100% { opacity: 1; }
                50% { opacity: 0.3; }
            }
            
            .dashicons.spin {
                animation: spin 2s linear infinite;
            }
            
            @keyframes spin {
                100% { transform: rotate(360deg); }
            }
        ');
        
        // Add inline JavaScript for tab switching and sync management
        wp_add_inline_script('jquery', '
            jQuery(document).ready(function($) {
                // Tab switching - Enhanced in v1.1.0.0
                $(".nav-tab").on("click", function(e) {
                    e.preventDefault();
                    
                    var tab = $(this).data("tab");
                    
                    // Update active tab
                    $(".nav-tab").removeClass("nav-tab-active");
                    $(this).addClass("nav-tab-active");
                    
                    // Show corresponding content
                    $(".delcampe-tab-content").removeClass("active");
                    $("#tab-" + tab).addClass("active");
                    
                    // Update URL without reload
                    var url = window.location.href.split("&tab=")[0];
                    window.history.pushState({}, "", url + "&tab=" + tab);
                    
                    // Log tab switch for debugging (v1.1.0.0)
                    console.log("Switched to tab: " + tab);
                });
                
                // Sync management functions (v1.2.0.0)
                $("#manual-sync-btn").on("click", function() {
                    var $btn = $(this);
                    var originalText = $btn.text();
                    
                    $btn.prop("disabled", true).text("Processing...");
                    $("#sync-result").html("");
                    
                    $.post(ajaxurl, {
                        action: "delcampe_process_sync_queue",
                        nonce: "' . wp_create_nonce('delcampe_admin_nonce') . '"
                    }, function(response) {
                        $btn.prop("disabled", false).text(originalText);
                        
                        if (response.success) {
                            $("#sync-result").html(
                                "<div class=\"notice notice-success\"><p>" + response.data.message + "</p></div>"
                            );
                            // Reload queue status
                            loadQueueStatus();
                        } else {
                            $("#sync-result").html(
                                "<div class=\"notice notice-error\"><p>" + response.data.message + "</p></div>"
                            );
                        }
                    });
                });
                
                // Clear queue function
                $("#clear-queue-btn").on("click", function() {
                    if (!confirm("Are you sure you want to clear the sync queue?")) {
                        return;
                    }
                    
                    var $btn = $(this);
                    var originalText = $btn.text();
                    
                    $btn.prop("disabled", true).text("Clearing...");
                    
                    $.post(ajaxurl, {
                        action: "delcampe_clear_sync_queue",
                        nonce: "' . wp_create_nonce('delcampe_admin_nonce') . '"
                    }, function(response) {
                        $btn.prop("disabled", false).text(originalText);
                        
                        if (response.success) {
                            $("#sync-result").html(
                                "<div class=\"notice notice-success\"><p>Queue cleared successfully.</p></div>"
                            );
                            loadQueueStatus();
                        }
                    });
                });
                
                // Load queue status
                function loadQueueStatus() {
                    $("#queue-status").html("Loading...");
                    
                    $.post(ajaxurl, {
                        action: "delcampe_get_queue_status",
                        nonce: "' . wp_create_nonce('delcampe_admin_nonce') . '"
                    }, function(response) {
                        if (response.success) {
                            var html = "<div class=\"sync-queue-status\">";
                            
                            // Show current processing item if available
                            if (response.data.current_item) {
                                html += "<div style=\"background: #f0f8ff; border: 1px solid #0073aa; padding: 10px; margin-bottom: 10px; border-radius: 3px;\">";
                                html += "<p style=\"margin:0 0 5px 0; color:#0073aa;\"><strong>🔄 Currently Processing:</strong></p>";
                                html += "<p style=\"margin:0 0 3px 0;\"><strong>SKU:</strong> " + response.data.current_item.sku + "</p>";
                                html += "<p style=\"margin:0 0 3px 0;\"><strong>Title:</strong> " + response.data.current_item.title + "</p>";
                                html += "<p style=\"margin:0 0 3px 0;\"><strong>Action:</strong> " + response.data.current_item.action + "</p>";
                                html += "<p style=\"margin:0;\"><strong>Time:</strong> " + response.data.current_item.elapsed + " seconds</p>";
                                html += "</div>";
                            }
                            
                            // Queue statistics with visual indicators
                            html += "<div style=\"margin-top: 15px;\">";
                            html += "<p style=\"font-weight: bold; margin-bottom: 10px;\">Queue Statistics:</p>";
                            html += "<div style=\"display: grid; grid-template-columns: repeat(auto-fit, minmax(120px, 1fr)); gap: 10px;\">";
                            
                            // Pending
                            html += "<div style=\"background: #fff3cd; border: 1px solid #ffc107; padding: 10px; border-radius: 5px; text-align: center;\">";
                            html += "<div style=\"font-size: 24px; font-weight: bold; color: #856404;\">" + response.data.pending + "</div>";
                            html += "<div style=\"font-size: 12px; color: #856404;\">Pending</div>";
                            html += "</div>";
                            
                            // Processing  
                            var processingCount = response.data.processing ? 1 : 0;
                            var processingStyle = processingCount > 0 ? 
                                "background: #cfe2ff; border: 1px solid #0073aa; animation: pulse 2s infinite;" : 
                                "background: #f8f9fa; border: 1px solid #dee2e6;";
                            html += "<div style=\"" + processingStyle + " padding: 10px; border-radius: 5px; text-align: center;\">";
                            html += "<div style=\"font-size: 24px; font-weight: bold; color: #004085;\">" + processingCount + "</div>";
                            html += "<div style=\"font-size: 12px; color: #004085;\">Processing</div>";
                            html += "</div>";
                            
                            // Failed
                            var failedStyle = response.data.failed > 0 ? 
                                "background: #f8d7da; border: 1px solid #dc3545;" : 
                                "background: #f8f9fa; border: 1px solid #dee2e6;";
                            html += "<div style=\"" + failedStyle + " padding: 10px; border-radius: 5px; text-align: center;\">";
                            html += "<div style=\"font-size: 24px; font-weight: bold; color: #721c24;\">" + response.data.failed + "</div>";
                            html += "<div style=\"font-size: 12px; color: #721c24;\">Failed</div>";
                            html += "</div>";
                            
                            // Total
                            html += "<div style=\"background: #e9ecef; border: 1px solid #6c757d; padding: 10px; border-radius: 5px; text-align: center;\">";
                            html += "<div style=\"font-size: 24px; font-weight: bold; color: #495057;\">" + response.data.total + "</div>";
                            html += "<div style=\"font-size: 12px; color: #495057;\">Total</div>";
                            html += "</div>";
                            
                            html += "</div>";
                            html += "</div>";
                            
                            if (response.data.next_run) {
                                html += "<p><strong>Next sync-queue run:</strong> " + response.data.next_run + "</p>";
                            }
                            if (response.data.master_next_run) {
                                html += "<p><strong>Next master sync run:</strong> " + response.data.master_next_run + "</p>";
                            }
                            
                            html += "</div>";
                            $("#queue-status").html(html);
                        }
                    });
                }
                
                // Load queue status when sync tab is shown
                function checkAndLoadQueueStatus() {
                    if ($("#tab-sync").hasClass("active")) {
                        loadQueueStatus();
                        // Set up auto-refresh if not already set
                        if (!window.queueRefreshInterval) {
                            // Refresh every 5 seconds when processing, 30 seconds otherwise
                            var refreshRate = 30000;
                            $.post(ajaxurl, {
                                action: "delcampe_get_queue_status",
                                nonce: "' . wp_create_nonce('delcampe_admin_nonce') . '"
                            }, function(response) {
                                if (response.success && response.data.processing) {
                                    refreshRate = 5000; // 5 seconds when processing
                                }
                            });
                            window.queueRefreshInterval = setInterval(loadQueueStatus, refreshRate);
                        }
                    }
                }
                
                // Check on page load
                checkAndLoadQueueStatus();
                startAutoCheckIfNeeded();
                
                // Also check when tab is clicked
                $(".nav-tab").on("click", function() {
                    var clickedTab = $(this).data("tab");
                    
                    // Handle queue status refresh
                    setTimeout(checkAndLoadQueueStatus, 100); // Small delay to ensure tab is active
                    
                    // Handle auto-checking for processing listings
                    if (clickedTab === "sync") {
                        setTimeout(startAutoCheckIfNeeded, 200);
                    } else {
                        stopAutoCheck();
                    }
                });
                
                // Auto-check processing listings
                var processingCheckInterval = null;
                var processingListings = [];
                
                function collectProcessingListings() {
                    processingListings = [];
                    $(".check-delcampe-status").each(function() {
                        processingListings.push({
                            button: $(this),
                            productId: $(this).data("product-id"),
                            reference: $(this).data("reference")
                        });
                    });
                    return processingListings.length > 0;
                }
                
                function checkProcessingListing(listing, isAutoCheck = true) {
                    var $btn = listing.button;
                    var productId = listing.productId;
                    var reference = listing.reference;
                    
                    // Visual feedback for auto-checking
                    if (isAutoCheck) {
                        $btn.addClass("checking-auto").text("Auto-checking...");
                    } else {
                        $btn.prop("disabled", true).text("Checking...");
                    }
                    
                    $.post(ajaxurl, {
                        action: "delcampe_check_listing_status",
                        nonce: "' . wp_create_nonce('delcampe_check_status') . '",
                        product_id: productId,
                        reference: reference
                    }, function(response) {
                        if (response.success) {
                            if (response.data.found) {
                                // Update the status cell
                                var $statusCell = $btn.closest("tr").find("td:nth-child(3)");
                                $statusCell.html(\'<span style="color: #46b450;">✓ Live</span><br><small>ID: \' + response.data.delcampe_id + \'</small>\');
                                
                                // Replace check button with view button
                                $btn.replaceWith(\'<a href="https://www.delcampe.net/en_US/collectibles/item/\' + response.data.delcampe_id + \'.html" target="_blank" class="button button-small">View</a>\');
                                
                                // Show notification instead of alert for auto-checks
                                if (isAutoCheck) {
                                    showNotification("Listing " + reference + " is now live on Delcampe!", "success");
                                } else {
                                    alert("Listing found on Delcampe! ID: " + response.data.delcampe_id);
                                }
                            } else {
                                // Still processing, restore button
                                if (isAutoCheck) {
                                    $btn.removeClass("checking-auto").text("Check Status");
                                } else {
                                    $btn.prop("disabled", false).text("Check Status");
                                    alert("Listing not found on Delcampe yet. It may still be processing.");
                                }
                            }
                        } else {
                            // Error, restore button
                            if (isAutoCheck) {
                                $btn.removeClass("checking-auto").text("Check Status");
                            } else {
                                $btn.prop("disabled", false).text("Check Status");
                                alert("Error checking status: " + (response.data ? response.data : "Unknown error"));
                            }
                        }
                    }).fail(function() {
                        // Failed, restore button
                        if (isAutoCheck) {
                            $btn.removeClass("checking-auto").text("Check Status");
                        } else {
                            $btn.prop("disabled", false).text("Check Status");
                            alert("Failed to check status. Please try again.");
                        }
                    });
                }
                
                function autoCheckProcessingListings() {
                    if (!collectProcessingListings()) {
                        // No more processing listings, stop auto-checking
                        if (processingCheckInterval) {
                            clearInterval(processingCheckInterval);
                            processingCheckInterval = null;
                            console.log("No processing listings found, stopping auto-check");
                        }
                        return;
                    }
                    
                    console.log("Auto-checking " + processingListings.length + " processing listings...");
                    
                    // Check all processing listings sequentially with small delay
                    processingListings.forEach(function(listing, index) {
                        setTimeout(function() {
                            // Only check if button still exists and hasnt been replaced
                            if (listing.button.hasClass("check-delcampe-status")) {
                                checkProcessingListing(listing, true);
                            }
                        }, index * 1000); // 1 second delay between checks
                    });
                }
                
                function showNotification(message, type) {
                    // Create notification element if it doesnt exist
                    if ($("#delcampe-notifications").length === 0) {
                        $("body").append(\'<div id="delcampe-notifications" style="position:fixed;top:50px;right:20px;z-index:9999;"></div>\');
                    }
                    
                    var notificationClass = type === "success" ? "notice-success" : "notice-warning";
                    var $notification = $(\'<div class="notice \' + notificationClass + \' is-dismissible" style="margin:5px 0;padding:10px;background:#fff;border-left:4px solid #46b450;box-shadow:0 1px 3px rgba(0,0,0,0.1);"><p>\' + message + \'</p></div>\');
                    
                    $("#delcampe-notifications").append($notification);
                    
                    // Auto-remove after 5 seconds
                    setTimeout(function() {
                        $notification.fadeOut(function() {
                            $(this).remove();
                        });
                    }, 5000);
                }
                
                // Start auto-checking when sync tab is active
                function startAutoCheckIfNeeded() {
                    if ($("#tab-sync").hasClass("active") && collectProcessingListings()) {
                        if (!processingCheckInterval) {
                            // Show auto-checking indicator
                            $("#sync-auto-status").addClass("active").fadeIn();
                            
                            // Initial check after 3 seconds
                            setTimeout(autoCheckProcessingListings, 3000);
                            
                            // Then check every 15 seconds
                            processingCheckInterval = setInterval(autoCheckProcessingListings, 15000);
                            console.log("Started auto-checking processing listings");
                        }
                    }
                }
                
                // Stop auto-checking when leaving sync tab
                function stopAutoCheck() {
                    if (processingCheckInterval) {
                        clearInterval(processingCheckInterval);
                        processingCheckInterval = null;
                        
                        // Hide auto-checking indicator
                        $("#sync-auto-status").removeClass("active").fadeOut();
                        
                        console.log("Stopped auto-checking processing listings");
                    }
                }
                
                // Manual check button handler
                $(document).on("click", ".check-delcampe-status", function(e) {
                    e.preventDefault();
                    var listing = {
                        button: $(this),
                        productId: $(this).data("product-id"),
                        reference: $(this).data("reference")
                    };
                    checkProcessingListing(listing, false);
                });
                
                // Close Delcampe listing button handler
                $(document).on("click", ".close-delcampe-listing", function() {
                    var $btn = $(this);
                    var productId = $btn.data("product-id");
                    var listingId = $btn.data("listing-id");
                    
                    if (!confirm("Are you sure you want to close this listing on Delcampe? This action cannot be undone.")) {
                        return;
                    }
                    
                    var reason = prompt("Optional: Enter a reason for closing this listing (or leave blank):");
                    
                    var originalText = $btn.text();
                    $btn.prop("disabled", true).text("Closing...");
                    
                    $.post(ajaxurl, {
                        action: "delcampe_close_listing",
                        nonce: "' . wp_create_nonce('delcampe_nonce') . '",
                        product_id: productId,
                        listing_id: listingId,
                        reason: reason || ""
                    }, function(response) {
                        if (response.success) {
                            $btn.text("Closed");
                            setTimeout(function() {
                                location.reload(); // Reload to show updated status
                            }, 1000);
                        } else {
                            $btn.prop("disabled", false).text(originalText);
                            alert("Failed to close listing: " + (response.data.message || "Unknown error"));
                        }
                    }).fail(function() {
                        $btn.prop("disabled", false).text(originalText);
                        alert("Request failed. Please try again.");
                    });
                });
            });
        ');
    }

    /**
     * AJAX handler for testing Delcampe connection
     * 
     * @since 1.10.13.0
     */
    public static function ajax_test_connection() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'test_delcampe_connection')) {
            wp_send_json_error(array('message' => __('Security check failed', 'wc-delcampe-integration')));
        }
        
        // Test the connection
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            wp_send_json_error(array(
                'message' => sprintf(
                    __('Connection failed: %s', 'wc-delcampe-integration'),
                    $token->get_error_message()
                )
            ));
        }
        
        wp_send_json_success(array(
            'message' => __('Connection successful!', 'wc-delcampe-integration')
        ));
    }
    
    /**
     * Register all plugin settings
     * Enhanced in version 1.1.0.0 with additional validation
     */
    public static function register_settings() {
        // General settings - Core
        register_setting('delcampe_general_settings', 'delcampe_api_key');
        register_setting('delcampe_general_settings', 'delcampe_enable_logging');
        
        // General settings - Background Tasks (v1.10.13.0)
        register_setting('delcampe_general_settings', 'delcampe_background_sync');
        register_setting('delcampe_general_settings', 'delcampe_push_changes');
        register_setting('delcampe_general_settings', 'delcampe_handle_stock');
        
        // General settings - Orders (v1.10.13.0)
        register_setting('delcampe_general_settings', 'delcampe_create_orders');
        register_setting('delcampe_general_settings', 'delcampe_shipped_order_status');
        register_setting('delcampe_general_settings', 'delcampe_paid_order_status');
        register_setting('delcampe_general_settings', 'delcampe_unpaid_order_status');
        
        // Advanced settings - Fully implemented in v1.10.10.4
        // v1.10.35.16: Split sync settings for orders vs listings/inventory
        register_setting('delcampe_advanced_settings', 'delcampe_orders_sync_enabled');
        register_setting('delcampe_advanced_settings', 'delcampe_orders_sync_interval');
        register_setting('delcampe_advanced_settings', 'delcampe_listings_sync_enabled');
        register_setting('delcampe_advanced_settings', 'delcampe_listings_sync_interval');
        register_setting('delcampe_advanced_settings', 'delcampe_sync_prices');
        register_setting('delcampe_advanced_settings', 'delcampe_order_sync_days');
        register_setting('delcampe_advanced_settings', 'delcampe_batch_size');
        
        // Keep legacy settings for backward compatibility
        register_setting('delcampe_advanced_settings', 'delcampe_sync_enabled');
        register_setting('delcampe_advanced_settings', 'delcampe_sync_interval');
    }

    /**
     * Render the main settings page with tabs
     * Enhanced in version 1.1.2.1 to only load translations when rendering
     */
    public static function render_settings_page() {
        $current_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'general';
        
        // Ensure tab exists
        if (!in_array($current_tab, self::$tab_keys)) {
            $current_tab = 'general';
        }
        
        // Get translated labels only when rendering
        $tab_labels = self::get_tab_labels();
        ?>
        <div class="wrap">
            <h1>
                <?php echo esc_html(get_admin_page_title()); ?>
                <span class="delcampe-version">v<?php echo DELCAMPE_PLUGIN_VERSION; ?></span>
            </h1>
            
            <!-- Tab Navigation -->
            <h2 class="nav-tab-wrapper delcampe-tabs">
                <?php foreach (self::$tab_keys as $tab_key) : ?>
                    <a href="#" class="nav-tab <?php echo $current_tab === $tab_key ? 'nav-tab-active' : ''; ?>" 
                       data-tab="<?php echo esc_attr($tab_key); ?>">
                        <?php echo esc_html($tab_labels[$tab_key]); ?>
                    </a>
                <?php endforeach; ?>
            </h2>
            
            <!-- Tab Content -->
            <?php foreach (self::$tab_keys as $tab_key) : ?>
                <div id="tab-<?php echo esc_attr($tab_key); ?>" 
                     class="delcampe-tab-content <?php echo $current_tab === $tab_key ? 'active' : ''; ?>">
                    <?php self::render_tab_content($tab_key); ?>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
    }

    /**
     * Render content for each tab
     * Enhanced in version 1.3.0.0 with shipping tab
     */
    private static function render_tab_content($tab) {
        switch ($tab) {
            case 'general':
                self::render_general_tab();
                break;
            case 'categories':
                self::render_categories_tab();
                break;
            case 'sync':
                self::render_sync_tab();
                break;
            case 'shipping':
                self::render_shipping_tab();
                break;
            case 'webhooks':
                self::render_webhooks_tab();
                break;
            case 'advanced':
                self::render_advanced_tab();
                break;
            case 'developer':
                self::render_developer_tab();
                break;
            case 'license':
                self::render_license_tab();
                break;
        }
    }

    /**
     * Render General Settings tab
     * Enhanced in version 1.1.0.0 with better descriptions
     */
    private static function render_general_tab() {
        // Use enhanced settings template (v1.10.13.0)
        $template_file = DWC_PLUGIN_DIR . 'admin/templates/settings-general-enhanced.php';
        
        // Fall back to basic template if enhanced doesn't exist
        if (!file_exists($template_file)) {
            ?>
            <form method="post" action="options.php">
                <?php settings_fields('delcampe_general_settings'); ?>
                
                <div class="delcampe-section">
                    <h3><?php _e('API Configuration', 'wc-delcampe-integration'); ?></h3>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="delcampe_api_key"><?php _e('Delcampe API Key', 'wc-delcampe-integration'); ?></label>
                            </th>
                            <td>
                                <input type="text" id="delcampe_api_key" name="delcampe_api_key" 
                                       value="<?php echo esc_attr(get_option('delcampe_api_key', '')); ?>" 
                                       class="regular-text" />
                                <p class="description">
                                    <?php _e('Enter your Delcampe API key. This is required for authentication.', 'wc-delcampe-integration'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <?php submit_button(); ?>
            </form>
            <?php
        } else {
            // Include the enhanced template
            include $template_file;
        }
    }

    /**
     * Render Categories tab
     * Enhanced in version 1.1.0.0 with improved layout
     */
    private static function render_categories_tab() {
        // Include the category management functionality
        require_once plugin_dir_path(__FILE__) . 'class-admin-page.php';
        
        echo '<div class="delcampe-section">';
        echo '<h3>' . __('Category Management', 'wc-delcampe-integration') . '</h3>';
        echo '<p>' . __('Import and manage Delcampe categories for your products.', 'wc-delcampe-integration') . '</p>';
        echo '<p class="description">' . sprintf(__('Version %s - Enhanced category navigation', 'wc-delcampe-integration'), DELCAMPE_PLUGIN_VERSION) . '</p>';
        
        // Show category navigation interface
        Delcampe_Admin_Page::render_category_mapping_page();
        
        echo '</div>';
        
        // Show saved categories section
        echo '<div class="delcampe-section">';
        echo '<h3>' . __('Saved Categories', 'wc-delcampe-integration') . '</h3>';
        
        Delcampe_Admin_Page::render_saved_categories_page();
        
        echo '</div>';
    }

    /**
     * Render Sync Management tab
     * Added in version 1.2.0.0
     */
    private static function render_sync_tab() {
        // Get sync handler instance
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-sync-handler.php';
        $sync_handler = Delcampe_Sync_Handler::get_instance();
        ?>
        
        <div class="delcampe-section">
            <h3><?php _e('Sync Queue Status', 'wc-delcampe-integration'); ?></h3>
            
            <div id="queue-status">
                <?php _e('Loading queue status...', 'wc-delcampe-integration'); ?>
            </div>
            
            <p style="margin-top: 20px;">
                <button type="button" id="manual-sync-btn" class="button button-primary">
                    <?php _e('Process Sync Queue Now', 'wc-delcampe-integration'); ?>
                </button>
                <button type="button" id="clear-queue-btn" class="button">
                    <?php _e('Clear Queue', 'wc-delcampe-integration'); ?>
                </button>
                <button type="button" id="clear-lock-btn" class="button">
                    <?php _e('Clear Sync Lock', 'wc-delcampe-integration'); ?>
                </button>
                <span id="sync-status-message" style="margin-left: 10px;"></span>
            </p>

            <div id="sync-result" style="margin-top: 20px;"></div>
        </div>

        <!-- Requeue Stuck Listings -->
            <div class="delcampe-section">
                <h3><?php _e('Recovery Tools', 'wc-delcampe-integration'); ?></h3>
                <p class="description"><?php _e('If some listings are stuck in Processing without a Delcampe ID, requeue them here.', 'wc-delcampe-integration'); ?></p>
                <label for="requeue-minutes-sync"><strong><?php _e('Older than (minutes):', 'wc-delcampe-integration'); ?></strong></label>
                <input type="number" id="requeue-minutes-sync" value="10" min="0" style="width: 80px;" />
                <button type="button" id="requeue-stuck-btn-sync" class="button">
                    <?php _e('Requeue Stuck Listings', 'wc-delcampe-integration'); ?>
                </button>
                <span id="requeue-status-message-sync" style="margin-left: 10px;"></span>
                <div style="margin-top:10px;">
                    <button type="button" id="reconcile-open-btn" class="button">
                        <?php _e('Reconcile with Delcampe (Open Items)', 'wc-delcampe-integration'); ?>
                    </button>
                    <span id="reconcile-status-message" style="margin-left:10px;"></span>
                </div>
                <div style="margin-top:10px;">
                    <button type="button" id="rebuild-state-dry-btn" class="button">
                        <?php _e('Rebuild Local State (Dry Run)', 'wc-delcampe-integration'); ?>
                    </button>
                    <button type="button" id="rebuild-state-btn" class="button button-primary">
                        <?php _e('Rebuild Local State (Apply)', 'wc-delcampe-integration'); ?>
                    </button>
                    <span id="rebuild-status-message" style="margin-left:10px;"></span>
                </div>
            </div>

        <div class="delcampe-section">
            <?php
            // Get recent products with sync status with pagination
            global $wpdb;
            $listings_table = $wpdb->prefix . 'delcampe_listings';
            
            // Pagination setup
            $per_page_options = array(20, 50, 100);
            $per_page = isset($_GET['per_page']) && in_array(intval($_GET['per_page']), $per_page_options) 
                        ? intval($_GET['per_page']) 
                        : 20;
            $current_page = isset($_GET['sync_page']) ? max(1, intval($_GET['sync_page'])) : 1;
            $offset = ($current_page - 1) * $per_page;
            
            // Get total count first
            $total_count = $wpdb->get_var(
                "SELECT COUNT(DISTINCT p.ID)
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_delcampe_sync_status'
                LEFT JOIN {$listings_table} l ON p.ID = l.product_id
                WHERE p.post_type = 'product' 
                AND (pm1.meta_value IS NOT NULL OR l.id IS NOT NULL)"
            );
            ?>
            
            <h3>
                <?php _e('Recent Sync Activity', 'wc-delcampe-integration'); ?>
                <span id="sync-auto-status" class="sync-auto-status" style="display:none;">
                    <span class="dashicons dashicons-update spin"></span> Auto-checking active
                </span>
                <?php if ($total_count > 20) : ?>
                <span style="float: right; font-size: 13px; font-weight: normal;">
                    <label for="per-page-select"><?php _e('Show:', 'wc-delcampe-integration'); ?></label>
                    <select id="per-page-select" onchange="window.location.href='<?php echo admin_url('admin.php?page=delcampe-settings&tab=sync'); ?>&per_page=' + this.value">
                        <?php foreach ($per_page_options as $option) : ?>
                            <option value="<?php echo $option; ?>" <?php selected($per_page, $option); ?>>
                                <?php echo $option; ?> <?php _e('items', 'wc-delcampe-integration'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </span>
                <?php endif; ?>
            </h3>
            
            <?php
            // Get paginated results
            $recent_syncs = $wpdb->get_results($wpdb->prepare(
                "SELECT p.ID, p.post_title, 
                        pm1.meta_value as sync_status, 
                        pm2.meta_value as last_error,
                        l.delcampe_id,
                        l.personal_reference,
                        l.status as listing_status,
                        l.date_published,
                        l.date_updated
                FROM {$wpdb->posts} p
                LEFT JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_delcampe_sync_status'
                LEFT JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_delcampe_last_error'
                LEFT JOIN {$listings_table} l ON p.ID = l.product_id
                WHERE p.post_type = 'product' 
                AND (pm1.meta_value IS NOT NULL OR l.id IS NOT NULL)
                ORDER BY COALESCE(l.date_updated, p.post_modified) DESC
                LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ));
            
            $total_pages = ceil($total_count / $per_page);
            
            if ($recent_syncs) {
                ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Product', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Status', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Delcampe Status', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Last Error', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Actions', 'wc-delcampe-integration'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_syncs as $sync) : ?>
                            <tr>
                                <td>
                                    <a href="<?php echo get_edit_post_link($sync->ID); ?>">
                                        <?php echo esc_html($sync->post_title); ?>
                                    </a>
                                    <?php if ($sync->personal_reference) : ?>
                                        <br><small>Ref: <?php echo esc_html($sync->personal_reference); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php 
                                    // Use listing status if available, otherwise use sync status
                                    $display_status = $sync->listing_status ?: $sync->sync_status;
                                    $status_class = $display_status;
                                    
                                    // Map listing statuses to indicator classes
                                    if ($display_status === 'ended' || $display_status === 'deleted') {
                                        $status_class = 'ended';
                                    } elseif ($display_status === 'published' || $display_status === 'active') {
                                        $status_class = 'active';
                                    } elseif ($display_status === 'error') {
                                        $status_class = 'error';
                                    }
                                    ?>
                                    <span class="sync-status-indicator <?php echo esc_attr($status_class); ?>"></span>
                                    <?php echo esc_html(ucfirst($display_status)); ?>
                                </td>
                                <td>
                                    <?php if ($sync->delcampe_id) : ?>
                                        <?php if ($sync->listing_status === 'ended') : ?>
                                            <span style="color: #999;">⏹ Ended</span>
                                        <?php elseif ($sync->listing_status === 'deleted') : ?>
                                            <span style="color: #666;">✗ Deleted</span>
                                        <?php elseif ($sync->listing_status === 'sold') : ?>
                                            <span style="color: #0073aa;">✓ Sold</span>
                                        <?php else : ?>
                                            <span style="color: #46b450;">✓ Live</span>
                                        <?php endif; ?>
                                        <br><small>ID: <?php echo esc_html($sync->delcampe_id); ?></small>
                                    <?php elseif ($sync->listing_status === 'published') : ?>
                                        <span style="color: #f0ad4e;">⏳ Processing</span>
                                        <br><small>Submitted to Delcampe</small>
                                    <?php elseif ($sync->sync_status === 'error') : ?>
                                        <span style="color: #dc3232;">✗ Error</span>
                                    <?php else : ?>
                                        <span style="color: #999;">—</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php echo $sync->last_error ? esc_html($sync->last_error) : '-'; ?>
                                </td>
                                <td>
                                    <a href="<?php echo get_edit_post_link($sync->ID); ?>" class="button button-small">
                                        <?php _e('Edit', 'wc-delcampe-integration'); ?>
                                    </a>
                                    <?php if ($sync->listing_status === 'published' && !$sync->delcampe_id && $sync->personal_reference) : ?>
                                        <button class="button button-small check-delcampe-status" 
                                                data-product-id="<?php echo esc_attr($sync->ID); ?>"
                                                data-reference="<?php echo esc_attr($sync->personal_reference); ?>">
                                            <?php _e('Check Status', 'wc-delcampe-integration'); ?>
                                        </button>
                                    <?php endif; ?>
                                    <?php if ($sync->delcampe_id) : ?>
                                        <a href="https://www.delcampe.net/en_US/collectibles/item/<?php echo esc_attr($sync->delcampe_id); ?>.html" 
                                           target="_blank" 
                                           class="button button-small">
                                            <?php _e('View', 'wc-delcampe-integration'); ?>
                                        </a>
                                        <?php if ($sync->listing_status === 'published' || $sync->listing_status === 'active' || $sync->sync_status === 'active') : ?>
                                            <button class="button button-small button-link-delete close-delcampe-listing" 
                                                    data-product-id="<?php echo esc_attr($sync->ID); ?>"
                                                    data-listing-id="<?php echo esc_attr($sync->delcampe_id); ?>"
                                                    style="color: #a00;">
                                                <?php _e('Close', 'wc-delcampe-integration'); ?>
                                            </button>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <?php
                // Add pagination controls if there's more than one page
                if ($total_pages > 1) {
                    $base_url = admin_url('admin.php?page=delcampe-settings&tab=sync');
                    if ($per_page != 20) {
                        $base_url .= '&per_page=' . $per_page;
                    }
                    ?>
                    <div class="tablenav bottom">
                        <div class="tablenav-pages">
                            <span class="displaying-num"><?php printf(__('%d items', 'wc-delcampe-integration'), $total_count); ?></span>
                            <span class="pagination-links">
                                <?php if ($current_page > 1) : ?>
                                    <a class="first-page button" href="<?php echo esc_url($base_url); ?>">
                                        <span aria-hidden="true">«</span>
                                    </a>
                                    <a class="prev-page button" href="<?php echo esc_url($base_url . '&sync_page=' . ($current_page - 1)); ?>">
                                        <span aria-hidden="true">‹</span>
                                    </a>
                                <?php else : ?>
                                    <span class="tablenav-pages-navspan button disabled" aria-hidden="true">«</span>
                                    <span class="tablenav-pages-navspan button disabled" aria-hidden="true">‹</span>
                                <?php endif; ?>
                                
                                <span class="paging-input">
                                    <span class="tablenav-paging-text">
                                        <?php printf(__('%d of %d', 'wc-delcampe-integration'), $current_page, $total_pages); ?>
                                    </span>
                                </span>
                                
                                <?php if ($current_page < $total_pages) : ?>
                                    <a class="next-page button" href="<?php echo esc_url($base_url . '&sync_page=' . ($current_page + 1)); ?>">
                                        <span aria-hidden="true">›</span>
                                    </a>
                                    <a class="last-page button" href="<?php echo esc_url($base_url . '&sync_page=' . $total_pages); ?>">
                                        <span aria-hidden="true">»</span>
                                    </a>
                                <?php else : ?>
                                    <span class="tablenav-pages-navspan button disabled" aria-hidden="true">›</span>
                                    <span class="tablenav-pages-navspan button disabled" aria-hidden="true">»</span>
                                <?php endif; ?>
                            </span>
                        </div>
                    </div>
                    <?php
                }
                ?>
                <?php
            } else {
                echo '<p>' . __('No sync activity found.', 'wc-delcampe-integration') . '</p>';
            }
            ?>
        </div>
        
        <div class="delcampe-section">
            <h3><?php _e('Sync Settings', 'wc-delcampe-integration'); ?></h3>
            
            <p><?php _e('The sync queue processes automatically every 5 minutes via WordPress cron.', 'wc-delcampe-integration'); ?></p>
            <p><?php _e('Products are added to the queue when you assign a Delcampe profile in the product edit screen.', 'wc-delcampe-integration'); ?></p>
            
            <h4><?php _e('Batch Processing', 'wc-delcampe-integration'); ?></h4>
            <ul>
                <li><?php _e('Batch size: 5 products per run', 'wc-delcampe-integration'); ?></li>
                <li><?php _e('Maximum retry attempts: 3', 'wc-delcampe-integration'); ?></li>
                <li><?php _e('Process lock timeout: 5 minutes', 'wc-delcampe-integration'); ?></li>
            </ul>
        </div>
        <?php
    }

    /**
     * Render Queue tab by embedding Queue Monitor UI.
     */
    private static function render_queue_tab() {
        // Enqueue queue assets
        wp_enqueue_style('delcampe-queue-admin', DWC_PLUGIN_URL . 'assets/css/queue-admin.css', array(), DWC_VERSION);
        wp_enqueue_script('delcampe-queue-admin', DWC_PLUGIN_URL . 'assets/js/queue-admin.js', array('jquery'), DWC_VERSION, true);
        echo '<div class="delcampe-section">';
        echo '<h3>' . esc_html__('Queue Monitor', 'wc-delcampe-integration') . '</h3>';
        if (class_exists('Delcampe_Queue_Admin')) {
            Delcampe_Queue_Admin::get_instance()->render_embedded();
        } else {
            echo '<p>' . esc_html__('Queue system not available.', 'wc-delcampe-integration') . '</p>';
        }
        echo '</div>';
    }

    /**
     * Render Shipping tab
     * Added in version 1.3.0.0
     */
    private static function render_shipping_tab() {
        // Include and initialize shipping admin
        require_once plugin_dir_path(__FILE__) . 'class-delcampe-shipping-admin.php';
        $shipping_admin = Delcampe_Shipping_Admin::get_instance();
        
        // Render the shipping tab content
        $shipping_admin->render_shipping_tab();
    }

    /**
     * Render Webhooks tab
     * Added in version 1.10.14.2
     */
    private static function render_webhooks_tab() {
        // Include webhook registration class if not already loaded
        if (!class_exists('Delcampe_Webhook_Registration')) {
            require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-webhook-registration.php';
        }
        
        // Get instance and render the settings page content
        $webhook_registration = Delcampe_Webhook_Registration::get_instance();
        
        // The render_settings_page method outputs the webhook interface
        $webhook_registration->render_settings_page();
    }

    /**
     * Render Advanced tab
     * Enhanced in version 1.10.10.4 with comprehensive sync settings
     */
    private static function render_advanced_tab() {
        // v1.10.35.16: Split sync settings for orders vs listings/inventory
        // Get current settings
        $sync_interval = get_option('delcampe_sync_interval', 'hourly'); // Legacy setting - kept for compatibility
        
        // Separate sync settings for orders and listings/inventory
        $orders_sync_enabled = get_option('delcampe_orders_sync_enabled', 'yes'); // Default: enabled
        $orders_sync_interval = get_option('delcampe_orders_sync_interval', 'every_15_minutes');
        
        $listings_sync_enabled = get_option('delcampe_listings_sync_enabled', 'no'); // Default: disabled
        $listings_sync_interval = get_option('delcampe_listings_sync_interval', 'hourly');
        
        $sync_prices = get_option('delcampe_sync_prices', 'no');
        $order_sync_days = get_option('delcampe_order_sync_days', '7');
        $batch_size = get_option('delcampe_batch_size', '10');
        
        // Get last sync status
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-master-sync.php';
        $sync_status = Delcampe_Master_Sync::get_sync_status();
        ?>
        <form method="post" action="options.php">
            <?php settings_fields('delcampe_advanced_settings'); ?>
            
            <!-- Sync Status Section -->
            <div class="delcampe-section">
                <h3><?php _e('Sync Status', 'wc-delcampe-integration'); ?></h3>
                <?php if (!empty($sync_status['last_sync'])) : ?>
                    <div class="sync-status-info" style="background: #f0f0f1; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                        <p><strong><?php _e('Last Sync:', 'wc-delcampe-integration'); ?></strong> 
                            <?php echo esc_html($sync_status['last_sync']['end_time'] ?? 'Never'); ?>
                            <?php if ($sync_status['last_sync']['status'] === 'completed') : ?>
                                <span style="color: #46b450;">✓ <?php _e('Completed', 'wc-delcampe-integration'); ?></span>
                            <?php elseif ($sync_status['last_sync']['status'] === 'error') : ?>
                                <span style="color: #dc3232;">✗ <?php _e('Error', 'wc-delcampe-integration'); ?></span>
                            <?php endif; ?>
                        </p>
                        <?php if (!empty($sync_status['last_sync']['orders'])) : ?>
                            <p style="margin: 5px 0;">
                                <strong><?php _e('Orders:', 'wc-delcampe-integration'); ?></strong>
                                <?php echo sprintf(
                                    __('%d checked, %d imported', 'wc-delcampe-integration'),
                                    $sync_status['last_sync']['orders']['checked'],
                                    $sync_status['last_sync']['orders']['imported']
                                ); ?>
                            </p>
                        <?php endif; ?>
                        <?php if (!empty($sync_status['last_sync']['listings'])) : ?>
                            <p style="margin: 5px 0;">
                                <strong><?php _e('Listings:', 'wc-delcampe-integration'); ?></strong>
                                <?php echo sprintf(
                                    __('%d checked, %d updated', 'wc-delcampe-integration'),
                                    $sync_status['last_sync']['listings']['checked'],
                                    $sync_status['last_sync']['listings']['updated']
                                ); ?>
                            </p>
                        <?php endif; ?>
                        <p><strong><?php _e('Next Sync:', 'wc-delcampe-integration'); ?></strong> 
                            <?php echo esc_html($sync_status['next_sync']); ?>
                        </p>
                    </div>
                <?php endif; ?>
                
                <button type="button" id="run-master-sync-now" class="button button-primary">
                    <?php _e('Run Sync Now', 'wc-delcampe-integration'); ?>
                </button>
                <span id="sync-status-message" style="margin-left: 10px;"></span>

                <div style="margin-top: 10px;">
                    <label for="requeue-minutes"><strong><?php _e('Requeue stuck listings older than (minutes):', 'wc-delcampe-integration'); ?></strong></label>
                    <input type="number" id="requeue-minutes" value="10" min="1" style="width: 80px;" />
                    <button type="button" id="requeue-stuck-btn" class="button">
                        <?php _e('Requeue Stuck Listings', 'wc-delcampe-integration'); ?>
                    </button>
                    <span id="requeue-status-message" style="margin-left: 10px;"></span>
                </div>
            </div>
            
            <!-- Sync Settings Section -->
            <div class="delcampe-section">
                <h3><?php _e('Synchronization Settings', 'wc-delcampe-integration'); ?></h3>
                <p><?php _e('Configure how often and what data to synchronize with Delcampe.', 'wc-delcampe-integration'); ?></p>
                
                <table class="form-table">
                    <!-- Order Sync Settings -->
                    <tr>
                        <th scope="row">
                            <label for="delcampe_orders_sync_enabled"><?php _e('Enable Order Sync', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="delcampe_orders_sync_enabled" name="delcampe_orders_sync_enabled">
                                <option value="yes" <?php selected($orders_sync_enabled, 'yes'); ?>><?php _e('Yes', 'wc-delcampe-integration'); ?></option>
                                <option value="no" <?php selected($orders_sync_enabled, 'no'); ?>><?php _e('No', 'wc-delcampe-integration'); ?></option>
                            </select>
                            <p class="description">
                                <?php _e('Enable automatic order synchronization from Delcampe (Default: Yes)', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="delcampe_orders_sync_interval"><?php _e('Order Sync Interval', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="delcampe_orders_sync_interval" name="delcampe_orders_sync_interval">
                                <option value="every_15_minutes" <?php selected($orders_sync_interval, 'every_15_minutes'); ?>><?php _e('Every 15 Minutes', 'wc-delcampe-integration'); ?></option>
                                <option value="every_30_minutes" <?php selected($orders_sync_interval, 'every_30_minutes'); ?>><?php _e('Every 30 Minutes', 'wc-delcampe-integration'); ?></option>
                                <option value="hourly" <?php selected($orders_sync_interval, 'hourly'); ?>><?php _e('Hourly', 'wc-delcampe-integration'); ?></option>
                                <option value="twicedaily" <?php selected($orders_sync_interval, 'twicedaily'); ?>><?php _e('Twice Daily', 'wc-delcampe-integration'); ?></option>
                                <option value="daily" <?php selected($orders_sync_interval, 'daily'); ?>><?php _e('Daily', 'wc-delcampe-integration'); ?></option>
                            </select>
                            <p class="description">
                                <?php _e('How often to import orders from Delcampe', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    
                    <!-- Listings/Inventory Sync Settings -->
                    <tr>
                        <th scope="row">
                            <label for="delcampe_listings_sync_enabled"><?php _e('Enable Listings/Inventory Sync', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="delcampe_listings_sync_enabled" name="delcampe_listings_sync_enabled">
                                <option value="yes" <?php selected($listings_sync_enabled, 'yes'); ?>><?php _e('Yes', 'wc-delcampe-integration'); ?></option>
                                <option value="no" <?php selected($listings_sync_enabled, 'no'); ?>><?php _e('No', 'wc-delcampe-integration'); ?></option>
                            </select>
                            <p class="description">
                                <?php _e('Enable automatic listings and inventory synchronization (Default: No - reduces API calls)', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="delcampe_listings_sync_interval"><?php _e('Listings Sync Interval', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="delcampe_listings_sync_interval" name="delcampe_listings_sync_interval">
                                <option value="every_15_minutes" <?php selected($listings_sync_interval, 'every_15_minutes'); ?>><?php _e('Every 15 Minutes', 'wc-delcampe-integration'); ?></option>
                                <option value="every_30_minutes" <?php selected($listings_sync_interval, 'every_30_minutes'); ?>><?php _e('Every 30 Minutes', 'wc-delcampe-integration'); ?></option>
                                <option value="hourly" <?php selected($listings_sync_interval, 'hourly'); ?>><?php _e('Hourly', 'wc-delcampe-integration'); ?></option>
                                <option value="twicedaily" <?php selected($listings_sync_interval, 'twicedaily'); ?>><?php _e('Twice Daily', 'wc-delcampe-integration'); ?></option>
                                <option value="daily" <?php selected($listings_sync_interval, 'daily'); ?>><?php _e('Daily', 'wc-delcampe-integration'); ?></option>
                            </select>
                            <p class="description">
                                <?php _e('How often to sync listings and inventory with Delcampe (when enabled)', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="delcampe_sync_prices"><?php _e('Sync Price Changes', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="delcampe_sync_prices" name="delcampe_sync_prices">
                                <option value="yes" <?php selected($sync_prices, 'yes'); ?>><?php _e('Yes', 'wc-delcampe-integration'); ?></option>
                                <option value="no" <?php selected($sync_prices, 'no'); ?>><?php _e('No', 'wc-delcampe-integration'); ?></option>
                            </select>
                            <p class="description">
                                <?php _e('Automatically update prices on Delcampe when WooCommerce prices change', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="delcampe_order_sync_days"><?php _e('Order Sync Period', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="delcampe_order_sync_days" name="delcampe_order_sync_days" 
                                   value="<?php echo esc_attr($order_sync_days); ?>" 
                                   min="1" max="30" />
                            <span><?php _e('days', 'wc-delcampe-integration'); ?></span>
                            <p class="description">
                                <?php _e('Import orders from the last X days', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="delcampe_batch_size"><?php _e('Batch Size', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="delcampe_batch_size" name="delcampe_batch_size" 
                                   value="<?php echo esc_attr($batch_size); ?>" 
                                   min="0" />
                            <p class="description">
                                <?php _e('Number of items to process per batch (0 = process entire queue)', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- What Gets Synced Section -->
            <div class="delcampe-section">
                <h3><?php _e('What Gets Synchronized', 'wc-delcampe-integration'); ?></h3>
                <div style="background: #fff; border: 1px solid #ccc; padding: 15px; border-radius: 5px;">
                    <h4><?php _e('From Delcampe to WooCommerce:', 'wc-delcampe-integration'); ?></h4>
                    <ul style="list-style: disc; margin-left: 20px;">
                        <li><?php _e('✓ New orders (sold items)', 'wc-delcampe-integration'); ?></li>
                        <li><?php _e('✓ Closed/ended listings', 'wc-delcampe-integration'); ?></li>
                        <li><?php _e('✓ Listing status updates', 'wc-delcampe-integration'); ?></li>
                        <li><?php _e('✓ Inventory changes', 'wc-delcampe-integration'); ?></li>
                    </ul>
                    
                    <h4><?php _e('From WooCommerce to Delcampe:', 'wc-delcampe-integration'); ?></h4>
                    <ul style="list-style: disc; margin-left: 20px;">
                        <li><?php _e('✓ Stock level updates', 'wc-delcampe-integration'); ?></li>
                        <li><?php _e('✓ Price changes (if enabled)', 'wc-delcampe-integration'); ?></li>
                        <li><?php _e('✓ Product availability', 'wc-delcampe-integration'); ?></li>
                    </ul>
                </div>
            </div>
            
            <?php submit_button(); ?>
        </form>
        
        <script>
        jQuery(document).ready(function($) {
            // Run sync now button
            $('#run-master-sync-now').on('click', function() {
                var $btn = $(this);
                var $msg = $('#sync-status-message');
                
                $btn.prop('disabled', true).text('<?php _e('Running...', 'wc-delcampe-integration'); ?>');
                $msg.html('<span style="color: #f0ad4e;">⏳ <?php _e('Sync in progress...', 'wc-delcampe-integration'); ?></span>');
                
                $.post(ajaxurl, {
                    action: 'delcampe_run_master_sync',
                    nonce: '<?php echo wp_create_nonce('delcampe_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        $msg.html('<span style="color: #46b450;">✓ <?php _e('Sync completed successfully!', 'wc-delcampe-integration'); ?></span>');
                        // Reload page after 2 seconds to show updated status
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $msg.html('<span style="color: #dc3232;">✗ ' + (response.data || '<?php _e('Sync failed', 'wc-delcampe-integration'); ?>') + '</span>');
                    }
                    $btn.prop('disabled', false).text('<?php _e('Run Sync Now', 'wc-delcampe-integration'); ?>');
                }).fail(function() {
                    $msg.html('<span style="color: #dc3232;">✗ <?php _e('Request failed', 'wc-delcampe-integration'); ?></span>');
                    $btn.prop('disabled', false).text('<?php _e('Run Sync Now', 'wc-delcampe-integration'); ?>');
                });

                // Clear sync lock
                $("#clear-lock-btn").on("click", function() {
                    var $btn = $(this);
                    var originalText = $btn.text();
                    $btn.prop("disabled", true).text("Clearing...");
                    $.post(ajaxurl, {
                        action: "delcampe_clear_sync_lock",
                        nonce: "' . wp_create_nonce('delcampe_admin_nonce') . '"
                    }, function(response) {
                        $btn.prop("disabled", false).text(originalText);
                        if (response && response.success) {
                            $("#sync-result").html(
                                "<div class=\"notice notice-success\"><p>" + response.data.message + "</p></div>"
                            );
                            loadQueueStatus();
                        }
                    });
                });
            });

            // Requeue stuck listings
            $('#requeue-stuck-btn').on('click', function() {
                var $btn = $(this);
                var $msg = $('#requeue-status-message');
                var minutes = parseInt($('#requeue-minutes').val() || '10', 10);
                $btn.prop('disabled', true).text('<?php _e('Requeuing...', 'wc-delcampe-integration'); ?>');
                $msg.html('<span style="color: #f0ad4e;">⏳ <?php _e('Working...', 'wc-delcampe-integration'); ?></span>');
                $.post(ajaxurl, {
                    action: 'delcampe_requeue_stuck_listings',
                    minutes: minutes,
                    nonce: '<?php echo wp_create_nonce('delcampe_admin_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        $msg.html('<span style="color: #46b450;">✓ ' + (response.data.message || '<?php _e('Requeued and scheduled.', 'wc-delcampe-integration'); ?>') + '</span>');
                    } else {
                        $msg.html('<span style="color: #dc3232;">✗ ' + (response.data && response.data.message ? response.data.message : '<?php _e('Requeue failed', 'wc-delcampe-integration'); ?>') + '</span>');
                    }
                    $btn.prop('disabled', false).text('<?php _e('Requeue Stuck Listings', 'wc-delcampe-integration'); ?>');
                }).fail(function() {
                    $msg.html('<span style="color: #dc3232;">✗ <?php _e('Request failed', 'wc-delcampe-integration'); ?></span>');
                    $btn.prop('disabled', false).text('<?php _e('Requeue Stuck Listings', 'wc-delcampe-integration'); ?>');
                });
            });

            // Requeue stuck listings (Sync tab)
            $('#requeue-stuck-btn-sync').on('click', function() {
                var $btn = $(this);
                var $msg = $('#requeue-status-message-sync');
                var minutes = parseInt($('#requeue-minutes-sync').val() || '10', 10);
                $btn.prop('disabled', true).text('<?php _e('Requeuing...', 'wc-delcampe-integration'); ?>');
                $msg.html('<span style="color: #f0ad4e;">⏳ <?php _e('Working...', 'wc-delcampe-integration'); ?></span>');
                $.post(ajaxurl, {
                    action: 'delcampe_requeue_stuck_listings',
                    minutes: minutes,
                    nonce: '<?php echo wp_create_nonce('delcampe_admin_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        $msg.html('<span style="color: #46b450;">✓ ' + (response.data.message || '<?php _e('Requeued and scheduled.', 'wc-delcampe-integration'); ?>') + '</span>');
                    } else {
                        $msg.html('<span style="color: #dc3232;">✗ ' + (response.data && response.data.message ? response.data.message : '<?php _e('Requeue failed', 'wc-delcampe-integration'); ?>') + '</span>');
                    }
                    $btn.prop('disabled', false).text('<?php _e('Requeue Stuck Listings', 'wc-delcampe-integration'); ?>');
                }).fail(function() {
                    $msg.html('<span style="color: #dc3232;">✗ <?php _e('Request failed', 'wc-delcampe-integration'); ?></span>');
                    $btn.prop('disabled', false).text('<?php _e('Requeue Stuck Listings', 'wc-delcampe-integration'); ?>');
                });
            });

            // Reconcile open listings with Delcampe
            $('#reconcile-open-btn').on('click', function() {
                var $btn = $(this);
                var $msg = $('#reconcile-status-message');
                $btn.prop('disabled', true).text('<?php _e('Reconciling...', 'wc-delcampe-integration'); ?>');
                $msg.html('<span style="color: #f0ad4e;">⏳ <?php _e('Working...', 'wc-delcampe-integration'); ?></span>');
                $.post(ajaxurl, {
                    action: 'delcampe_reconcile_open_listings',
                    nonce: '<?php echo wp_create_nonce('delcampe_admin_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        var d = response.data || {};
                        var txt = d.message || ('<?php _e('Reconciled.', 'wc-delcampe-integration'); ?>');
                        $msg.html('<span style="color: #46b450;">✓ ' + txt + '</span>');
                    } else {
                        $msg.html('<span style="color: #dc3232;">✗ ' + (response.data && response.data.message ? response.data.message : '<?php _e('Reconcile failed', 'wc-delcampe-integration'); ?>') + '</span>');
                    }
                    $btn.prop('disabled', false).text('<?php _e('Reconcile with Delcampe (Open Items)', 'wc-delcampe-integration'); ?>');
                }).fail(function() {
                    $msg.html('<span style="color: #dc3232;">✗ <?php _e('Request failed', 'wc-delcampe-integration'); ?></span>');
                    $btn.prop('disabled', false).text('<?php _e('Reconcile with Delcampe (Open Items)', 'wc-delcampe-integration'); ?>');
                });
            });

            function runRebuildState(dry) {
                var $msg = $('#rebuild-status-message');
                var $btnApply = $('#rebuild-state-btn');
                var $btnDry = $('#rebuild-state-dry-btn');
                $btnApply.prop('disabled', true).text('<?php _e('Rebuilding...', 'wc-delcampe-integration'); ?>');
                $btnDry.prop('disabled', true).text('<?php _e('Rebuilding (Dry)...', 'wc-delcampe-integration'); ?>');
                $msg.html('<span style="color: #f0ad4e;">⏳ <?php _e('Working...', 'wc-delcampe-integration'); ?></span>');
                $.post(ajaxurl, {
                    action: 'delcampe_rebuild_local_state',
                    dry_run: !!dry,
                    nonce: '<?php echo wp_create_nonce('delcampe_admin_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        var d = response.data || {};
                        var txt = d.message || ('<?php _e('Rebuild complete.', 'wc-delcampe-integration'); ?>');
                        $msg.html('<span style="color: #46b450;">✓ ' + txt + '</span>');
                    } else {
                        $msg.html('<span style="color: #dc3232;">✗ ' + (response.data && response.data.message ? response.data.message : '<?php _e('Rebuild failed', 'wc-delcampe-integration'); ?>') + '</span>');
                    }
                    $btnApply.prop('disabled', false).text('<?php _e('Rebuild Local State (Apply)', 'wc-delcampe-integration'); ?>');
                    $btnDry.prop('disabled', false).text('<?php _e('Rebuild Local State (Dry Run)', 'wc-delcampe-integration'); ?>');
                }).fail(function() {
                    $msg.html('<span style="color: #dc3232;">✗ <?php _e('Request failed', 'wc-delcampe-integration'); ?></span>');
                    $btnApply.prop('disabled', false).text('<?php _e('Rebuild Local State (Apply)', 'wc-delcampe-integration'); ?>');
                    $btnDry.prop('disabled', false).text('<?php _e('Rebuild Local State (Dry Run)', 'wc-delcampe-integration'); ?>');
                });
            }

            $('#rebuild-state-dry-btn').on('click', function(){ runRebuildState(true); });
            $('#rebuild-state-btn').on('click', function(){ runRebuildState(false); });
        });
        </script>
        <?php
    }

    /**
     * Render Developer tab
     * Enhanced in version 1.1.0.0 with improved test sync display
     */
    private static function render_developer_tab() {
        echo '<div class="delcampe-section">';
        echo '<h3>' . __('Sync Logs', 'wc-delcampe-integration') . '</h3>';
        
        // Include sync status functionality
        require_once plugin_dir_path(__FILE__) . 'class-admin-page.php';
        Delcampe_Admin_Page::render_sync_status_page();
        
        echo '</div>';
        
        // Add test sync section
        echo '<div class="delcampe-section">';
        echo '<h3>' . __('Test Product Sync', 'wc-delcampe-integration') . '</h3>';
        echo '<p>' . __('Test synchronization with a single product.', 'wc-delcampe-integration') . '</p>';
        
        // Check if test sync class exists
        if (class_exists('Delcampe_Test_Sync')) {
            $test_sync = Delcampe_Test_Sync::get_instance();
            $test_sync->render_test_sync_page();
        } else {
            echo '<p>' . __('Test sync functionality not available. Please ensure all plugin files are properly installed.', 'wc-delcampe-integration') . '</p>';
        }
        
        echo '</div>';
    }

    /**
     * Render Updates tab
     * Enhanced in version 1.8.0.0 with complete changelog
     */
    private static function render_updates_tab() {
        ?>
        <div class="delcampe-section">
            <h3><?php _e('Plugin Information', 'wc-delcampe-integration'); ?></h3>
            
            <table class="form-table">
                <tr>
                    <th scope="row"><?php _e('Current Version', 'wc-delcampe-integration'); ?></th>
                    <td><strong><?php echo DELCAMPE_PLUGIN_VERSION; ?></strong></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('Last Updated', 'wc-delcampe-integration'); ?></th>
                    <td><?php echo date('F j, Y'); ?></td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('WooCommerce Required', 'wc-delcampe-integration'); ?></th>
                    <td>3.0 or higher</td>
                </tr>
                <tr>
                    <th scope="row"><?php _e('API Format', 'wc-delcampe-integration'); ?></th>
                    <td>XML (all Delcampe API responses)</td>
                </tr>
            </table>
        </div>
        
        <div class="delcampe-section">
            <h3><?php _e('Changelog', 'wc-delcampe-integration'); ?></h3>
            
            <div class="delcampe-changelog">
                <h4>Version 1.8.0.0 - Production Release</h4>
                <ul>
                    <li><strong>Fixed data isolation issue</strong> - Plugin now requires clean configuration on new installations</li>
                    <li>Added setup notice for first-time configuration</li>
                    <li>Removed all hardcoded data and API keys from plugin files</li>
                    <li>Enhanced security with proper data storage in WordPress database only</li>
                    <li>Added .gitignore to prevent sensitive data inclusion</li>
                    <li>Comprehensive documentation added throughout codebase</li>
                    <li>All user data now stored exclusively in database</li>
                    <li>Updated all version numbers to 1.8.0.0 for production release</li>
                </ul>
                
                <h4>Version 1.6.2.0</h4>
                <ul>
                    <li>Fixed database schema to include personal_reference column</li>
                    <li>Added database migration for existing installations</li>
                    <li>Fixed sync handler to properly handle personal_reference field</li>
                    <li>Updated version numbers across all affected files</li>
                    <li>Added comprehensive comments explaining the fix</li>
                </ul>
                
                <h4>Version 1.6.1.0</h4>
                <ul>
                    <li>Fixed Import Existing Listings button functionality</li>
                    <li>Removed conflicting debug JavaScript from listings admin page</li>
                    <li>Enhanced sync handler with custom debug logging</li>
                    <li>Added comprehensive error handling and JSON response headers</li>
                    <li>Updated all affected component version numbers for consistency</li>
                </ul>
                
                <h4>Version 1.6.0.0</h4>
                <ul>
                    <li>Added comprehensive listings management system</li>
                    <li>Created listings model with full database structure</li>
                    <li>Implemented listings table display similar to wp-lister-ebay</li>
                    <li>Added listings admin interface with bulk actions</li>
                    <li>Created listing statuses (prepared, verified, published, etc.)</li>
                    <li>Added preview and edit functionality for listings</li>
                    <li>Implemented lock/unlock mechanism for listings</li>
                    <li>Added product meta box for Delcampe listings</li>
                </ul>
                
                <h4>Version 1.5.0.0</h4>
                <ul>
                    <li>Added comprehensive inventory management system</li>
                    <li>Implemented real-time stock synchronization between WooCommerce and Delcampe</li>
                    <li>Added automatic stock reduction when Delcampe orders are imported</li>
                    <li>Created inventory monitoring with low stock and out of stock alerts</li>
                    <li>Added inventory check tool to compare stock levels across platforms</li>
                    <li>Implemented bulk inventory sync and update functionality</li>
                    <li>Created dedicated inventory admin interface with statistics</li>
                </ul>
                
                <h4>Version 1.4.0.0</h4>
                <ul>
                    <li>Added comprehensive order management system</li>
                    <li>Implemented order import from Delcampe to WooCommerce</li>
                    <li>Added bi-directional order status synchronization</li>
                    <li>Created order admin interface with tabbed navigation</li>
                    <li>Added automatic order import via cron</li>
                    <li>Implemented customer creation from Delcampe buyers</li>
                    <li>Added order statistics and reporting</li>
                </ul>
                
                <h4>Version 1.3.0.1</h4>
                <ul>
                    <li>Fixed shipping models API parsing</li>
                    <li>Corrected XML response structure parsing</li>
                    <li>Fixed case sensitivity in API endpoint (shippingModels)</li>
                    <li>Updated field mapping (n for name instead of name)</li>
                </ul>
                
                <h4>Version 1.3.0.0</h4>
                <ul>
                    <li>Added Shipping tab for managing shipping models</li>
                    <li>Implemented shipping models manager for fetching and caching Delcampe shipping models</li>
                    <li>Created shipping admin interface for configuration</li>
                    <li>Added mapping between WooCommerce and Delcampe shipping methods</li>
                    <li>Integrated shipping model selection into profiles</li>
                    <li>Added shipping cost calculation functionality</li>
                    <li>Implemented test shipping calculation tool</li>
                </ul>
                
                <h4>Version 1.2.2.1</h4>
                <ul>
                    <li>Added development utilities for dev.fdstamps.com environment</li>
                    <li>Auto-detects development site and disables URL replacement</li>
                    <li>Added development mode indicators in admin</li>
                </ul>
                
                <h4>Version 1.2.2.0</h4>
                <ul>
                    <li>Updated API implementation to match actual Delcampe API</li>
                    <li>Fixed authentication to use POST /seller endpoint</li>
                    <li>Fixed listing creation to use form data instead of XML</li>
                    <li>Now properly creates listings on Delcampe</li>
                </ul>
                
                <h4>Version 1.2.1.0</h4>
                <ul>
                    <li>Added Listing API and Sync Handler modules</li>
                    <li>Implemented actual listing creation functionality</li>
                    <li>Added queue-based sync processing</li>
                    <li>Products can now be synced to Delcampe</li>
                </ul>
                
                <h4>Version 1.2.0.0</h4>
                <ul>
                    <li>Added Sync Management tab for queue monitoring</li>
                    <li>Implemented manual sync trigger functionality</li>
                    <li>Added queue status AJAX endpoints</li>
                    <li>Enhanced sync handler with better error reporting</li>
                    <li>Improved listing API implementation</li>
                </ul>
                
                <h4>Version 1.1.5.0</h4>
                <ul>
                    <li>Completed profiles functionality with working category selection</li>
                    <li>Simplified category selection to use only saved categories</li>
                    <li>Fixed all JavaScript and AJAX issues</li>
                    <li>Profiles system now fully functional</li>
                </ul>
                
                <h4>Version 1.1.2.1</h4>
                <ul>
                    <li>Fixed translation loading issue during plugin activation</li>
                    <li>Removed duplicate activation hook from admin page class</li>
                    <li>Used static tab keys to avoid early translation calls</li>
                </ul>
                
                <h4>Version 1.1.2.0</h4>
                <ul>
                    <li>Enhanced documentation across all files</li>
                    <li>Added new utility methods and error handling</li>
                    <li>Improved translation support throughout</li>
                    <li>Added category deletion and bulk actions</li>
                </ul>
                
                <h4>Version 1.1.1.0</h4>
                <ul>
                    <li>Fixed translation loading timing issue (WordPress 6.7 compatibility)</li>
                    <li>Created centralized version constants file</li>
                    <li>Fixed version display in WordPress plugins list</li>
                    <li>Improved initialization sequence to prevent early translation calls</li>
                </ul>
                
                <h4>Version 1.1.0.0</h4>
                <ul>
                    <li>Added comprehensive code comments throughout all PHP files</li>
                    <li>Enhanced logging with version numbers in log entries</li>
                    <li>Improved error handling and validation</li>
                    <li>Added version display throughout the interface</li>
                    <li>Enhanced code documentation with detailed PHPDoc blocks</li>
                    <li>Improved XML parsing with better error handling</li>
                    <li>Updated User-Agent headers to reflect new version</li>
                </ul>
                
                <h4>Version 1.0.9.0</h4>
                <ul>
                    <li>Restructured admin interface with professional tabbed layout</li>
                    <li>Moved category management to Categories tab under Settings</li>
                    <li>Added Developer tab with logs and test sync</li>
                    <li>Improved user experience and navigation</li>
                    <li>Fixed database table creation issues</li>
                </ul>
                
                <h4>Version 1.0.8.0</h4>
                <ul>
                    <li>Added comprehensive code documentation</li>
                    <li>Enhanced category branch import functionality</li>
                    <li>Improved error handling and logging</li>
                    <li>Fixed category navigation issues</li>
                </ul>
                
                <h4>Version 1.0.7.7</h4>
                <ul>
                    <li>Initial stable release</li>
                    <li>Basic authentication and category management</li>
                    <li>Test sync functionality</li>
                </ul>
            </div>
        </div>
        <?php
    }

    /**
     * Handle category save (moved from admin page)
     * Enhanced in version 1.1.0.0 with better delegation
     */
    public static function handle_category_save() {
        require_once plugin_dir_path(__FILE__) . 'class-admin-page.php';
        Delcampe_Admin_Page::handle_category_save();
    }

    /**
     * Handle branch import (moved from admin page)
     * Enhanced in version 1.1.0.0 with better delegation
     */
    public static function handle_branch_import() {
        require_once plugin_dir_path(__FILE__) . 'class-admin-page.php';
        Delcampe_Admin_Page::handle_branch_import();
    }

    /**
     * Render License tab
     * Added in version 1.10.5.5
     */
    private static function render_license_tab() {
        // Get license manager instance
        $license_manager = Delcampe_License_Manager::get_instance();
        
        // Set up variables needed by the license page view
        $license = $license_manager->get_license_data();
        $is_valid = $license_manager->is_license_valid();
        
        // Include the license page view (fixed version to avoid early translation loading)
        $license_page = plugin_dir_path( dirname( __FILE__ ) ) . 'includes/license/views/license-page-fixed.php';
        if ( file_exists( $license_page ) ) {
            require_once $license_page;
        } else {
            ?>
            <div class="delcampe-section">
                <h3><?php _e('License Management', 'wc-delcampe-integration'); ?></h3>
                <p><?php _e('License page not found. Please reinstall the plugin.', 'wc-delcampe-integration'); ?></p>
            </div>
            <?php
        }
    }

    /**
     * AJAX subcategory counter (simplified)
     * Enhanced in version 1.1.0.0 with version info
     */
    public static function ajax_count_subcategories() {
        require_once plugin_dir_path(__FILE__) . 'class-admin-page.php';
        Delcampe_Admin_Page::ajax_count_subcategories();
    }
}

// DO NOT initialize here - it will be initialized from main.php after translations are loaded
// Delcampe_Admin_Tabbed_Page::init();
