<?php
/**
 * Admin Menu Loader for Delcampe Plugin
 * 
 * Registers top-level and sub-menu pages for the plugin
 * Uses a tabbed interface for better organization
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  Admin
 * @since       1.0.0
 * @version     1.8.0.0
 * 
 * @author      Frank Kahle
 * @copyright   2024 Frank Kahle
 * @license     Proprietary
 * 
 * Changelog:
 * 1.8.0.0 - Production release with comprehensive documentation
 *         - Added detailed DocBlock comments throughout
 *         - Enhanced security with proper capability checks
 *         - Improved error handling and logging
 *         - Standardized code formatting per WordPress standards
 *         - Added inline documentation for complex logic
 *         - Enhanced initialization sequence for reliability
 * 1.6.0.2 - Updated version number for consistency with import listings fix
 * 1.6.0.1 - Fixed listings admin initialization and display_page method call
 * 1.6.0.0 - Added Listings management submenu
 * 1.5.0.0 - Added Inventory management submenu
 * 1.4.0.0 - Added Orders management submenu
 * 1.2.0.0 - Updated for product integration feature
 * 1.1.4.1 - Fixed profiles admin AJAX handlers by ensuring early instantiation
 * 1.1.3.0 - Added Profiles management page
 * 1.1.2.1 - Fixed translation loading by removing translations from menu registration
 * 1.1.2.0 - Enhanced documentation and improved error handling
 * 1.1.1.0 - Fixed translation loading timing
 * 1.1.0.0 - Continued tabbed interface from 1.0.9.0
 * 1.0.9.0 - Introduced simplified menu with tabbed interface
 * 1.0.0.0 - Initial release with multi-submenu structure
 */

// Exit if accessed directly to prevent direct file access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Delcampe_Admin_Menu
 * 
 * Handles the creation and management of admin menu items for the Delcampe integration.
 * Uses singleton pattern to ensure only one instance manages the admin menu.
 * This class is responsible for:
 * - Registering the main plugin menu
 * - Creating submenu pages for different features
 * - Initializing admin interfaces for AJAX handling
 * - Rendering main dashboard and delegating to feature-specific admin classes
 * 
 * @since   1.0.0
 * @version 1.8.0.0
 */
class Delcampe_Admin_Menu {

    /**
     * Singleton instance of the class
     * Ensures only one menu structure is created
     * 
     * @var Delcampe_Admin_Menu|null
     * @since 1.0.0
     */
    private static $instance = null;

    /**
     * Profiles admin instance
     * Manages profile-related functionality
     * 
     * @var Delcampe_Profiles_Admin|null
     * @since 1.1.4.1
     */
    private $profiles_admin = null;

    /**
     * Orders admin instance
     * Manages order synchronization interface
     * 
     * @var Delcampe_Orders_Admin|null
     * @since 1.4.0.0
     */
    private $orders_admin = null;

    /**
     * Inventory admin instance
     * Manages inventory synchronization interface
     * 
     * @var Delcampe_Inventory_Admin|null
     * @since 1.5.0.0
     */
    private $inventory_admin = null;

    /**
     * Listings admin instance
     * Manages listings interface similar to wp-lister
     * 
     * @var Delcampe_Listings_Admin|null
     * @since 1.6.0.1
     */
    private $listings_admin = null;

    /**
     * Get singleton instance
     * 
     * Ensures only one instance of the admin menu manager exists.
     * This prevents duplicate menu registrations and maintains consistency.
     * Uses lazy loading to create instance only when needed.
     * 
     * @since  1.0.0
     * @version 1.8.0.0
     * @return Delcampe_Admin_Menu The singleton instance
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor to enforce singleton pattern
     * 
     * Loads required dependencies and hooks into WordPress admin menu system.
     * Initializes admin interfaces early to ensure AJAX handlers are registered.
     * Uses admin_init for module initialization to avoid timing issues.
     * 
     * @since  1.0.0
     * @version 1.8.0.0
     */
    private function __construct() {
        // Load tabbed admin page class for settings interface
        require_once plugin_dir_path( __FILE__ ) . 'class-admin-tabbed-page.php';
        
        // Load admin page renderer class (for backward compatibility and table creation)
        require_once plugin_dir_path( __FILE__ ) . 'class-admin-page.php';
        
        // Load authentication class for API token management
        require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/class-delcampe-auth.php';

        // Hook into WordPress admin menu system
        // Priority 10 is default, ensures proper menu ordering
        add_action( 'admin_menu', array( $this, 'register_menu' ) );

        // Initialize all admin modules early for AJAX handler registration
        // This ensures AJAX endpoints are available on all admin requests
        add_action( 'admin_init', array( $this, 'init_profiles_admin' ) );
        add_action( 'admin_init', array( $this, 'init_orders_admin' ) );
        add_action( 'admin_init', array( $this, 'init_inventory_admin' ) );
        add_action( 'admin_init', array( $this, 'init_listings_admin' ) );
    }

    /**
     * Initialize profiles admin
     * 
     * Ensures the profiles admin class is instantiated early enough
     * for AJAX handlers to be properly registered. This must happen
     * on every admin request, not just when viewing the profiles page.
     * 
     * @since  1.1.4.1
     * @version 1.8.0.0
     * @return void
     */
    public function init_profiles_admin() {
        // Only initialize if in admin area and not already initialized
        if ( is_admin() && $this->profiles_admin === null ) {
            // Load profiles admin class
            require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/profiles/class-delcampe-profiles-admin.php';
            
            // Create instance - this will register AJAX handlers in the constructor
            $this->profiles_admin = new Delcampe_Profiles_Admin();
            
            // Log initialization for debugging if debug mode is enabled
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                delcampe_log( '[Delcampe Admin Menu v1.8.0.0] Profiles admin initialized for AJAX handlers', 'debug' );
            }
        }
    }

    /**
     * Initialize orders admin
     * 
     * Ensures the orders admin class is instantiated early enough
     * for AJAX handlers to be properly registered. Checks if the
     * required file exists before attempting to load.
     * 
     * @since  1.4.0.0
     * @version 1.8.0.0
     * @return void
     */
    public function init_orders_admin() {
        // Only initialize if in admin area and not already initialized
        if ( is_admin() && $this->orders_admin === null ) {
            // Load orders admin class if file exists
            $orders_file = plugin_dir_path( dirname(__FILE__) ) . 'includes/class-delcampe-orders-admin.php';
            if ( file_exists( $orders_file ) ) {
                require_once $orders_file;
                
                // Create instance - this will register AJAX handlers in the constructor
                $this->orders_admin = new Delcampe_Orders_Admin();
                
                // Log initialization for debugging if debug mode is enabled
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    delcampe_log( '[Delcampe Admin Menu v1.8.0.0] Orders admin initialized for AJAX handlers', 'debug' );
                }
            }
        }
    }

    /**
     * Initialize inventory admin
     * 
     * Ensures the inventory admin class is instantiated early enough
     * for AJAX handlers to be properly registered. Uses singleton
     * pattern if the class implements it.
     * 
     * @since  1.5.0.0
     * @version 1.8.0.0
     * @return void
     */
    public function init_inventory_admin() {
        // Only initialize if in admin area and not already initialized
        if ( is_admin() && $this->inventory_admin === null ) {
            // Load inventory admin class if file exists
            $inventory_file = plugin_dir_path( dirname(__FILE__) ) . 'includes/class-delcampe-inventory-admin.php';
            if ( file_exists( $inventory_file ) ) {
                require_once $inventory_file;
                
                // Create instance if class exists
                if ( class_exists( 'Delcampe_Inventory_Admin' ) ) {
                    $this->inventory_admin = Delcampe_Inventory_Admin::get_instance();
                    
                    // Log initialization for debugging if debug mode is enabled
                    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                        delcampe_log( '[Delcampe Admin Menu v1.8.0.0] Inventory admin initialized for AJAX handlers', 'debug' );
                    }
                }
            }
        }
    }

    /**
     * Initialize listings admin
     * 
     * Ensures the listings admin class is instantiated early enough
     * for AJAX handlers and proper page display. This module provides
     * functionality similar to wp-lister-ebay for managing listings.
     * 
     * @since  1.6.0.1
     * @version 1.8.0.0
     * @return void
     */
    public function init_listings_admin() {
        // Only initialize if in admin area and not already initialized
        if ( is_admin() && $this->listings_admin === null ) {
            // Load listings admin class if file exists
            $listings_file = plugin_dir_path( __FILE__ ) . 'class-delcampe-listings-admin.php';
            if ( file_exists( $listings_file ) ) {
                require_once $listings_file;
                
                // Create instance if class exists
                if ( class_exists( 'Delcampe_Listings_Admin' ) ) {
                    $this->listings_admin = new Delcampe_Listings_Admin();
                    
                    // Log initialization for debugging if debug mode is enabled
                    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                        delcampe_log( '[Delcampe Admin Menu v1.8.0.0] Listings admin initialized for page display', 'debug' );
                    }
                }
            }
        }
    }

    /**
     * Register admin menu and submenu items
     * 
     * Creates the main plugin menu and all submenu pages.
     * Uses hardcoded strings during initial registration to avoid
     * translation loading issues during plugin activation.
     * The actual displayed text is properly translated when rendered.
     * 
     * @since  1.0.0
     * @version 1.8.0.0
     * @return void
     */
    public function register_menu() {
        // Log menu registration attempt for debugging
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            delcampe_log( '[Delcampe Admin Menu v1.8.0.0] Registering menu structure', 'debug' );
        }

        /**
         * Add main menu page
         * This creates the top-level menu item in WordPress admin sidebar
         * 
         * Position 56 places it after Plugins menu
         * Using 'dashicons-store' for appropriate marketplace icon
         * 
         * Note: Using hardcoded strings to avoid translation loading issues during activation
         */
        add_menu_page(
            'Delcampe Integration',                  // Page title in browser (hardcoded)
            'Delcampe',                             // Menu title in sidebar (hardcoded)
            'manage_options',                       // User must be able to manage options
            DELCAMPE_MENU_MAIN,                    // Unique menu slug from constants
            array( $this, 'render_main_page' ),     // Render callback
            'dashicons-store',                      // Store icon (appropriate for marketplace)
            56                                      // Position after Plugins menu
        );
        
        // Add Summary as first submenu item (replaces the duplicate "Delcampe" item)
        add_submenu_page(
            DELCAMPE_MENU_MAIN,                     // Parent menu slug
            'Delcampe Summary',                     // Page title (hardcoded)
            'Summary',                              // Menu title (hardcoded)
            'manage_options',                       // Required capability
            DELCAMPE_MENU_MAIN,                     // Same slug as parent to replace it
            array( $this, 'render_main_page' )      // Render callback
        );

        /**
         * Add Settings submenu with tabbed interface
         * This replaces multiple submenu items with one organized page
         * All functionality is organized into tabs for better UX
         * 
         * Note: Using hardcoded strings to avoid translation loading issues during activation
         */
        add_submenu_page(
            DELCAMPE_MENU_MAIN,                     // Parent menu slug
            'Delcampe Settings',                    // Page title (hardcoded)
            'Settings',                             // Menu title (hardcoded)
            'manage_options',                       // Required capability
            DELCAMPE_MENU_SETTINGS,                 // Menu slug from constants
            array( 'Delcampe_Admin_Tabbed_Page', 'render_settings_page' ) // Static callback
        );

        /**
         * Add Profiles submenu
         * For managing category mappings and listing profiles
         * 
         * Note: Using hardcoded strings to avoid translation loading issues during activation
         */
        add_submenu_page(
            DELCAMPE_MENU_MAIN,                     // Parent menu slug
            'Delcampe Profiles',                    // Page title (hardcoded)
            'Profiles',                             // Menu title (hardcoded)
            'manage_options',                       // Required capability
            DELCAMPE_MENU_PROFILES,                 // Menu slug from constants
            array( $this, 'render_profiles_page' )  // Render callback
        );

        /**
         * Add Listings submenu
         * For managing Delcampe listings with wp-lister-like interface
         * 
         * @since 1.6.0.0
         * @updated 1.6.0.1 - Fixed callback method
         * Note: Using hardcoded strings to avoid translation loading issues during activation
         */
        add_submenu_page(
            DELCAMPE_MENU_MAIN,                     // Parent menu slug
            'Delcampe Listings',                    // Page title (hardcoded)
            'Listings',                             // Menu title (hardcoded)
            'manage_woocommerce',                   // Required capability (WooCommerce specific)
            DELCAMPE_MENU_LISTINGS,                 // Menu slug from constants
            array( $this, 'render_listings_page' )  // Render callback
        );

        /**
         * Add Orders submenu (conditionally)
         * For managing Delcampe orders and synchronization
         * Only added if the orders module is available
         * 
         * @since 1.4.0.0
         * Note: Using hardcoded strings to avoid translation loading issues during activation
         */
        if ( class_exists( 'Delcampe_Orders_Admin' ) ) {
            add_submenu_page(
                DELCAMPE_MENU_MAIN,                 // Parent menu slug
                'Delcampe Orders',                  // Page title (hardcoded)
                'Orders',                           // Menu title (hardcoded)
                'manage_options',                   // Required capability
                DELCAMPE_MENU_ORDERS,               // Menu slug from constants
                array( $this, 'render_orders_page' ) // Render callback
            );
        }

        /**
         * Add Inventory submenu (conditionally)
         * For managing inventory synchronization
         * Only added if the inventory module is available
         * 
         * @since 1.5.0.0
         * Note: Using hardcoded strings to avoid translation loading issues during activation
         */
        if ( class_exists( 'Delcampe_Inventory_Admin' ) ) {
            add_submenu_page(
                DELCAMPE_MENU_MAIN,                     // Parent menu slug
                'Delcampe Inventory',                   // Page title (hardcoded)
                'Inventory',                            // Menu title (hardcoded)
                'manage_options',                       // Required capability
                DELCAMPE_MENU_INVENTORY,                // Menu slug from constants
                array( $this, 'render_inventory_page' ) // Render callback
            );
        }

        // Log successful menu registration
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            delcampe_log( '[Delcampe Admin Menu v1.8.0.0] Menu registered successfully', 'debug' );
        }
    }

    /**
     * Render main plugin page
     * 
     * Shows overview dashboard with:
     * - Plugin status and version information
     * - Quick links to different sections
     * - Statistics for listings, orders, inventory
     * - Configuration status checks
     * 
     * @since  1.0.0
     * @version 1.8.0.0
     * @return void
     */
    public function render_main_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Delcampe Integration', 'wc-delcampe-integration' ); ?></h1>
            
            <div class="welcome-panel" style="padding: 20px; background: #fff; border: 1px solid #c3c4c7;">
                <h2 style="color: #1d2327; margin-top: 0;"><?php esc_html_e( 'Welcome to WooCommerce Delcampe Integration', 'wc-delcampe-integration' ); ?></h2>
                <p class="about-description" style="color: #50575e; font-size: 14px;">
                    <?php 
                    // Display current version from constants
                    printf(
                        esc_html__( 'Version %s - Seamlessly sync your WooCommerce products with Delcampe marketplace.', 'wc-delcampe-integration' ),
                        esc_html( DELCAMPE_PLUGIN_VERSION )
                    ); 
                    ?>
                </p>
                
                <div class="welcome-panel-column-container">
                    <!-- Get Started Column -->
                    <div class="welcome-panel-column">
                        <h3 style="color: #1d2327;"><?php esc_html_e( 'Get Started', 'wc-delcampe-integration' ); ?></h3>
                        <a class="button button-primary button-hero" href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_SETTINGS ) ); ?>">
                            <?php esc_html_e( 'Configure Settings', 'wc-delcampe-integration' ); ?>
                        </a>
                        <p class="description" style="margin-top: 10px; color: #646970;">
                            <?php esc_html_e( 'Set up your API key and configure synchronization options.', 'wc-delcampe-integration' ); ?>
                        </p>
                    </div>
                    
                    <!-- Quick Links Column -->
                    <div class="welcome-panel-column">
                        <h3 style="color: #1d2327;"><?php esc_html_e( 'Quick Links', 'wc-delcampe-integration' ); ?></h3>
                        <ul>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_LISTINGS ) ); ?>">
                                    <?php esc_html_e( 'Manage Listings', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                            <?php if ( class_exists( 'Delcampe_Orders_Admin' ) ): ?>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_ORDERS ) ); ?>">
                                    <?php esc_html_e( 'Manage Orders', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                            <?php endif; ?>
                            <?php if ( class_exists( 'Delcampe_Inventory_Admin' ) ): ?>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_INVENTORY ) ); ?>">
                                    <?php esc_html_e( 'Inventory Sync', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                            <?php endif; ?>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_PROFILES ) ); ?>">
                                    <?php esc_html_e( 'Manage Profiles', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'edit.php?post_type=product' ) ); ?>">
                                    <?php esc_html_e( 'Assign Profiles to Products', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_SETTINGS . '&tab=categories' ) ); ?>">
                                    <?php esc_html_e( 'Manage Categories', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_SETTINGS . '&tab=test-sync' ) ); ?>">
                                    <?php esc_html_e( 'Test Sync', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_SETTINGS . '&tab=developer' ) ); ?>">
                                    <?php esc_html_e( 'View Logs', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                            <li>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_SETTINGS . '&tab=updates' ) ); ?>">
                                    <?php esc_html_e( 'Check Updates', 'wc-delcampe-integration' ); ?>
                                </a>
                            </li>
                        </ul>
                    </div>
                    
                    <!-- Status Column -->
                    <div class="welcome-panel-column welcome-panel-last">
                        <h3 style="color: #1d2327;"><?php esc_html_e( 'Status', 'wc-delcampe-integration' ); ?></h3>
                        <?php
                        // Check API key configuration status
                        $api_key = get_option( DELCAMPE_API_KEY_OPTION, '' );
                        if ( ! empty( $api_key ) ) {
                            echo '<p style="color: green;">✓ ' . esc_html__( 'API Key Configured', 'wc-delcampe-integration' ) . '</p>';
                            
                            // Check if we can authenticate
                            $auth = Delcampe_Auth::get_instance();
                            $token = $auth->get_auth_token();
                            if ( $token && ! is_wp_error( $token ) ) {
                                echo '<p style="color: green;">✓ ' . esc_html__( 'Authentication Active', 'wc-delcampe-integration' ) . '</p>';
                            } else {
                                echo '<p style="color: orange;">⚠ ' . esc_html__( 'Authentication Failed', 'wc-delcampe-integration' ) . '</p>';
                            }
                        } else {
                            echo '<p style="color: red;">✗ ' . esc_html__( 'API Key Not Set', 'wc-delcampe-integration' ) . '</p>';
                        }
                        
                        // Check WooCommerce status
                        if ( class_exists( 'WooCommerce' ) ) {
                            echo '<p style="color: green;">✓ ' . esc_html__( 'WooCommerce Active', 'wc-delcampe-integration' ) . '</p>';
                        } else {
                            echo '<p style="color: red;">✗ ' . esc_html__( 'WooCommerce Not Active', 'wc-delcampe-integration' ) . '</p>';
                        }
                        
                        // Check listings status
                        if ( class_exists( 'Delcampe_Listings_Model' ) ) {
                            $summary = Delcampe_Listings_Model::get_status_summary();
                            if ( $summary && $summary->total_items > 0 ) {
                                echo '<p style="color: green;">✓ ' . 
                                     sprintf( esc_html__( '%d Total Listings', 'wc-delcampe-integration' ), intval( $summary->total_items ) ) . 
                                     '</p>';
                                if ( $summary->published > 0 ) {
                                    echo '<p style="color: green;">✓ ' . 
                                         sprintf( esc_html__( '%d Published', 'wc-delcampe-integration' ), intval( $summary->published ) ) . 
                                         '</p>';
                                }
                            } else {
                                echo '<p style="color: orange;">⚠ ' . esc_html__( 'No Listings Created', 'wc-delcampe-integration' ) . '</p>';
                            }
                        }
                        
                        // Check product integration status
                        if ( class_exists( 'Delcampe_Product_Integration' ) ) {
                            global $wpdb;
                            
                            // Count products with profiles
                            $products_with_profiles = $wpdb->get_var(
                                "SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta} 
                                WHERE meta_key = '_delcampe_profile_id' AND meta_value != ''"
                            );
                            
                            if ( $products_with_profiles > 0 ) {
                                echo '<p style="color: green;">✓ ' . 
                                     sprintf( esc_html__( '%d Products with Profiles', 'wc-delcampe-integration' ), intval( $products_with_profiles ) ) . 
                                     '</p>';
                            } else {
                                echo '<p style="color: orange;">⚠ ' . esc_html__( 'No Products Assigned', 'wc-delcampe-integration' ) . '</p>';
                            }
                        }
                        
                        // Check order management status
                        if ( class_exists( 'Delcampe_Order_Manager' ) ) {
                            global $wpdb;
                            $orders_table = $wpdb->prefix . DELCAMPE_TABLE_ORDERS;
                            $table_exists = $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $orders_table ) ) === $orders_table;
                            
                            if ( $table_exists ) {
                                $orders_count = $wpdb->get_var( "SELECT COUNT(*) FROM $orders_table" );
                                if ( $orders_count > 0 ) {
                                    echo '<p style="color: green;">✓ ' . 
                                         sprintf( esc_html__( '%d Orders Synced', 'wc-delcampe-integration' ), intval( $orders_count ) ) . 
                                         '</p>';
                                } else {
                                    echo '<p style="color: orange;">⚠ ' . esc_html__( 'No Orders Imported', 'wc-delcampe-integration' ) . '</p>';
                                }
                            }
                        }
                        ?>
                    </div>
                </div>
            </div>
            
            <!-- About This Plugin Information Panel -->
            <div class="card" style="margin-top: 20px; max-width: 100%;">
                <h3><?php esc_html_e( 'About Delcampe Sync', 'wc-delcampe-integration' ); ?></h3>
                <p>
                    <?php esc_html_e( 'Enterprise-grade integration between WooCommerce and Delcampe marketplace. Seamlessly manage your collectibles business with automated listings, real-time order processing, and intelligent inventory synchronization across both platforms.', 'wc-delcampe-integration' ); ?>
                </p>
                <p>
                    <strong><?php echo sprintf( esc_html__( 'Version %s Features:', 'wc-delcampe-integration' ), DELCAMPE_PLUGIN_VERSION ); ?></strong>
                </p>
                <ul style="list-style: disc; margin-left: 20px;">
                    <li><?php esc_html_e( 'Automated bi-directional product synchronization', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Advanced Batch Queue System with atomic locking', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Duplicate prevention with SKU verification', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Secure license key encryption (AES-256-CBC)', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Real-time order import with webhook support', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Smart inventory management with oversell prevention', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Daily reconciliation system with audit trails', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Business Event Logger - Production-ready event tracking', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Bulk operations for mass listing management', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Profile templates for different product categories', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Category mapping with hierarchical support', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Automatic status checking and updates', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Price synchronization (optional bi-directional)', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Image management and synchronization', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'Deadlock prevention for large catalogs', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( '14-day free trial with full features', 'wc-delcampe-integration' ); ?></li>
                    <li><?php esc_html_e( 'License tiers from Basic to Enterprise', 'wc-delcampe-integration' ); ?></li>
                </ul>
            </div>

            <!-- What's New Panel -->
            <div class="card" style="margin-top: 20px; max-width: 100%; background: #f0f8ff; border-left: 4px solid #0073aa;">
                <h3 style="color: #0073aa;"><?php esc_html_e( "What's New in Recent Updates", 'wc-delcampe-integration' ); ?></h3>
                <ul style="list-style: disc; margin-left: 20px;">
                    <li><strong><?php esc_html_e( 'UI/UX Improvements (v1.10.24):', 'wc-delcampe-integration' ); ?></strong> <?php esc_html_e( 'Batch Queue moved to Inventory Management, tab persistence fixed', 'wc-delcampe-integration' ); ?></li>
                    <li><strong><?php esc_html_e( 'Duplicate Prevention (v1.10.23):', 'wc-delcampe-integration' ); ?></strong> <?php esc_html_e( 'Enhanced atomic locking, SKU verification, extended idempotency window', 'wc-delcampe-integration' ); ?></li>
                    <li><strong><?php esc_html_e( 'Security Enhancement (v1.10.22):', 'wc-delcampe-integration' ); ?></strong> <?php esc_html_e( 'License keys now encrypted with AES-256-CBC and HMAC authentication', 'wc-delcampe-integration' ); ?></li>
                    <li><strong><?php esc_html_e( 'Critical Security Fixes (v1.10.21):', 'wc-delcampe-integration' ); ?></strong> <?php esc_html_e( 'Fixed SQL injection vulnerabilities, webhook authentication, AJAX authorization', 'wc-delcampe-integration' ); ?></li>
                    <li><strong><?php esc_html_e( 'Data Integrity (v1.10.20):', 'wc-delcampe-integration' ); ?></strong> <?php esc_html_e( 'Database constraints prevent duplicates, one-active-listing-per-product enforcement', 'wc-delcampe-integration' ); ?></li>
                    <li><strong><?php esc_html_e( 'Batch Queue System (v1.10.18):', 'wc-delcampe-integration' ); ?></strong> <?php esc_html_e( 'Advanced batch processing with retry logic and recovery mechanisms', 'wc-delcampe-integration' ); ?></li>
                </ul>
                <p style="margin-top: 15px;">
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=delcampe&tab=business-events' ) ); ?>" class="button button-primary">
                        <?php esc_html_e( 'View Business Events', 'wc-delcampe-integration' ); ?>
                    </a>
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=delcampe&tab=advanced' ) ); ?>" class="button">
                        <?php esc_html_e( 'Configure Sync Settings', 'wc-delcampe-integration' ); ?>
                    </a>
                </p>
            </div>
            
            <!-- Listings Statistics Panel -->
            <?php if ( class_exists( 'Delcampe_Listings_Model' ) ): ?>
            <div class="card" style="margin-top: 20px; max-width: 100%;">
                <h3><?php esc_html_e( 'Listings Statistics', 'wc-delcampe-integration' ); ?></h3>
                <?php
                $summary = Delcampe_Listings_Model::get_status_summary();
                ?>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px;">
                    <div>
                        <h4><?php esc_html_e( 'Total Listings', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $summary->total_items ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'Published', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $summary->published ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'Prepared', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $summary->prepared ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'Changed', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $summary->changed ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'Ended', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $summary->ended ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'Sold', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $summary->sold ); ?></p>
                    </div>
                </div>
                <p style="margin-top: 20px;">
                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=' . DELCAMPE_MENU_LISTINGS ) ); ?>" class="button button-primary">
                        <?php esc_html_e( 'Manage Listings', 'wc-delcampe-integration' ); ?>
                    </a>
                </p>
            </div>
            <?php endif; ?>
            
            <!-- Product Integration Statistics Panel -->
            <?php if ( class_exists( 'Delcampe_Product_Integration' ) ): ?>
            <div class="card" style="margin-top: 20px; max-width: 100%;">
                <h3><?php esc_html_e( 'Product Integration Statistics', 'wc-delcampe-integration' ); ?></h3>
                <?php
                global $wpdb;
                
                // Get profile statistics
                $profiles_count = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}" . DELCAMPE_TABLE_PROFILES );
                
                // Get products with profiles
                $products_with_profiles = $wpdb->get_var(
                    "SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta} 
                    WHERE meta_key = '_delcampe_profile_id' AND meta_value != ''"
                );
                
                // Get sync status counts
                $sync_stats = $wpdb->get_results(
                    "SELECT meta_value as status, COUNT(*) as count 
                    FROM {$wpdb->postmeta} 
                    WHERE meta_key = '_delcampe_sync_status' 
                    GROUP BY meta_value"
                );
                ?>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px;">
                    <div>
                        <h4><?php esc_html_e( 'Total Profiles', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $profiles_count ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'Products with Profiles', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $products_with_profiles ); ?></p>
                    </div>
                    <?php foreach ( $sync_stats as $stat ): ?>
                    <div>
                        <h4><?php echo esc_html( ucfirst( $stat->status ) ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $stat->count ); ?></p>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Order Management Statistics Panel -->
            <?php if ( class_exists( 'Delcampe_Order_Manager' ) ): ?>
            <div class="card" style="margin-top: 20px; max-width: 100%;">
                <h3><?php esc_html_e( 'Order Management Statistics', 'wc-delcampe-integration' ); ?></h3>
                <?php
                $order_manager = Delcampe_Order_Manager::get_instance();
                $order_stats = $order_manager->get_order_statistics();
                ?>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px;">
                    <div>
                        <h4><?php esc_html_e( 'Total Orders', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $order_stats['total'] ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'Orders Today', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $order_stats['today'] ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'This Month', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;"><?php echo intval( $order_stats['this_month'] ); ?></p>
                    </div>
                    <div>
                        <h4><?php esc_html_e( 'Total Revenue', 'wc-delcampe-integration' ); ?></h4>
                        <p style="font-size: 24px; margin: 0;">
                            <?php echo wc_price( $order_stats['total_revenue'] ); ?>
                        </p>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render profiles page
     * 
     * Displays the profiles management interface.
     * Uses the already-instantiated profiles admin instance
     * to ensure AJAX handlers are properly registered.
     * 
     * @since  1.1.3.0
     * @version 1.8.0.0
     * @return void
     */
    public function render_profiles_page() {
        // Use the already-instantiated profiles admin
        if ( $this->profiles_admin ) {
            $this->profiles_admin->render_page();
        } else {
            // Fallback: create instance if somehow not already created
            require_once plugin_dir_path( dirname(__FILE__) ) . 'includes/profiles/class-delcampe-profiles-admin.php';
            $profiles_admin = new Delcampe_Profiles_Admin();
            $profiles_admin->render_page();
        }
    }

    /**
     * Render listings page
     * 
     * Displays the listings management interface.
     * Delegates to the listings admin class display_page method.
     * Provides wp-lister-like functionality for managing listings.
     * 
     * @since  1.6.0.0
     * @version 1.8.0.0
     * @return void
     */
    public function render_listings_page() {
        // Use the already-instantiated listings admin
        if ( $this->listings_admin ) {
            $this->listings_admin->display_page();
        } else {
            // Fallback: create instance if somehow not already created
            $listings_file = plugin_dir_path( __FILE__ ) . 'class-delcampe-listings-admin.php';
            if ( file_exists( $listings_file ) ) {
                require_once $listings_file;
                $listings_admin = new Delcampe_Listings_Admin();
                $listings_admin->display_page();
            } else {
                echo '<div class="notice notice-error"><p>' . 
                     esc_html__( 'Listings module not found.', 'wc-delcampe-integration' ) . 
                     '</p></div>';
            }
        }
    }

    /**
     * Render orders page
     * 
     * Displays the orders management interface.
     * Uses the already-instantiated orders admin instance
     * to ensure proper functionality and AJAX handling.
     * 
     * @since  1.4.0.0
     * @version 1.8.0.0
     * @return void
     */
    public function render_orders_page() {
        // Use the already-instantiated orders admin
        if ( $this->orders_admin ) {
            $this->orders_admin->render_page();
        } else {
            // Fallback: create instance if somehow not already created
            $orders_file = plugin_dir_path( dirname(__FILE__) ) . 'includes/class-delcampe-orders-admin.php';
            if ( file_exists( $orders_file ) ) {
                require_once $orders_file;
                $orders_admin = new Delcampe_Orders_Admin();
                $orders_admin->render_page();
            } else {
                echo '<div class="notice notice-error"><p>' . 
                     esc_html__( 'Orders module not found.', 'wc-delcampe-integration' ) . 
                     '</p></div>';
            }
        }
    }

    /**
     * Render inventory page
     * 
     * Displays the inventory management interface.
     * Uses singleton pattern if the inventory admin class implements it.
     * Provides real-time inventory synchronization controls.
     * 
     * @since  1.5.0.0
     * @version 1.8.0.0
     * @return void
     */
    public function render_inventory_page() {
        // Use the already-instantiated inventory admin
        if ( $this->inventory_admin ) {
            $this->inventory_admin->render_page();
        } else {
            // Fallback: create instance if somehow not already created
            $inventory_file = plugin_dir_path( dirname(__FILE__) ) . 'includes/class-delcampe-inventory-admin.php';
            if ( file_exists( $inventory_file ) && class_exists( 'Delcampe_Inventory_Admin' ) ) {
                $inventory_admin = Delcampe_Inventory_Admin::get_instance();
                $inventory_admin->render_page();
            } else {
                echo '<div class="notice notice-error"><p>' . 
                     esc_html__( 'Inventory module not found.', 'wc-delcampe-integration' ) . 
                     '</p></div>';
            }
        }
    }

    /**
     * Get the main menu slug
     * 
     * Utility method to retrieve the main menu slug.
     * Useful when other components need to reference the menu.
     * Returns the constant value for consistency.
     * 
     * @since  1.0.0
     * @version 1.8.0.0
     * @return string The main menu slug
     */
    public function get_menu_slug() {
        return DELCAMPE_MENU_MAIN;
    }

    /**
     * Get all submenu slugs
     * 
     * Returns an array of all registered submenu slugs.
     * Includes conditional pages that may not be available
     * in all installations (orders, inventory).
     * 
     * @since  1.0.0
     * @version 1.8.0.0
     * @return array Array of submenu slugs
     */
    public function get_submenu_slugs() {
        $slugs = array(
            DELCAMPE_MENU_SETTINGS,  // Settings page with tabs
            DELCAMPE_MENU_PROFILES,  // Profiles management page
            DELCAMPE_MENU_LISTINGS,  // Listings management page
        );
        
        // Add conditional pages if they exist
        if ( class_exists( 'Delcampe_Orders_Admin' ) ) {
            $slugs[] = DELCAMPE_MENU_ORDERS;     // Orders management page
        }
        
        if ( class_exists( 'Delcampe_Inventory_Admin' ) ) {
            $slugs[] = DELCAMPE_MENU_INVENTORY;  // Inventory management page
        }
        
        return $slugs;
    }
}
