<?php
/**
 * Delcampe Listings Admin Page
 *
 * Manages the listings page in WordPress admin
 *
 * @package WooCommerce_Delcampe_Integration
 * @version 1.6.1.0
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Delcampe_Listings_Admin class
 * 
 * Handles the administration interface for Delcampe listings.
 * Provides functionality for viewing, editing, and managing listings.
 * 
 * @since 1.6.0.0
 * @version 1.6.1.0 - Fixed Import Existing Listings button functionality
 *                    Fixed script/CSS paths
 */
class Delcampe_Listings_Admin {
    
    /**
     * List table instance
     *
     * @var Delcampe_Listings_List_Table
     */
    private $list_table;
    
    /**
     * Constructor
     * 
     * Sets up hooks for admin functionality.
     * Menu registration is handled by Delcampe_Admin_Menu class.
     * 
     * @since 1.6.0.0
     * @version 1.6.1.0
     */
    public function __construct() {
        // Don't add menu item here - it's handled by Delcampe_Admin_Menu
        // add_action('admin_menu', array($this, 'add_menu_item'), 20);
        
        // Initialize action handlers for listings management
        add_action('admin_init', array($this, 'handle_actions'));
        
        // Enqueue necessary scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // Handle screen options for per-page settings
        add_filter('set-screen-option', array($this, 'set_screen_option'), 10, 3);
        
        // Add screen options hook for the listings page
        add_action('load-delcampe-integration_page_delcampe-listings', array($this, 'screen_options'));
        
        // Add meta box actions for WooCommerce products
        add_action('add_meta_boxes', array($this, 'add_product_meta_box'));
        add_action('woocommerce_process_product_meta', array($this, 'save_product_meta_box'));
        
        // AJAX handler for importing existing listings (v1.6.0.2)
        add_action('wp_ajax_delcampe_import_existing_listings', array($this, 'ajax_import_existing_listings'));
    }
    
    /**
     * Screen options
     * 
     * Sets up screen options for the listings page.
     * Allows users to configure how many listings to show per page.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    public function screen_options() {
        $option = 'per_page';
        $args = array(
            'label'   => __('Listings per page', 'wc-delcampe-integration'),
            'default' => 50,  // Increased default from 20 to 50
            'option'  => 'delcampe_listings_per_page'
        );
        
        add_screen_option($option, $args);
        
        // Create list table instance
        $this->list_table = new Delcampe_Listings_List_Table();
    }
    
    /**
     * Set screen option
     * 
     * Handles saving of screen options.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param mixed $status Current status
     * @param string $option Option name
     * @param mixed $value Option value
     * @return mixed Updated value or original status
     */
    public function set_screen_option($status, $option, $value) {
        if ('delcampe_listings_per_page' == $option) {
            return $value;
        }
        
        return $status;
    }
    
    /**
     * Enqueue scripts
     * 
     * Loads necessary JavaScript and CSS files for the listings admin page.
     * 
     * @since 1.6.0.0
     * @version 1.6.1.0 - Fixed script/CSS paths
     * @param string $hook Current admin page hook
     */
    public function enqueue_scripts($hook) {
        // Debug logging
        delcampe_debug('[Delcampe Listings Admin] enqueue_scripts called with hook: ' . $hook);
        
        // Try multiple hook variations in case there's a difference
        $valid_hooks = array(
            'delcampe-integration_page_delcampe-listings',
            'delcampe_page_delcampe-listings',
            'toplevel_page_delcampe-listings'
        );
        
        // Check if we're on the listings page
        $is_listings_page = false;
        if (in_array($hook, $valid_hooks)) {
            $is_listings_page = true;
        }
        
        // Also check if the page parameter matches
        if (!$is_listings_page && isset($_GET['page']) && $_GET['page'] === 'delcampe-listings') {
            $is_listings_page = true;
            delcampe_debug('[Delcampe Listings Admin] Detected listings page via $_GET parameter');
        }
        
        if (!$is_listings_page) {
            delcampe_debug('[Delcampe Listings Admin] Not on listings page, skipping script enqueue');
            return;
        }
        
        delcampe_debug('[Delcampe Listings Admin] On listings page, enqueuing scripts');
        
        // Enqueue thickbox for preview functionality
        add_thickbox();
        
        // Get the correct plugin URL - this is the fix for the path issue
        $plugin_url = plugin_dir_url(dirname(__FILE__)); // This gets us to the plugin root
        
        // Enqueue custom styles for listings admin
        $css_url = $plugin_url . 'admin/css/listings-admin.css';
        wp_enqueue_style(
            'delcampe-listings-admin',
            $css_url,
            array(),
            '1.6.1.0'
        );
        delcampe_debug('[Delcampe Listings Admin] Enqueued CSS: ' . $css_url);
        
        // Enqueue custom scripts for listings admin
        $js_url = $plugin_url . 'admin/js/listings-admin.js';
        wp_enqueue_script(
            'delcampe-listings-admin',
            $js_url,
            array('jquery'),
            '1.6.1.0',
            true
        );
        delcampe_debug('[Delcampe Listings Admin] Enqueued JS: ' . $js_url);
        
        // Localize script for AJAX and translations
        $localize_data = array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('delcampe_listings_nonce'),
            'admin_nonce' => wp_create_nonce('delcampe_admin_nonce'), // For import function
            'i18n'     => array(
                'confirm_delete' => __('Are you sure you want to delete this listing?', 'wc-delcampe-integration'),
                'confirm_end'    => __('Are you sure you want to end this listing?', 'wc-delcampe-integration'),
                'importing'      => __('Importing existing listings...', 'wc-delcampe-integration'),
                'import_complete' => __('Import complete!', 'wc-delcampe-integration'),
                'import_error'   => __('Import failed. Please check the error log.', 'wc-delcampe-integration'),
            )
        );
        wp_localize_script('delcampe-listings-admin', 'delcampe_listings', $localize_data);
        delcampe_debug('[Delcampe Listings Admin] Localized script with data');
    }
    
    /**
     * AJAX handler for importing existing listings
     * 
     * @since 1.6.0.2
     */
    public function ajax_import_existing_listings() {
        // This is just a wrapper - the actual work is done by the sync handler
        $sync_handler = Delcampe_Sync_Handler::get_instance();
        $sync_handler->ajax_import_existing_listings();
    }
    
    /**
     * Handle actions
     * 
     * Processes various listing actions like publish, end, delete, etc.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    public function handle_actions() {
        if (!isset($_GET['page']) || $_GET['page'] !== 'delcampe-listings') {
            return;
        }
        
        if (!isset($_GET['action'])) {
            return;
        }
        
        $action = sanitize_text_field($_GET['action']);
        $listing_id = isset($_GET['listing']) ? intval($_GET['listing']) : 0;
        
        if (!$listing_id) {
            return;
        }
        
        // Handle single listing actions
        switch ($action) {
            case 'prepare':
                if (check_admin_referer('prepare_listing')) {
                    $this->prepare_listing($listing_id);
                }
                break;
                
            case 'verify':
                if (check_admin_referer('verify_listing')) {
                    $this->verify_listing($listing_id);
                }
                break;
                
            case 'publish':
                if (check_admin_referer('publish_listing')) {
                    $this->publish_listing($listing_id);
                }
                break;
                
            case 'revise':
                if (check_admin_referer('revise_listing')) {
                    $this->revise_listing($listing_id);
                }
                break;
                
            case 'end':
                if (check_admin_referer('end_listing')) {
                    $this->end_listing($listing_id);
                }
                break;
                
            case 'relist':
                if (check_admin_referer('relist_listing')) {
                    $this->relist_listing($listing_id);
                }
                break;
                
            case 'archive':
                if (check_admin_referer('archive_listing')) {
                    $this->archive_listing($listing_id);
                }
                break;
                
            case 'delete':
                if (check_admin_referer('delete_listing')) {
                    $this->delete_listing($listing_id);
                }
                break;
                
            case 'lock':
                if (check_admin_referer('lock_listing')) {
                    $this->lock_listing($listing_id);
                }
                break;
                
            case 'unlock':
                if (check_admin_referer('unlock_listing')) {
                    $this->unlock_listing($listing_id);
                }
                break;
        }
    }
    
    /**
     * Display page
     * 
     * Main method for rendering the listings admin page.
     * Routes to appropriate sub-pages based on action.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    public function display_page() {
        // Check for edit action
        if (isset($_GET['action']) && $_GET['action'] == 'edit' && isset($_GET['listing'])) {
            $this->display_edit_page();
            return;
        }
        
        // Check for preview action
        if (isset($_GET['action']) && $_GET['action'] == 'preview' && isset($_GET['listing'])) {
            $this->display_preview();
            return;
        }
        
        // Check for new products to prepare
        if ($this->check_for_new_products()) {
            $this->display_prepare_page();
            return;
        }
        
        // Display listings table
        $this->display_listings_page();
    }
    
    /**
     * Display listings page
     * 
     * Shows the main listings table with filters and bulk actions.
     * 
     * @since 1.6.0.0
     * @version 1.6.1.0 - Removed debug JavaScript since the path issue is fixed
     */
    private function display_listings_page() {
        if (!$this->list_table) {
            $this->list_table = new Delcampe_Listings_List_Table();
        }
        
        $this->list_table->prepare_items();
        
        // Get status summary for messages
        $summary = Delcampe_Listings_Model::get_status_summary();
        
        // Check if we need to show import button (v1.6.0.2)
        global $wpdb;
        $sql = $wpdb->prepare(
            "SELECT COUNT(DISTINCT p1.post_id)
             FROM {$wpdb->postmeta} p1
             LEFT JOIN {$wpdb->prefix}delcampe_listings dl ON p1.post_id = dl.product_id
             WHERE p1.meta_key = %s
             AND p1.meta_value IN (%s, %s)
             AND dl.id IS NULL",
            '_delcampe_sync_status', 'active', 'pending'
        );
        $synced_without_listings = $wpdb->get_var($sql);
        
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php _e('Delcampe Listings', 'wc-delcampe-integration'); ?></h1>
            
            <?php if ($synced_without_listings > 0): ?>
                <a href="#" id="import-existing-listings" class="page-title-action">
                    <?php printf(__('Import %d Existing Listings', 'wc-delcampe-integration'), $synced_without_listings); ?>
                </a>
            <?php endif; ?>
            
            <?php if (isset($_GET['message'])): ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php echo esc_html($this->get_message($_GET['message'])); ?></p>
                </div>
            <?php endif; ?>
            <?php if (isset($_GET['error'])): ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo esc_html(wp_unslash($_GET['error'])); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if ($summary->changed > 0): ?>
                <div class="notice notice-warning">
                    <p>
                        <?php printf(
                            __('There are %d changed items that need to be revised on Delcampe.', 'wc-delcampe-integration'),
                            $summary->changed
                        ); ?>
                        <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=delcampe-listings&action=revise_all'), 'revise_all_listings'); ?>" class="button">
                            <?php _e('Revise all changed items', 'wc-delcampe-integration'); ?>
                        </a>
                    </p>
                </div>
            <?php endif; ?>
            
            <div id="import-results" class="notice" style="display: none;">
                <p></p>
            </div>
            
            <form method="get">
                <input type="hidden" name="page" value="delcampe-listings" />
                
                <?php $this->list_table->views(); ?>
                
                <?php $this->list_table->search_box(__('Search listings', 'wc-delcampe-integration'), 'search'); ?>
                
                <?php $this->list_table->display(); ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * Display edit page
     * 
     * Shows the edit form for a specific listing.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    private function display_edit_page() {
        $listing_id = intval($_GET['listing']);
        $listing = Delcampe_Listings_Model::get_listing($listing_id);
        
        if (!$listing) {
            wp_die(__('Listing not found.', 'wc-delcampe-integration'));
        }
        
        // Handle form submission
        if (isset($_POST['save_listing'])) {
            check_admin_referer('edit_listing');
            $this->save_listing($listing_id);
            $listing = Delcampe_Listings_Model::get_listing($listing_id); // Reload
        }
        
        ?>
        <div class="wrap">
            <h1><?php _e('Edit Listing', 'wc-delcampe-integration'); ?></h1>
            
            <form method="post" action="">
                <?php wp_nonce_field('edit_listing'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="listing_title"><?php _e('Title', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="listing_title" name="listing_title" value="<?php echo esc_attr($listing->listing_title); ?>" class="regular-text" />
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="quantity"><?php _e('Quantity', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="quantity" name="quantity" value="<?php echo esc_attr($listing->quantity); ?>" min="0" />
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="price"><?php _e('Price', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="price" name="price" value="<?php echo esc_attr($listing->price); ?>" />
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="status"><?php _e('Status', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="status" name="status">
                                <option value="prepared" <?php selected($listing->status, 'prepared'); ?>><?php _e('Prepared', 'wc-delcampe-integration'); ?></option>
                                <option value="verified" <?php selected($listing->status, 'verified'); ?>><?php _e('Verified', 'wc-delcampe-integration'); ?></option>
                                <option value="published" <?php selected($listing->status, 'published'); ?>><?php _e('Published', 'wc-delcampe-integration'); ?></option>
                                <option value="changed" <?php selected($listing->status, 'changed'); ?>><?php _e('Changed', 'wc-delcampe-integration'); ?></option>
                                <option value="ended" <?php selected($listing->status, 'ended'); ?>><?php _e('Ended', 'wc-delcampe-integration'); ?></option>
                                <option value="sold" <?php selected($listing->status, 'sold'); ?>><?php _e('Sold', 'wc-delcampe-integration'); ?></option>
                                <option value="archived" <?php selected($listing->status, 'archived'); ?>><?php _e('Archived', 'wc-delcampe-integration'); ?></option>
                            </select>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <input type="submit" name="save_listing" class="button-primary" value="<?php _e('Save Changes', 'wc-delcampe-integration'); ?>" />
                    <a href="<?php echo admin_url('admin.php?page=delcampe-listings'); ?>" class="button"><?php _e('Cancel', 'wc-delcampe-integration'); ?></a>
                </p>
            </form>
        </div>
        <?php
    }
    
    /**
     * Display preview
     * 
     * Shows a preview of how the listing will appear on Delcampe.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    private function display_preview() {
        $listing_id = intval($_GET['listing']);
        $listing = Delcampe_Listings_Model::get_listing($listing_id);
        
        if (!$listing) {
            wp_die(__('Listing not found.', 'wc-delcampe-integration'));
        }
        
        $product = wc_get_product($listing->product_id);
        
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <title><?php echo esc_html($listing->listing_title); ?></title>
            <style>
                body { font-family: Arial, sans-serif; padding: 20px; }
                .preview-container { max-width: 800px; margin: 0 auto; }
                .preview-title { font-size: 24px; font-weight: bold; margin-bottom: 20px; }
                .preview-info { margin-bottom: 20px; }
                .preview-info label { font-weight: bold; display: inline-block; width: 120px; }
                .preview-description { border-top: 1px solid #ddd; padding-top: 20px; }
            </style>
        </head>
        <body>
            <div class="preview-container">
                <h1 class="preview-title"><?php echo esc_html($listing->listing_title); ?></h1>
                
                <div class="preview-info">
                    <p><label><?php _e('Price:', 'wc-delcampe-integration'); ?></label> <?php echo wc_price($listing->price); ?></p>
                    <p><label><?php _e('Quantity:', 'wc-delcampe-integration'); ?></label> <?php echo intval($listing->quantity); ?></p>
                    <p><label><?php _e('Status:', 'wc-delcampe-integration'); ?></label> <?php echo esc_html($listing->status); ?></p>
                    <?php if ($listing->delcampe_id): ?>
                        <p><label><?php _e('Delcampe ID:', 'wc-delcampe-integration'); ?></label> <?php echo esc_html($listing->delcampe_id); ?></p>
                    <?php endif; ?>
                </div>
                
                <?php if ($product): ?>
                    <div class="preview-description">
                        <h2><?php _e('Description', 'wc-delcampe-integration'); ?></h2>
                        <?php echo wpautop($product->get_description()); ?>
                    </div>
                <?php endif; ?>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
    
    /**
     * Check for new products to prepare
     * 
     * Determines if there are products that need to be prepared for listing.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @return bool True if there are products to prepare
     */
    private function check_for_new_products() {
        // Check if we have selected products in the session or query
        if (isset($_GET['prepare_products'])) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Display prepare page
     * 
     * Shows the interface for preparing new products for Delcampe listing.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    private function display_prepare_page() {
        $product_ids = array();
        
        if (isset($_GET['prepare_products'])) {
            $product_ids = array_map('intval', explode(',', $_GET['prepare_products']));
        }
        
        // Get profiles
        global $wpdb;
        $profiles_table = $wpdb->prefix . 'delcampe_profiles';
        $profiles = $wpdb->get_results("SELECT * FROM $profiles_table WHERE status = 'active' ORDER BY profile_name");
        
        ?>
        <div class="wrap">
            <h1><?php _e('Prepare Products for Delcampe', 'wc-delcampe-integration'); ?></h1>
            
            <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
                <input type="hidden" name="action" value="delcampe_prepare_listings" />
                <?php wp_nonce_field('prepare_listings'); ?>
                
                <?php foreach ($product_ids as $product_id): ?>
                    <input type="hidden" name="product_ids[]" value="<?php echo $product_id; ?>" />
                <?php endforeach; ?>
                
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Product', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('SKU', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Price', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Stock', 'wc-delcampe-integration'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($product_ids as $product_id): 
                            $product = wc_get_product($product_id);
                            if (!$product) continue;
                        ?>
                            <tr>
                                <td><?php echo esc_html($product->get_name()); ?></td>
                                <td><?php echo esc_html($product->get_sku()); ?></td>
                                <td><?php echo wc_price($product->get_price()); ?></td>
                                <td><?php echo intval($product->get_stock_quantity()); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <h2><?php _e('Select Profile', 'wc-delcampe-integration'); ?></h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="profile_id"><?php _e('Listing Profile', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="profile_id" name="profile_id" required>
                                <option value=""><?php _e('Select a profile', 'wc-delcampe-integration'); ?></option>
                                <?php foreach ($profiles as $profile): ?>
                                    <option value="<?php echo $profile->id; ?>"><?php echo esc_html($profile->profile_name); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description"><?php _e('Select the profile to use for these listings.', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <input type="submit" class="button-primary" value="<?php _e('Prepare Listings', 'wc-delcampe-integration'); ?>" />
                    <a href="<?php echo admin_url('admin.php?page=delcampe-listings'); ?>" class="button"><?php _e('Cancel', 'wc-delcampe-integration'); ?></a>
                </p>
            </form>
        </div>
        <?php
    }
    
    /**
     * Save listing
     * 
     * Saves changes to a listing from the edit form.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to save
     */
    private function save_listing($listing_id) {
        $data = array(
            'listing_title' => sanitize_text_field($_POST['listing_title']),
            'quantity'      => intval($_POST['quantity']),
            'price'         => floatval($_POST['price']),
            'status'        => sanitize_text_field($_POST['status']),
        );
        
        Delcampe_Listings_Model::update_listing($listing_id, $data);
        
        wp_redirect(add_query_arg(array(
            'page'    => 'delcampe-listings',
            'action'  => 'edit',
            'listing' => $listing_id,
            'message' => 'saved'
        ), admin_url('admin.php')));
        exit;
    }
    
    /**
     * Verify listing
     * 
     * Verifies a listing with Delcampe API.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to verify
     */
    private function verify_listing($listing_id) {
        $listing = Delcampe_Listings_Model::get_listing($listing_id);
        if (!$listing) {
            $this->redirect_with_error(__('Listing not found.', 'wc-delcampe-integration'));
        }
        // Minimal local validation: require product and profile
        if (empty($listing->product_id) || empty($listing->profile_id)) {
            $this->redirect_with_error(__('Listing is missing product or profile information.', 'wc-delcampe-integration'));
        }
        // Mark as verified; Delcampe has no verify endpoint. Deep validation occurs on publish.
        Delcampe_Listings_Model::update_status($listing_id, 'verified');
        $this->redirect_with_message('verified');
    }
    
    /**
     * Publish listing
     * 
     * Publishes a listing to Delcampe.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to publish
     */
    private function publish_listing($listing_id) {
        $listing = Delcampe_Listings_Model::get_listing($listing_id);
        if (!$listing) {
            $this->redirect_with_error(__('Listing not found.', 'wc-delcampe-integration'));
        }
        if (empty($listing->product_id) || empty($listing->profile_id)) {
            $this->redirect_with_error(__('Listing is missing product or profile information.', 'wc-delcampe-integration'));
        }

        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        $api = Delcampe_Listing_API::get_instance();
        $result = $api->create_listing((int)$listing->product_id, (int)$listing->profile_id);

        if (is_wp_error($result)) {
            $this->redirect_with_error($result->get_error_message());
        }
        if (empty($result['success'])) {
            $msg = isset($result['error']) ? $result['error'] : __('Unknown error during publish.', 'wc-delcampe-integration');
            $this->redirect_with_error($msg);
        }

        // Update local record with publish info
        $data = array(
            'status' => 'published',
            'date_published' => current_time('mysql'),
        );
        if (!empty($result['id'])) {
            $data['delcampe_id'] = sanitize_text_field($result['id']);
        }
        if (!empty($result['personal_reference'])) {
            $data['personal_reference'] = sanitize_text_field($result['personal_reference']);
        }
        Delcampe_Listings_Model::update_listing($listing_id, $data);

        $this->redirect_with_message('published');
    }
    
    /**
     * Revise listing
     * 
     * Revises a listing on Delcampe with updated information.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to revise
     */
    private function revise_listing($listing_id) {
        $listing = Delcampe_Listings_Model::get_listing($listing_id);
        if (!$listing) {
            $this->redirect_with_error(__('Listing not found.', 'wc-delcampe-integration'));
        }
        if (empty($listing->product_id) || empty($listing->delcampe_id)) {
            $this->redirect_with_error(__('Cannot revise: missing product or Delcampe ID.', 'wc-delcampe-integration'));
        }
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        $api = Delcampe_Listing_API::get_instance();
        $result = $api->update_listing((int)$listing->product_id, (string)$listing->delcampe_id);
        if (is_wp_error($result)) {
            $this->redirect_with_error($result->get_error_message());
        }
        if (empty($result['success'])) {
            $msg = isset($result['error']) ? $result['error'] : __('Unknown error during revise.', 'wc-delcampe-integration');
            $this->redirect_with_error($msg);
        }
        Delcampe_Listings_Model::update_status($listing_id, 'published');
        $this->redirect_with_message('revised');
    }
    
    /**
     * End listing
     * 
     * Ends a listing on Delcampe.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to end
     */
    private function end_listing($listing_id) {
        $listing = Delcampe_Listings_Model::get_listing($listing_id);
        if (!$listing) {
            $this->redirect_with_error(__('Listing not found.', 'wc-delcampe-integration'));
        }
        $delcampe_id = $listing->delcampe_id;
        if (empty($delcampe_id)) {
            // Fallback to product meta
            $delcampe_id = get_post_meta((int)$listing->product_id, '_delcampe_listing_id', true);
        }
        if (empty($delcampe_id)) {
            $this->redirect_with_error(__('No Delcampe listing ID is associated with this listing.', 'wc-delcampe-integration'));
        }
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        $api = Delcampe_Listing_API::get_instance();
        // Use allowed close reason aligned with API guidance
        $result = $api->close_listing((string)$delcampe_id, 'NotAvailableItem');
        if (is_wp_error($result)) {
            $this->redirect_with_error($result->get_error_message());
        }
        if (empty($result['success'])) {
            $msg = isset($result['error']) ? $result['error'] : __('Unknown error during end operation.', 'wc-delcampe-integration');
            $this->redirect_with_error($msg);
        }
        Delcampe_Listings_Model::update_listing($listing_id, array(
            'status' => 'ended',
            'date_finished' => current_time('mysql'),
        ));
        $this->redirect_with_message('ended');
    }
    
    /**
     * Relist listing
     * 
     * Relists an ended or sold listing on Delcampe.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to relist
     */
    private function relist_listing($listing_id) {
        $listing = Delcampe_Listings_Model::get_listing($listing_id);
        if (!$listing) {
            $this->redirect_with_error(__('Listing not found.', 'wc-delcampe-integration'));
        }
        if (empty($listing->product_id) || empty($listing->profile_id)) {
            $this->redirect_with_error(__('Listing is missing product or profile information.', 'wc-delcampe-integration'));
        }
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        $api = Delcampe_Listing_API::get_instance();
        $result = $api->create_listing((int)$listing->product_id, (int)$listing->profile_id);
        if (is_wp_error($result)) {
            $this->redirect_with_error($result->get_error_message());
        }
        if (empty($result['success'])) {
            $msg = isset($result['error']) ? $result['error'] : __('Unknown error during relist.', 'wc-delcampe-integration');
            $this->redirect_with_error($msg);
        }
        $data = array(
            'status' => 'published',
            'date_published' => current_time('mysql'),
            'date_finished' => null,
        );
        if (!empty($result['id'])) {
            $data['delcampe_id'] = sanitize_text_field($result['id']);
        }
        if (!empty($result['personal_reference'])) {
            $data['personal_reference'] = sanitize_text_field($result['personal_reference']);
        }
        Delcampe_Listings_Model::update_listing($listing_id, $data);
        $this->redirect_with_message('relisted');
    }
    
    /**
     * Archive listing
     * 
     * Archives a listing, removing it from active views.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to archive
     */
    private function archive_listing($listing_id) {
        Delcampe_Listings_Model::update_status($listing_id, 'archived');
        
        $this->redirect_with_message('archived');
    }
    
    /**
     * Delete listing
     * 
     * Permanently deletes a listing from the database.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to delete
     */
    private function delete_listing($listing_id) {
        Delcampe_Listings_Model::delete_listing($listing_id);
        
        $this->redirect_with_message('deleted');
    }
    
    /**
     * Lock listing
     * 
     * Locks a listing to prevent automatic updates.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to lock
     */
    private function lock_listing($listing_id) {
        Delcampe_Listings_Model::set_locked($listing_id, true);
        
        $this->redirect_with_message('locked');
    }
    
    /**
     * Unlock listing
     * 
     * Unlocks a listing to allow automatic updates.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $listing_id Listing ID to unlock
     */
    private function unlock_listing($listing_id) {
        Delcampe_Listings_Model::set_locked($listing_id, false);
        
        $this->redirect_with_message('unlocked');
    }
    
    /**
     * Get message text
     * 
     * Returns user-friendly message text for various actions.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param string $message_id Message identifier
     * @return string Translated message text
     */
    private function get_message($message_id) {
        $messages = array(
            'saved'     => __('Listing saved successfully.', 'wc-delcampe-integration'),
            'verified'  => __('Listing verified successfully.', 'wc-delcampe-integration'),
            'published' => __('Listing published successfully.', 'wc-delcampe-integration'),
            'revised'   => __('Listing revised successfully.', 'wc-delcampe-integration'),
            'ended'     => __('Listing ended successfully.', 'wc-delcampe-integration'),
            'relisted'  => __('Listing relisted successfully.', 'wc-delcampe-integration'),
            'archived'  => __('Listing archived successfully.', 'wc-delcampe-integration'),
            'deleted'   => __('Listing deleted successfully.', 'wc-delcampe-integration'),
            'locked'    => __('Listing locked successfully.', 'wc-delcampe-integration'),
            'unlocked'  => __('Listing unlocked successfully.', 'wc-delcampe-integration'),
        );
        
        return isset($messages[$message_id]) ? $messages[$message_id] : '';
    }
    
    /**
     * Redirect with message
     * 
     * Helper method to redirect with a success message.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param string $message Message identifier
     */
    private function redirect_with_message($message) {
        wp_redirect(add_query_arg(array(
            'page'    => 'delcampe-listings',
            'message' => $message
        ), admin_url('admin.php')));
        exit;
    }

    /**
     * Redirect with error message
     *
     * @param string $error_message
     */
    private function redirect_with_error($error_message) {
        wp_redirect(add_query_arg(array(
            'page'  => 'delcampe-listings',
            'error' => rawurlencode($error_message)
        ), admin_url('admin.php')));
        exit;
    }
    
    /**
     * Add product meta box
     * 
     * Adds a meta box to the product edit page for Delcampe listings.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     */
    public function add_product_meta_box() {
        add_meta_box(
            'delcampe_listings',
            __('Delcampe Listings', 'wc-delcampe-integration'),
            array($this, 'display_product_meta_box'),
            'product',
            'side',
            'default'
        );
    }
    
    /**
     * Display product meta box
     * 
     * Shows Delcampe listings information in the product edit page.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param WP_Post $post Current post object
     */
    public function display_product_meta_box($post) {
        $listings = Delcampe_Listings_Model::get_listings_by_product_id($post->ID);
        
        if (empty($listings)) {
            echo '<p>' . __('No Delcampe listings for this product.', 'wc-delcampe-integration') . '</p>';
            echo '<a href="' . admin_url('admin.php?page=delcampe-listings&prepare_products=' . $post->ID) . '" class="button">' . __('Create Listing', 'wc-delcampe-integration') . '</a>';
        } else {
            echo '<ul>';
            foreach ($listings as $listing) {
                $edit_url = admin_url('admin.php?page=delcampe-listings&action=edit&listing=' . $listing->id);
                echo '<li>';
                echo '<a href="' . esc_url($edit_url) . '">' . esc_html($listing->listing_title) . '</a>';
                echo ' (' . esc_html($listing->status) . ')';
                echo '</li>';
            }
            echo '</ul>';
        }
    }
    
    /**
     * Save product meta box
     * 
     * Placeholder for saving product meta box data if needed in the future.
     * 
     * @since 1.6.0.0
     * @version 1.6.0.1
     * @param int $post_id Post ID being saved
     */
    public function save_product_meta_box($post_id) {
        // Nothing to save for now - listings are managed separately
    }
}

// Note: Instantiation is now handled by Delcampe_Admin_Menu class
// Removed: new Delcampe_Listings_Admin();
