<?php
/**
 * Cleanup Orphaned Listings Admin Page
 */

if (!defined('ABSPATH')) {
    exit;
}

// Handle cleanup action
$cleanup_results = null;
$orphaned_analysis = null;

if (isset($_POST['cleanup_all']) && check_admin_referer('delcampe_cleanup_orphaned')) {
    $cleanup_results = delcampe_cleanup_all_orphaned_listings();
} elseif (isset($_POST['cleanup_specific']) && check_admin_referer('delcampe_cleanup_orphaned')) {
    $sku = sanitize_text_field($_POST['sku']);
    $product_id = intval($_POST['product_id']);
    if ($sku || $product_id) {
        $cleanup_results = delcampe_cleanup_specific_listing($sku, $product_id);
    }
} elseif (isset($_POST['cleanup_listings_table']) && check_admin_referer('delcampe_cleanup_orphaned')) {
    $cleanup_results = delcampe_cleanup_listings_table();
} elseif (isset($_POST['analyze_orphaned']) && check_admin_referer('delcampe_cleanup_orphaned')) {
    $orphaned_analysis = delcampe_analyze_orphaned_metadata();
} elseif (isset($_POST['cleanup_orphaned_metadata']) && check_admin_referer('delcampe_cleanup_orphaned')) {
    $cleanup_results = delcampe_cleanup_orphaned_metadata();
}

/**
 * Cleanup all orphaned listings
 */
function delcampe_cleanup_all_orphaned_listings() {
    global $wpdb;
    $results = array(
        'products_cleaned' => 0,
        'listings_updated' => 0,
        'queue_cleared' => false
    );
    
    // Clean product meta
    $args = array(
        'post_type' => 'product',
        'posts_per_page' => -1,
        'meta_query' => array(
            array(
                'key' => '_delcampe_sync_status',
                'compare' => 'EXISTS'
            )
        )
    );
    
    $products = get_posts($args);
    foreach ($products as $product) {
        delete_post_meta($product->ID, '_delcampe_sync_status');
        delete_post_meta($product->ID, '_delcampe_id');
        delete_post_meta($product->ID, '_delcampe_last_sync');
        delete_post_meta($product->ID, '_delcampe_sync_error');
        $results['products_cleaned']++;
    }
    
    // Clear sync queue
    $queue_table = $wpdb->prefix . 'delcampe_sync_queue';
    if ($wpdb->get_var("SHOW TABLES LIKE '$queue_table'") == $queue_table) {
        $wpdb->query("TRUNCATE TABLE $queue_table");
        $results['queue_cleared'] = true;
    }
    
    // Mark listings as ended
    $listings_table = $wpdb->prefix . 'delcampe_listings';
    if ($wpdb->get_var("SHOW TABLES LIKE '$listings_table'") == $listings_table) {
        $results['listings_updated'] = $wpdb->update(
            $listings_table,
            array('status' => 'ended'),
            array('status' => 'open')
        );
    }
    
    return $results;
}

/**
 * Cleanup specific listing
 */
function delcampe_cleanup_specific_listing($sku, $product_id) {
    global $wpdb;
    
    if ($sku) {
        $product_id = wc_get_product_id_by_sku($sku);
    }
    
    if (!$product_id) {
        return array('error' => 'Product not found');
    }
    
    $product = wc_get_product($product_id);
    if (!$product) {
        return array('error' => 'Invalid product ID');
    }
    
    $results = array(
        'product_id' => $product_id,
        'sku' => $product->get_sku(),
        'name' => $product->get_name(),
        'cleaned' => false
    );
    
    // Get current status
    $results['old_status'] = array(
        'sync_status' => get_post_meta($product_id, '_delcampe_sync_status', true),
        'delcampe_id' => get_post_meta($product_id, '_delcampe_id', true)
    );
    
    // Clean meta
    delete_post_meta($product_id, '_delcampe_sync_status');
    delete_post_meta($product_id, '_delcampe_id');
    delete_post_meta($product_id, '_delcampe_last_sync');
    delete_post_meta($product_id, '_delcampe_sync_error');
    
    // Clean database entries
    $queue_table = $wpdb->prefix . 'delcampe_sync_queue';
    if ($wpdb->get_var("SHOW TABLES LIKE '$queue_table'") == $queue_table) {
        $wpdb->delete($queue_table, array('product_id' => $product_id));
    }
    
    $listings_table = $wpdb->prefix . 'delcampe_listings';
    if ($wpdb->get_var("SHOW TABLES LIKE '$listings_table'") == $listings_table) {
        $wpdb->update(
            $listings_table,
            array('status' => 'ended'),
            array('product_id' => $product_id)
        );
    }
    
    $results['cleaned'] = true;
    return $results;
}

/**
 * Cleanup listings table
 */
function delcampe_cleanup_listings_table() {
    global $wpdb;
    $listings_table = $wpdb->prefix . 'delcampe_listings';
    
    $results = array(
        'deleted' => 0,
        'error' => null
    );
    
    if ($wpdb->get_var("SHOW TABLES LIKE '$listings_table'") == $listings_table) {
        // Delete all listings - this is a complete cleanup
        $deleted = $wpdb->query("TRUNCATE TABLE $listings_table");
        $results['deleted'] = 'all';
    } else {
        $results['error'] = 'Listings table not found';
    }
    
    return $results;
}

/**
 * Analyze orphaned metadata
 */
function delcampe_analyze_orphaned_metadata() {
    global $wpdb;
    
    // Count products with sync status but no listing record
    $sql = $wpdb->prepare(
        "SELECT COUNT(DISTINCT p1.post_id)
         FROM {$wpdb->postmeta} p1
         LEFT JOIN {$wpdb->prefix}delcampe_listings dl ON p1.post_id = dl.product_id
         WHERE p1.meta_key = %s
         AND p1.meta_value IN (%s, %s)
         AND dl.id IS NULL",
        '_delcampe_sync_status', 'active', 'pending'
    );
    $orphaned_count = $wpdb->get_var($sql);
    
    // Get examples
    $examples = $wpdb->get_results($wpdb->prepare(
        "SELECT DISTINCT p1.post_id, p1.meta_value as status
         FROM {$wpdb->postmeta} p1
         LEFT JOIN {$wpdb->prefix}delcampe_listings dl ON p1.post_id = dl.product_id
         WHERE p1.meta_key = %s
         AND p1.meta_value IN (%s, %s)
         AND dl.id IS NULL
         LIMIT 10",
        '_delcampe_sync_status', 'active', 'pending'
    ));
    
    $example_products = array();
    foreach ($examples as $ex) {
        $product = wc_get_product($ex->post_id);
        if ($product) {
            $example_products[] = array(
                'id' => $ex->post_id,
                'name' => $product->get_name(),
                'sku' => $product->get_sku(),
                'status' => $ex->status
            );
        }
    }
    
    // Count all Delcampe metadata
    $meta_counts = $wpdb->get_results("
        SELECT meta_key, COUNT(*) as count 
        FROM {$wpdb->postmeta} 
        WHERE meta_key LIKE '%delcampe%'
        GROUP BY meta_key
        ORDER BY count DESC
    ");
    
    return array(
        'orphaned_count' => $orphaned_count,
        'examples' => $example_products,
        'meta_counts' => $meta_counts
    );
}

/**
 * Cleanup orphaned metadata
 */
function delcampe_cleanup_orphaned_metadata() {
    global $wpdb;
    
    $results = array(
        'sync_status_deleted' => 0,
        'other_meta_deleted' => array(),
        'total_deleted' => 0
    );
    
    // Delete orphaned sync status metadata
    $deleted = $wpdb->query($wpdb->prepare(
        "DELETE p1 FROM {$wpdb->postmeta} p1
         LEFT JOIN {$wpdb->prefix}delcampe_listings dl ON p1.post_id = dl.product_id
         WHERE p1.meta_key = %s
         AND p1.meta_value IN (%s, %s)
         AND dl.id IS NULL",
        '_delcampe_sync_status', 'active', 'pending'
    ));
    
    $results['sync_status_deleted'] = $deleted;
    $results['total_deleted'] += $deleted;
    
    // Clean up other orphaned Delcampe metadata
    $other_meta = array(
        '_delcampe_listing_id',
        '_delcampe_item_id',
        '_delcampe_personal_reference',
        '_delcampe_last_sync',
        '_delcampe_last_error'
    );
    
    foreach ($other_meta as $meta_key) {
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE p1 FROM {$wpdb->postmeta} p1
             LEFT JOIN {$wpdb->prefix}delcampe_listings dl ON p1.post_id = dl.product_id
             WHERE p1.meta_key = %s
             AND dl.id IS NULL",
            $meta_key
        ));
        
        if ($deleted > 0) {
            $results['other_meta_deleted'][$meta_key] = $deleted;
            $results['total_deleted'] += $deleted;
        }
    }
    
    return $results;
}
?>

<div class="wrap">
    <h1>Cleanup Orphaned Delcampe Listings</h1>
    
    <div class="notice notice-warning">
        <p><strong>Warning:</strong> This tool will remove Delcampe sync data from products. Use this when you have listings showing as "Live" that don't actually exist on Delcampe.</p>
    </div>
    
    <?php if ($cleanup_results): ?>
        <div class="notice notice-success is-dismissible">
            <p><strong>Cleanup Complete!</strong></p>
            <?php if (isset($cleanup_results['error'])): ?>
                <p>Error: <?php echo esc_html($cleanup_results['error']); ?></p>
            <?php elseif (isset($cleanup_results['deleted'])): ?>
                <p>Deleted all records from the listings table.</p>
            <?php elseif (isset($cleanup_results['products_cleaned'])): ?>
                <ul>
                    <li>Products cleaned: <?php echo intval($cleanup_results['products_cleaned']); ?></li>
                    <li>Listings marked as ended: <?php echo intval($cleanup_results['listings_updated']); ?></li>
                    <li>Sync queue cleared: <?php echo $cleanup_results['queue_cleared'] ? 'Yes' : 'No'; ?></li>
                </ul>
            <?php else: ?>
                <p>Cleaned product: <?php echo esc_html($cleanup_results['name']); ?> (SKU: <?php echo esc_html($cleanup_results['sku']); ?>)</p>
            <?php endif; ?>
        </div>
    <?php endif; ?>
    
    <div style="background: white; padding: 20px; margin-top: 20px;">
        <h2>Cleanup Specific Product</h2>
        <form method="post" action="">
            <?php wp_nonce_field('delcampe_cleanup_orphaned'); ?>
            <table class="form-table">
                <tr>
                    <th><label for="sku">Product SKU</label></th>
                    <td>
                        <input type="text" name="sku" id="sku" class="regular-text" placeholder="e.g., CA-J31III-PB">
                        <p class="description">Enter the SKU of the product to clean</p>
                    </td>
                </tr>
                <tr>
                    <th><label for="product_id">OR Product ID</label></th>
                    <td>
                        <input type="number" name="product_id" id="product_id" class="regular-text" placeholder="e.g., 1234">
                        <p class="description">Enter the Product ID if you don't know the SKU</p>
                    </td>
                </tr>
            </table>
            <p class="submit">
                <input type="submit" name="cleanup_specific" class="button button-primary" value="Cleanup Specific Product">
            </p>
        </form>
    </div>
    
    <div style="background: white; padding: 20px; margin-top: 20px;">
        <h2>Cleanup All Product Sync Data</h2>
        <p>This will remove ALL Delcampe sync metadata from ALL products. The listings table records will remain for historical reference.</p>
        <form method="post" action="" onsubmit="return confirm('Are you sure you want to clean up ALL product sync data? This cannot be undone.');">
            <?php wp_nonce_field('delcampe_cleanup_orphaned'); ?>
            <p class="submit">
                <input type="submit" name="cleanup_all" class="button button-primary" value="Cleanup All Product Sync Data" style="background: #dc3232;">
            </p>
        </form>
    </div>
    
    <div style="background: white; padding: 20px; margin-top: 20px; border: 2px solid #ff9800;">
        <h2 style="color: #ff9800;">🔍 Orphaned Metadata Cleanup</h2>
        <p>This tool identifies and removes orphaned Delcampe metadata - products that show as "Active" or "Pending" in sync status but have no corresponding listing in the database.</p>
        <p>This often causes the "Import X Existing Listings" issue even when no listings exist.</p>
        
        <?php if ($orphaned_analysis): ?>
            <div style="background: #f0f0f0; padding: 15px; margin: 15px 0; border-radius: 5px;">
                <h3>Analysis Results</h3>
                <p><strong>Products with orphaned metadata: <?php echo intval($orphaned_analysis['orphaned_count']); ?></strong></p>
                
                <?php if (!empty($orphaned_analysis['examples'])): ?>
                    <h4>Example Products:</h4>
                    <table class="widefat" style="margin: 10px 0;">
                        <thead>
                            <tr>
                                <th>Product ID</th>
                                <th>Name</th>
                                <th>SKU</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($orphaned_analysis['examples'] as $product): ?>
                                <tr>
                                    <td>#<?php echo esc_html($product['id']); ?></td>
                                    <td><?php echo esc_html($product['name']); ?></td>
                                    <td><?php echo esc_html($product['sku']); ?></td>
                                    <td><?php echo esc_html($product['status']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
                
                <?php if (!empty($orphaned_analysis['meta_counts'])): ?>
                    <h4>All Delcampe Metadata Counts:</h4>
                    <ul>
                        <?php foreach ($orphaned_analysis['meta_counts'] as $meta): ?>
                            <li><code><?php echo esc_html($meta->meta_key); ?></code>: <?php echo intval($meta->count); ?> entries</li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
                
                <?php if ($orphaned_analysis['orphaned_count'] > 0): ?>
                    <form method="post" action="" style="margin-top: 15px;">
                        <?php wp_nonce_field('delcampe_cleanup_orphaned'); ?>
                        <p class="submit">
                            <input type="submit" name="cleanup_orphaned_metadata" class="button button-primary" value="Clean Up Orphaned Metadata" onclick="return confirm('This will remove all orphaned Delcampe metadata. Continue?');">
                        </p>
                    </form>
                <?php else: ?>
                    <p style="color: green;"><strong>✓ No orphaned metadata found!</strong></p>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <form method="post" action="">
                <?php wp_nonce_field('delcampe_cleanup_orphaned'); ?>
                <p class="submit">
                    <input type="submit" name="analyze_orphaned" class="button button-secondary" value="Analyze Orphaned Metadata">
                </p>
            </form>
        <?php endif; ?>
        
        <?php if (isset($cleanup_results['sync_status_deleted'])): ?>
            <div class="notice notice-success is-dismissible">
                <p><strong>Orphaned Metadata Cleanup Complete!</strong></p>
                <ul>
                    <li>Sync status entries deleted: <?php echo intval($cleanup_results['sync_status_deleted']); ?></li>
                    <?php if (!empty($cleanup_results['other_meta_deleted'])): ?>
                        <?php foreach ($cleanup_results['other_meta_deleted'] as $key => $count): ?>
                            <li><?php echo esc_html($key); ?> entries deleted: <?php echo intval($count); ?></li>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    <li><strong>Total metadata entries deleted: <?php echo intval($cleanup_results['total_deleted']); ?></strong></li>
                </ul>
            </div>
        <?php endif; ?>
    </div>
    
    <div style="background: white; padding: 20px; margin-top: 20px; border: 2px solid #dc3232;">
        <h2 style="color: #dc3232;">⚠️ Clear Entire Listings Table</h2>
        <p><strong>DANGER:</strong> This will completely DELETE ALL records from the Delcampe listings table. This removes all historical listing data and cannot be undone!</p>
        <p>Only use this if you want to completely start fresh with no listing history.</p>
        <form method="post" action="" onsubmit="return confirm('⚠️ WARNING: This will DELETE ALL listings from the database table!\n\nThis action CANNOT be undone!\n\nAre you absolutely sure you want to delete all listing records?');">
            <?php wp_nonce_field('delcampe_cleanup_orphaned'); ?>
            <p class="submit">
                <input type="submit" name="cleanup_listings_table" class="button" value="DELETE All Listings Table Records" style="background: #dc3232; color: white; font-weight: bold;">
            </p>
        </form>
    </div>
</div>