#!/usr/bin/php
<?php
/**
 * Clean up stuck queue items
 * v1.10.35.12
 */

require_once(dirname(dirname(dirname(dirname(dirname(__FILE__))))) . '/wp-load.php');

echo "\n=== CLEANING STUCK QUEUE ITEMS ===\n";
echo "Time: " . current_time('Y-m-d H:i:s') . "\n\n";

global $wpdb;

// 1. Check what we have
$sync_queue_table = $wpdb->prefix . 'delcampe_queue';
$batch_queue_table = $wpdb->prefix . 'delcampe_batch_queue';

// Count stuck items in sync queue
$stuck_published = $wpdb->get_var(
    "SELECT COUNT(*) FROM $sync_queue_table 
     WHERE state = 'published' 
     AND delcampe_listing_id IS NULL"
);

echo "Found $stuck_published items in 'published' state with no Delcampe ID\n";
echo "These should NOT be in the queue - they're completed but stuck.\n\n";

// Get details
$sample = $wpdb->get_results(
    "SELECT * FROM $sync_queue_table 
     WHERE state = 'published' 
     LIMIT 5"
);

if ($sample) {
    echo "Sample items:\n";
    foreach ($sample as $item) {
        echo "  ID: {$item->id}, Product: {$item->product_id}, Action: {$item->action}, Created: {$item->created_at}\n";
    }
    echo "\n";
}

// Ask for confirmation
echo "Do you want to clean up these stuck items? (yes/no): ";
$handle = fopen("php://stdin", "r");
$line = fgets($handle);
$answer = trim(strtolower($line));
fclose($handle);

if ($answer !== 'yes' && $answer !== 'y') {
    echo "\nCleanup cancelled.\n";
    exit(0);
}

echo "\n=== CLEANING UP ===\n";

// 1. Remove published items with no Delcampe ID (they're completed)
$deleted = $wpdb->query(
    "DELETE FROM $sync_queue_table 
     WHERE state = 'published' 
     AND (delcampe_listing_id IS NULL OR delcampe_listing_id = '')"
);

echo "✅ Removed $deleted stuck 'published' items with no Delcampe ID\n";

// 2. Remove very old pending items (>7 days)
$old_deleted = $wpdb->query(
    "DELETE FROM $sync_queue_table 
     WHERE state IN ('pending', 'processing') 
     AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)"
);

if ($old_deleted > 0) {
    echo "✅ Removed $old_deleted old pending/processing items (>7 days)\n";
}

// 3. Clean up batch queue
$batch_cleaned = $wpdb->query(
    "DELETE FROM $batch_queue_table 
     WHERE status IN ('completed', 'cancelled', 'failed') 
     AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)"
);

if ($batch_cleaned > 0) {
    echo "✅ Removed $batch_cleaned old completed/cancelled/failed batch items\n";
}

// 4. Reset any stuck processing items to pending
$reset = $wpdb->query(
    "UPDATE $sync_queue_table 
     SET state = 'pending' 
     WHERE state = 'processing' 
     AND updated_at < DATE_SUB(NOW(), INTERVAL 1 HOUR)"
);

if ($reset > 0) {
    echo "✅ Reset $reset stuck 'processing' items to 'pending'\n";
}

// Log the cleanup
delcampe_log('[Queue Cleanup] Removed ' . $deleted . ' stuck published items, ' . $old_deleted . ' old items');

// Final status
echo "\n=== FINAL QUEUE STATUS ===\n";

$final_count = $wpdb->get_results(
    "SELECT state, COUNT(*) as count 
     FROM $sync_queue_table 
     GROUP BY state"
);

if ($final_count) {
    foreach ($final_count as $row) {
        echo "State '{$row->state}': {$row->count} items\n";
    }
} else {
    echo "✅ Sync queue is now EMPTY\n";
}

echo "\n✅ Cleanup complete!\n\n";