<?php
/**
 * Category Mapping Admin Template
 *
 * Handles the display and management of category mapping between
 * WooCommerce and Delcampe platforms. Provides an intuitive interface
 * for administrators to map categories and manage synchronization rules.
 *
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Admin/Templates
 * @version 1.0.8.0
 * @since 1.0.0
 */

// Prevent direct file access
if (!defined('ABSPATH')) {
    exit;
}

// Get WooCommerce categories
$wc_categories = get_terms(array(
    'taxonomy' => 'product_cat',
    'hide_empty' => false,
));

// Get stored mappings
$category_mappings = get_option('delcampe_category_mappings', array());

// Get Delcampe categories from the API
try {
    $delcampe_api = Delcampe_API_Client::get_instance();
    $delcampe_categories = $delcampe_api->get_categories();
} catch (Exception $e) {
    $delcampe_categories = array();
    add_settings_error(
        'delcampe_category_mapping',
        'api_error',
        sprintf(
            __('Error fetching Delcampe categories: %s', 'wc-delcampe-integration'),
            $e->getMessage()
        )
    );
}
?>

<div class="wrap">
    <h1><?php echo esc_html__('Category Mapping', 'wc-delcampe-integration'); ?></h1>

    <?php settings_errors(); ?>

    <div class="card">
        <h2><?php echo esc_html__('Map WooCommerce Categories to Delcampe', 'wc-delcampe-integration'); ?></h2>
        
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" id="category-mapping-form">
            <input type="hidden" name="action" value="save_delcampe_category_mapping">
            <?php wp_nonce_field('save_delcampe_category_mapping', 'delcampe_category_mapping_nonce'); ?>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th scope="col" class="manage-column column-wc-category">
                            <?php echo esc_html__('WooCommerce Category', 'wc-delcampe-integration'); ?>
                        </th>
                        <th scope="col" class="manage-column column-delcampe-category">
                            <?php echo esc_html__('Delcampe Category', 'wc-delcampe-integration'); ?>
                        </th>
                        <th scope="col" class="manage-column column-sync-enabled">
                            <?php echo esc_html__('Enable Sync', 'wc-delcampe-integration'); ?>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($wc_categories)) : ?>
                        <?php foreach ($wc_categories as $category) : ?>
                            <tr>
                                <td class="column-wc-category">
                                    <?php echo esc_html($category->name); ?>
                                    <input type="hidden" 
                                           name="wc_categories[]" 
                                           value="<?php echo esc_attr($category->term_id); ?>">
                                </td>
                                <td class="column-delcampe-category">
                                    <select name="delcampe_category_map[<?php echo esc_attr($category->term_id); ?>]"
                                            class="delcampe-category-select">
                                        <option value=""><?php echo esc_html__('-- Select Category --', 'wc-delcampe-integration'); ?></option>
                                        <?php
                                        if (!empty($delcampe_categories)) {
                                            foreach ($delcampe_categories as $dc_cat) {
                                                $selected = isset($category_mappings[$category->term_id]) 
                                                          && $category_mappings[$category->term_id] === $dc_cat['id'];
                                                echo sprintf(
                                                    '<option value="%s" %s>%s</option>',
                                                    esc_attr($dc_cat['id']),
                                                    selected($selected, true, false),
                                                    esc_html($dc_cat['name'])
                                                );
                                            }
                                        }
                                        ?>
                                    </select>
                                </td>
                                <td class="column-sync-enabled">
                                    <input type="checkbox" 
                                           name="sync_enabled[<?php echo esc_attr($category->term_id); ?>]" 
                                           value="1"
                                           <?php checked(isset($category_mappings[$category->term_id . '_enabled']) && $category_mappings[$category->term_id . '_enabled']); ?>>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr>
                            <td colspan="3">
                                <?php echo esc_html__('No WooCommerce categories found.', 'wc-delcampe-integration'); ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>

            <?php submit_button(__('Save Mappings', 'wc-delcampe-integration')); ?>
        </form>
    </div>

    <div class="card">
        <h2><?php echo esc_html__('Category Sync Status', 'wc-delcampe-integration'); ?></h2>
        <table class="wp-list-table widefat fixed">
            <thead>
                <tr>
                    <th><?php echo esc_html__('Category', 'wc-delcampe-integration'); ?></th>
                    <th><?php echo esc_html__('Last Sync', 'wc-delcampe-integration'); ?></th>
                    <th><?php echo esc_html__('Status', 'wc-delcampe-integration'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php
                $sync_status = get_option('delcampe_category_sync_status', array());
                if (!empty($sync_status)) :
                    foreach ($sync_status as $cat_id => $status) : 
                        $category = get_term($cat_id, 'product_cat');
                        if (!$category) continue;
                        ?>
                        <tr>
                            <td><?php echo esc_html($category->name); ?></td>
                            <td>
                                <?php 
                                echo isset($status['last_sync']) 
                                    ? esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $status['last_sync']))
                                    : esc_html__('Never', 'wc-delcampe-integration');
                                ?>
                            </td>
                            <td>
                                <?php
                                $status_text = isset($status['status']) ? $status['status'] : 'pending';
                                $status_class = 'status-' . sanitize_html_class($status_text);
                                echo sprintf(
                                    '<span class="status-indicator %s">%s</span>',
                                    esc_attr($status_class),
                                    esc_html(ucfirst($status_text))
                                );
                                ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else : ?>
                    <tr>
                        <td colspan="3"><?php echo esc_html__('No sync history available.', 'wc-delcampe-integration'); ?></td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<style>
.status-indicator {
    display: inline-block;
    padding: 3px 8px;
    border-radius: 3px;
    font-size: 12px;
    font-weight: 500;
}
.status-success { background: #dff0d8; color: #3c763d; }
.status-error { background: #f2dede; color: #a94442; }
.status-pending { background: #fcf8e3; color: #8a6d3b; }
.status-syncing { background: #d9edf7; color: #31708f; }
</style>

<script type="text/javascript">
jQuery(document).ready(function($) {
    // Enable select2 for category dropdowns if available
    if ($.fn.select2) {
        $('.delcampe-category-select').select2({
            width: '100%',
            placeholder: '<?php echo esc_js(__('Select a category', 'wc-delcampe-integration')); ?>'
        });
    }

    // Form submission handling
    $('#category-mapping-form').on('submit', function(e) {
        var hasUnmappedEnabled = false;
        $('.delcampe-category-select').each(function() {
            var $select = $(this);
            var $checkbox = $select.closest('tr').find('input[type="checkbox"]');
            
            if ($checkbox.is(':checked') && !$select.val()) {
                hasUnmappedEnabled = true;
                return false;
            }
        });

        if (hasUnmappedEnabled) {
            e.preventDefault();
            alert('<?php echo esc_js(__('Please select Delcampe categories for all enabled mappings.', 'wc-delcampe-integration')); ?>');
        }
    });
});
</script>
