/**
 * Batch Queue Monitor JavaScript
 * 
 * @package WC_Delcampe_Integration
 * @since 1.10.18.0
 */

(function($) {
    'use strict';
    
    var DelcampeBatchMonitor = {
        
        refreshInterval: null,
        isRefreshing: false,
        
        /**
         * Initialize
         */
        init: function() {
            this.bindEvents();
            this.startAutoRefresh();
        },
        
        /**
         * Bind events
         */
        bindEvents: function() {
            var self = this;
            
            // Refresh button
            $('#refresh-queue-stats').on('click', function() {
                self.refreshStats();
            });
            
            // Auto-refresh toggle
            $('#auto-refresh').on('change', function() {
                if ($(this).is(':checked')) {
                    self.startAutoRefresh();
                } else {
                    self.stopAutoRefresh();
                }
            });
            
            // Pause/Resume processing
            $('#pause-processing').on('click', function() {
                self.pauseProcessing();
            });
            
            $('#resume-processing').on('click', function() {
                self.resumeProcessing();
            });
            
            // Cancel batch
            $(document).on('click', '.cancel-batch', function() {
                var batchId = $(this).data('batch-id');
                if (confirm(delcampeBatchMonitor.strings.confirm_cancel)) {
                    self.cancelBatch(batchId);
                }
            });
            
            // Retry batch
            $(document).on('click', '.retry-batch', function() {
                var batchId = $(this).data('batch-id');
                if (confirm(delcampeBatchMonitor.strings.confirm_retry)) {
                    self.retryBatch(batchId);
                }
            });

            // Force unlock processing batch
            $(document).on('click', '.force-unlock-batch', function() {
                var batchId = $(this).data('batch-id');
                self.forceUnlockBatch(batchId);
            });

            // Run sync queue now
            $('#run-sync-now').on('click', function() {
                self.runSyncNow();
            });

            // Clear sync lock
            $('#clear-sync-lock').on('click', function() {
                self.clearSyncLock();
            });
        },
        
        /**
         * Start auto-refresh
         */
        startAutoRefresh: function() {
            var self = this;
            
            if (this.refreshInterval) {
                return;
            }
            
            this.refreshInterval = setInterval(function() {
                if (!self.isRefreshing) {
                    self.refreshStats(true);
                }
            }, delcampeBatchMonitor.refresh_interval);
        },
        
        /**
         * Stop auto-refresh
         */
        stopAutoRefresh: function() {
            if (this.refreshInterval) {
                clearInterval(this.refreshInterval);
                this.refreshInterval = null;
            }
        },
        
        /**
         * Refresh statistics
         */
        refreshStats: function(silent) {
            var self = this;
            
            if (this.isRefreshing) {
                return;
            }
            
            this.isRefreshing = true;
            
            if (!silent) {
                $('#refresh-queue-stats').prop('disabled', true).addClass('updating-message');
            }
            
            $.ajax({
                url: delcampeBatchMonitor.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_get_queue_stats',
                    nonce: delcampeBatchMonitor.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $('#queue-stats-container').html(response.data.stats_html);
                        $('#active-batches-container').html(response.data.active_html);
                        $('#recent-batches-container').html(response.data.recent_html);
                        
                        // Update pause/resume button
                        if (response.data.is_paused) {
                            $('#pause-processing').hide();
                            $('#resume-processing').show();
                        } else {
                            $('#pause-processing').show();
                            $('#resume-processing').hide();
                        }
                    }
                },
                complete: function() {
                    self.isRefreshing = false;
                    if (!silent) {
                        $('#refresh-queue-stats').prop('disabled', false).removeClass('updating-message');
                    }
                }
            });
        },
        
        /**
         * Cancel batch
         */
        cancelBatch: function(batchId) {
            var self = this;
            
            $.ajax({
                url: delcampeBatchMonitor.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_cancel_batch',
                    batch_id: batchId,
                    nonce: delcampeBatchMonitor.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.refreshStats();
                    } else {
                        alert(response.data || 'Failed to cancel batch');
                    }
                }
            });
        },
        
        /**
         * Retry batch
         */
        retryBatch: function(batchId) {
            var self = this;
            
            $.ajax({
                url: delcampeBatchMonitor.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_retry_batch',
                    batch_id: batchId,
                    nonce: delcampeBatchMonitor.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Also clear any batch pause and nudge processors
                        self.runSyncNow();
                        self.refreshStats();
                    } else {
                        alert(response.data || 'Failed to retry batch');
                    }
                }
            });
        },

        /**
         * Force run the sync queue now
         */
        runSyncNow: function() {
            var self = this;
            $.ajax({
                url: delcampeBatchMonitor.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_process_sync_queue',
                    nonce: delcampeBatchMonitor.admin_nonce
                },
                success: function(response) {
                    // Optionally display a brief status; always refresh
                    self.refreshStats(true);
                }
            });
        },

        /**
         * Clear the sync lock and trigger a run
         */
        clearSyncLock: function() {
            var self = this;
            $.ajax({
                url: delcampeBatchMonitor.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_clear_sync_lock',
                    nonce: delcampeBatchMonitor.admin_nonce
                },
                success: function(response) {
                    self.refreshStats(true);
                    if (response && response.success) {
                        alert(delcampeBatchMonitor.strings.clear_lock);
                    }
                }
            });
        },

        /**
         * Force unlock a processing batch
         */
        forceUnlockBatch: function(batchId) {
            var self = this;
            $.ajax({
                url: delcampeBatchMonitor.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_force_unlock_batch',
                    batch_id: batchId,
                    nonce: delcampeBatchMonitor.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.refreshStats();
                    } else {
                        alert(response.data || 'Failed to unlock batch');
                    }
                }
            });
        },
        
        /**
         * Pause processing
         */
        pauseProcessing: function() {
            var self = this;
            
            $.ajax({
                url: delcampeBatchMonitor.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_pause_processing',
                    nonce: delcampeBatchMonitor.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(delcampeBatchMonitor.strings.processing_paused);
                        self.refreshStats();
                    }
                }
            });
        },
        
        /**
         * Resume processing
         */
        resumeProcessing: function() {
            var self = this;
            
            $.ajax({
                url: delcampeBatchMonitor.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_resume_processing',
                    nonce: delcampeBatchMonitor.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(delcampeBatchMonitor.strings.processing_resumed);
                        self.refreshStats();
                    }
                }
            });
        }
    };
    
    // Initialize on document ready
    $(document).ready(function() {
        DelcampeBatchMonitor.init();
    });
    
})(jQuery);
