/**
 * Delcampe Product Integration JavaScript
 * Version: 1.2.0.0
 * 
 * Handles client-side functionality for product-profile integration
 * Manages AJAX sync requests and UI updates
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.2.0.0
 */

(function($) {
    'use strict';

    /**
     * Product Integration Handler
     * 
     * @since 1.2.0.0
     */
    var DelcampeProductIntegration = {
        
        /**
         * Initialize the handler
         * 
         * @since 1.2.0.0
         */
        init: function() {
            // Bind events
            this.bindEvents();
            
            // Log initialization
            console.log('[Delcampe Product Integration v1.2.0.0] JavaScript initialized');
        },
        
        /**
         * Bind event handlers
         * 
         * @since 1.2.0.0
         */
        bindEvents: function() {
            // Sync now button
            $(document).on('click', '#delcampe-sync-now', this.handleSyncNow.bind(this));
            
            // View listing button
            $(document).on('click', '#delcampe-view-listing', this.handleViewListing.bind(this));
            
            // Close listing button
            $(document).on('click', '#delcampe-close-listing', this.handleCloseListing.bind(this));
            
            // Profile selection change
            $(document).on('change', '#delcampe_profile_id', this.handleProfileChange.bind(this));
        },
        
        /**
         * Handle sync now button click
         * 
         * @since 1.2.0.0
         * @param {Event} e Click event
         */
        handleSyncNow: function(e) {
            e.preventDefault();
            
            var $button = $(e.target);
            var productId = $button.data('product-id');
            
            // Confirm action
            if (!confirm(delcampe_product.strings.sync_confirm)) {
                return;
            }
            
            // Update button state
            $button.prop('disabled', true).text(delcampe_product.strings.syncing);
            
            // Send AJAX request
            $.ajax({
                url: delcampe_product.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_sync_product',
                    product_id: productId,
                    nonce: delcampe_product.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Update button
                        $button.text(delcampe_product.strings.sync_complete)
                               .removeClass('button-primary')
                               .addClass('button-secondary');
                        
                        // Update sync status display if present
                        if (response.data.sync_status) {
                            DelcampeProductIntegration.updateSyncStatus(response.data);
                        }
                        
                        // Re-enable button after delay
                        setTimeout(function() {
                            $button.prop('disabled', false)
                                   .text(delcampe_product.strings.sync_now || 'Sync Now')
                                   .removeClass('button-secondary')
                                   .addClass('button-primary');
                        }, 3000);
                    } else {
                        // Show error
                        alert(response.data.message || delcampe_product.strings.sync_error);
                        $button.prop('disabled', false).text(delcampe_product.strings.sync_now || 'Sync Now');
                    }
                },
                error: function() {
                    alert(delcampe_product.strings.sync_error);
                    $button.prop('disabled', false).text(delcampe_product.strings.sync_now || 'Sync Now');
                }
            });
        },
        
        /**
         * Handle view listing button click
         * 
         * @since 1.2.0.0
         * @param {Event} e Click event
         */
        handleViewListing: function(e) {
            e.preventDefault();
            
            var listingId = $(e.target).data('listing-id');
            if (listingId) {
                // Open Delcampe listing in new window
                // Note: URL structure may need adjustment based on Delcampe's actual URL format
                window.open('https://www.delcampe.net/en_US/collectibles/item/' + listingId + '.html', '_blank');
            }
        },
        
        /**
         * Handle close listing button click
         * 
         * @since 1.10.5.0
         * @param {Event} e Click event
         */
        handleCloseListing: function(e) {
            e.preventDefault();
            
            var $button = $(e.target);
            var productId = $button.data('product-id');
            var listingId = $button.data('listing-id');
            
            if (!confirm('Are you sure you want to close this listing on Delcampe? This action cannot be undone.')) {
                return;
            }
            
            var reason = prompt('Optional: Enter a reason for closing this listing (or leave blank):');
            
            // Update button state
            $button.prop('disabled', true).text('Closing...');
            
            // Send AJAX request
            $.ajax({
                url: delcampe_product.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_close_listing',
                    product_id: productId,
                    listing_id: listingId,
                    reason: reason || '',
                    nonce: delcampe_product.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Check if there's a note about asynchronous processing
                        var message = response.data.message || 'Listing closed successfully.';
                        if (response.data.note) {
                            message += '\n\n' + response.data.note;
                        }
                        
                        alert(message);
                        $button.text('Close Requested');
                        // Reload page to update status
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        alert('Failed to close listing: ' + (response.data.message || 'Unknown error'));
                        $button.prop('disabled', false).text('Close Listing');
                    }
                },
                error: function() {
                    alert('Request failed. Please try again.');
                    $button.prop('disabled', false).text('Close Listing');
                }
            });
        },
        
        /**
         * Handle profile selection change
         * 
         * @since 1.2.0.0
         * @param {Event} e Change event
         */
        handleProfileChange: function(e) {
            var $select = $(e.target);
            var profileId = $select.val();
            
            // Show/hide profile details based on selection
            if (profileId) {
                // Could load profile details via AJAX in future enhancement
                console.log('[Delcampe Product Integration] Profile selected:', profileId);
            }
        },
        
        /**
         * Update sync status display
         * 
         * @since 1.2.0.0
         * @param {Object} data Sync status data
         */
        updateSyncStatus: function(data) {
            var $statusContainer = $('.sync-status');
            
            if ($statusContainer.length === 0) {
                // Create status container if it doesn't exist
                var $metabox = $('.delcampe-product-meta');
                if ($metabox.length) {
                    $('<hr style="margin: 15px 0;">').appendTo($metabox);
                    $statusContainer = $('<div class="sync-status"></div>').appendTo($metabox);
                }
            }
            
            // Update status display
            if (data.listing_id && $statusContainer.find('code').length === 0) {
                $('<p><strong>Delcampe Listing ID:</strong><br><code>' + data.listing_id + '</code></p>')
                    .prependTo($statusContainer);
            }
            
            // Update or create status span
            var $statusSpan = $statusContainer.find('.sync-status-' + data.sync_status);
            if ($statusSpan.length === 0) {
                // Remove old status
                $statusContainer.find('[class^="sync-status-"]').parent().remove();
                
                // Add new status
                $('<p><strong>Sync Status:</strong><br><span class="sync-status-' + data.sync_status + '">' + 
                  data.sync_status.charAt(0).toUpperCase() + data.sync_status.slice(1) + '</span></p>')
                    .appendTo($statusContainer);
            }
            
            // Update last sync time
            if (data.last_sync) {
                var $lastSync = $statusContainer.find('p:contains("Last Sync")');
                if ($lastSync.length) {
                    $lastSync.html('<strong>Last Sync:</strong><br>' + data.last_sync_formatted);
                } else {
                    $('<p><strong>Last Sync:</strong><br>' + data.last_sync_formatted + '</p>')
                        .appendTo($statusContainer);
                }
            }
        }
    };
    
    // Initialize when document is ready
    $(document).ready(function() {
        DelcampeProductIntegration.init();
    });
    
})(jQuery);
