/**
 * Delcampe Profiles Admin JavaScript
 * Version: 1.1.4.2
 * 
 * CHANGELOG:
 * Version 1.1.4.2 - Added debug logging to troubleshoot button clicks
 * Version 1.1.4.1 - Fixed AJAX handlers and modal functionality
 *                 - Added better error handling
 *                 - Improved category selection logic
 */

(function($) {
    'use strict';

    console.log('[Delcampe Profiles Admin] Script loaded');

    var DelcampeProfilesAdmin = {
        
        /**
         * Current selected category data
         */
        selectedCategory: null,
        
        /**
         * Category breadcrumb trail
         */
        breadcrumb: [],
        
        /**
         * Initialize
         */
        init: function() {
            console.log('[Delcampe Profiles Admin] Initializing...');
            this.bindEvents();
            this.initializeForm();
            console.log('[Delcampe Profiles Admin] Initialization complete');
        },
        
        /**
         * Bind event handlers
         */
        bindEvents: function() {
            console.log('[Delcampe Profiles Admin] Binding events...');
            
            // Category selection button
            $('#select-delcampe-category').on('click', this.openCategoryModal.bind(this));
            console.log('[Delcampe Profiles Admin] Bound click event to #select-delcampe-category');
            
            // Select from saved categories button
            $('#select-from-saved').on('click', this.openSavedCategoriesModal.bind(this));
            console.log('[Delcampe Profiles Admin] Bound click event to #select-from-saved');
            
            // Modal close button
            $('.delcampe-modal-close, #delcampe-category-cancel').on('click', this.closeCategoryModal.bind(this));
            
            // Saved categories modal close
            $('.saved-categories-close').on('click', this.closeSavedCategoriesModal.bind(this));
            
            // Modal select button
            $('#delcampe-category-select').on('click', this.selectCategory.bind(this));
            
            // Quantity mode toggle
            $('#quantity_mode').on('change', this.toggleQuantityMode.bind(this));
            
            // Auto sync toggle
            $('#auto_sync').on('change', this.toggleAutoSync.bind(this));
            
            // Click outside modal to close
            $('#delcampe-category-modal').on('click', function(e) {
                if (e.target === this) {
                    this.closeCategoryModal();
                }
            }.bind(this));
            
            $('#saved-categories-modal').on('click', function(e) {
                if (e.target === this) {
                    this.closeSavedCategoriesModal();
                }
            }.bind(this));
            
            // Prevent modal content clicks from closing
            $('.delcampe-modal-content').on('click', function(e) {
                e.stopPropagation();
            });
            
            // Debug: Check if buttons exist
            if ($('#select-delcampe-category').length) {
                console.log('[Delcampe Profiles Admin] Select Delcampe Category button found');
            } else {
                console.error('[Delcampe Profiles Admin] Select Delcampe Category button NOT found');
            }
            
            if ($('#select-from-saved').length) {
                console.log('[Delcampe Profiles Admin] Select from Saved button found');
            } else {
                console.error('[Delcampe Profiles Admin] Select from Saved button NOT found');
            }
        },
        
        /**
         * Initialize form state
         */
        initializeForm: function() {
            // Set initial visibility states
            this.toggleQuantityMode();
            this.toggleAutoSync();
        },
        
        /**
         * Open category selection modal
         */
        openCategoryModal: function(e) {
            console.log('[Delcampe Profiles Admin] openCategoryModal called');
            e.preventDefault();
            
            console.log('[Delcampe Profiles Admin] Opening category modal...');
            $('#delcampe-category-modal').show();
            this.loadCategories(''); // Load root categories (empty string for root)
            this.breadcrumb = [];
            this.updateBreadcrumb();
        },
        
        /**
         * Open saved categories modal
         */
        openSavedCategoriesModal: function(e) {
            console.log('[Delcampe Profiles Admin] openSavedCategoriesModal called');
            e.preventDefault();
            
            console.log('[Delcampe Profiles Admin] Opening saved categories modal...');
            $('#saved-categories-modal').show();
            this.loadSavedCategories();
        },
        
        /**
         * Close category selection modal
         */
        closeCategoryModal: function() {
            $('#delcampe-category-modal').hide();
            this.selectedCategory = null;
            $('#delcampe-category-select').prop('disabled', true);
        },
        
        /**
         * Close saved categories modal
         */
        closeSavedCategoriesModal: function() {
            $('#saved-categories-modal').hide();
        },
        
        /**
         * Load categories from server
         */
        loadCategories: function(parentId) {
            console.log('[Delcampe Profiles Admin] Loading categories for parent:', parentId);
            var self = this;
            var $list = $('#delcampe-category-list');
            
            // Show loading state
            $list.html('<p class="loading">' + delcampe_profiles.strings.loading + '</p>');
            
            // Check if delcampe_profiles is defined
            if (typeof delcampe_profiles === 'undefined') {
                console.error('[Delcampe Profiles Admin] delcampe_profiles is not defined!');
                $list.html('<p class="error">Configuration error: delcampe_profiles is not defined</p>');
                return;
            }
            
            console.log('[Delcampe Profiles Admin] AJAX URL:', delcampe_profiles.ajax_url);
            console.log('[Delcampe Profiles Admin] Nonce:', delcampe_profiles.nonce);
            
            // Make AJAX request
            $.ajax({
                url: delcampe_profiles.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_get_categories',
                    parent_id: parentId,
                    nonce: delcampe_profiles.nonce
                },
                success: function(response) {
                    console.log('[Delcampe Profiles Admin] Categories loaded:', response);
                    if (response.success) {
                        self.displayCategories(response.data);
                    } else {
                        $list.html('<p class="error">' + (response.data || delcampe_profiles.strings.error) + '</p>');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('[Delcampe Profiles Admin] AJAX Error:', error);
                    console.error('[Delcampe Profiles Admin] Response:', xhr.responseText);
                    $list.html('<p class="error">' + delcampe_profiles.strings.error + '</p>');
                }
            });
        },
        
        /**
         * Load saved categories
         */
        loadSavedCategories: function() {
            console.log('[Delcampe Profiles Admin] Loading saved categories...');
            var self = this;
            var $list = $('#saved-categories-list');
            
            // Show loading state
            $list.html('<p class="loading">' + delcampe_profiles.strings.loading + '</p>');
            
            // Check if delcampe_profiles is defined
            if (typeof delcampe_profiles === 'undefined') {
                console.error('[Delcampe Profiles Admin] delcampe_profiles is not defined!');
                $list.html('<p class="error">Configuration error: delcampe_profiles is not defined</p>');
                return;
            }
            
            // Make AJAX request
            $.ajax({
                url: delcampe_profiles.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_get_saved_categories',
                    nonce: delcampe_profiles.nonce
                },
                success: function(response) {
                    console.log('[Delcampe Profiles Admin] Saved categories loaded:', response);
                    if (response.success && response.data && response.data.length > 0) {
                        self.displaySavedCategories(response.data);
                    } else {
                        $list.html('<p>' + delcampe_profiles.strings.no_saved_categories + '</p>');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('[Delcampe Profiles Admin] AJAX Error:', error);
                    console.error('[Delcampe Profiles Admin] Response:', xhr.responseText);
                    $list.html('<p class="error">' + delcampe_profiles.strings.error + '</p>');
                }
            });
        },
        
        /**
         * Display categories in the list
         */
        displayCategories: function(categories) {
            var self = this;
            var $list = $('#delcampe-category-list');
            
            if (!categories || categories.length === 0) {
                $list.html('<p>' + delcampe_profiles.strings.no_categories + '</p>');
                return;
            }
            
            $list.empty();
            
            $.each(categories, function(index, category) {
                var $item = $('<div class="delcampe-category-item"></div>');
                
                if (category.has_children) {
                    $item.addClass('has-children');
                }
                
                var $name = $('<span class="delcampe-category-name"></span>').text(category.name);
                var $id = $('<span class="delcampe-category-id"></span>').text('ID: ' + category.id);
                
                $item.append($name).append($id);
                
                // Click handler
                $item.on('click', function() {
                    if (category.has_children) {
                        // Navigate to subcategories
                        self.breadcrumb.push({
                            id: category.id,
                            name: category.name
                        });
                        self.loadCategories(category.id);
                        self.updateBreadcrumb();
                    } else {
                        // Select this category
                        $('.delcampe-category-item').removeClass('selected');
                        $(this).addClass('selected');
                        self.selectedCategory = category;
                        $('#delcampe-category-select').prop('disabled', false);
                    }
                });
                
                // Also allow selection of categories with children
                $item.on('dblclick', function() {
                    $('.delcampe-category-item').removeClass('selected');
                    $(this).addClass('selected');
                    self.selectedCategory = category;
                    $('#delcampe-category-select').prop('disabled', false);
                });
                
                $list.append($item);
            });
            
            // Add instruction
            $list.append('<p style="margin-top: 10px; font-style: italic; color: #666;">' + 
                'Click to browse subcategories, double-click to select a category.' + 
                '</p>');
        },
        
        /**
         * Display saved categories with hierarchical tree UI
         */
        displaySavedCategories: function(categories) {
            var self = this;
            var $list = $('#saved-categories-list');
            
            this.allSavedCategories = categories || [];

            var html = '';
            html += '<div class="saved-categories-controls">';
            html += '<input type="text" id="saved-categories-search" placeholder="Search categories..." class="widefat" />';
            html += '<div class="saved-categories-stats">';
            html += '<span class="category-count">' + this.allSavedCategories.length + ' saved categories</span>';
            html += '</div>';
            html += '</div>';
            html += '<div class="saved-categories-tree" id="saved-categories-tree"></div>';
            $list.html(html);

            // Build and render tree
            this.savedTree = this.buildSavedTree(this.allSavedCategories);
            this.renderSavedTree(this.savedTree);

            // Bind search
            $('#saved-categories-search').on('input', function() {
                var term = $(this).val().toLowerCase();
                self.renderSavedTree(self.savedTree, term);
            });
        },

        /** Build a hierarchical tree from saved categories list */
        buildSavedTree: function(categories) {
            var root = { name: 'Root', children: {}, leaf: false };
            (categories || []).forEach(function(cat){
                var path = (cat.parent_path || '').trim();
                var parts = path ? path.split(' > ') : [];
                // Remove leading 'Root' if present
                if (parts.length && parts[0].toLowerCase() === 'root') parts.shift();
                // Last segment is the leaf name (should match category_name)
                var current = root;
                for (var i = 0; i < parts.length; i++) {
                    var seg = parts[i];
                    if (!seg) continue;
                    if (!current.children[seg]) {
                        current.children[seg] = { name: seg, children: {}, leaf: false };
                    }
                    current = current.children[seg];
                }
                // Create leaf entry
                var leafName = cat.category_name || (parts.length ? parts[parts.length-1] : '');
                if (!current.children[leafName]) {
                    current.children[leafName] = { name: leafName, children: {}, leaf: true, data: cat };
                } else {
                    // If an intermediate node has same name, force a leaf child entry with unique key
                    if (!current.children[leafName].leaf) {
                        var key = leafName + ' (leaf)';
                        current.children[key] = { name: leafName, children: {}, leaf: true, data: cat };
                    } else {
                        current.children[leafName].data = cat;
                    }
                }
            });
            return root;
        },

        /** Render the saved categories tree into the modal */
        renderSavedTree: function(tree, searchTerm) {
            var self = this;
            var $container = $('#saved-categories-tree');
            $container.empty();

            function createNode(name, node, level) {
                var matches = !searchTerm || name.toLowerCase().indexOf(searchTerm) !== -1;
                var hasChildren = node && node.children && Object.keys(node.children).length > 0;
                var $li = $('<li></li>').addClass('tree-node').attr('data-level', level);
                var $row = $('<div class="tree-row"></div>');
                var icon = node.leaf ? '📄' : '📁';
                var $label = $('<span class="tree-label"></span>').text(name);
                var $icon = $('<span class="tree-icon"></span>').text(icon);
                $row.append($icon).append($label);

                // If leaf, attach selection behavior
                if (node.leaf && node.data) {
                    var isParentFlag = (node.data.has_children === true);
                    if (isParentFlag) {
                        $row.addClass('is-parent');
                        $row.append('<span class="tree-status parent">⚠️</span>');
                        $row.on('click', function(){
                            self.showErrorFeedback('Cannot select "' + node.data.category_name + '" - it has subcategories');
                        });
                    } else {
                        $row.addClass('is-leaf');
                        $row.on('click', function(){
                            // Select and update main form
                            $('#delcampe_category_id').val(node.data.category_id);
                            $('#delcampe_category_name').val(node.data.category_name);
                            $('#delcampe_category_path').val(node.data.parent_path || '');
                            var $display = $('#delcampe-category-display');
                            $display.html(
                                '<div class="delcampe-category-selected">' +
                                '<strong>' + self.escapeHtml(node.data.category_name) + '</strong>' +
                                (node.data.parent_path ? '<br><small>' + self.escapeHtml(node.data.parent_path) + '</small>' : '') +
                                '</div>'
                            );
                            self.showSelectionFeedback(node.data.category_name);
                            setTimeout(function(){ self.closeSavedCategoriesModal(); }, 400);
                        });
                    }
                } else {
                    // Expand/collapse behavior for folders
                    $row.on('click', function(){
                        $li.toggleClass('collapsed');
                    });
                }

                $li.append($row);

                if (hasChildren) {
                    var $ul = $('<ul class="tree-children"></ul>');
                    // Sort children by name
                    var keys = Object.keys(node.children).sort(function(a,b){ return a.localeCompare(b); });
                    keys.forEach(function(childName){
                        var child = node.children[childName];
                        // Filter by search: include if matches here or any descendant matches
                        if (!searchTerm || childName.toLowerCase().indexOf(searchTerm) !== -1) {
                            $ul.append(createNode(childName, child, level+1));
                        } else {
                            // Check descendants
                            var stack = [child];
                            var found = false;
                            while (stack.length && !found) {
                                var x = stack.pop();
                                if (x.leaf && x.data && x.data.category_name && x.data.category_name.toLowerCase().indexOf(searchTerm) !== -1) { found = true; break; }
                                if (x.children) {
                                    Object.keys(x.children).forEach(function(k){ stack.push(x.children[k]); });
                                }
                            }
                            if (found) {
                                $ul.append(createNode(childName, child, level+1));
                            }
                        }
                    });
                    $li.append($ul);
                }

                return $li;
            }

            // Top-level list (hide the artificial root row, show its children)
            var $top = $('<ul class="tree-root"></ul>');
            var topKeys = Object.keys(tree.children || {}).sort(function(a,b){ return a.localeCompare(b); });
            topKeys.forEach(function(name){
                $top.append(createNode(name, tree.children[name], 0));
            });
            $container.append($top);
        },
        
        /**
         * Render saved categories in grid layout
         */
        renderSavedCategoriesGrid: function(categories) {
            var self = this;
            var $grid = $('#saved-categories-grid');
            
            $grid.empty();
            
            if (categories.length === 0) {
                $grid.html('<div class="no-results">No categories found matching your search.</div>');
                return;
            }
            
            // Group categories by hierarchical path (all but the leaf name)
            var grouped = {};
            $.each(categories, function(index, category) {
                var path = (category.parent_path || '').trim();
                // path may equal leaf name only; split by ' > ' and drop last segment to get container path
                var parts = path ? path.split(' > ') : [];
                var groupKey = 'Root';
                if (parts.length > 1) {
                    parts.pop(); // remove leaf
                    groupKey = parts.join(' > ');
                } else if (parts.length === 1) {
                    groupKey = 'Root';
                }
                if (!grouped[groupKey]) {
                    grouped[groupKey] = [];
                }
                grouped[groupKey].push(category);
            });

            // Sort groups by path
            var sortedGroups = Object.keys(grouped).sort(function(a,b){
                return a.localeCompare(b);
            });

            $.each(sortedGroups, function(index, pathHeader) {
                var $group = $('<div class="category-group"></div>');
                $group.append('<div class="group-header">' + self.escapeHtml(pathHeader) + '</div>');
                
                var $groupItems = $('<div class="group-items"></div>');
                
                $.each(grouped[letter], function(catIndex, category) {
                    var $card = $('<div class="category-card"></div>');
                    
                    // Check if this is a parent category (has children)
                    var hasFlag = (typeof category.has_children !== 'undefined');
                    var isParent = hasFlag && category.has_children === true;
                    var isLeaf   = hasFlag && category.has_children === false;
                    var isCheckError = category.check_error === true;
                    
                    // Add class for parent categories
                    if (isParent) {
                        $card.addClass('parent-category');
                    }
                    if (isCheckError) {
                        $card.addClass('check-error');
                    }
                    
                    // Category icon based on path and status
                    var icon = '📁';
                    if (isParent) {
                        icon = '⚠️'; // Warning icon for parent categories
                    } else if (isLeaf) {
                        icon = '✅';
                    } else if (category.parent_path) {
                        if (category.parent_path.toLowerCase().includes('stamp')) icon = '📮';
                        else if (category.parent_path.toLowerCase().includes('coin')) icon = '🪙';
                        else if (category.parent_path.toLowerCase().includes('post')) icon = '✉️';
                        else if (category.parent_path.toLowerCase().includes('cover')) icon = '📧';
                    }
                    
                    var cardHtml = '';
                    cardHtml += '<div class="card-icon">' + icon + '</div>';
                    cardHtml += '<div class="card-content">';
                    cardHtml += '<div class="card-title">' + self.escapeHtml(category.category_name) + '</div>';
                    cardHtml += '<div class="card-id">ID: ' + category.category_id + '</div>';
                    
                    // Add status indicator (do not show persistent verifying by default)
                    if (isParent) {
                        cardHtml += '<div class="card-status parent-warning">⚠️ Has subcategories - cannot select</div>';
                    } else if (isLeaf) {
                        cardHtml += '<div class="card-status leaf-ok">✓ Leaf category - safe to use</div>';
                    } else if (isCheckError) {
                        cardHtml += '<div class="card-status check-warning">⚠️ Could not verify - use with caution</div>';
                    }
                    
                    if (category.parent_path) {
                        cardHtml += '<div class="card-path" title="' + self.escapeHtml(category.parent_path) + '">';
                        cardHtml += self.truncatePath(category.parent_path, 60);
                        cardHtml += '</div>';
                    }
                    cardHtml += '</div>';
                    
                    $card.html(cardHtml);
                    
                    // Click handler - prevent selection of parent categories and lazily verify unknowns
                    $card.on('click', function() {
                        var $this = $(this);
                        if (isParent) {
                            self.showErrorFeedback('Cannot select "' + category.category_name + '" - it has subcategories');
                            return;
                        }

                        // If has_children flag is unknown, verify via AJAX before selecting
                        if (!hasFlag) {
                            $this.addClass('verifying');
                            $.ajax({
                                url: delcampe_profiles.ajax_url,
                                type: 'POST',
                                data: {
                                    action: 'delcampe_check_category_leaf',
                                    nonce: delcampe_profiles.nonce,
                                    category_id: category.category_id
                                },
                                success: function(resp) {
                                    $this.removeClass('verifying');
                                    if (resp && resp.success) {
                                        // If API returns null (unknown), allow selection with caution
                                        var hasChildren = (resp.data && resp.data.has_children === true);
                                        if (hasChildren) {
                                            self.showErrorFeedback('Cannot select "' + category.category_name + '" - it has subcategories');
                                            // Update UI to parent state
                                            $card.addClass('parent-category');
                                            $card.find('.card-status').removeClass('leaf-ok').addClass('parent-warning').text('⚠️ Has subcategories - cannot select');
                                            return;
                                        }
                                        // For unknown or leaf, proceed; reflect UI if leaf
                                        if (resp.data && resp.data.has_children === false) {
                                            $card.removeClass('parent-category');
                                            $card.find('.card-status').removeClass('parent-warning').addClass('leaf-ok').text('✓ Leaf category - safe to use');
                                        } else if (resp.data && resp.data.warning) {
                                            // Show a subtle inline hint but proceed
                                            if ($card.find('.card-status').length === 0) {
                                                $card.append('<div class="card-status check-warning">⚠️ Could not verify - using selection</div>');
                                            } else {
                                                $card.find('.card-status').removeClass('parent-warning').addClass('check-warning').text('⚠️ Could not verify - using selection');
                                            }
                                        }
                                        proceedSelect();
                                    } else {
                                        self.showErrorFeedback(resp && resp.data ? resp.data : (delcampe_profiles.strings.error || 'Error'));
                                    }
                                },
                                error: function() {
                                    $this.removeClass('verifying');
                                    self.showErrorFeedback(delcampe_profiles.strings.error || 'Error');
                                }
                            });
                            return;
                        }

                        proceedSelect();

                        function proceedSelect() {
                            // Remove previous selection
                            $('.category-card').removeClass('selected');
                            $card.addClass('selected');
                            // Update the main form fields
                            $('#delcampe_category_id').val(category.category_id);
                            $('#delcampe_category_name').val(category.category_name);
                            $('#delcampe_category_path').val(category.parent_path || '');
                            // Update display
                            var $display = $('#delcampe-category-display');
                            $display.html(
                                '<div class="delcampe-category-selected">' +
                                '<strong>' + self.escapeHtml(category.category_name) + '</strong>' +
                                (category.parent_path ? '<br><small>' + self.escapeHtml(category.parent_path) + '</small>' : '') +
                                '</div>'
                            );
                            // Show success feedback
                            self.showSelectionFeedback(category.category_name);
                            // Close modal after short delay
                            setTimeout(function() { self.closeSavedCategoriesModal(); }, 500);
                        }
                    });
                    
                    $groupItems.append($card);
                });
                
                $group.append($groupItems);
                $grid.append($group);
            });
        },
        
        /**
         * Filter saved categories based on search term
         */
        filterSavedCategories: function(searchTerm) {
            var self = this;
            searchTerm = searchTerm.toLowerCase();
            
            if (!searchTerm) {
                this.renderSavedCategoriesGrid(this.allSavedCategories);
                $('.category-count').text(this.allSavedCategories.length + ' saved categories');
                return;
            }
            
            var filtered = $.grep(this.allSavedCategories, function(category) {
                return category.category_name.toLowerCase().includes(searchTerm) ||
                       category.category_id.toString().includes(searchTerm) ||
                       (category.parent_path && category.parent_path.toLowerCase().includes(searchTerm));
            });
            
            this.renderSavedCategoriesGrid(filtered);
            $('.category-count').text(filtered.length + ' of ' + this.allSavedCategories.length + ' categories');
        },
        
        /**
         * Show selection feedback
         */
        showSelectionFeedback: function(categoryName) {
            var $feedback = $('<div class="selection-feedback">✓ Selected: ' + this.escapeHtml(categoryName) + '</div>');
            $('#saved-categories-modal .delcampe-modal-body').append($feedback);
            
            setTimeout(function() {
                $feedback.fadeOut(function() {
                    $(this).remove();
                });
            }, 2000);
        },
        
        /**
         * Show error feedback
         */
        showErrorFeedback: function(message) {
            var $feedback = $('<div class="error-feedback">✕ ' + this.escapeHtml(message) + '</div>');
            $('#saved-categories-modal .delcampe-modal-body').append($feedback);
            
            setTimeout(function() {
                $feedback.fadeOut(function() {
                    $(this).remove();
                });
            }, 3000);
        },
        
        /**
         * Truncate long paths
         */
        truncatePath: function(path, maxLength) {
            if (path.length <= maxLength) return this.escapeHtml(path);
            
            var parts = path.split(' > ');
            var result = parts[0];
            
            for (var i = 1; i < parts.length; i++) {
                var testResult = result + ' > ' + parts[i];
                if (testResult.length > maxLength && i > 1) {
                    return this.escapeHtml(result + ' > ... > ' + parts[parts.length - 1]);
                }
                result = testResult;
            }
            
            return this.escapeHtml(result);
        },
        
        /**
         * Escape HTML for security
         */
        escapeHtml: function(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        },
        
        /**
         * Update breadcrumb navigation
         */
        updateBreadcrumb: function() {
            var self = this;
            var $breadcrumb = $('#delcampe-breadcrumb');
            
            $breadcrumb.empty();
            
            // Add root link
            var $root = $('<a class="delcampe-breadcrumb-item">Root</a>');
            $root.on('click', function() {
                self.breadcrumb = [];
                self.loadCategories('');
                self.updateBreadcrumb();
            });
            $breadcrumb.append($root);
            
            // Add breadcrumb items
            $.each(this.breadcrumb, function(index, item) {
                $breadcrumb.append('<span class="delcampe-breadcrumb-separator">→</span>');
                
                var $item = $('<a class="delcampe-breadcrumb-item"></a>').text(item.name);
                
                // Click handler to go back to this level
                $item.on('click', function() {
                    self.breadcrumb = self.breadcrumb.slice(0, index + 1);
                    self.loadCategories(item.id);
                    self.updateBreadcrumb();
                });
                
                $breadcrumb.append($item);
            });
        },
        
        /**
         * Select category and close modal
         */
        selectCategory: function() {
            if (!this.selectedCategory) {
                return;
            }
            
            // Update hidden fields
            $('#delcampe_category_id').val(this.selectedCategory.id);
            $('#delcampe_category_name').val(this.selectedCategory.name);
            
            // Build and store category path
            var path = this.breadcrumb.map(function(item) {
                return item.name;
            }).join(' > ');
            if (path) {
                path += ' > ';
            }
            path += this.selectedCategory.name;
            $('#delcampe_category_path').val(path);
            
            // Update display
            var $display = $('#delcampe-category-display');
            $display.html(
                '<div class="delcampe-category-selected">' +
                '<strong>' + this.selectedCategory.name + '</strong>' +
                '<br><small>' + path + '</small>' +
                '</div>'
            );
            
            // Close modal
            this.closeCategoryModal();
        },
        
        /**
         * Toggle quantity mode fields
         */
        toggleQuantityMode: function() {
            var mode = $('#quantity_mode').val();
            if (mode === 'fixed') {
                $('#fixed-quantity-wrapper').show();
            } else {
                $('#fixed-quantity-wrapper').hide();
            }
        },
        
        /**
         * Toggle auto sync fields
         */
        toggleAutoSync: function() {
            var enabled = $('#auto_sync').is(':checked');
            if (enabled) {
                $('#sync-interval-wrapper').show();
            } else {
                $('#sync-interval-wrapper').hide();
            }
        }
    };
    
    // Initialize on document ready
    $(document).ready(function() {
        console.log('[Delcampe Profiles Admin] Document ready');
        
        // Check if we're on the profiles page
        if ($('#delcampe-profile-form').length) {
            console.log('[Delcampe Profiles Admin] Profile form found, initializing...');
            DelcampeProfilesAdmin.init();
        } else {
            console.log('[Delcampe Profiles Admin] Profile form not found');
        }
    });
    
})(jQuery);
