/**
 * Delcampe Orders Admin JavaScript
 * 
 * Handles AJAX operations and UI interactions for order management
 * 
 * @version 1.4.0.0
 */

(function($) {
    'use strict';

    /**
     * Orders Admin Manager
     */
    var DelcampeOrdersAdmin = {
        
        /**
         * Initialize
         */
        init: function() {
            this.bindEvents();
        },
        
        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Import orders button
            $('#import-orders-btn').on('click', this.handleImportOrders.bind(this));
            
            // Update pending orders button (both locations)
            $('#update-pending-orders-btn, #update-orders-btn-main').on('click', this.handleUpdatePendingOrders.bind(this));
            
            // Sync closed sales button
            $('#sync-closed-sales-btn').on('click', this.handleSyncClosedSales.bind(this));
            
            // Cleanup listings button
            $('#cleanup-listings-btn').on('click', this.handleCleanupListings.bind(this));
            
            // Sync order buttons
            $(document).on('click', '.sync-order', this.handleSyncOrder.bind(this));
            
            // View details buttons
            $(document).on('click', '.view-details', this.handleViewDetails.bind(this));
            
            // Close modal
            $(document).on('click', '.order-details-close', this.closeDetailsModal.bind(this));
        },
        
        /**
         * Handle import orders
         */
        handleImportOrders: function(e) {
            e.preventDefault();
            
            if (!confirm(delcampe_orders.strings.confirm_import)) {
                return;
            }
            
            var $button = $(e.target);
            var $progress = $('#import-progress');
            var $results = $('#import-results');
            var limit = $('#import-limit').val();
            
            // Disable button and show progress
            $button.prop('disabled', true);
            $progress.show();
            $results.hide().empty();
            
            // Make AJAX request
            $.ajax({
                url: delcampe_orders.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_import_orders',
                    nonce: delcampe_orders.nonce,
                    limit: limit
                },
                success: function(response) {
                    if (response.success) {
                        $results.html(
                            '<div class="notice notice-success"><p>' + 
                            response.data.message + 
                            '</p></div>'
                        ).show();
                        
                        // Reload page after 2 seconds to show new orders
                        setTimeout(function() {
                            window.location.reload();
                        }, 2000);
                    } else {
                        $results.html(
                            '<div class="notice notice-error"><p>' + 
                            (response.data.message || delcampe_orders.strings.import_error) + 
                            '</p></div>'
                        ).show();
                    }
                },
                error: function() {
                    $results.html(
                        '<div class="notice notice-error"><p>' + 
                        delcampe_orders.strings.import_error + 
                        '</p></div>'
                    ).show();
                },
                complete: function() {
                    $button.prop('disabled', false);
                    $progress.hide();
                }
            });
        },
        
        /**
         * Handle update pending orders
         */
        handleUpdatePendingOrders: function(e) {
            e.preventDefault();
            
            var $button = $(e.target);
            var isMainPage = $button.attr('id') === 'update-orders-btn-main';
            var $progress = isMainPage ? $('<div class="notice notice-info"><p>Updating orders...</p></div>') : $('#import-progress');
            var $results = isMainPage ? $('<div></div>') : $('#import-results');
            
            // Disable button and show progress
            $button.prop('disabled', true);
            
            if (isMainPage) {
                $button.after($progress);
            } else {
                $progress.show();
                $results.hide().empty();
                $('.import-message').text('Updating pending orders...');
            }
            
            // Make AJAX request
            $.ajax({
                url: delcampe_orders.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_update_pending_orders',
                    nonce: delcampe_orders.nonce
                },
                success: function(response) {
                    if (response.success) {
                        var successMsg = '<div class="notice notice-success"><p>' + 
                            response.data.message + 
                            '</p></div>';
                        
                        if (isMainPage) {
                            $progress.replaceWith(successMsg);
                        } else {
                            $results.html(successMsg).show();
                        }
                        
                        // Reload page after 2 seconds to show updated orders
                        if (response.data.data && response.data.data.updated > 0) {
                            setTimeout(function() {
                                window.location.reload();
                            }, 2000);
                        }
                    } else {
                        var errorMsg = '<div class="notice notice-error"><p>' + 
                            (response.data.message || 'Error updating orders.') + 
                            '</p></div>';
                        
                        if (isMainPage) {
                            $progress.replaceWith(errorMsg);
                        } else {
                            $results.html(errorMsg).show();
                        }
                    }
                },
                error: function() {
                    var errorMsg = '<div class="notice notice-error"><p>Error updating pending orders.</p></div>';
                    
                    if (isMainPage) {
                        $progress.replaceWith(errorMsg);
                    } else {
                        $results.html(errorMsg).show();
                    }
                },
                complete: function() {
                    $button.prop('disabled', false);
                    if (!isMainPage) {
                        $progress.hide();
                        $('.import-message').text(delcampe_orders.strings.importing);
                    }
                }
            });
        },
        
        /**
         * Handle sync closed sales
         */
        handleSyncClosedSales: function(e) {
            e.preventDefault();
            
            var $button = $(e.target);
            var $progress = $('#closed-sales-progress');
            var $results = $('#closed-sales-results');
            var limit = $('#closed-sales-limit').val();
            
            // Disable button and show progress
            $button.prop('disabled', true);
            $progress.show();
            $results.hide().empty();
            
            // Make AJAX request
            $.ajax({
                url: delcampe_orders.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_sync_closed_sales',
                    nonce: delcampe_orders.nonce,
                    limit: limit
                },
                success: function(response) {
                    if (response.success) {
                        $results.html(
                            '<div class="notice notice-success"><p>' + 
                            response.data.message + 
                            '</p></div>'
                        ).show();
                        
                        // Reload page after 2 seconds to show updated orders
                        setTimeout(function() {
                            window.location.href = '?page=delcampe-orders&tab=orders';
                        }, 2000);
                    } else {
                        $results.html(
                            '<div class="notice notice-error"><p>' + 
                            (response.data.message || 'Error syncing closed sales.') + 
                            '</p></div>'
                        ).show();
                    }
                },
                error: function() {
                    $results.html(
                        '<div class="notice notice-error"><p>' + 
                        'Error syncing closed sales. Please try again.' + 
                        '</p></div>'
                    ).show();
                },
                complete: function() {
                    $button.prop('disabled', false);
                    $progress.hide();
                }
            });
        },
        
        /**
         * Handle cleanup listings
         */
        handleCleanupListings: function(e) {
            e.preventDefault();
            
            if (!confirm('Are you sure you want to cleanup all listings? This will check all your listings against Delcampe and mark any that are no longer active as ended.')) {
                return;
            }
            
            var $button = $(e.target);
            var $progress = $('#cleanup-progress');
            var $results = $('#cleanup-results');
            
            // Disable button and show progress
            $button.prop('disabled', true);
            $progress.show();
            $results.hide().empty();
            
            // Make AJAX request
            $.ajax({
                url: delcampe_orders.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_cleanup_listings',
                    nonce: delcampe_orders.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $results.html(
                            '<div class="notice notice-success"><p>' + 
                            response.data.message + 
                            '</p></div>'
                        ).show();
                        
                        // Reload page after 3 seconds to show updated listings
                        setTimeout(function() {
                            window.location.href = '?page=delcampe-listings';
                        }, 3000);
                    } else {
                        $results.html(
                            '<div class="notice notice-error"><p>' + 
                            (response.data.message || 'Error cleaning up listings.') + 
                            '</p></div>'
                        ).show();
                    }
                },
                error: function() {
                    $results.html(
                        '<div class="notice notice-error"><p>' + 
                        'Error cleaning up listings. Please try again.' + 
                        '</p></div>'
                    ).show();
                },
                complete: function() {
                    $button.prop('disabled', false);
                    $progress.hide();
                }
            });
        },
        
        /**
         * Handle sync single order
         */
        handleSyncOrder: function(e) {
            e.preventDefault();
            
            var $button = $(e.target);
            var orderId = $button.data('order-id');
            var originalText = $button.text();
            
            // Update button state
            $button.text(delcampe_orders.strings.syncing).prop('disabled', true);
            
            // Make AJAX request
            $.ajax({
                url: delcampe_orders.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_sync_order',
                    nonce: delcampe_orders.nonce,
                    order_id: orderId
                },
                success: function(response) {
                    if (response.success) {
                        // Show success message
                        $button.text(delcampe_orders.strings.sync_complete)
                               .addClass('button-primary');
                        
                        // Reload row after delay
                        setTimeout(function() {
                            window.location.reload();
                        }, 1500);
                    } else {
                        alert(response.data.message || delcampe_orders.strings.sync_error);
                        $button.text(originalText).prop('disabled', false);
                    }
                },
                error: function() {
                    alert(delcampe_orders.strings.sync_error);
                    $button.text(originalText).prop('disabled', false);
                }
            });
        },
        
        /**
         * Handle view order details
         */
        handleViewDetails: function(e) {
            e.preventDefault();
            
            var $button = $(e.target);
            var orderId = $button.data('order-id');
            
            // Show loading
            this.showLoading();
            
            // Make AJAX request
            $.ajax({
                url: delcampe_orders.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_get_order_details',
                    nonce: delcampe_orders.nonce,
                    order_id: orderId
                },
                success: function(response) {
                    if (response.success) {
                        DelcampeOrdersAdmin.showOrderDetails(response.data);
                    } else {
                        alert(response.data.message || 'Error loading order details');
                    }
                },
                error: function() {
                    alert('Error loading order details');
                },
                complete: function() {
                    DelcampeOrdersAdmin.hideLoading();
                }
            });
        },
        
        /**
         * Show order details in modal
         */
        showOrderDetails: function(data) {
            var order = data.order;
            var wcOrder = data.wc_order;
            
            // Build modal content
            var modalHtml = '<div class="order-details-modal" id="order-details-modal">' +
                '<div class="order-details-content">' +
                '<span class="order-details-close">&times;</span>' +
                '<h2>Order Details</h2>' +
                '<table class="form-table">' +
                '<tr><th>Delcampe Order ID:</th><td>' + order.delcampe_order_id + '</td></tr>' +
                '<tr><th>Buyer:</th><td>' + order.buyer_nickname + ' (' + order.buyer_email + ')</td></tr>' +
                '<tr><th>Sale Price:</th><td>' + order.sale_price + ' ' + order.currency + '</td></tr>' +
                '<tr><th>Sale Date:</th><td>' + order.sale_date + '</td></tr>' +
                '<tr><th>Payment Status:</th><td>' + order.payment_status + '</td></tr>' +
                '<tr><th>Shipping Status:</th><td>' + order.shipping_status + '</td></tr>' +
                '<tr><th>Sync Status:</th><td>' + order.sync_status + '</td></tr>';
            
            if (wcOrder) {
                modalHtml += '<tr><th>WooCommerce Order:</th><td><a href="' + wcOrder.view_url + '" target="_blank">#' + wcOrder.id + '</a></td></tr>';
                modalHtml += '<tr><th>WC Status:</th><td>' + wcOrder.status + '</td></tr>';
            }
            
            modalHtml += '</table></div></div>';
            
            // Add modal to page
            $('body').append(modalHtml);
            
            // Show modal
            $('#order-details-modal').show();
        },
        
        /**
         * Close details modal
         */
        closeDetailsModal: function() {
            $('#order-details-modal').remove();
        },
        
        /**
         * Show loading overlay
         */
        showLoading: function() {
            $('body').append(
                '<div class="delcampe-loading-overlay">' +
                '<div class="spinner is-active"></div>' +
                '</div>'
            );
        },
        
        /**
         * Hide loading overlay
         */
        hideLoading: function() {
            $('.delcampe-loading-overlay').remove();
        }
    };
    
    // Initialize when document is ready
    $(document).ready(function() {
        DelcampeOrdersAdmin.init();
    });
    
})(jQuery);
