<?php
/**
 * Delcampe API Client (lightweight wrapper)
 *
 * Provides a minimal facade used by legacy/admin templates expecting
 * an array-based category structure rather than raw XML.
 *
 * Version: 1.0.0
 * @package WooCommerce_Delcampe_Integration
 */

if ( ! defined( 'ABSPATH' ) ) exit;

class Delcampe_API_Client {
    private static $instance = null;

    /**
     * Singleton accessor
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Get root categories as an array of [id, name]
     * Falls back to empty array on error; caller can show notice.
     *
     * @param string $lang E=English, F=French, N=Dutch
     * @param int $show_only_leaves 1 to return only leaf categories
     * @return array<int, array{id:int,name:string}>
     */
    public function get_categories( $lang = 'E', $show_only_leaves = 0 ) {
        if ( ! class_exists( 'Delcampe_Category_API' ) ) {
            return array();
        }

        $xml = Delcampe_Category_API::get_categories( null, $lang, $show_only_leaves );

        if ( is_wp_error( $xml ) || empty( $xml ) ) {
            return array();
        }

        $out = array();
        if ( isset( $xml->category ) ) {
            foreach ( $xml->category as $cat ) {
                $id = 0;
                if ( isset( $cat->id_category ) ) {
                    $id = (int) $cat->id_category;
                } elseif ( isset( $cat->id ) ) {
                    $id = (int) $cat->id;
                }
                $name = isset( $cat->name ) ? (string) $cat->name : '';
                if ( $id && $name !== '' ) {
                    $out[] = array( 'id' => $id, 'name' => $name );
                }
            }
        }

        return $out;
    }

    /**
     * Get subcategories for a parent category.
     * Returns same array format as get_categories().
     *
     * @param int $parent_id
     * @param string $lang
     * @param int $show_only_leaves
     * @return array
     */
    public function get_subcategories( $parent_id, $lang = 'E', $show_only_leaves = 0 ) {
        if ( ! class_exists( 'Delcampe_Category_API' ) ) {
            return array();
        }

        $xml = Delcampe_Category_API::get_categories( (int) $parent_id, $lang, $show_only_leaves );
        if ( is_wp_error( $xml ) || empty( $xml ) ) {
            return array();
        }

        $out = array();
        if ( isset( $xml->category ) ) {
            foreach ( $xml->category as $cat ) {
                $id = 0;
                if ( isset( $cat->id_category ) ) {
                    $id = (int) $cat->id_category;
                } elseif ( isset( $cat->id ) ) {
                    $id = (int) $cat->id;
                }
                $name = isset( $cat->name ) ? (string) $cat->name : '';
                if ( $id && $name !== '' ) {
                    $out[] = array( 'id' => $id, 'name' => $name );
                }
            }
        }
        return $out;
    }
}
