<?php
/**
 * Database Upgrade Script for Delcampe Integration
 * Version: 1.1.4.0
 * 
 * Handles database schema updates for the plugin
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Database
 * @since 1.0.8.0
 * @version 1.1.4.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_DB_Upgrade
 * 
 * Manages database schema updates and migrations
 * Enhanced in v1.1.4.0 to handle profiles table upgrades
 */
class Delcampe_DB_Upgrade {
    
    /**
     * Current database version
     */
    const DB_VERSION = '1.10.20.3';
    
    /**
     * Option name for storing database version
     */
    const VERSION_OPTION = 'delcampe_db_version';
    
    /**
     * Run database upgrades if needed
     */
    public static function run_upgrades() {
        $current_version = get_option(self::VERSION_OPTION, '0');
        
        // Check if upgrade is needed
        if (version_compare($current_version, self::DB_VERSION, '<')) {
            // Run version-specific upgrades
            if (version_compare($current_version, '1.0.8.0', '<')) {
                self::upgrade_to_1_0_8_0();
            }
            if (version_compare($current_version, '1.1.4.0', '<')) {
                self::upgrade_to_1_1_4_0();
            }
            if (version_compare($current_version, '1.10.20.3', '<')) {
                self::upgrade_to_1_10_20_3();
            }
            
            update_option(self::VERSION_OPTION, self::DB_VERSION);
        }
    }
    
    /**
     * Upgrade database to version 1.0.8.0
     * 
     * Adds new columns for enhanced category management:
     * - category_name: Store the category name
     * - parent_path: Store the full hierarchy path
     * - import_method: Track how the category was imported (manual/branch)
     * - depth: Track the depth in the category tree
     */
    private static function upgrade_to_1_0_8_0() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_selected_categories';
        
        // SECURITY FIX v1.10.21.0: Use prepared statement to prevent SQL injection
        // Check if table exists
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;
        
        if ($table_exists) {
            // Add new columns if they don't exist
            
            // Add category_name column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'category_name'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN category_name VARCHAR(255) DEFAULT NULL AFTER category_id");
            }
            
            // Add parent_path column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'parent_path'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN parent_path TEXT DEFAULT NULL AFTER category_name");
            }
            
            // Add import_method column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'import_method'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN import_method VARCHAR(20) DEFAULT 'manual' AFTER parent_path");
            }
            
            // Add depth column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'depth'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN depth INT DEFAULT 0 AFTER import_method");
            }
            
            // Add unique constraint on category_id if not exists
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $index_exists = $wpdb->get_var("SHOW INDEX FROM `$safe_table` WHERE Key_name = 'unique_category'");
            if (!$index_exists) {
                // First remove any duplicates
                $wpdb->query("
                    DELETE c1 FROM `$safe_table` c1
                    INNER JOIN `$safe_table` c2 
                    WHERE c1.id > c2.id 
                    AND c1.category_id = c2.category_id
                ");
                
                // Then add unique constraint
                $wpdb->query("ALTER TABLE `$safe_table` ADD UNIQUE KEY unique_category (category_id)");
            }
            
            error_log('[Delcampe DB Upgrade] Successfully upgraded table to version 1.0.8.0');
        }
    }
    
    /**
     * Upgrade database to version 1.1.4.0
     * 
     * Adds new columns for stamp-specific settings to profiles table:
     * - stamp_settings: JSON field for stamp-specific defaults
     * - shipping_settings: JSON field for shipping configuration
     * - payment_settings: JSON field for payment options
     * - description_settings: JSON field for description templates
     * 
     * @since 1.1.4.0
     */
    private static function upgrade_to_1_1_4_0() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_profiles';
        
        // SECURITY FIX v1.10.21.0: Use prepared statement to prevent SQL injection
        // Check if table exists
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;
        
        if ($table_exists) {
            // Add stamp_settings column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'stamp_settings'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN stamp_settings LONGTEXT DEFAULT NULL AFTER sync_settings");
                error_log('[Delcampe DB Upgrade v1.1.4.0] Added stamp_settings column');
            }
            
            // Add shipping_settings column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'shipping_settings'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN shipping_settings LONGTEXT DEFAULT NULL AFTER stamp_settings");
                error_log('[Delcampe DB Upgrade v1.1.4.0] Added shipping_settings column');
            }
            
            // Add payment_settings column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'payment_settings'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN payment_settings LONGTEXT DEFAULT NULL AFTER shipping_settings");
                error_log('[Delcampe DB Upgrade v1.1.4.0] Added payment_settings column');
            }
            
            // Add description_settings column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'description_settings'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN description_settings LONGTEXT DEFAULT NULL AFTER payment_settings");
                error_log('[Delcampe DB Upgrade v1.1.4.0] Added description_settings column');
            }
            
            // Update existing profiles with default values for new fields
            $existing_profiles = $wpdb->get_results("SELECT profile_id FROM $table_name");
            
            if (!empty($existing_profiles)) {
                // Load stamp fields helper to get defaults
                if (file_exists(plugin_dir_path(__FILE__) . 'profiles/stamp-profile-fields.php')) {
                    require_once plugin_dir_path(__FILE__) . 'profiles/stamp-profile-fields.php';
                    $default_fields = delcampe_get_stamp_profile_fields();
                    
                    foreach ($existing_profiles as $profile) {
                        $update_data = array();
                        
                        // Check if fields are empty and set defaults
                        $stamp_settings = $wpdb->get_var($wpdb->prepare(
                            "SELECT stamp_settings FROM $table_name WHERE profile_id = %d",
                            $profile->profile_id
                        ));
                        
                        if (empty($stamp_settings) || $stamp_settings === 'null') {
                            $update_data['stamp_settings'] = wp_json_encode($default_fields['stamp_settings'] ?? array());
                        }
                        
                        $shipping_settings = $wpdb->get_var($wpdb->prepare(
                            "SELECT shipping_settings FROM $table_name WHERE profile_id = %d",
                            $profile->profile_id
                        ));
                        
                        if (empty($shipping_settings) || $shipping_settings === 'null') {
                            $update_data['shipping_settings'] = wp_json_encode($default_fields['shipping_settings'] ?? array());
                        }
                        
                        $payment_settings = $wpdb->get_var($wpdb->prepare(
                            "SELECT payment_settings FROM $table_name WHERE profile_id = %d",
                            $profile->profile_id
                        ));
                        
                        if (empty($payment_settings) || $payment_settings === 'null') {
                            $update_data['payment_settings'] = wp_json_encode($default_fields['payment_settings'] ?? array());
                        }
                        
                        $description_settings = $wpdb->get_var($wpdb->prepare(
                            "SELECT description_settings FROM $table_name WHERE profile_id = %d",
                            $profile->profile_id
                        ));
                        
                        if (empty($description_settings) || $description_settings === 'null') {
                            $update_data['description_settings'] = wp_json_encode($default_fields['description_settings'] ?? array());
                        }
                        
                        // Update profile if we have data to update
                        if (!empty($update_data)) {
                            $wpdb->update(
                                $table_name,
                                $update_data,
                                array('profile_id' => $profile->profile_id)
                            );
                        }
                    }
                    
                    error_log('[Delcampe DB Upgrade v1.1.4.0] Updated existing profiles with default values');
                }
            }
            
            error_log('[Delcampe DB Upgrade v1.1.4.0] Successfully upgraded profiles table');
        }
    }
    
    /**
     * Upgrade database to version 1.10.20.3
     * 
     * Adds stock caching columns to listings table for performance optimization:
     * - delcampe_stock_cached: Cached stock value from Delcampe
     * - delcampe_stock_checked_at: Timestamp of last stock check
     * 
     * @since 1.10.20.3
     */
    private static function upgrade_to_1_10_20_3() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        // SECURITY FIX v1.10.21.0: Use prepared statement to prevent SQL injection
        // Check if table exists
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;
        
        if ($table_exists) {
            // Add delcampe_stock_cached column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'delcampe_stock_cached'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN delcampe_stock_cached INT NULL DEFAULT NULL AFTER quantity");
                error_log('[Delcampe DB Upgrade v1.10.20.3] Added delcampe_stock_cached column');
            }
            
            // Add delcampe_stock_checked_at column
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $column_exists = $wpdb->get_var("SHOW COLUMNS FROM `$safe_table` LIKE 'delcampe_stock_checked_at'");
            if (!$column_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD COLUMN delcampe_stock_checked_at DATETIME NULL DEFAULT NULL AFTER delcampe_stock_cached");
                error_log('[Delcampe DB Upgrade v1.10.20.3] Added delcampe_stock_checked_at column');
            }
            
            // Add index on delcampe_stock_checked_at for efficient cache refresh queries
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $index_exists = $wpdb->get_var("SHOW INDEX FROM `$safe_table` WHERE Key_name = 'idx_stock_freshness'");
            if (!$index_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD INDEX idx_stock_freshness (status, delcampe_stock_checked_at)");
                error_log('[Delcampe DB Upgrade v1.10.20.3] Added idx_stock_freshness index');
            }
            
            // Add index on delcampe_stock_cached for mismatch queries
            // SECURITY FIX v1.10.21.0: Table name sanitized for SQL safety
            $safe_table = esc_sql($table_name);
            $index_exists = $wpdb->get_var("SHOW INDEX FROM `$safe_table` WHERE Key_name = 'idx_stock_cached'");
            if (!$index_exists) {
                $wpdb->query("ALTER TABLE `$safe_table` ADD INDEX idx_stock_cached (delcampe_stock_cached)");
                error_log('[Delcampe DB Upgrade v1.10.20.3] Added idx_stock_cached index');
            }
            
            error_log('[Delcampe DB Upgrade v1.10.20.3] Successfully upgraded listings table for stock caching');
        } else {
            error_log('[Delcampe DB Upgrade v1.10.20.3] Warning: wp_delcampe_listings table not found');
        }
    }
}

// Hook into admin_init to run upgrades
add_action('admin_init', array('Delcampe_DB_Upgrade', 'run_upgrades'));
