<?php
/**
 * Delcampe Batch Queue Monitor
 * 
 * Provides admin interface for monitoring and managing batch queue
 * 
 * @package WC_Delcampe_Integration
 * @since 1.10.18.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Batch_Monitor {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Batch queue instance
     */
    private $batch_queue;
    
    /**
     * Constructor
     */
    private function __construct() {
        // Load batch queue
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-batch-queue.php';
        $this->batch_queue = Delcampe_Batch_Queue::get_instance();
        
        // v1.10.24.0 - Removed standalone menu, now integrated in Inventory Management
        // add_action('admin_menu', array($this, 'add_admin_menu'), 15);
        
        // AJAX handlers
        add_action('wp_ajax_delcampe_get_batch_status', array($this, 'ajax_get_batch_status'));
        add_action('wp_ajax_delcampe_get_queue_stats', array($this, 'ajax_get_queue_stats'));
        add_action('wp_ajax_delcampe_cancel_batch', array($this, 'ajax_cancel_batch'));
        add_action('wp_ajax_delcampe_retry_batch', array($this, 'ajax_retry_batch'));
        add_action('wp_ajax_delcampe_force_unlock_batch', array($this, 'ajax_force_unlock_batch'));
        add_action('wp_ajax_delcampe_recover_stuck_batches', array($this, 'ajax_recover_stuck_batches'));
        add_action('wp_ajax_delcampe_pause_processing', array($this, 'ajax_pause_processing'));
        add_action('wp_ajax_delcampe_resume_processing', array($this, 'ajax_resume_processing'));
        
        // Enqueue scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Add admin menu
     * 
     * @deprecated 1.10.24.0 Moved to Inventory Management tab
     */
    public function add_admin_menu() {
        // Removed in v1.10.24.0 - Batch Queue is now a tab in Inventory Management
    }
    
    /**
     * Enqueue scripts
     */
    public function enqueue_scripts($hook) {
        // v1.10.24.0 - Also load on inventory page when batch-queue tab is active
        if (strpos($hook, 'delcampe-batch-queue') === false && 
            !(strpos($hook, 'delcampe-inventory') !== false && isset($_GET['tab']) && $_GET['tab'] === 'batch-queue')) {
            return;
        }
        
        wp_enqueue_script(
            'delcampe-batch-monitor',
            plugin_dir_url(dirname(__FILE__)) . 'assets/js/batch-monitor.js',
            array('jquery'),
            defined('DELCAMPE_PLUGIN_VERSION') ? DELCAMPE_PLUGIN_VERSION : time(),
            true
        );
        
        wp_localize_script('delcampe-batch-monitor', 'delcampeBatchMonitor', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('delcampe_batch_monitor'),
            'admin_nonce' => wp_create_nonce('delcampe_admin_nonce'),
            'refresh_interval' => 5000, // Refresh every 5 seconds
            'strings' => array(
                'confirm_cancel' => __('Are you sure you want to cancel this batch?', 'wc-delcampe-integration'),
                'confirm_retry' => __('Are you sure you want to retry this batch?', 'wc-delcampe-integration'),
                'processing_paused' => __('Processing paused successfully', 'wc-delcampe-integration'),
                'processing_resumed' => __('Processing resumed successfully', 'wc-delcampe-integration'),
                'run_sync_now' => __('Run Sync Queue Now', 'wc-delcampe-integration'),
                'clear_lock' => __('Sync lock cleared and run triggered', 'wc-delcampe-integration')
            )
        ));
        
        wp_enqueue_style(
            'delcampe-batch-monitor',
            plugin_dir_url(dirname(__FILE__)) . 'assets/css/batch-monitor.css',
            array(),
            defined('DELCAMPE_PLUGIN_VERSION') ? DELCAMPE_PLUGIN_VERSION : time()
        );
    }
    
    /**
     * Render admin page (standalone)
     * 
     * @deprecated 1.10.24.0 Use render_batch_queue_content() for tab integration
     */
    public function render_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Batch Queue Monitor', 'wc-delcampe-integration'); ?></h1>
            <?php $this->render_batch_queue_content(); ?>
        </div>
        <?php
    }
    
    /**
     * Render batch queue content (for tab integration)
     * 
     * @since 1.10.24.0
     */
    public function render_batch_queue_content() {
        // Get current stats
        $stats = $this->batch_queue->get_queue_stats();
        $is_paused = get_transient('delcampe_batch_processing_paused');
        
        ?>
        <!-- Queue Controls -->
        <div class="batch-queue-controls">
            <button type="button" class="button" id="refresh-queue-stats">
                <?php esc_html_e('Refresh', 'wc-delcampe-integration'); ?>
            </button>

            <button type="button" class="button" id="run-sync-now">
                <?php esc_html_e('Run Sync Queue Now', 'wc-delcampe-integration'); ?>
            </button>
            
            <?php if ($is_paused): ?>
                <button type="button" class="button button-primary" id="resume-processing">
                    <?php esc_html_e('Resume Processing', 'wc-delcampe-integration'); ?>
                </button>
            <?php else: ?>
                <button type="button" class="button" id="pause-processing">
                    <?php esc_html_e('Pause Processing', 'wc-delcampe-integration'); ?>
                </button>
            <?php endif; ?>
            
            <label>
                <input type="checkbox" id="auto-refresh" checked>
                <?php esc_html_e('Auto-refresh', 'wc-delcampe-integration'); ?>
            </label>
        </div>
        
        <!-- Queue Statistics -->
        <div class="batch-queue-stats">
            <h2><?php esc_html_e('Queue Statistics', 'wc-delcampe-integration'); ?></h2>
            <div id="queue-stats-container">
                <?php $this->render_stats($stats); ?>
            </div>
        </div>
        
        <!-- Active Batches -->
        <div class="batch-queue-active">
            <h2><?php esc_html_e('Active Batches', 'wc-delcampe-integration'); ?></h2>
            <div id="active-batches-container">
                <?php $this->render_active_batches(); ?>
            </div>
        </div>
        
        <!-- Recent Batches -->
        <div class="batch-queue-recent">
            <h2><?php esc_html_e('Recent Batches', 'wc-delcampe-integration'); ?></h2>
            <div id="recent-batches-container">
                <?php $this->render_recent_batches(); ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render statistics
     */
    private function render_stats($stats) {
        $status_labels = array(
            'pending' => __('Pending', 'wc-delcampe-integration'),
            'queued' => __('Queued', 'wc-delcampe-integration'),
            'processing' => __('Processing', 'wc-delcampe-integration'),
            'completed' => __('Completed', 'wc-delcampe-integration'),
            'failed' => __('Failed', 'wc-delcampe-integration'),
            'retry' => __('Retry', 'wc-delcampe-integration'),
            'cancelled' => __('Cancelled', 'wc-delcampe-integration')
        );
        
        ?>
        <table class="widefat">
            <thead>
                <tr>
                    <th><?php esc_html_e('Status', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Count (24h)', 'wc-delcampe-integration'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($status_labels as $status => $label): ?>
                    <tr>
                        <td><?php echo esc_html($label); ?></td>
                        <td><?php echo esc_html($stats[$status] ?? 0); ?></td>
                    </tr>
                <?php endforeach; ?>
                <tr>
                    <td><strong><?php esc_html_e('Queue Length', 'wc-delcampe-integration'); ?></strong></td>
                    <td><strong><?php echo esc_html($stats['queue_length'] ?? 0); ?></strong></td>
                </tr>
                <tr>
                    <td><?php esc_html_e('Avg. Processing Time', 'wc-delcampe-integration'); ?></td>
                    <td><?php echo $this->format_duration($stats['avg_processing_time'] ?? 0); ?></td>
                </tr>
            </tbody>
        </table>
        <?php
    }
    
    /**
     * Render active batches
     */
    private function render_active_batches() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_batch_queue';
        $active_batches = $wpdb->get_results("
            SELECT * FROM {$table_name}
            WHERE status IN ('processing', 'queued', 'retry')
            ORDER BY 
                CASE status 
                    WHEN 'processing' THEN 0 
                    WHEN 'queued' THEN 1 
                    ELSE 2 
                END,
                queue_position ASC,
                created_at DESC
            LIMIT 10
        ");
        
        if (empty($active_batches)) {
            echo '<p>' . esc_html__('No active batches', 'wc-delcampe-integration') . '</p>';
            return;
        }
        
        ?>
        <table class="widefat">
            <thead>
                <tr>
                    <th><?php esc_html_e('Batch ID', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Status', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Queue Position', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Progress', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Created', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Actions', 'wc-delcampe-integration'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($active_batches as $batch): ?>
                    <tr data-batch-id="<?php echo esc_attr($batch->batch_id); ?>">
                        <td><?php echo esc_html($batch->batch_id); ?></td>
                        <td>
                            <span class="batch-status status-<?php echo esc_attr($batch->status); ?>">
                                <?php echo esc_html(ucfirst($batch->status)); ?>
                            </span>
                        </td>
                        <td><?php echo esc_html($batch->queue_position ?? '-'); ?></td>
                        <td>
                            <?php 
                            $progress = ($batch->total_count > 0) 
                                ? round(($batch->processed_count / $batch->total_count) * 100) 
                                : 0;
                            ?>
                            <div class="progress-bar">
                                <div class="progress-fill" style="width: <?php echo $progress; ?>%"></div>
                                <span class="progress-text">
                                    <?php echo esc_html($batch->processed_count . '/' . $batch->total_count); ?>
                                    (<?php echo $progress; ?>%)
                                </span>
                            </div>
                        </td>
                        <td><?php echo esc_html($this->format_time_ago($batch->created_at)); ?></td>
                        <td>
                            <button type="button" class="button button-small cancel-batch" 
                                    data-batch-id="<?php echo esc_attr($batch->batch_id); ?>">
                                <?php esc_html_e('Cancel', 'wc-delcampe-integration'); ?>
                            </button>
                            
                            <?php if ($batch->status === 'retry'): ?>
                                <button type="button" class="button button-small retry-batch" 
                                        data-batch-id="<?php echo esc_attr($batch->batch_id); ?>">
                                    <?php esc_html_e('Retry Now', 'wc-delcampe-integration'); ?>
                                </button>
                            <?php elseif ($batch->status === 'processing'): ?>
                                <button type="button" class="button button-small force-unlock-batch" 
                                        data-batch-id="<?php echo esc_attr($batch->batch_id); ?>">
                                    <?php esc_html_e('Force Unlock', 'wc-delcampe-integration'); ?>
                                </button>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }
    
    /**
     * Render recent batches
     */
    private function render_recent_batches() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_batch_queue';
        $recent_batches = $wpdb->get_results("
            SELECT * FROM {$table_name}
            WHERE status IN ('completed', 'failed', 'cancelled')
            ORDER BY updated_at DESC
            LIMIT 20
        ");
        
        if (empty($recent_batches)) {
            echo '<p>' . esc_html__('No recent batches', 'wc-delcampe-integration') . '</p>';
            return;
        }
        
        ?>
        <table class="widefat">
            <thead>
                <tr>
                    <th><?php esc_html_e('Batch ID', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Status', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Items', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Duration', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Completed', 'wc-delcampe-integration'); ?></th>
                    <th><?php esc_html_e('Error', 'wc-delcampe-integration'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($recent_batches as $batch): ?>
                    <tr>
                        <td><?php echo esc_html($batch->batch_id); ?></td>
                        <td>
                            <span class="batch-status status-<?php echo esc_attr($batch->status); ?>">
                                <?php echo esc_html(ucfirst($batch->status)); ?>
                            </span>
                        </td>
                        <td><?php echo esc_html($batch->total_count); ?></td>
                        <td>
                            <?php 
                            if ($batch->processing_started && $batch->processing_completed) {
                                $duration = strtotime($batch->processing_completed) - strtotime($batch->processing_started);
                                echo $this->format_duration($duration);
                            } else {
                                echo '-';
                            }
                            ?>
                        </td>
                        <td><?php echo esc_html($this->format_time_ago($batch->updated_at)); ?></td>
                        <td>
                            <?php if ($batch->last_error): ?>
                                <span class="error-message" title="<?php echo esc_attr($batch->last_error); ?>">
                                    <?php echo esc_html(substr($batch->last_error, 0, 50) . '...'); ?>
                                </span>
                            <?php else: ?>
                                -
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }
    
    /**
     * AJAX: Get batch status
     */
    public function ajax_get_batch_status() {
        check_ajax_referer('delcampe_batch_monitor', 'nonce');
        
        $batch_id = sanitize_text_field($_POST['batch_id'] ?? '');
        
        if (!$batch_id) {
            wp_send_json_error('Invalid batch ID');
        }
        
        $batch = $this->batch_queue->get_batch_status($batch_id);
        
        if (!$batch) {
            wp_send_json_error('Batch not found');
        }
        
        wp_send_json_success($batch);
    }
    
    /**
     * AJAX: Get queue statistics
     */
    public function ajax_get_queue_stats() {
        check_ajax_referer('delcampe_batch_monitor', 'nonce');
        
        $stats = $this->batch_queue->get_queue_stats();
        
        ob_start();
        $this->render_stats($stats);
        $stats_html = ob_get_clean();
        
        ob_start();
        $this->render_active_batches();
        $active_html = ob_get_clean();
        
        ob_start();
        $this->render_recent_batches();
        $recent_html = ob_get_clean();
        
        wp_send_json_success(array(
            'stats_html' => $stats_html,
            'active_html' => $active_html,
            'recent_html' => $recent_html,
            'is_paused' => (bool)get_transient('delcampe_batch_processing_paused')
        ));
    }
    
    /**
     * AJAX: Cancel batch
     */
    public function ajax_cancel_batch() {
        check_ajax_referer('delcampe_batch_monitor', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Permission denied');
        }
        
        $batch_id = sanitize_text_field($_POST['batch_id'] ?? '');
        
        if (!$batch_id) {
            wp_send_json_error('Invalid batch ID');
        }
        
        $result = $this->batch_queue->cancel_batch($batch_id);
        
        if ($result) {
            wp_send_json_success('Batch cancelled successfully');
        } else {
            wp_send_json_error('Failed to cancel batch');
        }
    }
    
    /**
     * AJAX: Retry batch
     */
    public function ajax_retry_batch() {
        check_ajax_referer('delcampe_batch_monitor', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Permission denied');
        }
        
        $batch_id = sanitize_text_field($_POST['batch_id'] ?? '');
        
        if (!$batch_id) {
            wp_send_json_error('Invalid batch ID');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_batch_queue';
        
        // Reset batch for immediate retry: set to queued, reset counters and clear last errors
        $result = $wpdb->update(
            $table_name,
            array(
                'status' => 'queued',
                'processed_count' => 0,
                'retry_count' => 0,
                'next_retry_at' => null,
                'last_error' => null,
                'error_details' => null,
                'queue_position' => 0 // Priority
            ),
            array('batch_id' => $batch_id)
        );

        if ($result !== false) {
            // Trigger processing immediately (Action Scheduler if present, else cron + direct)
            if (function_exists('as_enqueue_async_action')) {
                as_enqueue_async_action('delcampe_process_batch_queue', array(), 'delcampe');
            }
            wp_schedule_single_event(time(), 'delcampe_process_batch_queue');
            // Ensure batch processing is not paused
            delete_transient('delcampe_batch_processing_paused');
            do_action('delcampe_process_batch_queue');
            wp_send_json_success('Batch queued for retry');
        } else {
            wp_send_json_error('Failed to retry batch');
        }
    }

    /**
     * AJAX: Force unlock a processing batch (set back to queued)
     */
    public function ajax_force_unlock_batch() {
        check_ajax_referer('delcampe_batch_monitor', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Permission denied');
        }
        
        $batch_id = sanitize_text_field($_POST['batch_id'] ?? '');
        if (!$batch_id) {
            wp_send_json_error('Invalid batch ID');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_batch_queue';
        
        $result = $wpdb->update(
            $table_name,
            array(
                'status' => 'queued',
                'processing_started' => null,
                'processing_completed' => null,
                'processing_pid' => null,
                'next_retry_at' => null,
                'queue_position' => 0
            ),
            array('batch_id' => $batch_id)
        );
        
        if ($result !== false) {
            delete_transient('delcampe_batch_processing_paused');
            if (function_exists('as_enqueue_async_action')) {
                as_enqueue_async_action('delcampe_process_batch_queue', array(), 'delcampe');
            }
            wp_schedule_single_event(time(), 'delcampe_process_batch_queue');
            do_action('delcampe_process_batch_queue');
            wp_send_json_success('Batch unlocked and queued');
        } else {
            wp_send_json_error('Failed to unlock batch');
        }
    }

    /**
     * AJAX: Recover stuck batches (older than timeout)
     */
    public function ajax_recover_stuck_batches() {
        check_ajax_referer('delcampe_batch_monitor', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Permission denied');
        }
        
        $count = $this->batch_queue->recover_stuck_batches();
        if ($count > 0) {
            if (function_exists('as_enqueue_async_action')) {
                as_enqueue_async_action('delcampe_process_batch_queue', array(), 'delcampe');
            }
            wp_schedule_single_event(time(), 'delcampe_process_batch_queue');
        }
        wp_send_json_success(array('recovered' => $count));
    }
    
    /**
     * AJAX: Pause processing
     */
    public function ajax_pause_processing() {
        check_ajax_referer('delcampe_batch_monitor', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Permission denied');
        }
        
        set_transient('delcampe_batch_processing_paused', true, HOUR_IN_SECONDS);
        wp_send_json_success('Processing paused');
    }
    
    /**
     * AJAX: Resume processing
     */
    public function ajax_resume_processing() {
        check_ajax_referer('delcampe_batch_monitor', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Permission denied');
        }
        
        delete_transient('delcampe_batch_processing_paused');
        
        // Trigger immediate processing
        wp_schedule_single_event(time(), 'delcampe_process_batch_queue');
        
        wp_send_json_success('Processing resumed');
    }
    
    /**
     * Format duration
     */
    private function format_duration($seconds) {
        if ($seconds < 60) {
            return sprintf(__('%d seconds', 'wc-delcampe-integration'), $seconds);
        } elseif ($seconds < 3600) {
            return sprintf(__('%d minutes', 'wc-delcampe-integration'), round($seconds / 60));
        } else {
            return sprintf(__('%.1f hours', 'wc-delcampe-integration'), $seconds / 3600);
        }
    }
    
    /**
     * Format time ago
     */
    private function format_time_ago($datetime) {
        // Convert database UTC time to WordPress time
        $timestamp = strtotime($datetime . ' UTC');
        $current_time = current_time('timestamp');
        $diff = $current_time - $timestamp;
        
        // Handle negative differences (future times)
        if ($diff < 0) {
            return __('Just now', 'wc-delcampe-integration');
        }
        
        if ($diff < 60) {
            return __('Just now', 'wc-delcampe-integration');
        } elseif ($diff < 3600) {
            return sprintf(__('%d minutes ago', 'wc-delcampe-integration'), round($diff / 60));
        } elseif ($diff < 86400) {
            return sprintf(__('%d hours ago', 'wc-delcampe-integration'), round($diff / 3600));
        } else {
            return sprintf(__('%d days ago', 'wc-delcampe-integration'), round($diff / 86400));
        }
    }
}

// Initialize
Delcampe_Batch_Monitor::get_instance();
