<?php
/**
 * Delcampe Batch Queue v2 - Proper batch management without immediate sync
 * 
 * This implementation follows the specification:
 * - Batches are created and queued but NOT immediately synced
 * - Products are marked with _delcampe_pending_batch to prevent duplicates
 * - Proper preflight checking before batch creation
 * - Monitoring and control capabilities
 * 
 * @package Delcampe_Sync
 * @since 1.10.28.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Batch_Queue_V2 {
    
    private static $instance = null;
    private $table_name;
    
    // Batch statuses
    const STATUS_PENDING = 'pending';
    const STATUS_QUEUED = 'queued';
    const STATUS_PROCESSING = 'processing';
    const STATUS_PAUSED = 'paused';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_CANCELLED = 'cancelled';
    
    // Processing constants
    const DEFAULT_BATCH_SIZE = 10;
    const MAX_BATCH_SIZE = 100;
    const PROCESSING_TIMEOUT = 600; // 10 minutes
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'delcampe_batch_queue';
    }
    
    /**
     * Preflight check - determine which products are eligible
     * 
     * @param array $product_ids Products to check
     * @param int $profile_id Profile to use
     * @return array Preflight results with eligible and skipped items
     */
    public function preflight_check($product_ids, $profile_id) {
        global $wpdb;
        
        $results = array(
            'eligible' => array(),
            'skipped' => array(),
            'skip_reasons' => array(
                'already_published' => array(),
                'in_flight' => array(),
                'not_publishable' => array(),
                'missing_profile' => array(),
                'other' => array()
            ),
            'counts' => array(
                'selected' => count($product_ids),
                'eligible' => 0,
                'skipped' => 0
            )
        );
        
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        foreach ($product_ids as $product_id) {
            $skip_reason = null;
            
            // 1. Check if already published
            $listing_id = get_post_meta($product_id, '_delcampe_listing_id', true);
            if (!empty($listing_id)) {
                $skip_reason = 'already_published';
            }
            
            // Check listings table
            if (!$skip_reason) {
                $listing_status = $wpdb->get_var($wpdb->prepare(
                    "SELECT status FROM {$listings_table} 
                     WHERE product_id = %d 
                     AND status IN ('active', 'verified', 'published', 'processing', 'changed')",
                    $product_id
                ));
                if ($listing_status) {
                    $skip_reason = 'already_published';
                }
            }
            
            // 2. Check if in-flight
            if (!$skip_reason) {
                // Check if in another batch
                $pending_batch = get_post_meta($product_id, '_delcampe_pending_batch', true);
                if (!empty($pending_batch)) {
                    // Verify batch is still active
                    $batch_status = $wpdb->get_var($wpdb->prepare(
                        "SELECT status FROM {$this->table_name} WHERE batch_id = %s",
                        $pending_batch
                    ));
                    if (in_array($batch_status, array('pending', 'queued', 'processing', 'paused'))) {
                        $skip_reason = 'in_flight';
                    } else {
                        // Clean up stale batch reference
                        delete_post_meta($product_id, '_delcampe_pending_batch');
                    }
                }
                
                // Check sync queue
                if (!$skip_reason) {
                    $sync_queue = get_option('delcampe_sync_queue', array());
                    if (isset($sync_queue[$product_id])) {
                        $skip_reason = 'in_flight';
                    }
                }
                
                // Check sync status
                if (!$skip_reason) {
                    $sync_status = get_post_meta($product_id, '_delcampe_sync_status', true);
                    if (in_array($sync_status, array('pending', 'processing', 'queued'))) {
                        $skip_reason = 'in_flight';
                    }
                }
            }
            
            // 3. Check if publishable
            if (!$skip_reason) {
                $post_status = get_post_status($product_id);
                if ($post_status !== 'publish') {
                    $skip_reason = 'not_publishable';
                }
            }
            
            // 4. Check profile (if not using provided profile)
            if (!$skip_reason) {
                $product_profile = get_post_meta($product_id, '_delcampe_profile_id', true);
                if (empty($product_profile) && empty($profile_id)) {
                    $skip_reason = 'missing_profile';
                }
            }
            
            // Add to appropriate list
            if ($skip_reason) {
                $product = wc_get_product($product_id);
                $results['skipped'][] = array(
                    'product_id' => $product_id,
                    'title' => $product ? $product->get_name() : "Product #{$product_id}",
                    'sku' => $product ? $product->get_sku() : '',
                    'reason' => $skip_reason
                );
                $results['skip_reasons'][$skip_reason][] = $product_id;
                $results['counts']['skipped']++;
            } else {
                $results['eligible'][] = $product_id;
                $results['counts']['eligible']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Create a batch with eligible products only
     * 
     * @param array $product_ids Eligible product IDs
     * @param int $profile_id Profile to use
     * @param array $metadata Additional metadata
     * @return array|WP_Error Batch info or error
     */
    public function create_batch($product_ids, $profile_id, $metadata = array()) {
        global $wpdb;
        
        if (empty($product_ids)) {
            return new WP_Error('no_products', 'No eligible products to process');
        }
        
        // Generate batch ID
        $batch_id = 'batch_' . time() . '_' . wp_generate_password(8, false);
        
        // Prepare batch data
        $batch_data = array(
            'batch_id' => $batch_id,
            'product_ids' => json_encode(array_values($product_ids)),
            'profile_id' => $profile_id,
            'action' => 'create',
            'status' => self::STATUS_QUEUED,
            'batch_size' => min(self::MAX_BATCH_SIZE, count($product_ids)),
            'processed_count' => 0,
            'total_count' => count($product_ids),
            'created_by' => get_current_user_id(),
            'metadata' => json_encode(array_merge($metadata, array(
                'created_at' => current_time('mysql'),
                'source' => isset($metadata['source']) ? $metadata['source'] : 'bulk_publish'
            )))
        );
        
        // Insert batch
        $result = $wpdb->insert($this->table_name, $batch_data);
        
        if ($result === false) {
            return new WP_Error('db_error', 'Failed to create batch: ' . $wpdb->last_error);
        }
        
        // Mark products as pending in this batch
        foreach ($product_ids as $product_id) {
            update_post_meta($product_id, '_delcampe_pending_batch', $batch_id);
            update_post_meta($product_id, '_delcampe_sync_status', 'pending');
            if ($profile_id) {
                update_post_meta($product_id, '_delcampe_profile_id', $profile_id);
            }
        }
        
        // Get queue position
        $queue_position = $this->get_queue_position($batch_id);
        
        delcampe_log('[Batch Queue] Created batch ' . $batch_id . ' with ' . count($product_ids) . ' products');
        
        return array(
            'batch_id' => $batch_id,
            'queue_position' => $queue_position,
            'status' => self::STATUS_QUEUED,
            'total_count' => count($product_ids),
            'message' => sprintf('Batch created with %d products', count($product_ids))
        );
    }
    
    /**
     * Get queue position for a batch
     */
    private function get_queue_position($batch_id) {
        global $wpdb;
        
        $position = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) + 1 
             FROM {$this->table_name} 
             WHERE status IN ('queued', 'processing', 'paused') 
             AND created_at < (SELECT created_at FROM {$this->table_name} WHERE batch_id = %s)",
            $batch_id
        ));
        
        return $position ?: 1;
    }
    
    /**
     * Process next batch in queue (called by cron or manual trigger)
     * This ONLY prepares items for sync, doesn't actually sync them
     */
    public function process_next_batch() {
        global $wpdb;
        
        // Check for paused processing
        if (get_transient('delcampe_batch_processing_paused')) {
            return;
        }
        
        // Get next queued batch
        $batch = $wpdb->get_row("
            SELECT * FROM {$this->table_name}
            WHERE status = 'queued'
            ORDER BY created_at ASC
            LIMIT 1
        ");
        
        if (!$batch) {
            return;
        }
        
        // Mark as processing
        $wpdb->update(
            $this->table_name,
            array(
                'status' => self::STATUS_PROCESSING,
                'processing_started' => current_time('mysql')
            ),
            array('batch_id' => $batch->batch_id)
        );
        
        delcampe_log('[Batch Queue] Processing batch ' . $batch->batch_id);
        
        // Process the batch
        $this->process_batch_items($batch);
    }
    
    /**
     * Process items in a batch
     * This prepares them for sync but doesn't trigger sync
     */
    private function process_batch_items($batch) {
        global $wpdb;
        
        $product_ids = json_decode($batch->product_ids, true);
        $processed = $batch->processed_count;
        $errors = array();
        
        // Get sync handler
        $sync_handler = Delcampe_Sync_Handler::get_instance();
        
        // Process in chunks
        $chunk_size = min($batch->batch_size, 10);
        $chunk = array_slice($product_ids, $processed, $chunk_size);
        
        foreach ($chunk as $product_id) {
            // Re-validate eligibility
            $preflight = $this->preflight_check(array($product_id), $batch->profile_id);
            
            if (!empty($preflight['eligible'])) {
                // Add to sync queue (but don't trigger sync)
                $result = $sync_handler->add_to_queue($product_id, $batch->profile_id, 'create');
                
                if (is_wp_error($result)) {
                    $errors[] = array(
                        'product_id' => $product_id,
                        'error' => $result->get_error_message()
                    );
                }
            } else {
                // Product no longer eligible
                $errors[] = array(
                    'product_id' => $product_id,
                    'error' => 'No longer eligible: ' . $preflight['skipped'][0]['reason']
                );
            }
            
            $processed++;
            
            // Update progress
            $wpdb->update(
                $this->table_name,
                array('processed_count' => $processed),
                array('batch_id' => $batch->batch_id)
            );
        }
        
        // Check if batch is complete
        if ($processed >= $batch->total_count) {
            $status = empty($errors) ? self::STATUS_COMPLETED : self::STATUS_COMPLETED;
            
            $wpdb->update(
                $this->table_name,
                array(
                    'status' => $status,
                    'processing_completed' => current_time('mysql'),
                    'error_details' => empty($errors) ? null : json_encode($errors)
                ),
                array('batch_id' => $batch->batch_id)
            );
            
            delcampe_log('[Batch Queue] Completed batch ' . $batch->batch_id);
        } else {
            // More to process - will be picked up on next cron run
            delcampe_log('[Batch Queue] Batch ' . $batch->batch_id . ' progress: ' . $processed . '/' . $batch->total_count);
        }
    }
    
    /**
     * Trigger sync for items in completed batches
     * This is a separate action that actually syncs to Delcampe
     */
    public function trigger_batch_sync($batch_id = null) {
        global $wpdb;
        
        if ($batch_id) {
            // Sync specific batch
            $batch = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->table_name} WHERE batch_id = %s",
                $batch_id
            ));
            
            if ($batch && $batch->status === self::STATUS_COMPLETED) {
                // Trigger sync queue processing
                do_action('delcampe_process_sync_queue');
                
                delcampe_log('[Batch Queue] Triggered sync for batch ' . $batch_id);
            }
        } else {
            // Trigger sync for all completed batches
            do_action('delcampe_process_sync_queue');
        }
    }
    
    /**
     * Cancel a batch
     */
    public function cancel_batch($batch_id) {
        global $wpdb;
        
        // Update batch status
        $wpdb->update(
            $this->table_name,
            array(
                'status' => self::STATUS_CANCELLED,
                'processing_completed' => current_time('mysql')
            ),
            array('batch_id' => $batch_id)
        );
        
        // Clear pending batch markers
        $batch = $wpdb->get_row($wpdb->prepare(
            "SELECT product_ids FROM {$this->table_name} WHERE batch_id = %s",
            $batch_id
        ));
        
        if ($batch) {
            $product_ids = json_decode($batch->product_ids, true);
            foreach ($product_ids as $product_id) {
                $current_batch = get_post_meta($product_id, '_delcampe_pending_batch', true);
                if ($current_batch === $batch_id) {
                    delete_post_meta($product_id, '_delcampe_pending_batch');
                    delete_post_meta($product_id, '_delcampe_sync_status');
                }
            }
        }
        
        delcampe_log('[Batch Queue] Cancelled batch ' . $batch_id);
    }
    
    /**
     * Pause/resume batch processing
     */
    public function pause_processing($pause = true) {
        if ($pause) {
            set_transient('delcampe_batch_processing_paused', true, HOUR_IN_SECONDS);
        } else {
            delete_transient('delcampe_batch_processing_paused');
        }
    }
    
    /**
     * Get batch status for monitoring
     */
    public function get_batch_status($batch_id = null) {
        global $wpdb;
        
        if ($batch_id) {
            return $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->table_name} WHERE batch_id = %s",
                $batch_id
            ), ARRAY_A);
        }
        
        // Get all recent batches
        return $wpdb->get_results("
            SELECT * FROM {$this->table_name}
            WHERE created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
            ORDER BY created_at DESC
            LIMIT 20
        ", ARRAY_A);
    }
    
    /**
     * Export batch report as CSV
     */
    public function export_batch_report($batch_id) {
        global $wpdb;
        
        $batch = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE batch_id = %s",
            $batch_id
        ));
        
        if (!$batch) {
            return new WP_Error('not_found', 'Batch not found');
        }
        
        $product_ids = json_decode($batch->product_ids, true);
        $errors = json_decode($batch->error_details, true) ?: array();
        
        $csv_data = array();
        $csv_data[] = array('Batch ID', 'Product ID', 'Product Title', 'SKU', 'Result', 'Delcampe ID', 'Error');
        
        foreach ($product_ids as $product_id) {
            $product = wc_get_product($product_id);
            $listing_id = get_post_meta($product_id, '_delcampe_listing_id', true);
            $sync_status = get_post_meta($product_id, '_delcampe_sync_status', true);
            
            $error = '';
            foreach ($errors as $err) {
                if ($err['product_id'] == $product_id) {
                    $error = $err['error'];
                    break;
                }
            }
            
            $csv_data[] = array(
                $batch_id,
                $product_id,
                $product ? $product->get_name() : '',
                $product ? $product->get_sku() : '',
                $sync_status ?: 'pending',
                $listing_id ?: '',
                $error
            );
        }
        
        return $csv_data;
    }
}