<?php
/**
 * Delcampe Business Event Logger
 * 
 * Production-focused logging system that tracks meaningful business events
 * like orders, stock changes, and synchronization activities.
 * 
 * @package Delcampe_Sync
 * @since 1.10.12.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Business_Logger {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Database table name
     */
    private $table_name;
    
    /**
     * Event types
     */
    const EVENT_ORDER_FROM_DELCAMPE = 'order_from_delcampe';
    const EVENT_ORDER_FROM_SITE = 'order_from_site';
    const EVENT_STOCK_UPDATED = 'stock_updated';
    const EVENT_STOCK_SYNCED_TO_DELCAMPE = 'stock_synced_to_delcampe';
    const EVENT_LISTING_CREATED = 'listing_created';
    const EVENT_LISTING_UPDATED = 'listing_updated';
    const EVENT_LISTING_ENDED = 'listing_ended';
    const EVENT_PRICE_UPDATED = 'price_updated';
    const EVENT_SYNC_ERROR = 'sync_error';
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'delcampe_business_events';
        
        // Hook into various WordPress/WooCommerce events
        $this->setup_hooks();
        
        // Create table if needed
        $this->maybe_create_table();
    }
    
    /**
     * Setup WordPress hooks
     */
    private function setup_hooks() {
        // Stock change hooks
        add_action('woocommerce_product_set_stock', array($this, 'log_stock_change'), 10, 1);
        add_action('woocommerce_variation_set_stock', array($this, 'log_stock_change'), 10, 1);
        
        // Order hooks
        add_action('woocommerce_new_order', array($this, 'log_new_order'), 10, 2);
        add_action('woocommerce_order_status_changed', array($this, 'log_order_status_change'), 10, 4);
        
        // Custom hooks for Delcampe events
        add_action('delcampe_order_imported_with_details', array($this, 'log_delcampe_order'), 10, 2);
        add_action('delcampe_stock_synced', array($this, 'log_delcampe_sync'), 10, 3);
        add_action('delcampe_listing_created', array($this, 'log_listing_created'), 10, 2);
        add_action('delcampe_listing_updated', array($this, 'log_listing_updated'), 10, 2);
        add_action('delcampe_sync_error', array($this, 'log_sync_error'), 10, 2);
    }
    
    /**
     * Create database table
     */
    private function maybe_create_table() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            event_type varchar(50) NOT NULL,
            event_source varchar(50) DEFAULT NULL,
            product_id bigint(20) DEFAULT NULL,
            order_id bigint(20) DEFAULT NULL,
            delcampe_id varchar(100) DEFAULT NULL,
            sku varchar(100) DEFAULT NULL,
            product_name varchar(255) DEFAULT NULL,
            quantity_before int(11) DEFAULT NULL,
            quantity_after int(11) DEFAULT NULL,
            quantity_change int(11) DEFAULT NULL,
            price decimal(10,2) DEFAULT NULL,
            details text,
            status varchar(50) DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_event_type (event_type),
            KEY idx_product_id (product_id),
            KEY idx_order_id (order_id),
            KEY idx_created_at (created_at),
            KEY idx_sku (sku)
        ) $charset_collate";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Log a business event
     */
    public function log_event($event_type, $data = array()) {
        global $wpdb;
        
        // Prepare insert data
        $insert_data = array(
            'event_type' => $event_type,
            'event_source' => isset($data['source']) ? $data['source'] : $this->detect_source(),
            'product_id' => isset($data['product_id']) ? $data['product_id'] : null,
            'order_id' => isset($data['order_id']) ? $data['order_id'] : null,
            'delcampe_id' => isset($data['delcampe_id']) ? $data['delcampe_id'] : null,
            'sku' => isset($data['sku']) ? $data['sku'] : null,
            'product_name' => isset($data['product_name']) ? $data['product_name'] : null,
            'quantity_before' => isset($data['quantity_before']) ? $data['quantity_before'] : null,
            'quantity_after' => isset($data['quantity_after']) ? $data['quantity_after'] : null,
            'quantity_change' => isset($data['quantity_change']) ? $data['quantity_change'] : null,
            'price' => isset($data['price']) ? $data['price'] : null,
            'details' => isset($data['details']) ? json_encode($data['details']) : null,
            'status' => isset($data['status']) ? $data['status'] : 'success',
            'created_at' => current_time('mysql')
        );
        
        // Insert the event
        $wpdb->insert($this->table_name, $insert_data);
        
        // Trigger sync to Delcampe if needed
        $this->maybe_trigger_delcampe_sync($event_type, $data);
    }
    
    /**
     * Detect the source of the event
     */
    private function detect_source() {
        // Check if it's from Delcampe webhook or import
        if (did_action('delcampe_webhook_received') || get_transient('delcampe_importing_order')) {
            return 'delcampe';
        }
        
        // Check if it's from a cron job
        if (wp_doing_cron()) {
            return 'system';
        }
        
        // Check if it's from admin
        if (is_admin() && !wp_doing_ajax()) {
            return 'admin';
        }
        
        // Check if it's from AJAX
        if (wp_doing_ajax()) {
            return 'ajax';
        }
        
        // Check if it's from frontend
        if (!is_admin()) {
            return 'website';
        }
        
        // Default
        return 'system';
    }
    
    /**
     * Log stock change
     */
    public function log_stock_change($product) {
        $product_id = $product->get_id();
        
        // Don't log stock changes for non-published products (v1.10.17.8)
        $product_status = get_post_status($product_id);
        if ($product_status !== 'publish') return;
        
        $sku = $product->get_sku();
        $stock_quantity = $product->get_stock_quantity();
        
        // Get previous quantity from meta
        $previous_quantity = get_post_meta($product_id, '_stock_before_change', true);
        if ($previous_quantity === '') {
            $previous_quantity = $stock_quantity;
        }
        
        // Calculate change
        $quantity_change = $stock_quantity - $previous_quantity;
        
        // Only log if there's an actual change
        if ($quantity_change != 0) {
            $this->log_event(self::EVENT_STOCK_UPDATED, array(
                'product_id' => $product_id,
                'sku' => $sku,
                'product_name' => $product->get_name(),
                'quantity_before' => $previous_quantity,
                'quantity_after' => $stock_quantity,
                'quantity_change' => $quantity_change,
                'details' => array(
                    'source' => $this->detect_source(),
                    'user_id' => get_current_user_id(),
                    'reason' => $quantity_change < 0 ? 'stock_reduced' : 'stock_increased'
                )
            ));
            
            // Update the before value for next time
            update_post_meta($product_id, '_stock_before_change', $stock_quantity);
        }
    }
    
    /**
     * Log new order
     */
    public function log_new_order($order_id, $order) {
        // Check if this is a Delcampe order
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        
        // Determine event type - only Delcampe vs Website matters
        if (!empty($delcampe_order_id)) {
            $event_type = self::EVENT_ORDER_FROM_DELCAMPE;
            $source = 'delcampe';
        } else {
            // Everything else is a website order (including eBay, Amazon, etc.)
            $event_type = self::EVENT_ORDER_FROM_SITE;
            $source = 'website';
        }
        
        // Log each line item
        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            $product = $item->get_product();
            
            if ($product) {
                $this->log_event($event_type, array(
                    'order_id' => $order_id,
                    'product_id' => $product_id,
                    'sku' => $product->get_sku(),
                    'product_name' => $item->get_name(),
                    'quantity_change' => -$item->get_quantity(),
                    'price' => $item->get_total(),
                    'source' => $source,
                    'details' => array(
                        'order_number' => $order->get_order_number(),
                        'customer_email' => $order->get_billing_email(),
                        'payment_method' => $order->get_payment_method(),
                        'delcampe_order_id' => $delcampe_order_id ?: null
                    )
                ));
            }
        }
    }
    
    /**
     * Log order status change
     */
    public function log_order_status_change($order_id, $old_status, $new_status, $order) {
        // Only log if moving to completed or cancelled
        if ($new_status === 'completed' || $new_status === 'cancelled') {
            $this->log_event('order_status_changed', array(
                'order_id' => $order_id,
                'details' => array(
                    'old_status' => $old_status,
                    'new_status' => $new_status,
                    'order_number' => $order->get_order_number()
                )
            ));
        }
    }
    
    /**
     * Log Delcampe order import
     */
    public function log_delcampe_order($order_id, $delcampe_data) {
        $order = wc_get_order($order_id);
        if (!$order) return;
        
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if ($product) {
                $this->log_event(self::EVENT_ORDER_FROM_DELCAMPE, array(
                    'order_id' => $order_id,
                    'product_id' => $product->get_id(),
                    'delcampe_id' => isset($delcampe_data['item_id']) ? $delcampe_data['item_id'] : null,
                    'sku' => $product->get_sku(),
                    'product_name' => $item->get_name(),
                    'quantity_change' => -$item->get_quantity(),
                    'price' => $item->get_total(),
                    'details' => array(
                        'delcampe_order_id' => isset($delcampe_data['id_order']) ? $delcampe_data['id_order'] : null,
                        'delcampe_buyer' => isset($delcampe_data['buyer_pseudo']) ? $delcampe_data['buyer_pseudo'] : null,
                        'import_time' => current_time('mysql')
                    )
                ));
            }
        }
    }
    
    /**
     * Log Delcampe sync
     */
    public function log_delcampe_sync($product_id, $new_quantity, $result) {
        $product = wc_get_product($product_id);
        if (!$product) return;
        
        // Don't log sync for non-published products (v1.10.17.8)
        $product_status = get_post_status($product_id);
        if ($product_status !== 'publish') return;
        
        $this->log_event(self::EVENT_STOCK_SYNCED_TO_DELCAMPE, array(
            'product_id' => $product_id,
            'sku' => $product->get_sku(),
            'product_name' => $product->get_name(),
            'quantity_after' => $new_quantity,
            'status' => $result ? 'success' : 'failed',
            'details' => array(
                'sync_time' => current_time('mysql'),
                'result' => $result
            )
        ));
    }
    
    /**
     * Log listing created
     */
    public function log_listing_created($product_id, $delcampe_id) {
        $product = wc_get_product($product_id);
        if (!$product) return;
        
        $this->log_event(self::EVENT_LISTING_CREATED, array(
            'product_id' => $product_id,
            'delcampe_id' => $delcampe_id,
            'sku' => $product->get_sku(),
            'product_name' => $product->get_name(),
            'quantity_after' => $product->get_stock_quantity(),
            'price' => $product->get_price(),
            'details' => array(
                'created_time' => current_time('mysql')
            )
        ));
    }
    
    /**
     * Log listing updated
     */
    public function log_listing_updated($product_id, $changes) {
        $product = wc_get_product($product_id);
        if (!$product) return;
        
        $this->log_event(self::EVENT_LISTING_UPDATED, array(
            'product_id' => $product_id,
            'sku' => $product->get_sku(),
            'product_name' => $product->get_name(),
            'details' => $changes
        ));
    }
    
    /**
     * Log sync error
     */
    public function log_sync_error($message, $context) {
        $this->log_event(self::EVENT_SYNC_ERROR, array(
            'status' => 'error',
            'details' => array(
                'error_message' => $message,
                'context' => $context,
                'time' => current_time('mysql')
            )
        ));
    }
    
    /**
     * Maybe trigger Delcampe sync based on event
     */
    private function maybe_trigger_delcampe_sync($event_type, $data) {
        // Skip if the event is already from Delcampe
        if (isset($data['source']) && $data['source'] === 'delcampe') {
            return;
        }
        
        // Trigger sync for stock changes and website orders
        if (in_array($event_type, array(
            self::EVENT_STOCK_UPDATED,
            self::EVENT_ORDER_FROM_SITE
        ))) {
            if (isset($data['product_id'])) {
                // Schedule immediate inventory sync for this product
                as_schedule_single_action(
                    time(),
                    'delcampe_sync_single_product_inventory',
                    array('product_id' => $data['product_id']),
                    'delcampe-sync'
                );
            }
        }
    }
    
    /**
     * Get recent events
     */
    public function get_recent_events($limit = 100, $offset = 0, $filters = array()) {
        global $wpdb;
        
        $where = array('1=1');
        
        // Apply filters
        if (!empty($filters['event_type'])) {
            $where[] = $wpdb->prepare('event_type = %s', $filters['event_type']);
        }
        
        if (!empty($filters['product_id'])) {
            $where[] = $wpdb->prepare('product_id = %d', $filters['product_id']);
        }
        
        if (!empty($filters['date_from'])) {
            $where[] = $wpdb->prepare('created_at >= %s', $filters['date_from']);
        }
        
        if (!empty($filters['date_to'])) {
            $where[] = $wpdb->prepare('created_at <= %s', $filters['date_to']);
        }
        
        $where_clause = implode(' AND ', $where);
        
        $sql = "SELECT * FROM {$this->table_name} 
                WHERE {$where_clause} 
                ORDER BY created_at DESC 
                LIMIT %d OFFSET %d";
        
        return $wpdb->get_results($wpdb->prepare($sql, $limit, $offset));
    }
    
    /**
     * Get event statistics
     */
    public function get_statistics($days = 7) {
        global $wpdb;
        
        $date_from = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $sql = "SELECT 
                    event_type,
                    COUNT(*) as count,
                    DATE(created_at) as date
                FROM {$this->table_name}
                WHERE created_at >= %s
                GROUP BY event_type, DATE(created_at)
                ORDER BY date DESC, count DESC";
        
        $results = $wpdb->get_results($wpdb->prepare($sql, $date_from));
        
        // Format statistics
        $stats = array(
            'by_type' => array(),
            'by_date' => array(),
            'totals' => array()
        );
        
        foreach ($results as $row) {
            // By type
            if (!isset($stats['by_type'][$row->event_type])) {
                $stats['by_type'][$row->event_type] = 0;
            }
            $stats['by_type'][$row->event_type] += $row->count;
            
            // By date
            if (!isset($stats['by_date'][$row->date])) {
                $stats['by_date'][$row->date] = array();
            }
            $stats['by_date'][$row->date][$row->event_type] = $row->count;
        }
        
        // Calculate totals
        $stats['totals'] = array(
            'orders_from_delcampe' => isset($stats['by_type'][self::EVENT_ORDER_FROM_DELCAMPE]) ? $stats['by_type'][self::EVENT_ORDER_FROM_DELCAMPE] : 0,
            'orders_from_site' => isset($stats['by_type'][self::EVENT_ORDER_FROM_SITE]) ? $stats['by_type'][self::EVENT_ORDER_FROM_SITE] : 0,
            'stock_updates' => isset($stats['by_type'][self::EVENT_STOCK_UPDATED]) ? $stats['by_type'][self::EVENT_STOCK_UPDATED] : 0,
            'sync_errors' => isset($stats['by_type'][self::EVENT_SYNC_ERROR]) ? $stats['by_type'][self::EVENT_SYNC_ERROR] : 0
        );
        
        return $stats;
    }
    
    /**
     * Clean old events (retention policy)
     */
    public function clean_old_events($days_to_keep = 90) {
        global $wpdb;
        
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days_to_keep} days"));
        
        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE created_at < %s",
            $cutoff_date
        ));
        
        return $wpdb->rows_affected;
    }
}

// Initialize the logger
add_action('init', function() {
    Delcampe_Business_Logger::get_instance();
});

// Add cleanup cron
add_action('init', function() {
    if (!wp_next_scheduled('delcampe_clean_old_business_events')) {
        wp_schedule_event(time(), 'daily', 'delcampe_clean_old_business_events');
    }
});

add_action('delcampe_clean_old_business_events', function() {
    $logger = Delcampe_Business_Logger::get_instance();
    $logger->clean_old_events(90); // Keep 90 days of history
});