<?php
/**
 * Delcampe Close Reasons Handler
 * 
 * Fetches and manages allowed close reasons from Delcampe API
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage API
 * @since 1.10.20.4
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_Close_Reasons
 * 
 * Handles fetching and caching of allowed close reasons from Delcampe
 */
class Delcampe_Close_Reasons {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Cache key for close reasons
     */
    const CACHE_KEY = 'delcampe_close_reasons';
    
    /**
     * Cache duration (24 hours)
     */
    const CACHE_DURATION = 86400;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Get allowed close reasons from Delcampe API
     * 
     * @param bool $force_refresh Force API call instead of using cache
     * @return array|WP_Error Array of reasons or error
     */
    public function get_allowed_reasons($force_refresh = false) {
        // Check cache first
        if (!$force_refresh) {
            $cached = get_transient(self::CACHE_KEY);
            if ($cached !== false) {
                return $cached;
            }
        }
        
        // Get authentication token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        // Build API URL - correct endpoint from documentation
        $url = DELCAMPE_API_URL . '/item/allowedclosereasons?token=' . $token;
        
        // Make API request
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/xml',
                'User-Agent' => DELCAMPE_USER_AGENT
            )
        ));
        
        if (is_wp_error($response)) {
            error_log('[Delcampe Close Reasons] API request failed: ' . $response->get_error_message());
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code !== 200) {
            error_log('[Delcampe Close Reasons] API returned status ' . $status_code);
            return new WP_Error('api_error', 'Failed to fetch close reasons. Status: ' . $status_code);
        }
        
        // Parse XML response
        $xml = simplexml_load_string($body);
        if (!$xml) {
            return new WP_Error('parse_error', 'Failed to parse XML response');
        }
        
        // Extract reasons from XML
        $reasons = $this->parse_reasons_from_xml($xml);
        
        // Cache the results
        if (!empty($reasons)) {
            set_transient(self::CACHE_KEY, $reasons, self::CACHE_DURATION);
        }
        
        return $reasons;
    }
    
    /**
     * Parse close reasons from XML response
     * 
     * @param SimpleXMLElement $xml XML response
     * @return array Array of reasons
     */
    private function parse_reasons_from_xml($xml) {
        $reasons = array();
        
        // Check for error in response
        if (isset($xml->error) || isset($xml->e)) {
            $error = isset($xml->error) ? (string)$xml->error : (string)$xml->e;
            error_log('[Delcampe Close Reasons] API error: ' . $error);
            return $reasons;
        }
        
        // Try different possible XML structures
        // Structure 1: Direct reasons list
        if (isset($xml->reason)) {
            foreach ($xml->reason as $reason) {
                $reasons[] = array(
                    'code' => (string)$reason['code'] ?? (string)$reason,
                    'label' => (string)$reason['label'] ?? (string)$reason,
                    'value' => (string)$reason
                );
            }
        }
        
        // Structure 2: Wrapped in body
        if (isset($xml->body->reasons->reason)) {
            foreach ($xml->body->reasons->reason as $reason) {
                $reasons[] = array(
                    'code' => (string)$reason['code'] ?? (string)$reason,
                    'label' => (string)$reason['label'] ?? (string)$reason,
                    'value' => (string)$reason
                );
            }
        }
        
        // Structure 3: As items
        if (isset($xml->item)) {
            foreach ($xml->item as $item) {
                $reasons[] = array(
                    'code' => (string)$item['code'] ?? (string)$item,
                    'label' => (string)$item['label'] ?? (string)$item,
                    'value' => (string)$item
                );
            }
        }
        
        // If no structured data found but we have a simple list
        if (empty($reasons) && $xml->count() > 0) {
            foreach ($xml->children() as $child) {
                $reasons[] = array(
                    'code' => $child->getName(),
                    'label' => (string)$child,
                    'value' => (string)$child
                );
            }
        }
        
        // Add default reasons if API doesn't provide them
        if (empty($reasons)) {
            $reasons = $this->get_default_reasons();
        }
        
        return $reasons;
    }
    
    /**
     * Get default close reasons (fallback)
     * 
     * @return array Default reasons based on common scenarios
     */
    private function get_default_reasons() {
        return array(
            array(
                'code' => 'sold_elsewhere',
                'label' => __('Sold elsewhere', 'wc-delcampe-integration'),
                'value' => 'sold_elsewhere'
            ),
            array(
                'code' => 'no_longer_available',
                'label' => __('No longer available', 'wc-delcampe-integration'),
                'value' => 'no_longer_available'
            ),
            array(
                'code' => 'listing_error',
                'label' => __('Listing error', 'wc-delcampe-integration'),
                'value' => 'listing_error'
            ),
            array(
                'code' => 'other',
                'label' => __('Other reason', 'wc-delcampe-integration'),
                'value' => 'other'
            )
        );
    }
    
    /**
     * Get formatted reasons for dropdown
     * 
     * @return array Associative array for select dropdown
     */
    public function get_reasons_for_dropdown() {
        $reasons = $this->get_allowed_reasons();
        
        if (is_wp_error($reasons)) {
            // Return default options on error
            return array(
                '' => __('-- Select reason --', 'wc-delcampe-integration'),
                'sold_elsewhere' => __('Sold elsewhere', 'wc-delcampe-integration'),
                'no_longer_available' => __('No longer available', 'wc-delcampe-integration'),
                'other' => __('Other reason', 'wc-delcampe-integration')
            );
        }
        
        $dropdown = array(
            '' => __('-- Select reason --', 'wc-delcampe-integration')
        );
        
        foreach ($reasons as $reason) {
            $key = $reason['code'] ?? $reason['value'];
            $label = $reason['label'] ?? $reason['value'];
            $dropdown[$key] = $label;
        }
        
        return $dropdown;
    }
    
    /**
     * Clear cached reasons
     */
    public function clear_cache() {
        delete_transient(self::CACHE_KEY);
    }
}

// Initialize on admin
if (is_admin()) {
    add_action('init', array('Delcampe_Close_Reasons', 'get_instance'));
}