<?php
/**
 * Delcampe Deadlock Prevention
 * 
 * Helps prevent database deadlocks when using Action Scheduler
 * 
 * @package WC_Delcampe_Integration
 * @version 1.10.10.1
 * @since   1.10.10.1
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Delcampe_Deadlock_Prevention class
 *
 * @since 1.10.10.1
 */
class Delcampe_Deadlock_Prevention {
    
    /**
     * Singleton instance
     * @var Delcampe_Deadlock_Prevention
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     *
     * @return Delcampe_Deadlock_Prevention
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init();
    }
    
    /**
     * Initialize deadlock prevention
     */
    private function init() {
        // Add retry logic for Action Scheduler
        add_filter('action_scheduler_queue_runner_batch_size', array($this, 'reduce_batch_size'));
        add_filter('action_scheduler_queue_runner_time_limit', array($this, 'reduce_time_limit'));
        
        // Handle deadlock errors
        add_action('action_scheduler_failed_action', array($this, 'handle_failed_action'), 10, 2);
        
        // Optimize Action Scheduler tables
        add_action('action_scheduler_before_process_queue', array($this, 'optimize_tables'));
        
        // Add custom retry logic for our plugin's scheduled actions
        add_filter('action_scheduler_retry_schedule', array($this, 'custom_retry_schedule'), 10, 2);
        
        // Reduce concurrent runners
        add_filter('action_scheduler_queue_runner_concurrent_batches', array($this, 'reduce_concurrent_batches'));
    }
    
    /**
     * Reduce batch size to prevent deadlocks
     *
     * @param int $batch_size Current batch size
     * @return int Modified batch size
     */
    public function reduce_batch_size($batch_size) {
        // Reduce from default 25 to 10 to minimize lock contention
        return min($batch_size, 10);
    }
    
    /**
     * Reduce time limit to prevent long-running transactions
     *
     * @param int $time_limit Current time limit in seconds
     * @return int Modified time limit
     */
    public function reduce_time_limit($time_limit) {
        // Reduce from default 30 to 20 seconds
        return min($time_limit, 20);
    }
    
    /**
     * Reduce concurrent batches to prevent deadlocks
     *
     * @param int $concurrent Current number of concurrent batches
     * @return int Modified concurrent batches
     */
    public function reduce_concurrent_batches($concurrent) {
        // Allow only 1 concurrent batch to minimize deadlocks
        return 1;
    }
    
    /**
     * Handle failed actions that might be due to deadlocks
     *
     * @param int $action_id Action ID
     * @param Exception $exception Exception that caused failure
     */
    public function handle_failed_action($action_id, $exception) {
        $message = $exception->getMessage();
        
        // Check if it's a deadlock error
        if (stripos($message, 'deadlock') !== false || stripos($message, 'lock wait timeout') !== false) {
            // Log the deadlock
            if (function_exists('delcampe_log')) {
                delcampe_log('Action Scheduler deadlock detected for action ' . $action_id . ': ' . $message, 'warning');
            }
            
            // Get the action
            $action = ActionScheduler::store()->fetch_action($action_id);
            if ($action) {
                $hook = $action->get_hook();
                
                // If it's one of our hooks, reschedule with delay
                if (strpos($hook, 'delcampe_') === 0) {
                    $this->reschedule_with_delay($action_id, $hook, $action->get_args());
                }
            }
        }
    }
    
    /**
     * Reschedule action with delay to avoid deadlock
     *
     * @param int $action_id Original action ID
     * @param string $hook Action hook
     * @param array $args Action arguments
     */
    private function reschedule_with_delay($action_id, $hook, $args) {
        // Add random delay between 5-15 seconds to avoid collision
        $delay = rand(5, 15);
        
        // Schedule new action
        as_schedule_single_action(time() + $delay, $hook, $args, 'delcampe');
        
        // Log rescheduling
        if (function_exists('delcampe_log')) {
            delcampe_log("Rescheduled action {$hook} with {$delay} second delay due to deadlock", 'info');
        }
    }
    
    /**
     * Custom retry schedule for our actions
     *
     * @param int $retry_schedule Current retry schedule
     * @param ActionScheduler_Action $action The action
     * @return int Modified retry schedule
     */
    public function custom_retry_schedule($retry_schedule, $action) {
        $hook = $action->get_hook();
        
        // For Delcampe actions, use exponential backoff
        if (strpos($hook, 'delcampe_') === 0) {
            $attempts = $action->get_attempts();
            
            // Exponential backoff: 10s, 30s, 60s, 120s, 300s
            $delays = array(10, 30, 60, 120, 300);
            $delay = isset($delays[$attempts]) ? $delays[$attempts] : 300;
            
            return time() + $delay;
        }
        
        return $retry_schedule;
    }
    
    /**
     * Optimize Action Scheduler tables periodically
     * 
     * DISABLED: OPTIMIZE TABLE causes severe performance issues with large tables
     * Production sites can have millions of rows in Action Scheduler tables (1.5GB+)
     * OPTIMIZE locks tables for extended periods causing timeouts and 200%+ CPU usage
     * 
     * @since 1.10.10.1
     * @deprecated 1.10.11.2 Disabled due to performance issues on production
     */
    public function optimize_tables() {
        // DISABLED: OPTIMIZE TABLE causes severe performance issues with large tables
        // The Action Scheduler tables have millions of rows and OPTIMIZE locks them
        // for extended periods causing timeouts and high CPU usage
        return;
    }
    
    /**
     * Add index to improve Action Scheduler performance
     * This should be called on plugin activation
     */
    public static function add_indexes() {
        global $wpdb;
        
        $table = $wpdb->prefix . 'actionscheduler_actions';
        
        // Check if table exists
        $table_exists = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = %s AND table_name = %s",
            DB_NAME,
            $table
        ));
        
        if (!$table_exists) {
            return;
        }
        
        // Add composite index for better query performance
        $index_exists = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM information_schema.statistics 
             WHERE table_schema = %s AND table_name = %s AND index_name = %s",
            DB_NAME,
            $table,
            'idx_hook_status_scheduled'
        ));
        
        if (!$index_exists) {
            // SECURITY FIX v1.10.21.2: Properly escape table names in schema operations
            $safe_table = esc_sql($table);
            $wpdb->query("ALTER TABLE `$safe_table` ADD INDEX idx_hook_status_scheduled (hook(191), status(20), scheduled_date_gmt)");
        }
        
        // Add index for group_id if not exists
        $group_index_exists = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM information_schema.statistics 
             WHERE table_schema = %s AND table_name = %s AND index_name = %s",
            DB_NAME,
            $table,
            'idx_group_status'
        ));
        
        if (!$group_index_exists) {
            // SECURITY FIX v1.10.21.2: Properly escape table names in schema operations
            $safe_table = esc_sql($table);
            $wpdb->query("ALTER TABLE `$safe_table` ADD INDEX idx_group_status (group_id, status(20))");
        }
    }
    
    /**
     * Clean up old completed actions to prevent table bloat
     * 
     * Improved in v1.10.11.2:
     * - Added time limit to prevent timeouts
     * - Reduced batch size for better performance
     * - Added processing counter to track cleanup progress
     */
    public static function cleanup_old_actions() {
        if (!function_exists('as_get_scheduled_actions')) {
            return;
        }
        
        // Set a time limit for this operation (max 20 seconds)
        $start_time = time();
        $max_time = 20;
        $deleted_count = 0;
        
        // Delete completed actions older than 7 days
        $cutoff = strtotime('-7 days');
        
        // Process in smaller batches to avoid memory issues
        $args = array(
            'status' => ActionScheduler_Store::STATUS_COMPLETE,
            'modified' => $cutoff,
            'modified_compare' => '<',
            'per_page' => 20,  // Reduced from 100 to 20 for better performance
            'orderby' => 'none'
        );
        
        // Keep processing until time limit or no more actions
        while ((time() - $start_time) < $max_time) {
            $actions = as_get_scheduled_actions($args);
            
            if (empty($actions)) {
                break;
            }
            
            foreach ($actions as $action_id => $action) {
                ActionScheduler::store()->delete_action($action_id);
                $deleted_count++;
                
                // Check time limit within the loop
                if ((time() - $start_time) >= $max_time) {
                    break 2;
                }
            }
        }
        
        // Only process failed actions if we have time left
        if ((time() - $start_time) < ($max_time - 5)) {
            // Also clean up failed actions older than 30 days
            $failed_cutoff = strtotime('-30 days');
            
            $failed_args = array(
                'status' => ActionScheduler_Store::STATUS_FAILED,
                'modified' => $failed_cutoff,
                'modified_compare' => '<',
                'per_page' => 20,  // Reduced batch size
                'orderby' => 'none'
            );
            
            $failed_actions = as_get_scheduled_actions($failed_args);
            
            foreach ($failed_actions as $action_id => $action) {
                ActionScheduler::store()->delete_action($action_id);
                $deleted_count++;
                
                // Check time limit
                if ((time() - $start_time) >= $max_time) {
                    break;
                }
            }
        }
        
        // Log cleanup results if any actions were deleted
        if ($deleted_count > 0 && function_exists('delcampe_log')) {
            delcampe_log("Cleaned up {$deleted_count} old Action Scheduler actions", 'debug');
        }
    }
}

// Initialize the class
add_action('init', array('Delcampe_Deadlock_Prevention', 'get_instance'));

// Schedule cleanup of old actions
add_action('init', function() {
    if (!wp_next_scheduled('delcampe_cleanup_action_scheduler')) {
        wp_schedule_event(time(), 'daily', 'delcampe_cleanup_action_scheduler');
    }
});

// Hook the cleanup function
add_action('delcampe_cleanup_action_scheduler', array('Delcampe_Deadlock_Prevention', 'cleanup_old_actions'));