<?php
/**
 * Delcampe Development Utilities
 * Version: 1.0.0.1
 * 
 * Helper functions for development environment
 * Now fully dynamic - no hardcoded URLs
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.0.0.0
 * @version 1.0.0.1
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Check if we're running on a development or local site
 * 
 * @since 1.0.0.0
 * @version 1.0.0.1 - Now checks for any non-production indicators
 * @return bool
 */
function delcampe_is_dev_site() {
    // Check if WP_DEBUG is on (common for dev sites)
    if (defined('WP_DEBUG') && WP_DEBUG) {
        return true;
    }
    
    // Check if WP_ENVIRONMENT_TYPE is set to local or development
    if (defined('WP_ENVIRONMENT_TYPE') && in_array(WP_ENVIRONMENT_TYPE, array('local', 'development'))) {
        return true;
    }
    
    // Check environment variable
    if (getenv('WP_ENV') && in_array(getenv('WP_ENV'), array('development', 'local', 'dev'))) {
        return true;
    }
    
    // Get the site URL
    $site_url = get_site_url();
    $parsed_url = parse_url($site_url);
    $host = isset($parsed_url['host']) ? $parsed_url['host'] : '';
    
    // Check for common development indicators in the domain
    $dev_indicators = array(
        '.local',
        '.dev',
        '.test',
        '.localhost',
        'staging.',
        'dev.',
        'test.',
        'demo.',
        'localhost'
    );
    
    foreach ($dev_indicators as $indicator) {
        if (stripos($host, $indicator) !== false) {
            return true;
        }
    }
    
    // Check for private IP ranges
    if (preg_match('/^(?:192\.168|10\.|172\.(?:1[6-9]|2[0-9]|3[01]))\.|^127\.|^::1$/i', $host)) {
        return true;
    }
    
    // Check for port numbers (common in development)
    if (isset($parsed_url['port']) && !in_array($parsed_url['port'], array(80, 443))) {
        return true;
    }
    
    return false;
}

/**
 * Get a user-friendly site type description
 * 
 * @since 1.0.0.1
 * @return string
 */
function delcampe_get_site_type() {
    if (delcampe_is_dev_site()) {
        return __('Development/Local Site', 'wc-delcampe-integration');
    }
    return __('Production Site', 'wc-delcampe-integration');
}

/**
 * Site information notice disabled - moved to Developer tab
 * 
 * @since 1.0.0.0
 * @version 1.10.8.0 - Disabled to reduce admin notice clutter
 */
/* Disabled - Site information moved to Developer tab in Settings
add_action('admin_notices', function() {
    if (isset($_GET['page']) && strpos($_GET['page'], 'delcampe') !== false) {
        $site_url = get_site_url();
        $site_type = delcampe_get_site_type();
        $notice_class = delcampe_is_dev_site() ? 'notice-info' : 'notice-success';
        ?>
        <div class="notice <?php echo esc_attr($notice_class); ?>">
            <p>
                <strong><?php _e('Site Information:', 'wc-delcampe-integration'); ?></strong> 
                <?php echo esc_html($site_type); ?> - 
                <?php echo esc_html($site_url); ?>
                <?php if (delcampe_is_dev_site()): ?>
                    <br>
                    <em><?php _e('Note: Ensure your site is publicly accessible for Delcampe to fetch images.', 'wc-delcampe-integration'); ?></em>
                <?php endif; ?>
            </p>
        </div>
        <?php
    }
});
*/

/**
 * Add debug information to sync logs
 * 
 * @since 1.0.0.0
 * @version 1.0.0.1 - Shows actual site URL
 */
add_action('delcampe_before_sync', function($product_id) {
    $site_url = get_site_url();
    $site_type = delcampe_is_dev_site() ? 'DEV' : 'PROD';
    error_log(sprintf(
        '[Delcampe %s] Syncing product %d from: %s',
        $site_type,
        $product_id,
        $site_url
    ));
});

/**
 * Filter to check if site is externally accessible
 * Can be overridden by developers
 * 
 * @since 1.0.0.1
 * @return bool
 */
add_filter('delcampe_site_is_public', function($is_public) {
    // Allow override via constant
    if (defined('DELCAMPE_FORCE_PUBLIC') && DELCAMPE_FORCE_PUBLIC) {
        return true;
    }
    
    // Check if site appears to be private
    $site_url = get_site_url();
    $parsed_url = parse_url($site_url);
    $host = isset($parsed_url['host']) ? $parsed_url['host'] : '';
    
    // Private IP check
    if (preg_match('/^(?:192\.168|10\.|172\.(?:1[6-9]|2[0-9]|3[01]))\.|^127\.|^::1$/i', $host)) {
        return false;
    }
    
    // Localhost check
    if (in_array(strtolower($host), array('localhost', 'localhost.localdomain'))) {
        return false;
    }
    
    return $is_public;
}, 10, 1);

/**
 * Add settings field for forcing public mode
 * Useful for sites behind tunnels or proxies
 * 
 * @since 1.0.0.1
 */
add_action('delcampe_settings_advanced', function() {
    ?>
    <tr>
        <th scope="row">
            <label for="delcampe_force_public">
                <?php _e('Force Public Mode', 'wc-delcampe-integration'); ?>
            </label>
        </th>
        <td>
            <input type="checkbox" 
                   id="delcampe_force_public" 
                   name="delcampe_force_public" 
                   value="1"
                   <?php checked(get_option('delcampe_force_public'), '1'); ?>>
            <p class="description">
                <?php _e('Check this if your site is accessible via tunnel (ngrok, Cloudflare, etc.) but appears as local.', 'wc-delcampe-integration'); ?>
            </p>
        </td>
    </tr>
    <?php
});

// Save the force public setting
add_action('admin_init', function() {
    if (isset($_POST['delcampe_force_public'])) {
        update_option('delcampe_force_public', '1');
    } elseif (isset($_POST['delcampe_save_settings'])) {
        delete_option('delcampe_force_public');
    }
});

// Define constant based on setting
add_action('init', function() {
    if (!defined('DELCAMPE_FORCE_PUBLIC')) {
        define('DELCAMPE_FORCE_PUBLIC', get_option('delcampe_force_public') === '1');
    }
});
