<?php
/**
 * Delcampe Duplicate Monitor
 * 
 * Enhanced logging and monitoring for duplicate detection
 * Part of Phase 1 duplicate prevention implementation
 * 
 * @package    Delcampe_Sync
 * @subpackage Delcampe_Sync/includes
 * @since      1.10.35.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Delcampe_Duplicate_Monitor {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Log file path
     */
    private $log_file;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $upload_dir = wp_upload_dir();
        $log_dir = $upload_dir['basedir'] . '/delcampe-logs';
        
        // Create log directory if it doesn't exist
        if ( ! file_exists( $log_dir ) ) {
            wp_mkdir_p( $log_dir );
        }
        
        $this->log_file = $log_dir . '/duplicate-monitor.log';
        
        // Hook into listing creation attempts
        add_action( 'delcampe_duplicate_prevented', array( $this, 'log_duplicate_prevention' ), 10, 2 );
        add_action( 'delcampe_listing_created', array( $this, 'log_successful_creation' ), 10, 2 );
        
        // Admin notices
        add_action( 'admin_notices', array( $this, 'display_duplicate_notices' ) );
        
        // Daily report
        add_action( 'delcampe_daily_duplicate_report', array( $this, 'send_daily_report' ) );
        
        // Schedule daily report if not already scheduled
        if ( ! wp_next_scheduled( 'delcampe_daily_duplicate_report' ) ) {
            wp_schedule_event( strtotime( 'tomorrow 6:00 AM' ), 'daily', 'delcampe_daily_duplicate_report' );
        }
    }
    
    /**
     * Log duplicate prevention event
     */
    public function log_duplicate_prevention( $product_id, $details = array() ) {
        $log_entry = array(
            'timestamp' => current_time( 'mysql' ),
            'event' => 'duplicate_prevented',
            'product_id' => $product_id,
            'details' => $details,
            'user_id' => get_current_user_id(),
            'source' => $this->get_request_source()
        );
        
        $this->write_log( $log_entry );
        
        // Store in database for reporting
        global $wpdb;
        $wpdb->insert(
            $wpdb->prefix . 'delcampe_duplicate_log',
            array(
                'product_id' => $product_id,
                'event_type' => 'prevented',
                'details' => json_encode( $details ),
                'user_id' => get_current_user_id(),
                'source' => $this->get_request_source(),
                'created_at' => current_time( 'mysql' )
            )
        );
        
        // Increment counter for admin notice
        $count = get_transient( 'delcampe_duplicates_prevented_today' );
        set_transient( 'delcampe_duplicates_prevented_today', intval( $count ) + 1, DAY_IN_SECONDS );
    }
    
    /**
     * Log successful listing creation
     */
    public function log_successful_creation( $product_id, $response ) {
        $log_entry = array(
            'timestamp' => current_time( 'mysql' ),
            'event' => 'listing_created',
            'product_id' => $product_id,
            'delcampe_id' => isset( $response['id'] ) ? $response['id'] : '',
            'personal_reference' => isset( $response['personal_reference'] ) ? $response['personal_reference'] : '',
            'user_id' => get_current_user_id(),
            'source' => $this->get_request_source()
        );
        
        $this->write_log( $log_entry );
    }
    
    /**
     * Get request source (batch, manual, webhook, cron, etc)
     */
    private function get_request_source() {
        // Check if it's a cron job
        if ( defined( 'DOING_CRON' ) && DOING_CRON ) {
            return 'cron';
        }
        
        // Check if it's an AJAX request
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
            if ( isset( $_POST['action'] ) ) {
                if ( strpos( $_POST['action'], 'batch' ) !== false ) {
                    return 'batch_queue';
                }
                if ( strpos( $_POST['action'], 'bulk' ) !== false ) {
                    return 'bulk_action';
                }
            }
            return 'ajax';
        }
        
        // Check if it's a webhook
        if ( isset( $_SERVER['REQUEST_URI'] ) && strpos( $_SERVER['REQUEST_URI'], 'webhook' ) !== false ) {
            return 'webhook';
        }
        
        // Check if it's from admin
        if ( is_admin() ) {
            return 'admin_manual';
        }
        
        return 'unknown';
    }
    
    /**
     * Write to log file
     */
    private function write_log( $entry ) {
        $log_line = sprintf(
            "[%s] %s | Product: %d | User: %d | Source: %s | Details: %s\n",
            $entry['timestamp'],
            $entry['event'],
            isset( $entry['product_id'] ) ? $entry['product_id'] : 0,
            isset( $entry['user_id'] ) ? $entry['user_id'] : 0,
            isset( $entry['source'] ) ? $entry['source'] : 'unknown',
            json_encode( isset( $entry['details'] ) ? $entry['details'] : $entry )
        );
        
        error_log( $log_line, 3, $this->log_file );
    }
    
    /**
     * Display admin notices for duplicate prevention
     */
    public function display_duplicate_notices() {
        // Only show on Delcampe admin pages
        if ( ! isset( $_GET['page'] ) || strpos( $_GET['page'], 'delcampe' ) === false ) {
            return;
        }
        
        $prevented_today = get_transient( 'delcampe_duplicates_prevented_today' );
        
        if ( $prevented_today > 0 ) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong>Delcampe Duplicate Prevention:</strong> 
                    <?php printf( 
                        __( '%d duplicate listing attempts were prevented today.', 'wc-delcampe-integration' ),
                        $prevented_today
                    ); ?>
                    <a href="<?php echo admin_url( 'admin.php?page=delcampe-duplicate-report' ); ?>">
                        <?php _e( 'View Details', 'wc-delcampe-integration' ); ?>
                    </a>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Send daily duplicate report
     */
    public function send_daily_report() {
        global $wpdb;
        
        // Get yesterday's data
        $yesterday = date( 'Y-m-d', strtotime( '-1 day' ) );
        
        $results = $wpdb->get_results( $wpdb->prepare(
            "SELECT 
                event_type,
                source,
                COUNT(*) as count,
                COUNT(DISTINCT product_id) as unique_products
             FROM {$wpdb->prefix}delcampe_duplicate_log
             WHERE DATE(created_at) = %s
             GROUP BY event_type, source
             ORDER BY count DESC",
            $yesterday
        ) );
        
        if ( empty( $results ) ) {
            return; // No duplicates yesterday
        }
        
        // Build report
        $report = "Delcampe Duplicate Prevention Report for $yesterday\n";
        $report .= "=" . str_repeat( "=", 50 ) . "\n\n";
        
        $total_prevented = 0;
        foreach ( $results as $row ) {
            if ( $row->event_type === 'prevented' ) {
                $total_prevented += $row->count;
                $report .= sprintf(
                    "Source: %s - %d attempts prevented (%d unique products)\n",
                    $row->source,
                    $row->count,
                    $row->unique_products
                );
            }
        }
        
        $report .= "\nTotal Duplicates Prevented: $total_prevented\n";
        
        // Get top products with duplicate attempts
        $top_products = $wpdb->get_results( $wpdb->prepare(
            "SELECT 
                product_id,
                COUNT(*) as attempts,
                GROUP_CONCAT(DISTINCT source) as sources
             FROM {$wpdb->prefix}delcampe_duplicate_log
             WHERE DATE(created_at) = %s
             AND event_type = 'prevented'
             GROUP BY product_id
             HAVING attempts > 1
             ORDER BY attempts DESC
             LIMIT 10",
            $yesterday
        ) );
        
        if ( ! empty( $top_products ) ) {
            $report .= "\nTop Products with Multiple Duplicate Attempts:\n";
            foreach ( $top_products as $product ) {
                $product_obj = wc_get_product( $product->product_id );
                $product_name = $product_obj ? $product_obj->get_name() : 'Unknown';
                $report .= sprintf(
                    "- Product #%d (%s): %d attempts from %s\n",
                    $product->product_id,
                    $product_name,
                    $product->attempts,
                    $product->sources
                );
            }
        }
        
        // Send email to admin
        $admin_email = get_option( 'admin_email' );
        $subject = sprintf( 
            '[%s] Delcampe Duplicate Prevention Report - %s',
            get_bloginfo( 'name' ),
            $yesterday
        );
        
        wp_mail( $admin_email, $subject, $report );
        
        // Also log to file
        $this->write_log( array(
            'timestamp' => current_time( 'mysql' ),
            'event' => 'daily_report_sent',
            'details' => array(
                'date' => $yesterday,
                'total_prevented' => $total_prevented
            )
        ) );
    }
    
    /**
     * Create database table for duplicate logging
     */
    public static function create_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_duplicate_log';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            product_id bigint(20) NOT NULL,
            event_type varchar(50) NOT NULL,
            details longtext,
            user_id bigint(20),
            source varchar(50),
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY event_type (event_type),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
    }
}