<?php
/**
 * Delcampe Feature Flags Manager
 * 
 * Manages feature flags for gradual rollout of new functionality
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Features
 * @since 1.10.20.3
 * @version 1.10.20.3
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_Feature_Flags
 * 
 * Provides centralized feature flag management for controlled rollouts
 */
class Delcampe_Feature_Flags {
    
    /**
     * Option name for storing feature flags
     */
    const OPTION_NAME = 'delcampe_feature_flags';
    
    /**
     * Default feature flags configuration
     */
    private static $default_flags = array(
        'stock_caching' => array(
            'enabled' => false,
            'description' => 'Enable stock caching system to reduce API calls',
            'rollout_percentage' => 0,
            'whitelist_users' => array(), // User IDs for testing
            'since_version' => '1.10.20.3'
        ),
        'batch_cache_refresh' => array(
            'enabled' => false,
            'description' => 'Enable automatic batch refresh of stock cache',
            'rollout_percentage' => 0,
            'whitelist_users' => array(),
            'since_version' => '1.10.20.3'
        ),
        'cache_ui_indicators' => array(
            'enabled' => false,
            'description' => 'Show cache freshness indicators in inventory UI',
            'rollout_percentage' => 0,
            'whitelist_users' => array(),
            'since_version' => '1.10.20.3'
        ),
        'cache_monitoring' => array(
            'enabled' => false,
            'description' => 'Enable cache performance monitoring and alerts',
            'rollout_percentage' => 0,
            'whitelist_users' => array(),
            'since_version' => '1.10.20.3'
        )
    );
    
    /**
     * Get all feature flags
     * 
     * @return array All feature flags with their status
     */
    public static function get_all_flags() {
        $saved_flags = get_option(self::OPTION_NAME, array());
        return wp_parse_args($saved_flags, self::$default_flags);
    }
    
    /**
     * Check if a feature is enabled
     * 
     * @param string $feature_name Feature flag name
     * @param int $user_id Optional user ID for user-specific checks
     * @return bool Whether the feature is enabled
     */
    public static function is_enabled($feature_name, $user_id = null) {
        $flags = self::get_all_flags();
        
        // Feature doesn't exist
        if (!isset($flags[$feature_name])) {
            return false;
        }
        
        $flag = $flags[$feature_name];
        
        // Feature globally enabled
        if (!empty($flag['enabled'])) {
            return true;
        }
        
        // Check user whitelist
        if ($user_id === null) {
            $user_id = get_current_user_id();
        }
        
        if ($user_id && !empty($flag['whitelist_users']) && in_array($user_id, $flag['whitelist_users'])) {
            return true;
        }
        
        // Check rollout percentage
        if (!empty($flag['rollout_percentage']) && $flag['rollout_percentage'] > 0) {
            return self::is_in_rollout($user_id, $flag['rollout_percentage']);
        }
        
        return false;
    }
    
    /**
     * Check if user is in rollout percentage
     * 
     * @param int $user_id User ID
     * @param int $percentage Rollout percentage (0-100)
     * @return bool Whether user is in rollout
     */
    private static function is_in_rollout($user_id, $percentage) {
        if ($percentage >= 100) {
            return true;
        }
        
        if ($percentage <= 0) {
            return false;
        }
        
        // Use consistent hash for user to ensure stable assignment
        $hash = crc32($user_id . 'delcampe_rollout');
        $bucket = $hash % 100;
        
        return $bucket < $percentage;
    }
    
    /**
     * Enable a feature flag
     * 
     * @param string $feature_name Feature flag name
     * @param array $options Optional configuration
     * @return bool Success status
     */
    public static function enable_feature($feature_name, $options = array()) {
        $flags = self::get_all_flags();
        
        if (!isset($flags[$feature_name])) {
            return false;
        }
        
        $flags[$feature_name]['enabled'] = true;
        
        // Apply additional options
        if (!empty($options['rollout_percentage'])) {
            $flags[$feature_name]['rollout_percentage'] = intval($options['rollout_percentage']);
        }
        
        if (!empty($options['whitelist_users'])) {
            $flags[$feature_name]['whitelist_users'] = array_map('intval', $options['whitelist_users']);
        }
        
        return update_option(self::OPTION_NAME, $flags);
    }
    
    /**
     * Disable a feature flag
     * 
     * @param string $feature_name Feature flag name
     * @return bool Success status
     */
    public static function disable_feature($feature_name) {
        $flags = self::get_all_flags();
        
        if (!isset($flags[$feature_name])) {
            return false;
        }
        
        $flags[$feature_name]['enabled'] = false;
        $flags[$feature_name]['rollout_percentage'] = 0;
        
        return update_option(self::OPTION_NAME, $flags);
    }
    
    /**
     * Set rollout percentage for a feature
     * 
     * @param string $feature_name Feature flag name
     * @param int $percentage Rollout percentage (0-100)
     * @return bool Success status
     */
    public static function set_rollout_percentage($feature_name, $percentage) {
        $flags = self::get_all_flags();
        
        if (!isset($flags[$feature_name])) {
            return false;
        }
        
        $percentage = max(0, min(100, intval($percentage)));
        $flags[$feature_name]['rollout_percentage'] = $percentage;
        
        // Auto-enable if percentage > 0
        if ($percentage > 0) {
            $flags[$feature_name]['enabled'] = false; // Use rollout instead of global enable
        }
        
        return update_option(self::OPTION_NAME, $flags);
    }
    
    /**
     * Add user to feature whitelist
     * 
     * @param string $feature_name Feature flag name
     * @param int $user_id User ID to whitelist
     * @return bool Success status
     */
    public static function add_to_whitelist($feature_name, $user_id) {
        $flags = self::get_all_flags();
        
        if (!isset($flags[$feature_name])) {
            return false;
        }
        
        $user_id = intval($user_id);
        if (!in_array($user_id, $flags[$feature_name]['whitelist_users'])) {
            $flags[$feature_name]['whitelist_users'][] = $user_id;
        }
        
        return update_option(self::OPTION_NAME, $flags);
    }
    
    /**
     * Remove user from feature whitelist
     * 
     * @param string $feature_name Feature flag name
     * @param int $user_id User ID to remove
     * @return bool Success status
     */
    public static function remove_from_whitelist($feature_name, $user_id) {
        $flags = self::get_all_flags();
        
        if (!isset($flags[$feature_name])) {
            return false;
        }
        
        $user_id = intval($user_id);
        $flags[$feature_name]['whitelist_users'] = array_diff(
            $flags[$feature_name]['whitelist_users'],
            array($user_id)
        );
        
        return update_option(self::OPTION_NAME, $flags);
    }
    
    /**
     * Reset all feature flags to defaults
     * 
     * @return bool Success status
     */
    public static function reset_to_defaults() {
        return update_option(self::OPTION_NAME, self::$default_flags);
    }
    
    /**
     * Get feature flag statistics
     * 
     * @return array Statistics about feature flags
     */
    public static function get_statistics() {
        $flags = self::get_all_flags();
        
        $stats = array(
            'total_features' => count($flags),
            'enabled_features' => 0,
            'rollout_features' => 0,
            'whitelist_features' => 0
        );
        
        foreach ($flags as $flag) {
            if (!empty($flag['enabled'])) {
                $stats['enabled_features']++;
            }
            
            if (!empty($flag['rollout_percentage']) && $flag['rollout_percentage'] > 0) {
                $stats['rollout_features']++;
            }
            
            if (!empty($flag['whitelist_users'])) {
                $stats['whitelist_features']++;
            }
        }
        
        return $stats;
    }
    
    /**
     * Log feature flag usage
     * 
     * @param string $feature_name Feature being checked
     * @param bool $enabled Whether it was enabled
     */
    private static function log_usage($feature_name, $enabled) {
        // Track feature flag checks for analytics
        $usage_key = 'delcampe_feature_usage_' . date('Y-m-d');
        $usage = get_transient($usage_key) ?: array();
        
        if (!isset($usage[$feature_name])) {
            $usage[$feature_name] = array('checks' => 0, 'enabled' => 0);
        }
        
        $usage[$feature_name]['checks']++;
        if ($enabled) {
            $usage[$feature_name]['enabled']++;
        }
        
        set_transient($usage_key, $usage, DAY_IN_SECONDS);
    }
    
    /**
     * Admin interface for managing feature flags
     */
    public static function render_admin_interface() {
        $flags = self::get_all_flags();
        ?>
        <div class="wrap">
            <h2>Delcampe Feature Flags</h2>
            
            <div class="notice notice-info">
                <p>Feature flags allow gradual rollout of new functionality. Use these controls to enable features for testing or production deployment.</p>
            </div>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Feature</th>
                        <th>Description</th>
                        <th>Status</th>
                        <th>Rollout %</th>
                        <th>Whitelisted Users</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($flags as $name => $flag): ?>
                    <tr>
                        <td><strong><?php echo esc_html($name); ?></strong></td>
                        <td><?php echo esc_html($flag['description']); ?></td>
                        <td>
                            <?php if ($flag['enabled']): ?>
                                <span class="dashicons dashicons-yes" style="color: green;"></span> Enabled
                            <?php elseif ($flag['rollout_percentage'] > 0): ?>
                                <span class="dashicons dashicons-clock" style="color: orange;"></span> Rolling Out
                            <?php else: ?>
                                <span class="dashicons dashicons-no" style="color: red;"></span> Disabled
                            <?php endif; ?>
                        </td>
                        <td><?php echo esc_html($flag['rollout_percentage']); ?>%</td>
                        <td><?php echo count($flag['whitelist_users']); ?> users</td>
                        <td>
                            <button class="button button-small toggle-feature" data-feature="<?php echo esc_attr($name); ?>">
                                <?php echo $flag['enabled'] ? 'Disable' : 'Enable'; ?>
                            </button>
                            <button class="button button-small configure-feature" data-feature="<?php echo esc_attr($name); ?>">
                                Configure
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <div class="feature-stats" style="margin-top: 20px;">
                <h3>Feature Flag Statistics</h3>
                <?php $stats = self::get_statistics(); ?>
                <ul>
                    <li>Total Features: <?php echo $stats['total_features']; ?></li>
                    <li>Enabled Features: <?php echo $stats['enabled_features']; ?></li>
                    <li>Features in Rollout: <?php echo $stats['rollout_features']; ?></li>
                    <li>Features with Whitelist: <?php echo $stats['whitelist_features']; ?></li>
                </ul>
            </div>
        </div>
        <?php
    }
}

// Convenience function for checking feature flags
if (!function_exists('delcampe_feature_enabled')) {
    function delcampe_feature_enabled($feature_name) {
        return Delcampe_Feature_Flags::is_enabled($feature_name);
    }
}