<?php
/**
 * Delcampe Image Manager
 * 
 * Handles image import/export between WooCommerce and Delcampe
 * 
 * @package WC_Delcampe_Integration
 * @version 1.10.10.0
 * @since   1.10.10.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Delcampe_Image_Manager class
 *
 * @since 1.10.10.0
 */
class Delcampe_Image_Manager {
    
    /**
     * Singleton instance
     * @var Delcampe_Image_Manager
     */
    private static $instance = null;
    
    /**
     * Auth instance
     * @var Delcampe_Auth
     */
    private $auth;
    
    /**
     * Logger instance
     * @var Delcampe_Sync_Logger
     */
    private $logger;
    
    /**
     * Maximum image size in bytes (10MB)
     * @var int
     */
    const MAX_IMAGE_SIZE = 10485760;
    
    /**
     * Allowed image types
     * @var array
     */
    const ALLOWED_TYPES = array('image/jpeg', 'image/jpg', 'image/png', 'image/gif');
    
    /**
     * Get singleton instance
     *
     * @return Delcampe_Image_Manager
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init();
    }
    
    /**
     * Initialize image manager
     */
    private function init() {
        // Initialize dependencies
        $this->auth = Delcampe_Auth::get_instance();
        $this->logger = Delcampe_Sync_Logger::get_instance();
        
        // Hook into listing creation/update
        add_filter('delcampe_listing_data', array($this, 'add_images_to_listing'), 10, 2);
        
        // Hook into product update to sync images
        add_action('woocommerce_update_product', array($this, 'sync_images_to_delcampe'), 20, 1);
        
        // Add AJAX handlers
        add_action('wp_ajax_delcampe_import_images', array($this, 'ajax_import_images'));
        add_action('wp_ajax_delcampe_export_images', array($this, 'ajax_export_images'));
        
        // Add webhook handler for image updates
        add_action('delcampe_webhook_image_update', array($this, 'handle_webhook_image_update'), 10, 2);
        
        // Add admin settings
        add_filter('delcampe_sync_settings', array($this, 'add_image_sync_settings'));
        
        // Add metabox for manual image sync
        add_action('add_meta_boxes', array($this, 'add_image_sync_metabox'));
    }
    
    /**
     * Add images to listing data
     *
     * @param array $listing_data Current listing data
     * @param int $product_id Product ID
     * @return array Modified listing data
     */
    public function add_images_to_listing($listing_data, $product_id) {
        // Check if image export is enabled
        if (get_option('delcampe_enable_image_export', 'yes') !== 'yes') {
            return $listing_data;
        }
        
        $product = wc_get_product($product_id);
        if (!$product) {
            return $listing_data;
        }
        
        $images = array();
        
        // Get main image
        $main_image_id = $product->get_image_id();
        if ($main_image_id) {
            $main_image_url = wp_get_attachment_url($main_image_id);
            if ($main_image_url) {
                $images[] = array(
                    'url' => $main_image_url,
                    'is_main' => true,
                    'position' => 0
                );
            }
        }
        
        // Get gallery images
        $gallery_image_ids = $product->get_gallery_image_ids();
        $position = 1;
        foreach ($gallery_image_ids as $image_id) {
            $image_url = wp_get_attachment_url($image_id);
            if ($image_url) {
                $images[] = array(
                    'url' => $image_url,
                    'is_main' => false,
                    'position' => $position++
                );
            }
        }
        
        // Add images to listing data
        if (!empty($images)) {
            $listing_data['images'] = $images;
            
            // Log image export
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'IMAGE_EXPORT_PREPARED',
                'product_id' => $product_id,
                'image_count' => count($images)
            ));
        }
        
        return $listing_data;
    }
    
    /**
     * Import images from Delcampe to WooCommerce
     *
     * @param int $product_id Product ID
     * @param string $delcampe_id Delcampe listing ID
     * @return bool Success status
     */
    public function import_images_from_delcampe($product_id, $delcampe_id) {
        try {
            // Get auth token
            $token = $this->auth->get_auth_token();
            if (!$token) {
                throw new Exception('Failed to get auth token');
            }
            
            // Get item details from Delcampe
            $api_url = 'https://rest.delcampe.net/item/' . $delcampe_id;
            
            $response = wp_remote_get($api_url, array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $token
                ),
                'timeout' => 30
            ));
            
            if (is_wp_error($response)) {
                throw new Exception($response->get_error_message());
            }
            
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (!isset($data['images']) || empty($data['images'])) {
                return false;
            }
            
            $product = wc_get_product($product_id);
            if (!$product) {
                throw new Exception('Product not found');
            }
            
            $imported_images = array();
            $main_image_set = false;
            
            foreach ($data['images'] as $index => $image) {
                $image_url = $image['url'] ?? $image['large_url'] ?? $image['medium_url'] ?? null;
                
                if (!$image_url) {
                    continue;
                }
                
                // Download and attach image
                $attachment_id = $this->download_and_attach_image($image_url, $product_id);
                
                if ($attachment_id) {
                    if (!$main_image_set && ($image['is_main'] ?? $index === 0)) {
                        // Set as main product image
                        $product->set_image_id($attachment_id);
                        $main_image_set = true;
                    } else {
                        // Add to gallery
                        $imported_images[] = $attachment_id;
                    }
                }
            }
            
            // Update gallery images
            if (!empty($imported_images)) {
                $existing_gallery = $product->get_gallery_image_ids();
                $merged_gallery = array_unique(array_merge($existing_gallery, $imported_images));
                $product->set_gallery_image_ids($merged_gallery);
            }
            
            $product->save();
            
            // Log success
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'IMAGES_IMPORTED',
                'product_id' => $product_id,
                'delcampe_id' => $delcampe_id,
                'image_count' => count($imported_images) + ($main_image_set ? 1 : 0)
            ));
            
            return true;
            
        } catch (Exception $e) {
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'IMAGE_IMPORT_ERROR',
                'product_id' => $product_id,
                'delcampe_id' => $delcampe_id,
                'error' => $e->getMessage()
            ));
            return false;
        }
    }
    
    /**
     * Download and attach image to product
     *
     * @param string $image_url Image URL
     * @param int $product_id Product ID
     * @return int|false Attachment ID or false on failure
     */
    private function download_and_attach_image($image_url, $product_id) {
        // Include required files
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        // Download image
        $tmp = download_url($image_url);
        
        if (is_wp_error($tmp)) {
            return false;
        }
        
        // Get file info
        $file_array = array(
            'name' => basename($image_url),
            'tmp_name' => $tmp
        );
        
        // Check file size
        if (filesize($tmp) > self::MAX_IMAGE_SIZE) {
            @unlink($tmp);
            return false;
        }
        
        // Check file type
        $file_type = wp_check_filetype($file_array['name']);
        if (!in_array($file_type['type'], self::ALLOWED_TYPES)) {
            @unlink($tmp);
            return false;
        }
        
        // Upload and attach to product
        $attachment_id = media_handle_sideload($file_array, $product_id);
        
        // Clean up temp file
        @unlink($tmp);
        
        if (is_wp_error($attachment_id)) {
            return false;
        }
        
        return $attachment_id;
    }
    
    /**
     * Sync images to Delcampe when product is updated
     *
     * @param int $product_id Product ID
     */
    public function sync_images_to_delcampe($product_id) {
        // Check if image sync is enabled
        if (get_option('delcampe_enable_image_sync', 'yes') !== 'yes') {
            return;
        }
        
        // Check if product is linked to Delcampe
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (empty($delcampe_id)) {
            return;
        }
        
        // Queue image update
        $this->queue_image_update($product_id, $delcampe_id);
    }
    
    /**
     * Queue image update for Delcampe
     *
     * @param int $product_id Product ID
     * @param string $delcampe_id Delcampe listing ID
     */
    private function queue_image_update($product_id, $delcampe_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_image_sync_queue';
        
        // Create table if it doesn't exist
        $this->create_image_sync_table();
        
        // Check if update already queued
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table_name} 
             WHERE product_id = %d AND status = 'pending'",
            $product_id
        ));
        
        if (!$existing) {
            // Insert new queue entry
            $wpdb->insert(
                $table_name,
                array(
                    'product_id' => $product_id,
                    'delcampe_id' => $delcampe_id,
                    'status' => 'pending',
                    'created_at' => current_time('mysql')
                ),
                array('%d', '%s', '%s', '%s')
            );
        }
    }
    
    /**
     * Process image sync queue
     */
    public function process_image_sync_queue() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_image_sync_queue';
        
        // Get pending updates
        $pending_updates = $wpdb->get_results(
            "SELECT * FROM {$table_name} 
             WHERE status = 'pending' 
             ORDER BY created_at ASC 
             LIMIT 5"
        );
        
        if (empty($pending_updates)) {
            return;
        }
        
        foreach ($pending_updates as $update) {
            $this->process_single_image_update($update);
        }
    }
    
    /**
     * Process single image update
     *
     * @param object $update Queue entry
     */
    private function process_single_image_update($update) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_image_sync_queue';
        
        try {
            // Get product
            $product = wc_get_product($update->product_id);
            if (!$product) {
                throw new Exception('Product not found');
            }
            
            // Prepare images data
            $images_data = array();
            
            // Get main image
            $main_image_id = $product->get_image_id();
            if ($main_image_id) {
                $main_image_url = wp_get_attachment_url($main_image_id);
                if ($main_image_url) {
                    $images_data[] = array(
                        'url' => $main_image_url,
                        'is_main' => true
                    );
                }
            }
            
            // Get gallery images
            $gallery_image_ids = $product->get_gallery_image_ids();
            foreach ($gallery_image_ids as $image_id) {
                $image_url = wp_get_attachment_url($image_id);
                if ($image_url) {
                    $images_data[] = array(
                        'url' => $image_url,
                        'is_main' => false
                    );
                }
            }
            
            // Update images on Delcampe
            $this->update_delcampe_images($update->delcampe_id, $images_data);
            
            // Update queue status
            $wpdb->update(
                $table_name,
                array(
                    'status' => 'completed',
                    'processed_at' => current_time('mysql')
                ),
                array('id' => $update->id),
                array('%s', '%s'),
                array('%d')
            );
            
            // Log success
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'IMAGE_SYNC_SUCCESS',
                'product_id' => $update->product_id,
                'delcampe_id' => $update->delcampe_id,
                'image_count' => count($images_data)
            ));
            
        } catch (Exception $e) {
            // Update queue status
            $wpdb->update(
                $table_name,
                array(
                    'status' => 'failed',
                    'error_message' => $e->getMessage()
                ),
                array('id' => $update->id),
                array('%s', '%s'),
                array('%d')
            );
            
            // Log error
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'IMAGE_SYNC_ERROR',
                'product_id' => $update->product_id,
                'delcampe_id' => $update->delcampe_id,
                'error' => $e->getMessage()
            ));
        }
    }
    
    /**
     * Update images on Delcampe
     *
     * @param string $delcampe_id Delcampe listing ID
     * @param array $images_data Images data
     * @throws Exception
     */
    private function update_delcampe_images($delcampe_id, $images_data) {
        // Get auth token
        $token = $this->auth->get_auth_token();
        if (!$token) {
            throw new Exception('Failed to get auth token');
        }
        
        // API endpoint for updating images
        $api_url = 'https://rest.delcampe.net/item/' . $delcampe_id . '/images';
        
        $response = wp_remote_request($api_url, array(
            'method' => 'PUT',
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array('images' => $images_data)),
            'timeout' => 60
        ));
        
        if (is_wp_error($response)) {
            throw new Exception($response->get_error_message());
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200 && $status_code !== 204) {
            $body = wp_remote_retrieve_body($response);
            throw new Exception('API error: ' . $body);
        }
    }
    
    /**
     * AJAX handler for importing images
     */
    public function ajax_import_images() {
        // Check nonce
        if (!check_ajax_referer('delcampe_ajax_nonce', 'nonce', false)) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $product_id = intval($_POST['product_id'] ?? 0);
        $delcampe_id = sanitize_text_field($_POST['delcampe_id'] ?? '');
        
        if (!$product_id || !$delcampe_id) {
            wp_send_json_error('Missing required parameters');
        }
        
        $result = $this->import_images_from_delcampe($product_id, $delcampe_id);
        
        if ($result) {
            wp_send_json_success('Images imported successfully');
        } else {
            wp_send_json_error('Failed to import images');
        }
    }
    
    /**
     * AJAX handler for exporting images
     */
    public function ajax_export_images() {
        // Check nonce
        if (!check_ajax_referer('delcampe_ajax_nonce', 'nonce', false)) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $product_id = intval($_POST['product_id'] ?? 0);
        
        if (!$product_id) {
            wp_send_json_error('Missing product ID');
        }
        
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (!$delcampe_id) {
            wp_send_json_error('Product not linked to Delcampe');
        }
        
        $this->queue_image_update($product_id, $delcampe_id);
        
        wp_send_json_success('Images queued for export');
    }
    
    /**
     * Add image sync metabox
     */
    public function add_image_sync_metabox() {
        add_meta_box(
            'delcampe_image_sync',
            __('Delcampe Image Sync', 'wc-delcampe-integration'),
            array($this, 'render_image_sync_metabox'),
            'product',
            'side',
            'default'
        );
    }
    
    /**
     * Render image sync metabox
     *
     * @param WP_Post $post Post object
     */
    public function render_image_sync_metabox($post) {
        $delcampe_id = get_post_meta($post->ID, '_delcampe_listing_id', true);
        
        if (!$delcampe_id) {
            echo '<p>' . esc_html__('Product not linked to Delcampe', 'wc-delcampe-integration') . '</p>';
            return;
        }
        
        wp_nonce_field('delcampe_image_sync', 'delcampe_image_sync_nonce');
        ?>
        <div class="delcampe-image-sync-box">
            <p><?php esc_html_e('Sync images between WooCommerce and Delcampe', 'wc-delcampe-integration'); ?></p>
            
            <button type="button" class="button" id="delcampe-import-images" data-product-id="<?php echo esc_attr($post->ID); ?>" data-delcampe-id="<?php echo esc_attr($delcampe_id); ?>">
                <?php esc_html_e('Import from Delcampe', 'wc-delcampe-integration'); ?>
            </button>
            
            <button type="button" class="button" id="delcampe-export-images" data-product-id="<?php echo esc_attr($post->ID); ?>">
                <?php esc_html_e('Export to Delcampe', 'wc-delcampe-integration'); ?>
            </button>
            
            <div class="delcampe-image-sync-status" style="margin-top: 10px;"></div>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#delcampe-import-images').on('click', function() {
                var button = $(this);
                var statusDiv = $('.delcampe-image-sync-status');
                
                button.prop('disabled', true);
                statusDiv.html('<span class="spinner is-active"></span> Importing images...');
                
                $.post(ajaxurl, {
                    action: 'delcampe_import_images',
                    product_id: button.data('product-id'),
                    delcampe_id: button.data('delcampe-id'),
                    nonce: '<?php echo wp_create_nonce('delcampe_ajax_nonce'); ?>'
                }, function(response) {
                    button.prop('disabled', false);
                    if (response.success) {
                        statusDiv.html('<span style="color: green;">✓ ' + response.data + '</span>');
                        setTimeout(function() { location.reload(); }, 2000);
                    } else {
                        statusDiv.html('<span style="color: red;">✗ ' + response.data + '</span>');
                    }
                });
            });
            
            $('#delcampe-export-images').on('click', function() {
                var button = $(this);
                var statusDiv = $('.delcampe-image-sync-status');
                
                button.prop('disabled', true);
                statusDiv.html('<span class="spinner is-active"></span> Exporting images...');
                
                $.post(ajaxurl, {
                    action: 'delcampe_export_images',
                    product_id: button.data('product-id'),
                    nonce: '<?php echo wp_create_nonce('delcampe_ajax_nonce'); ?>'
                }, function(response) {
                    button.prop('disabled', false);
                    if (response.success) {
                        statusDiv.html('<span style="color: green;">✓ ' + response.data + '</span>');
                    } else {
                        statusDiv.html('<span style="color: red;">✗ ' + response.data + '</span>');
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Handle webhook image update notification
     *
     * @param string $delcampe_id Delcampe listing ID
     * @param array $images Updated images data
     */
    public function handle_webhook_image_update($delcampe_id, $images) {
        // Find the linked product
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        $listing = $wpdb->get_row($wpdb->prepare(
            "SELECT product_id FROM {$table_name} WHERE delcampe_id = %s",
            $delcampe_id
        ));
        
        if (!$listing) {
            return;
        }
        
        // Import the updated images
        $this->import_images_from_delcampe($listing->product_id, $delcampe_id);
    }
    
    /**
     * Add image sync settings
     *
     * @param array $settings Current settings
     * @return array Modified settings
     */
    public function add_image_sync_settings($settings) {
        $settings['image_sync'] = array(
            'title' => __('Image Synchronization', 'wc-delcampe-integration'),
            'fields' => array(
                'delcampe_enable_image_sync' => array(
                    'title' => __('Enable Image Sync', 'wc-delcampe-integration'),
                    'type' => 'checkbox',
                    'description' => __('Automatically sync images when products are updated', 'wc-delcampe-integration'),
                    'default' => 'yes'
                ),
                'delcampe_enable_image_export' => array(
                    'title' => __('Export Images to Delcampe', 'wc-delcampe-integration'),
                    'type' => 'checkbox',
                    'description' => __('Include product images when creating Delcampe listings', 'wc-delcampe-integration'),
                    'default' => 'yes'
                ),
                'delcampe_enable_image_import' => array(
                    'title' => __('Import Images from Delcampe', 'wc-delcampe-integration'),
                    'type' => 'checkbox',
                    'description' => __('Import images when syncing from Delcampe', 'wc-delcampe-integration'),
                    'default' => 'yes'
                ),
                'delcampe_image_quality' => array(
                    'title' => __('Image Quality', 'wc-delcampe-integration'),
                    'type' => 'number',
                    'description' => __('JPEG compression quality (1-100)', 'wc-delcampe-integration'),
                    'default' => '85',
                    'custom_attributes' => array(
                        'min' => '1',
                        'max' => '100'
                    )
                ),
                'delcampe_max_images' => array(
                    'title' => __('Maximum Images', 'wc-delcampe-integration'),
                    'type' => 'number',
                    'description' => __('Maximum number of images to sync per product', 'wc-delcampe-integration'),
                    'default' => '10',
                    'custom_attributes' => array(
                        'min' => '1',
                        'max' => '20'
                    )
                )
            )
        );
        
        return $settings;
    }
    
    /**
     * Create image sync queue table
     */
    private function create_image_sync_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_image_sync_queue';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            product_id bigint(20) NOT NULL,
            delcampe_id varchar(50) NOT NULL,
            status varchar(20) DEFAULT 'pending',
            error_message text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            processed_at datetime,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY delcampe_id (delcampe_id),
            KEY status (status)
        ) {$charset_collate};";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
}

// Initialize the class only after all plugins are loaded
add_action('plugins_loaded', function() {
    if (class_exists('Delcampe_Sync_Logger') && class_exists('Delcampe_Auth')) {
        Delcampe_Image_Manager::get_instance();
    }
}, 20);

// Add cron processor after class is initialized
add_action('plugins_loaded', function() {
    if (class_exists('Delcampe_Image_Manager')) {
        add_action('delcampe_process_image_sync_queue', array(Delcampe_Image_Manager::get_instance(), 'process_image_sync_queue'));
    }
}, 25);

// Schedule image sync queue processor if not already scheduled
add_action('init', function() {
    if (!wp_next_scheduled('delcampe_process_image_sync_queue')) {
        wp_schedule_event(time(), 'delcampe_five_minutes', 'delcampe_process_image_sync_queue');
    }
});