<?php
/**
 * Delcampe Image Mapper
 * Version: 1.0.0.0
 *
 * Maps local development images to production URLs
 * Handles cases where production images are in different locations
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.0.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Delcampe_Image_Mapper
 * 
 * Maps development images to their production counterparts
 * 
 * @since 1.0.0.0
 */
class Delcampe_Image_Mapper {
    
    /**
     * Map a local image URL to production URL
     * 
     * @since 1.0.0.0
     * @param string $local_url Local image URL
     * @param WC_Product $product Product object for context
     * @return string Production URL
     */
    public static function map_to_production($local_url, $product = null) {
        // Extract filename from local URL
        $filename = basename($local_url);
        
        // Try different production paths where images might be
        $possible_paths = array(
            'https://www.fdstamps.com/wp-content/uploads/imported/1/' . $filename,
            'https://www.fdstamps.com/wp-content/uploads/imported/2/' . $filename,
            'https://www.fdstamps.com/wp-content/uploads/imported/3/' . $filename,
            'https://www.fdstamps.com/wp-content/uploads/' . date('Y') . '/' . date('m') . '/' . $filename,
            'https://www.fdstamps.com/wp-content/uploads/' . $filename,
        );
        
        // If we have a product, try to find by SKU or title pattern
        if ($product) {
            $sku = $product->get_sku();
            if ($sku) {
                // Try SKU-based patterns
                $possible_paths[] = 'https://www.fdstamps.com/wp-content/uploads/imported/1/' . $sku . '.jpg';
                $possible_paths[] = 'https://www.fdstamps.com/wp-content/uploads/imported/1/' . $sku . '.JPG';
                $possible_paths[] = 'https://www.fdstamps.com/wp-content/uploads/imported/1/' . $sku . '-1.jpg';
            }
        }
        
        // Check if any URL exists (optional - requires HTTP requests)
        foreach ($possible_paths as $url) {
            // For now, just return the first imported/1/ path
            // In production, you might want to check if URL exists
            return $possible_paths[0];
        }
        
        // Default fallback
        return str_replace(
            array('http://192.168.22.192', 'http://localhost'),
            'https://www.fdstamps.com',
            $local_url
        );
    }
    
    /**
     * Get production image URLs for a product
     * This can be manually configured per product
     * 
     * @since 1.0.0.0
     * @param int $product_id Product ID
     * @return array|false Array of production URLs or false if not configured
     */
    public static function get_manual_mapping($product_id) {
        $mapping = get_post_meta($product_id, '_delcampe_production_images', true);
        
        if (!empty($mapping) && is_array($mapping)) {
            return $mapping;
        }
        
        return false;
    }
    
    /**
     * Save manual image mapping for a product
     * 
     * @since 1.0.0.0
     * @param int $product_id Product ID
     * @param array $production_urls Array of production image URLs
     */
    public static function save_manual_mapping($product_id, $production_urls) {
        update_post_meta($product_id, '_delcampe_production_images', $production_urls);
    }
}
