<?php
/**
 * Delcampe Image Proxy
 * 
 * Provides a publicly accessible endpoint for serving images to Delcampe,
 * bypassing DNS and VPN issues.
 * 
 * @package Delcampe_Sync
 * @since 1.10.4.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Image Proxy Class
 */
class Delcampe_Image_Proxy {
    
    /**
     * Instance
     */
    private static $instance = null;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Register REST API endpoint
        add_action('rest_api_init', array($this, 'register_routes'));
        
        // Add rewrite rule for non-REST access
        add_action('init', array($this, 'add_rewrite_rules'));
        
        // Handle the image request
        add_action('template_redirect', array($this, 'handle_image_request'));
        
        // Filter to replace image URLs with proxy URLs
        add_filter('delcampe_product_images', array($this, 'filter_image_urls'), 20, 3);
    }
    
    /**
     * Register REST API routes
     */
    public function register_routes() {
        register_rest_route('delcampe/v1', '/image/(?P<id>\d+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'serve_image'),
            'permission_callback' => '__return_true', // Public access
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param, $request, $key) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));
    }
    
    /**
     * Add rewrite rules for direct image access
     */
    public function add_rewrite_rules() {
        add_rewrite_rule(
            'delcampe-image/([0-9]+)/?$',
            'index.php?delcampe_image=$1',
            'top'
        );
        
        // Add query var
        add_filter('query_vars', function($vars) {
            $vars[] = 'delcampe_image';
            return $vars;
        });
    }
    
    /**
     * Handle direct image requests
     */
    public function handle_image_request() {
        $image_id = get_query_var('delcampe_image');
        if ($image_id) {
            $this->output_image($image_id);
            exit;
        }
    }
    
    /**
     * Serve image via REST API
     */
    public function serve_image($request) {
        $image_id = $request['id'];
        $this->output_image($image_id);
        exit;
    }
    
    /**
     * Output image
     */
    private function output_image($attachment_id) {
        // Verify attachment exists
        $attachment = get_post($attachment_id);
        if (!$attachment || $attachment->post_type !== 'attachment') {
            status_header(404);
            exit('Image not found');
        }
        
        // Get file path
        $file_path = get_attached_file($attachment_id);
        if (!file_exists($file_path)) {
            status_header(404);
            exit('Image file not found');
        }
        
        // Get mime type
        $mime_type = get_post_mime_type($attachment_id);
        if (!$mime_type) {
            $mime_type = 'image/jpeg'; // Default
        }
        
        // Set headers
        header('Content-Type: ' . $mime_type);
        header('Content-Length: ' . filesize($file_path));
        header('Cache-Control: public, max-age=86400'); // Cache for 24 hours
        header('X-Served-By: Delcampe-Image-Proxy');
        
        // Output file
        readfile($file_path);
        exit;
    }
    
    /**
     * Filter image URLs to use proxy
     */
    public function filter_image_urls($images, $product, $profile) {
        // Only apply if we're on dev site or if explicitly enabled
        $use_proxy = apply_filters('delcampe_use_image_proxy', 
            strpos(home_url(), 'dev.fdstamps.com') !== false
        );
        
        if (!$use_proxy) {
            return $images;
        }
        
        $filtered_images = array();
        
        foreach ($images as $image_url) {
            // Get attachment ID from URL
            $attachment_id = attachment_url_to_postid($image_url);
            
            if ($attachment_id) {
                // Use REST API endpoint (more reliable)
                $proxy_url = rest_url('delcampe/v1/image/' . $attachment_id);
                
                // For external access, we might need to use the production URL
                if (defined('DELCAMPE_PROXY_BASE_URL')) {
                    $proxy_url = DELCAMPE_PROXY_BASE_URL . '/wp-json/delcampe/v1/image/' . $attachment_id;
                }
                
                $filtered_images[] = $proxy_url;
            } else {
                // Fallback to original URL if we can't find attachment ID
                $filtered_images[] = $image_url;
            }
        }
        
        return $filtered_images;
    }
    
    /**
     * Get proxy URL for an attachment
     */
    public function get_proxy_url($attachment_id) {
        return rest_url('delcampe/v1/image/' . $attachment_id);
    }
}