<?php
/**
 * Delcampe Inventory Manager
 * 
 * Manages inventory synchronization between WooCommerce and Delcampe
 * Handles real-time stock updates, monitoring, and inventory checking
 * 
 * IMPORTANT (v1.10.25.0): WooCommerce is the MASTER source of truth for inventory.
 * All sync operations push WooCommerce stock levels TO Delcampe, never the reverse.
 * Delcampe is treated as a sales channel that should mirror WooCommerce inventory.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Inventory
 * @since 1.5.0.0
 * @version 1.10.25.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_Inventory_Manager
 * 
 * Core inventory management functionality for Delcampe integration
 * Handles bi-directional inventory synchronization and stock monitoring
 * 
 * @since 1.5.0.0
 */
class Delcampe_Inventory_Manager {
    
    /**
     * Singleton instance
     * @var Delcampe_Inventory_Manager|null
     */
    private static $instance = null;
    
    /**
     * Database table name for inventory sync log
     * @var string
     */
    private $table_name;
    
    /**
     * Database table name for inventory alerts
     * @var string
     */
    private $alerts_table_name;
    
    /**
     * Stock cache manager instance
     * @var Delcampe_Stock_Cache_Manager|null
     * @since 1.10.20.3
     */
    private $cache_manager;
    
    /**
     * Get singleton instance
     * 
     * @return Delcampe_Inventory_Manager
     * @since 1.5.0.0
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     * 
     * Initialize inventory manager and set up hooks
     * 
     * @since 1.5.0.0
     */
    private function __construct() {
        global $wpdb;
        
        // Set table names
        $this->table_name = $wpdb->prefix . 'delcampe_inventory_sync';
        $this->alerts_table_name = $wpdb->prefix . 'delcampe_inventory_alerts';
        
        // Initialize cache manager if feature is enabled (v1.10.20.3)
        if (class_exists('Delcampe_Feature_Flags') && 
            class_exists('Delcampe_Stock_Cache_Manager')) {
            if (Delcampe_Feature_Flags::is_enabled('stock_caching')) {
                $this->cache_manager = Delcampe_Stock_Cache_Manager::get_instance();
            }
        }
        
        // Initialize hooks
        $this->init_hooks();
        
        // Inventory manager initialized
    }
    
    /**
     * Initialize WordPress hooks
     * 
     * @since 1.5.0.0
     */
    private function init_hooks() {
        // v1.10.35.12: DISABLED automatic stock sync on WooCommerce changes
        // This was causing API storms with hundreds of requests
        // Stock sync should only happen manually or in controlled batches
        
        // Check if auto-sync is explicitly enabled (defaults to 'no')
        $auto_sync_enabled = get_option('delcampe_auto_sync_on_stock_change', 'no');
        
        if ($auto_sync_enabled === 'yes') {
            // WooCommerce stock change hooks - ONLY if explicitly enabled
            add_action('woocommerce_product_set_stock', array($this, 'sync_stock_to_delcampe'));
            add_action('woocommerce_variation_set_stock', array($this, 'sync_stock_to_delcampe'));
            
            // Additional stock change hooks for better coverage
            add_action('woocommerce_product_set_stock_status', array($this, 'handle_stock_status_change'), 10, 3);
            add_action('woocommerce_variation_set_stock_status', array($this, 'handle_stock_status_change'), 10, 3);
        }
        
        // Hook into order creation to reduce stock properly
        add_action('woocommerce_new_order', array($this, 'check_delcampe_order_stock_reduction'), 10, 1);
        add_action('woocommerce_order_status_processing', array($this, 'ensure_stock_reduced'), 10, 1);
        
        // Low stock monitoring
        add_action('woocommerce_low_stock', array($this, 'handle_low_stock_alert'));
        add_action('woocommerce_no_stock', array($this, 'handle_no_stock_alert'));
        
        // Manual inventory check handlers (removed automatic cron)
        add_action('delcampe_inventory_check_continue', array($this, 'run_inventory_check'));
        
        // Migration: Remove automatic inventory sync (v1.10.34.7)
        // This ensures clean upgrade from versions that had automatic sync
        $manual_sync_migrated = get_option('delcampe_manual_sync_migrated', 'no');
        if ($manual_sync_migrated !== 'yes') {
            // Clear all existing scheduled inventory checks
            wp_clear_scheduled_hook('delcampe_inventory_check_cron');
            wp_clear_scheduled_hook('delcampe_inventory_check_continue');
            
            // Clean up old transients
            delete_transient('delcampe_inventory_last_processed');
            
            // Mark as migrated
            update_option('delcampe_manual_sync_migrated', 'yes');
            
            error_log('[Delcampe Inventory Manager] Migrated to manual sync - removed automatic schedules');
        }
        
        // Admin notices for inventory alerts
        add_action('admin_notices', array($this, 'display_inventory_alerts'));
        
        // AJAX handler for resolving alerts (v1.10.26.0)
        add_action('wp_ajax_delcampe_resolve_alert', array($this, 'handle_resolve_alert'));
    }

    /**
     * Sync a single product's stock to Delcampe (used by master sync)
     *
     * @param int $product_id
     * @return bool|WP_Error
     */
    public function sync_product_stock($product_id) {
        $product = wc_get_product($product_id);
        if (!$product) {
            return new WP_Error('invalid_product', 'Product not found');
        }

        // Try both meta keys to locate the Delcampe listing ID
        $delcampe_item_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (empty($delcampe_item_id)) {
            $delcampe_item_id = get_post_meta($product_id, '_delcampe_item_id', true);
        }

        if (empty($delcampe_item_id)) {
            // Not listed on Delcampe; nothing to sync
            return true;
        }

        // Determine stock quantity; if not managing stock, treat as 0/skip
        $qty = $product->get_stock_quantity();
        if ($qty === null) {
            // If stock not tracked, skip to avoid sending incorrect values
            return true;
        }

        // Use existing update method (POST /item/update) to keep behavior consistent here
        // Note: elsewhere we migrated to PUT /item/{id} with item[qty]
        // This path is used by master sync as a best-effort; failures are logged
        return $this->update_delcampe_stock($delcampe_item_id, (int)$qty);
    }
    
    /**
     * Create database tables
     * 
     * Creates tables for inventory sync log and alerts
     * 
     * @since 1.5.0.0
     */
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Inventory sync log table
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            variation_id bigint(20) unsigned DEFAULT 0,
            delcampe_item_id varchar(255) DEFAULT NULL,
            wc_stock int(11) NOT NULL,
            delcampe_stock int(11) DEFAULT NULL,
            sync_type varchar(50) NOT NULL COMMENT 'wc_to_delcampe, delcampe_to_wc, check',
            sync_status varchar(50) NOT NULL DEFAULT 'pending',
            sync_message text,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY variation_id (variation_id),
            KEY delcampe_item_id (delcampe_item_id),
            KEY sync_status (sync_status),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Inventory alerts table
        $sql_alerts = "CREATE TABLE IF NOT EXISTS {$this->alerts_table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            variation_id bigint(20) unsigned DEFAULT 0,
            alert_type varchar(50) NOT NULL COMMENT 'low_stock, no_stock, oversell, sync_error',
            alert_message text NOT NULL,
            current_stock int(11) DEFAULT NULL,
            threshold_stock int(11) DEFAULT NULL,
            is_resolved tinyint(1) DEFAULT 0,
            resolved_at datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY alert_type (alert_type),
            KEY is_resolved (is_resolved),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        dbDelta($sql_alerts);
        
        error_log('[Delcampe Inventory Manager v1.5.0.0] Database tables created');
    }
    
    /**
     * Sync stock to Delcampe when WooCommerce stock changes
     * 
     * @param WC_Product $product Product object
     * @since 1.5.0.0
     */
    public function sync_stock_to_delcampe($product) {
        // Check if product has Delcampe listing
        $delcampe_item_id = $product->get_meta('_delcampe_item_id');
        
        if (!$delcampe_item_id) {
            return; // Product not listed on Delcampe
        }
        
        // Get current stock
        $stock_quantity = $product->get_stock_quantity();
        
        // Log sync attempt
        $this->log_inventory_sync(
            $product->get_id(),
            $product->is_type('variation') ? $product->get_id() : 0,
            $delcampe_item_id,
            $stock_quantity,
            null,
            'wc_to_delcampe',
            'pending'
        );
        
        // Update stock on Delcampe
        $result = $this->update_delcampe_stock($delcampe_item_id, $stock_quantity);
        
        if (is_wp_error($result)) {
            // Log error
            $this->log_inventory_sync(
                $product->get_id(),
                $product->is_type('variation') ? $product->get_id() : 0,
                $delcampe_item_id,
                $stock_quantity,
                null,
                'wc_to_delcampe',
                'failed',
                $result->get_error_message()
            );
            
            // Create alert
            $this->create_inventory_alert(
                $product->get_id(),
                $product->is_type('variation') ? $product->get_id() : 0,
                'sync_error',
                sprintf(
                    __('Failed to sync stock to Delcampe: %s', 'wc-delcampe-integration'),
                    $result->get_error_message()
                ),
                $stock_quantity
            );
        } else {
            // Log success
            $this->log_inventory_sync(
                $product->get_id(),
                $product->is_type('variation') ? $product->get_id() : 0,
                $delcampe_item_id,
                $stock_quantity,
                $stock_quantity,
                'wc_to_delcampe',
                'success'
            );
        }
    }
    
    /**
     * Update stock quantity on Delcampe
     * 
     * @param string $delcampe_item_id Delcampe item ID
     * @param int $quantity New stock quantity
     * @return bool|WP_Error Success or error
     * @since 1.5.0.0
     */
    private function update_delcampe_stock($delcampe_item_id, $quantity) {
        // Get authentication token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        // Build API URL for updating item quantity - use PUT /item/{id}
        $api_url = DELCAMPE_API_URL . '/item/' . $delcampe_item_id;
        
        // Prepare update data - the API expects item[qty] format
        $data = array(
            'token' => $token,
            'item[qty]' => $quantity
        );
        
        // Make API request using PUT method (correct endpoint per API docs)
        $response = wp_remote_request($api_url, array(
            'method' => 'PUT',
            'timeout' => 30,
            'body' => $data,
            'headers' => array(
                'Accept' => 'application/xml',
                'Content-Type' => 'application/x-www-form-urlencoded'
            )
        ));
        
        if (is_wp_error($response)) {
            error_log('[Delcampe Inventory Manager] API request failed: ' . $response->get_error_message());
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        
        // Parse XML response
        $xml = simplexml_load_string($body);
        if ($xml === false) {
            return new WP_Error('xml_parse_error', 'Failed to parse XML response');
        }
        
        // Check for API errors
        if (isset($xml->e)) {
            return new WP_Error('api_error', (string)$xml->e);
        }
        
        // Success - update cache with the new value (v1.10.20.3)
        if ($this->cache_manager) {
            $this->cache_manager->write_cache($delcampe_item_id, $quantity, 'stock_push');
        }
        
        error_log('[Delcampe Inventory Manager] Updated stock for item ' . $delcampe_item_id . ' to ' . $quantity);
        return true;
    }
    
    /**
     * Handle stock status change
     * 
     * @param int $product_id Product ID
     * @param string $stock_status New stock status
     * @param WC_Product $product Product object
     * @since 1.5.0.0
     */
    public function handle_stock_status_change($product_id, $stock_status, $product) {
        // If product is out of stock, sync 0 quantity to Delcampe
        if ($stock_status === 'outofstock') {
            $this->sync_stock_to_delcampe($product);
        }
    }
    
    /**
     * Check if Delcampe order needs stock reduction
     * 
     * @param int $order_id Order ID
     * @since 1.5.0.0
     */
    public function check_delcampe_order_stock_reduction($order_id) {
        $order = wc_get_order($order_id);
        
        // Check if this is a Delcampe order
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        if (!$delcampe_order_id) {
            return; // Not a Delcampe order
        }
        
        // Check if created via delcampe_import
        if ($order->get_created_via() !== 'delcampe_import') {
            return;
        }
        
        // Process stock reduction for each item
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            
            if (!$product || !$product->managing_stock()) {
                continue;
            }
            
            $qty = $item->get_quantity();
            $old_stock = $product->get_stock_quantity();
            $new_stock = wc_update_product_stock($product, $qty, 'decrease');
            
            // Log the stock reduction
            error_log(sprintf(
                '[Delcampe Inventory Manager] Reduced stock for product #%d from %d to %d (Delcampe order #%s)',
                $product->get_id(),
                $old_stock,
                $new_stock,
                $delcampe_order_id
            ));
            
            // Log in inventory sync table
            $this->log_inventory_sync(
                $product->get_id(),
                $product->is_type('variation') ? $product->get_id() : 0,
                $product->get_meta('_delcampe_item_id'),
                $new_stock,
                $old_stock,
                'delcampe_order',
                'success',
                sprintf('Stock reduced by %d for Delcampe order #%s', $qty, $delcampe_order_id)
            );
        }
    }
    
    /**
     * Ensure stock is reduced when order status changes to processing
     * 
     * @param int $order_id Order ID
     * @since 1.5.0.0
     */
    public function ensure_stock_reduced($order_id) {
        $order = wc_get_order($order_id);
        
        // Check if this is a Delcampe order
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        if (!$delcampe_order_id) {
            return;
        }
        
        // Check if stock was already reduced
        if ($order->get_meta('_delcampe_stock_reduced') === 'yes') {
            return;
        }
        
        // Reduce stock
        wc_reduce_stock_levels($order_id);
        
        // Mark as reduced
        $order->update_meta_data('_delcampe_stock_reduced', 'yes');
        $order->save();
        
        error_log('[Delcampe Inventory Manager] Stock reduced for Delcampe order #' . $delcampe_order_id);
    }
    
    /**
     * Handle low stock alert
     * 
     * @param WC_Product $product Product object
     * @since 1.5.0.0
     */
    public function handle_low_stock_alert($product) {
        $stock = $product->get_stock_quantity();
        $threshold = get_option('woocommerce_notify_low_stock_amount', 2);
        
        $this->create_inventory_alert(
            $product->get_id(),
            $product->is_type('variation') ? $product->get_id() : 0,
            'low_stock',
            sprintf(
                __('Low stock alert: %s has only %d items remaining', 'wc-delcampe-integration'),
                $product->get_name(),
                $stock
            ),
            $stock,
            $threshold
        );
    }
    
    /**
     * Handle no stock alert
     * 
     * @param WC_Product $product Product object
     * @since 1.5.0.0
     */
    public function handle_no_stock_alert($product) {
        $this->create_inventory_alert(
            $product->get_id(),
            $product->is_type('variation') ? $product->get_id() : 0,
            'no_stock',
            sprintf(
                __('Out of stock alert: %s is now out of stock', 'wc-delcampe-integration'),
                $product->get_name()
            ),
            0,
            0
        );
    }
    
    /**
     * Run scheduled inventory check
     * 
     * Compares WooCommerce and Delcampe inventory levels
     * 
     * @since 1.5.0.0
     */
    public function run_inventory_check($batch_size = null) {
        error_log('[Delcampe Inventory Manager] Running scheduled inventory check');
        
        // Extend time limit for cron jobs (not for AJAX calls)
        if (defined('DOING_CRON') && DOING_CRON) {
            @set_time_limit(300); // 5 minutes for cron
        }
        
        // Default batch size - smaller for safety
        if ($batch_size === null) {
            $batch_size = apply_filters('delcampe_inventory_batch_size', 50);
        }
        
        // Get the last processed product ID from transient
        $last_product_id = get_transient('delcampe_inventory_last_processed');
        
        // Get products with ACTIVE Delcampe listings only
        // Join with delcampe_listings table to only get published/verified items
        global $wpdb;
        $product_ids = $wpdb->get_col($wpdb->prepare(
            "SELECT DISTINCT dl.product_id 
            FROM {$wpdb->prefix}delcampe_listings dl
            INNER JOIN {$wpdb->posts} p ON dl.product_id = p.ID
            WHERE dl.status IN ('published', 'verified')
            AND dl.delcampe_id IS NOT NULL
            AND dl.delcampe_id != ''
            AND p.post_type = 'product'
            AND p.post_status = 'publish'
            AND dl.product_id > %d
            ORDER BY dl.product_id ASC
            LIMIT %d",
            $last_product_id ? $last_product_id : 0,
            $batch_size
        ));
        
        if (empty($product_ids)) {
            delete_transient('delcampe_inventory_last_processed');
            error_log('[Delcampe Inventory Manager] Inventory check completed - processed all active listings');
            return;
        }
        
        // Get products with active listings
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => -1,
            'post__in' => $product_ids,
            'orderby' => 'ID',
            'order' => 'ASC'
        );
        
        $products = get_posts($args);
        
        $processed_count = 0;
        $last_id = 0;
        
        foreach ($products as $product_post) {
            $product = wc_get_product($product_post->ID);
            if (!$product) continue;
            
            // Check main product
            $this->check_single_product_inventory($product);
            $processed_count++;
            
            // Check variations if variable product
            if ($product->is_type('variable')) {
                $variations = $product->get_children();
                foreach ($variations as $variation_id) {
                    $variation = wc_get_product($variation_id);
                    if ($variation) {
                        $this->check_single_product_inventory($variation);
                        $processed_count++;
                    }
                }
            }
            
            $last_id = $product_post->ID;
            
            // Break if we're taking too long (for AJAX calls)
            if (!defined('DOING_CRON') || !DOING_CRON) {
                if ((time() - $_SERVER['REQUEST_TIME']) > 20) {
                    break;
                }
            }
        }
        
        // Save the last processed product ID
        if ($last_id > 0) {
            set_transient('delcampe_inventory_last_processed', $last_id, DAY_IN_SECONDS);
        }
        
        error_log(sprintf('[Delcampe Inventory Manager] Processed %d items in batch', $processed_count));
        
        // If running from cron and there are more products, schedule another run soon
        if (defined('DOING_CRON') && DOING_CRON && count($products) == $batch_size) {
            if (!wp_next_scheduled('delcampe_inventory_check_continue')) {
                wp_schedule_single_event(time() + 60, 'delcampe_inventory_check_continue');
            }
        }
    }
    
    /**
     * Check inventory for a single product
     * 
     * @param WC_Product $product Product to check
     * @since 1.5.0.0
     */
    private function check_single_product_inventory($product) {
        // Skip if stock sync is disabled
        if (!apply_filters('delcampe_enable_stock_sync', true)) {
            return;
        }
        
        $delcampe_item_id = $product->get_meta('_delcampe_item_id');
        if (!$delcampe_item_id) return;
        
        // Get WooCommerce stock
        $wc_stock = $product->get_stock_quantity();
        
        // Skip if stock hasn't changed recently (check last sync time)
        $last_sync_key = 'delcampe_last_stock_' . $product->get_id();
        $last_synced_stock = get_transient($last_sync_key);
        if ($last_synced_stock !== false && $last_synced_stock == $wc_stock) {
            // Stock hasn't changed since last check, skip API call
            return;
        }
        
        // Get Delcampe stock
        $delcampe_stock = $this->get_delcampe_stock($delcampe_item_id);
        
        if (is_wp_error($delcampe_stock)) {
            // Check if listing doesn't exist on Delcampe (v1.10.25.0)
            $error_code = $delcampe_stock->get_error_code();
            $error_message = $delcampe_stock->get_error_message();
            
            if (strpos($error_message, '404') !== false || strpos($error_message, 'not found') !== false) {
                // Listing missing on Delcampe - flag for re-listing
                $this->create_inventory_alert(
                    $product->get_id(),
                    $product->is_type('variation') ? $product->get_id() : 0,
                    'missing_listing',
                    sprintf(
                        __('Product "%s" has no active Delcampe listing but should (WC stock: %d). Needs re-listing.', 'wc-delcampe-integration'),
                        $product->get_name(),
                        $wc_stock
                    ),
                    $wc_stock
                );
                
                // Clear the invalid Delcampe ID
                $product->delete_meta_data('_delcampe_item_id');
                $product->save();
            } else {
                // Other API error
                $this->log_inventory_sync(
                    $product->get_id(),
                    $product->is_type('variation') ? $product->get_id() : 0,
                    $delcampe_item_id,
                    $wc_stock,
                    null,
                    'check',
                    'failed',
                    $error_message
                );
            }
            return;
        }
        
        // Log check
        $this->log_inventory_sync(
            $product->get_id(),
            $product->is_type('variation') ? $product->get_id() : 0,
            $delcampe_item_id,
            $wc_stock,
            $delcampe_stock,
            'check',
            'success'
        );
        
        // Check for discrepancies - WooCommerce is the master (v1.10.25.0)
        if ($wc_stock !== $delcampe_stock) {
            // Create alert showing that Delcampe needs updating
            $this->create_inventory_alert(
                $product->get_id(),
                $product->is_type('variation') ? $product->get_id() : 0,
                'sync_needed',
                sprintf(
                    __('Delcampe out of sync: Updating from %d to %d for %s', 'wc-delcampe-integration'),
                    $delcampe_stock,
                    $wc_stock,
                    $product->get_name()
                ),
                $wc_stock
            );
            
            // Always sync WooCommerce → Delcampe (WC is master)
            $sync_result = $this->sync_stock_to_delcampe($product);
            
            if (!is_wp_error($sync_result)) {
                // Log successful sync
                $this->log_inventory_sync(
                    $product->get_id(),
                    $product->is_type('variation') ? $product->get_id() : 0,
                    $delcampe_item_id,
                    $wc_stock,
                    $wc_stock, // After sync, Delcampe should match WC
                    'auto_sync',
                    'success',
                    'Automatically synced WooCommerce stock to Delcampe'
                );
                
                // Save the synced stock level to avoid unnecessary future API calls
                $last_sync_key = 'delcampe_last_stock_' . $product->get_id();
                set_transient($last_sync_key, $wc_stock, WEEK_IN_SECONDS);
            }
        } else {
            // Stocks match, save this to avoid future API calls
            $last_sync_key = 'delcampe_last_stock_' . $product->get_id();
            set_transient($last_sync_key, $wc_stock, WEEK_IN_SECONDS);
        }
    }
    
    /**
     * Get stock quantity from Delcampe
     * 
     * @param string $delcampe_item_id Delcampe item ID
     * @return int|WP_Error Stock quantity or error
     * @since 1.5.0.0
     */
    private function get_delcampe_stock($delcampe_item_id) {
        // Try to use cache if available (v1.10.20.3)
        if ($this->cache_manager) {
            $cached_data = $this->cache_manager->get_stock($delcampe_item_id, false);
            if ($cached_data !== false) {
                // Log cache hit
                error_log('[Delcampe Inventory] Using cached stock for item ' . $delcampe_item_id . 
                         ' (source: ' . $cached_data['source'] . ', age: ' . $cached_data['age_seconds'] . 's)');
                return $cached_data['stock'];
            }
        }
        
        // Fallback to live API call
        // Get authentication token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        // Build API URL - using correct REST endpoint format
        // Correct format is /item/{id} not /item/get
        $url = DELCAMPE_API_URL . '/item/' . $delcampe_item_id . '?token=' . $token;
        
        // Make API request
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        
        // Parse XML response
        $xml = @simplexml_load_string($body);
        if ($xml === false) {
            return new WP_Error('xml_parse_error', 'Failed to parse XML response');
        }
        
        // Check for API errors
        if (isset($xml->e)) {
            return new WP_Error('api_error', (string)$xml->e);
        }
        
        // Robust quantity extraction across known formats
        $stock_value = null;
        
        // 1) Direct item node
        if (isset($xml->item)) {
            if (isset($xml->item->qty)) {
                $stock_value = (int)$xml->item->qty;
            }
            if (isset($xml->item->quantity)) {
                $stock_value = (int)$xml->item->quantity;
            }
        }
        // 2) Notification envelope
        if ($stock_value === null && isset($xml->Notification_Data->body->item)) {
            $it = $xml->Notification_Data->body->item;
            if (isset($it->qty)) {
                $stock_value = (int)$it->qty;
            }
            if (isset($it->quantity)) {
                $stock_value = (int)$it->quantity;
            }
        }
        // 3) XPath fallback
        if ($stock_value === null) {
            $nodes = $xml->xpath('//item');
            if ($nodes && isset($nodes[0])) {
                $it = $nodes[0];
                if (isset($it->qty)) {
                    $stock_value = (int)$it->qty;
                }
                if (isset($it->quantity)) {
                    $stock_value = (int)$it->quantity;
                }
            }
        }
        
        // If we found a stock value, update cache and return it
        if ($stock_value !== null) {
            // Update cache if cache manager is available (v1.10.20.3)
            if ($this->cache_manager) {
                $this->cache_manager->write_cache($delcampe_item_id, $stock_value, 'api_fetch');
            }
            return $stock_value;
        }
        
        return new WP_Error('no_quantity', 'Quantity not found in response');
    }
    
    /**
     * Log inventory sync attempt
     * 
     * @param int $product_id Product ID
     * @param int $variation_id Variation ID (0 for simple products)
     * @param string $delcampe_item_id Delcampe item ID
     * @param int $wc_stock WooCommerce stock level
     * @param int|null $delcampe_stock Delcampe stock level
     * @param string $sync_type Type of sync
     * @param string $sync_status Status of sync
     * @param string $sync_message Optional message
     * @since 1.5.0.0
     */
    private function log_inventory_sync($product_id, $variation_id, $delcampe_item_id, $wc_stock, $delcampe_stock, $sync_type, $sync_status, $sync_message = '') {
        global $wpdb;
        
        $wpdb->insert($this->table_name, array(
            'product_id' => $product_id,
            'variation_id' => $variation_id,
            'delcampe_item_id' => $delcampe_item_id,
            'wc_stock' => $wc_stock,
            'delcampe_stock' => $delcampe_stock,
            'sync_type' => $sync_type,
            'sync_status' => $sync_status,
            'sync_message' => $sync_message,
            'created_at' => current_time('mysql')
        ));
    }
    
    /**
     * Create inventory alert
     * 
     * @param int $product_id Product ID
     * @param int $variation_id Variation ID
     * @param string $alert_type Type of alert
     * @param string $alert_message Alert message
     * @param int|null $current_stock Current stock level
     * @param int|null $threshold_stock Threshold stock level
     * @since 1.5.0.0
     */
    private function create_inventory_alert($product_id, $variation_id, $alert_type, $alert_message, $current_stock = null, $threshold_stock = null) {
        global $wpdb;
        
        // Check if similar unresolved alert already exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$this->alerts_table_name} 
            WHERE product_id = %d AND variation_id = %d AND alert_type = %s AND is_resolved = 0
            ORDER BY created_at DESC LIMIT 1",
            $product_id, $variation_id, $alert_type
        ));
        
        if (!$existing) {
            $wpdb->insert($this->alerts_table_name, array(
                'product_id' => $product_id,
                'variation_id' => $variation_id,
                'alert_type' => $alert_type,
                'alert_message' => $alert_message,
                'current_stock' => $current_stock,
                'threshold_stock' => $threshold_stock,
                'created_at' => current_time('mysql')
            ));
        }
    }
    
    /**
     * Display inventory alerts in admin
     * 
     * @since 1.5.0.0
     */
    public function display_inventory_alerts() {
        global $wpdb;
        
        // Only show on Delcampe plugin pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'delcampe') === false) {
            return;
        }
        
        // Get unresolved alerts
        $alerts = $wpdb->get_results(
            "SELECT * FROM {$this->alerts_table_name} 
            WHERE is_resolved = 0 
            ORDER BY created_at DESC 
            LIMIT 5"
        );
        
        if (empty($alerts)) {
            return;
        }
        
        ?>
        <div class="notice notice-warning is-dismissible delcampe-inventory-alerts">
            <h3><?php _e('Delcampe Inventory Alerts', 'wc-delcampe-integration'); ?></h3>
            <ul>
                <?php foreach ($alerts as $alert): ?>
                    <li>
                        <strong><?php echo esc_html(ucfirst(str_replace('_', ' ', $alert->alert_type))); ?>:</strong>
                        <?php echo esc_html($alert->alert_message); ?>
                        <small>(<?php echo esc_html(human_time_diff(strtotime($alert->created_at), current_time('timestamp'))); ?> ago)</small>
                        <a href="#" class="resolve-alert" data-alert-id="<?php echo esc_attr($alert->id); ?>"><?php _e('Resolve', 'wc-delcampe-integration'); ?></a>
                    </li>
                <?php endforeach; ?>
            </ul>
            <p>
                <a href="<?php echo admin_url('admin.php?page=delcampe-inventory'); ?>" class="button button-primary">
                    <?php _e('View All Inventory Alerts', 'wc-delcampe-integration'); ?>
                </a>
            </p>
        </div>
        <script>
        jQuery(document).ready(function($) {
            $('.resolve-alert').on('click', function(e) {
                e.preventDefault();
                var alertId = $(this).data('alert-id');
                var $link = $(this);
                
                $.post(ajaxurl, {
                    action: 'delcampe_resolve_alert',
                    alert_id: alertId,
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_resolve_alert'); ?>'
                }, function(response) {
                    if (response.success) {
                        $link.closest('li').fadeOut();
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Get inventory statistics
     * 
     * @return array Inventory statistics
     * @since 1.5.0.0
     */
    public function get_inventory_statistics() {
        global $wpdb;
        
        $stats = array();
        
        // Total synced products
        $stats['total_synced'] = $wpdb->get_var(
            "SELECT COUNT(DISTINCT product_id) FROM {$this->table_name}"
        );
        
        // Recent sync activity (last 24 hours)
        $stats['recent_syncs'] = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name} 
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)"
        );
        
        // Failed syncs
        $stats['failed_syncs'] = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name} 
            WHERE sync_status = 'failed' AND created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)"
        );
        
        // Active alerts
        $stats['active_alerts'] = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->alerts_table_name} WHERE is_resolved = 0"
        );
        
        // Alert breakdown
        $alert_types = $wpdb->get_results(
            "SELECT alert_type, COUNT(*) as count 
            FROM {$this->alerts_table_name} 
            WHERE is_resolved = 0 
            GROUP BY alert_type"
        );
        
        foreach ($alert_types as $type) {
            $stats['alerts_by_type'][$type->alert_type] = $type->count;
        }
        
        // Inventory mismatches
        $stats['mismatches'] = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name} 
            WHERE sync_type = 'check' 
            AND wc_stock != delcampe_stock 
            AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)"
        );
        
        return $stats;
    }
    
    /**
     * Manual inventory check for specific product
     * 
     * @param int $product_id Product ID to check
     * @return array Check results
     * @since 1.5.0.0
     */
    public function manual_inventory_check($product_id) {
        $product = wc_get_product($product_id);
        if (!$product) {
            return array('error' => __('Product not found', 'wc-delcampe-integration'));
        }
        
        $results = array();
        
        // Check main product
        $this->check_single_product_inventory($product);
        $results['main'] = array(
            'product_id' => $product_id,
            'name' => $product->get_name(),
            'wc_stock' => $product->get_stock_quantity(),
            'checked' => true
        );
        
        // Check variations if variable product
        if ($product->is_type('variable')) {
            $variations = $product->get_children();
            foreach ($variations as $variation_id) {
                $variation = wc_get_product($variation_id);
                if ($variation) {
                    $this->check_single_product_inventory($variation);
                    $results['variations'][] = array(
                        'variation_id' => $variation_id,
                        'name' => $variation->get_name(),
                        'wc_stock' => $variation->get_stock_quantity(),
                        'checked' => true
                    );
                }
            }
        }
        
        return $results;
    }
    
    /**
     * Bulk update inventory
     * 
     * @param array $updates Array of product_id => stock_quantity
     * @return array Results of bulk update
     * @since 1.5.0.0
     */
    public function bulk_update_inventory($updates) {
        $results = array(
            'success' => 0,
            'failed' => 0,
            'errors' => array()
        );
        
        foreach ($updates as $product_id => $stock) {
            $product = wc_get_product($product_id);
            if (!$product) {
                $results['failed']++;
                $results['errors'][] = sprintf(__('Product #%d not found', 'wc-delcampe-integration'), $product_id);
                continue;
            }
            
            // Update WooCommerce stock
            $product->set_stock_quantity($stock);
            $product->save();
            
            // Sync to Delcampe
            $this->sync_stock_to_delcampe($product);
            
            $results['success']++;
        }
        
        return $results;
    }
    
    /**
     * Handle AJAX request to resolve inventory alert
     * 
     * @since 1.10.26.0
     */
    public function handle_resolve_alert() {
        // Verify nonce
        check_ajax_referer('delcampe_resolve_alert', '_ajax_nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(-1);
        }
        
        $alert_id = isset($_POST['alert_id']) ? intval($_POST['alert_id']) : 0;
        
        if (!$alert_id) {
            wp_send_json_error('Invalid alert ID');
        }
        
        global $wpdb;
        
        // Get alert details
        $alert = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->alerts_table_name} WHERE id = %d",
            $alert_id
        ));
        
        if (!$alert) {
            wp_send_json_error('Alert not found');
        }
        
        // If this is a sync_needed alert, trigger sync
        if ($alert->alert_type === 'sync_needed' || $alert->alert_type === 'sync_error') {
            $product = wc_get_product($alert->product_id);
            
            if ($product) {
                // Sync WooCommerce stock to Delcampe
                $sync_result = $this->sync_stock_to_delcampe($product);
                
                if (!is_wp_error($sync_result)) {
                    // Log successful sync
                    delcampe_log('[Inventory Alert] Resolved alert #' . $alert_id . ' - Synced product #' . $alert->product_id . ' to Delcampe');
                }
            }
        }
        
        // Mark alert as resolved
        $wpdb->update(
            $this->alerts_table_name,
            array('is_resolved' => 1, 'resolved_at' => current_time('mysql')),
            array('id' => $alert_id)
        );
        
        wp_send_json_success('Alert resolved and inventory synced');
    }
}
