<?php
/**
 * Simplified Close Listing Implementation
 * 
 * Since the Delcampe API doesn't appear to have a close/end listing endpoint,
 * this provides a way to mark listings as ended locally.
 * 
 * @package Delcampe_Sync
 * @since 1.10.5.1
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Mark a listing as ended locally
 * 
 * This function updates the local database to mark a listing as ended.
 * It doesn't actually close the listing on Delcampe since that API
 * endpoint doesn't appear to exist.
 * 
 * @param int $product_id Product ID
 * @param string $listing_id Delcampe listing ID
 * @param string $reason Optional reason for ending
 * @return array Result array
 */
function delcampe_mark_listing_ended($product_id, $listing_id, $reason = '') {
    // Update product meta
    update_post_meta($product_id, '_delcampe_sync_status', 'ended');
    update_post_meta($product_id, '_delcampe_end_reason', $reason);
    update_post_meta($product_id, '_delcampe_end_date', current_time('mysql'));
    
    // Update listings table if it exists
    global $wpdb;
    $table_name = $wpdb->prefix . 'delcampe_listings';
    
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name) {
        $wpdb->update(
            $table_name,
            array(
                'status' => 'ended',
                'end_date' => current_time('mysql')
            ),
            array('delcampe_id' => $listing_id),
            array('%s', '%s'),
            array('%s')
        );
    }
    
    // Log the action
    if (function_exists('delcampe_log')) {
        delcampe_log(sprintf('Listing %s marked as ended locally for product %d', $listing_id, $product_id), 'info');
    }
    
    // Trigger action for other plugins
    do_action('delcampe_listing_ended_locally', $product_id, $listing_id, $reason);
    
    return array(
        'success' => true,
        'listing_id' => $listing_id,
        'message' => __('Listing marked as ended. Note: This only updates your local records. To remove the listing from Delcampe, please log into your Delcampe account.', 'wc-delcampe-integration')
    );
}