<?php
/**
 * Delcampe Listing API
 * 
 * Handles the creation and management of listings on Delcampe
 * Takes WooCommerce product data and profile settings to create listings via Delcampe API
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  API
 * @since       1.2.0.0
 * @version     1.8.0.0
 * 
 * @author      Frank Kahle
 * @copyright   2024 Frank Kahle
 * @license     Proprietary
 * 
 * Changelog:
 * 1.8.0.0 - Production release with comprehensive documentation
 *         - Enhanced security with improved validation
 *         - Better error handling and user feedback
 *         - Added detailed inline documentation
 *         - Improved image URL validation
 *         - Better API response parsing
 *         - Enhanced logging capabilities
 * 1.3.0.0 - Updated to use shipping_model_id from profile instead of shipping_settings
 *         - Integrated with new shipping models manager
 * 1.2.2.6 - Removed hardcoded URLs, now uses WordPress site URL dynamically
 *         - Plugin is now portable and works on any domain
 *         - Properly validates if images are accessible externally
 * 1.2.2.5 - Added image URL replacement for development environments
 *         - Can now use production site images for testing
 * 1.2.2.4 - Made id_shipping_model mandatory for fixed price items
 *         - Added better error logging for debugging
 * 1.2.2.3 - Fixed field names to match API documentation (fixed_price, qty)
 *         - Removed date_end to avoid conflicts with duration
 * 1.2.2.2 - Fixed API response parsing to handle error responses correctly
 *         - Limited duration to maximum 30 days
 * 1.2.2.1 - Fixed description builder instantiation with proper parameters
 * 1.2.2.0 - Updated to match actual Delcampe API requirements
 *         - Changed from XML body to form data submission
 *         - Updated endpoints and field names
 *         - Fixed response parsing
 * 1.2.0.0 - Initial implementation of listing creation API
 */

// Exit if accessed directly to prevent direct file access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Delcampe_Listing_API
 * 
 * Manages the creation and updating of Delcampe listings.
 * This class handles:
 * - Product data preparation for Delcampe format
 * - Image URL validation and preparation
 * - API communication for listing creation
 * - Response parsing and error handling
 * - Listing status updates in WooCommerce
 * 
 * The Delcampe API expects form data submission with specific field names
 * and all responses are in XML format.
 * 
 * @since   1.2.0.0
 * @version 1.8.0.0
 */
class Delcampe_Listing_API {
    
    /**
     * Singleton instance
     * Ensures only one instance of the API handler exists
     * 
     * @var Delcampe_Listing_API|null
     * @since 1.2.0.0
     */
    private static $instance = null;
    
    /**
     * API endpoints for different operations
     * All endpoints are relative to the base API URL
     * 
     * @var array
     * @since 1.2.0.0
     */
    private $endpoints = array(
        'add_item'    => '/item',
        'update_item' => '/item/',
        'get_item'    => '/item/',
        'close_item'  => '/item/',
        'bulk_item'   => '/item/bulk'
    );
    
    /**
     * Get singleton instance
     * 
     * Ensures only one instance of the listing API exists
     * throughout the plugin lifecycle.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @return Delcampe_Listing_API The singleton instance
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Get product ID by listing ID
     * 
     * @param string $listing_id Delcampe listing ID
     * @return int|false Product ID or false if not found
     */
    private function get_product_id_by_listing( $listing_id ) {
        global $wpdb;
        
        // First try to find in post meta
        $product_id = $wpdb->get_var( $wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} 
             WHERE meta_key = '_delcampe_listing_id' 
             AND meta_value = %s 
             LIMIT 1",
            $listing_id
        ) );
        
        if ( $product_id ) {
            return intval( $product_id );
        }
        
        // Try the listings table if it exists
        $table_name = $wpdb->prefix . 'delcampe_listings';
        if ( $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" ) === $table_name ) {
            $product_id = $wpdb->get_var( $wpdb->prepare(
                "SELECT product_id FROM {$table_name} 
                 WHERE delcampe_id = %s 
                 LIMIT 1",
                $listing_id
            ) );
            
            if ( $product_id ) {
                return intval( $product_id );
            }
        }
        
        return false;
    }
    
    /**
     * Constructor
     * 
     * Private constructor to enforce singleton pattern.
     * Initializes the listing API handler.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     */
    private function __construct() {
        // Log initialization
        $this->log( 'Listing API initialized', 'info' );
    }
    
    /**
     * Create a new listing on Delcampe
     * 
     * Takes a WooCommerce product and Delcampe profile to create a listing.
     * Handles all aspects of listing creation including:
     * - Data preparation
     * - Validation
     * - API submission
     * - Response handling
     * - Status updates
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  int $product_id WooCommerce product ID
     * @param  int $profile_id Delcampe profile ID
     * @return array|WP_Error Result array with listing details or error
     */
    public function create_listing( $product_id, $profile_id ) {
        $this->log( sprintf( 'Creating listing for product ID: %d with profile ID: %d', $product_id, $profile_id ), 'info' );
        
        // PHASE 1 DUPLICATE PREVENTION - Enhanced pre-creation checks
        global $wpdb;
        
        // 1. Database-level duplicate check BEFORE any API calls
        $existing_active = $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}delcampe_listings 
             WHERE product_id = %d 
             AND status IN ('active', 'published', 'verified', 'pending')
             ORDER BY created_at DESC
             LIMIT 1",
            $product_id
        ) );
        
        if ( $existing_active ) {
            $this->log( sprintf(
                'DUPLICATE PREVENTED: Product %d already has %s listing (ID: %s, Created: %s)',
                $product_id,
                $existing_active->status,
                $existing_active->delcampe_id,
                $existing_active->created_at
            ), 'warning' );
            
            // Log to business events for tracking
            if ( class_exists( 'Delcampe_Business_Logger' ) ) {
                $logger = Delcampe_Business_Logger::get_instance();
                $logger->log_event( 'duplicate_prevented', array(
                    'product_id' => $product_id,
                    'existing_listing_id' => $existing_active->id,
                    'existing_delcampe_id' => $existing_active->delcampe_id,
                    'existing_status' => $existing_active->status,
                    'prevention_source' => 'database_check'
                ), 'high' );
            }
            
            return array(
                'success' => true,
                'no_op' => true,
                'reason' => 'duplicate_prevented_db',
                'message' => __( 'Active listing already exists in database.', 'wc-delcampe-integration' ),
                'id' => $existing_active->delcampe_id,
                'product_id' => $product_id
            );
        }
        
        // 2. Distributed locking to prevent concurrent creation
        // v1.10.35.3: Enhanced atomic locking to fix race condition
        $lock_key = 'delcampe_create_' . $product_id;
        $lock_timeout = 60; // 60 seconds lock
        $lock_value = wp_generate_uuid4(); // Unique lock identifier
        
        // Atomic lock acquisition using database transaction
        // This prevents the race condition where two processes check simultaneously
        $acquired_lock = false;
        
        // Try to acquire lock atomically
        $existing_lock = get_transient( $lock_key );
        if ( ! $existing_lock ) {
            // Use database query for true atomic operation
            $option_name = '_transient_' . $lock_key;
            $result = $wpdb->query( $wpdb->prepare(
                "INSERT IGNORE INTO {$wpdb->options} (option_name, option_value, autoload) 
                 VALUES (%s, %s, 'no')",
                $option_name,
                serialize( $lock_value )
            ) );
            
            if ( $result === 1 ) {
                // We successfully acquired the lock
                $acquired_lock = true;
                // Set expiration
                set_transient( $lock_key, $lock_value, $lock_timeout );
            }
        }
        
        if ( ! $acquired_lock ) {
            $this->log( sprintf(
                'DUPLICATE PREVENTED: Product %d creation already in progress (locked)',
                $product_id
            ), 'warning' );
            
            // Log duplicate prevention
            if ( class_exists( 'Delcampe_Duplicate_Monitor' ) ) {
                $monitor = Delcampe_Duplicate_Monitor::get_instance();
                $monitor->log_prevention( 'lock_conflict', $product_id, 0, array(
                    'lock_key' => $lock_key,
                    'source' => wp_doing_ajax() ? 'ajax' : (wp_doing_cron() ? 'cron' : 'direct')
                ) );
            }
            
            return array(
                'success' => true,
                'no_op' => true,
                'reason' => 'creation_in_progress',
                'message' => __( 'Listing creation already in progress.', 'wc-delcampe-integration' ),
                'product_id' => $product_id
            );
        }
        
        // Initialize sync logger
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-sync-logger.php';
        $sync_logger = Delcampe_Sync_Logger::get_instance();
        
        // Initialize listing tracker for idempotency and audit logging
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-listing-tracker.php';
        $tracker = Delcampe_Listing_Tracker::get_instance();
        
        // Check idempotency - prevent duplicate creation
        // Debug override: allow forcing API calls even if idempotency would block
        $force_publish = (defined('DELCAMPE_FORCE_API_PUBLISH') && DELCAMPE_FORCE_API_PUBLISH) || (get_option('delcampe_force_api_publish', 'no') === 'yes');
        if ( ! $force_publish && ! $tracker->should_create_listing( $product_id ) ) {
            $this->log( 'Listing creation blocked by idempotency check for product: ' . $product_id, 'info' );
            
            // Release the lock before returning
            delete_transient( $lock_key );
            
            // Return success with no-op flag instead of error to prevent queue retries
            $existing_id = get_post_meta( $product_id, '_delcampe_item_id', true );
            
            if (function_exists('delcampe_publish_audit')) {
                delcampe_publish_audit(array(
                    'phase' => 'create_noop',
                    'endpoint' => '/item',
                    'product_id' => (int)$product_id,
                    'reason' => 'duplicate_prevented',
                    'existing_id' => $existing_id ? (string)$existing_id : null
                ));
            }
            
            return array(
                'success' => true,
                'no_op' => true,
                'reason' => 'duplicate_prevented',
                'message' => __( 'Listing already exists or creation in progress.', 'wc-delcampe-integration' ),
                'id' => $existing_id ? $existing_id : '',
                'product_id' => $product_id
            );
        }
        
        // Validate product ID
        if ( ! is_numeric( $product_id ) || $product_id <= 0 ) {
            // Release lock before returning error
            delete_transient( $lock_key );
            return new WP_Error( 
                'invalid_product_id', 
                __( 'Invalid product ID provided.', 'wc-delcampe-integration' ) 
            );
        }
        
        // Get product data
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            // Release lock before returning error
            delete_transient( $lock_key );
            return new WP_Error( 
                'invalid_product', 
                __( 'Product not found.', 'wc-delcampe-integration' ) 
            );
        }
        
        // Log sync start
        $product_data = array(
            'sku' => $product->get_sku(),
            'title' => $product->get_name(),
            'price' => $product->get_price(),
            'personal_reference' => $product->get_sku()
        );
        $sync_logger->log_sync_start($product_id, $product_data);
        
        // Validate profile ID
        if ( ! is_numeric( $profile_id ) || $profile_id <= 0 ) {
            // Release lock before returning error
            delete_transient( $lock_key );
            return new WP_Error( 
                'invalid_profile_id', 
                __( 'Invalid profile ID provided.', 'wc-delcampe-integration' ) 
            );
        }
        
        // Get profile data
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/profiles/class-delcampe-profiles-model.php';
        $profiles_model = Delcampe_Profiles_Model::get_instance();
        $profile = $profiles_model->get_profile( $profile_id );
        
        if ( ! $profile ) {
            // Release lock before returning error
            delete_transient( $lock_key );
            return new WP_Error( 
                'invalid_profile', 
                __( 'Profile not found.', 'wc-delcampe-integration' ) 
            );
        }
        
        // Build listing data from product and profile
        $listing_data = $this->build_listing_data( $product, $profile );
        
        // CRITICAL: Pre-create SKU check ENABLED by default to prevent duplicates (v1.10.35.9)
        // This check prevents duplicate listings when reconciliation runs during batch processing
        $do_precheck = (bool) apply_filters('delcampe_precreate_sku_check', true, $listing_data, $product_id);
        if ($do_precheck && ! $force_publish && !empty($listing_data['personal_reference'])) {
            $existing_item = $this->check_sku_exists_on_delcampe($listing_data['personal_reference']);
            
            if ($existing_item !== false) {
                // SKU already exists on Delcampe - update local records and return success
                $this->log('SKU ' . $listing_data['personal_reference'] . ' already exists on Delcampe with ID: ' . $existing_item['id'], 'info');
                
                // Store the Delcampe ID locally
                $delcampe_id = isset($existing_item['id']) ? $existing_item['id'] : '';
                if (!empty($delcampe_id)) {
                    // Use tracker to store with verification
                    $tracker->store_listing_with_verification(
                        $product_id,
                        $delcampe_id,
                        array(
                            '_delcampe_personal_reference' => $listing_data['personal_reference'],
                            '_delcampe_sync_source' => 'sku_check_recovery'
                        )
                    );
                    
                    // Log the recovery
                    $tracker->log_api_audit('sku_recovery', $product_id, array(
                        'personal_reference' => $listing_data['personal_reference'],
                        'delcampe_id' => $delcampe_id,
                        'reason' => 'Found existing item on Delcampe during SKU check'
                    ));
                }
                
                // Return success with recovered flag
                if (function_exists('delcampe_publish_audit')) {
                    delcampe_publish_audit(array(
                        'phase' => 'create_sku_recovery',
                        'endpoint' => '/item/reference/{ref}',
                        'product_id' => (int)$product_id,
                        'delcampe_id' => (string)$delcampe_id,
                        'personal_reference' => (string)$listing_data['personal_reference']
                    ));
                }
                // Release lock before returning
                delete_transient( $lock_key );
                return array(
                    'success' => true,
                    'recovered' => true,
                    'reason' => 'sku_already_exists',
                    'message' => __('Item already exists on Delcampe, local records updated.', 'wc-delcampe-integration'),
                    'id' => $delcampe_id,
                    'product_id' => $product_id,
                    'personal_reference' => $listing_data['personal_reference']
                );
            }
        }

        
        

        
        // v1.10.35.7: CRITICAL FIX - Never modify the SKU/personal_reference
        // The SKU must remain unique and unchanged to prevent duplicates on Delcampe
        // Each product should have ONE unique SKU that never changes
        
        // Get the personal reference (SKU)
        $personal_ref = isset( $listing_data['personal_reference'] ) ? (string) $listing_data['personal_reference'] : '';
        
        // If no SKU provided, use product SKU or generate one
        if ( empty( $personal_ref ) ) {
            $personal_ref = $product->get_sku();
            if ( empty( $personal_ref ) ) {
                // Only as last resort, generate one based on product ID
                $personal_ref = 'WC-' . $product->get_id();
            }
        }
        
        // Sanitize to allowed characters (alphanumeric, dash, underscore)
        $personal_ref = preg_replace( '/[^A-Za-z0-9\-_]/', '-', $personal_ref );
        
        // Enforce maximum length (Delcampe limit)
        $MAX_LEN = 50; // Delcampe's actual limit
        if ( strlen( $personal_ref ) > $MAX_LEN ) {
            $personal_ref = substr( $personal_ref, 0, $MAX_LEN );
        }
        
        // Clean up any trailing dashes
        $personal_ref = rtrim( $personal_ref, '-' );
        
        // IMPORTANT: Check if this exact SKU already has an active listing
        // If it does, we should NOT create a duplicate
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-listings-model.php';
        $existing = Delcampe_Listings_Model::get_listings_by_product_id( (int) $product_id );
        
        if ( is_array( $existing ) && ! empty( $existing ) ) {
            // Check if there's already an active listing with this SKU
            foreach ( $existing as $row ) {
                if ( isset( $row->status ) && in_array( $row->status, array( 'active', 'published', 'verified', 'pending' ) ) ) {
                    // Active listing exists - should not create duplicate
                    $this->log( sprintf(
                        'DUPLICATE PREVENTED: Product %d (SKU: %s) already has active listing',
                        $product_id,
                        $personal_ref
                    ), 'warning' );
                    
                    // Return the existing listing info instead of creating duplicate
                    return array(
                        'success' => true,
                        'no_op' => true,
                        'reason' => 'active_listing_exists',
                        'message' => __( 'Active listing already exists for this SKU.', 'wc-delcampe-integration' ),
                        'id' => $row->delcampe_id,
                        'personal_reference' => $personal_ref,
                        'product_id' => $product_id
                    );
                }
            }
        }
        
        // Set the final personal reference - NEVER modified with suffixes
        $listing_data['personal_reference'] = $personal_ref;
        
        // Persist chosen personal reference to product meta for later reconciliation
        if ( ! empty( $listing_data['personal_reference'] ) ) {
            update_post_meta( $product_id, '_delcampe_personal_reference', $listing_data['personal_reference'] );
        }

        // Validate listing data before submission
        $validation = $this->validate_listing_data( $listing_data );
        if ( is_wp_error( $validation ) ) {
            $this->log( 'Listing validation failed: ' . $validation->get_error_message(), 'error' );
            return $validation;
        }
        
        // Check if images are accessible externally
        $image_check = $this->validate_image_urls( $listing_data['images'] );
        if ( is_wp_error( $image_check ) ) {
            $this->log( 'Image validation warning: ' . $image_check->get_error_message(), 'warning' );
            // Log warning but continue - let Delcampe handle the final validation
        }
        
        // Convert to Delcampe API format
        $post_data = $this->format_for_api( $listing_data );
        
        // Log the prepared data for debugging
        if ( delcampe_is_debug_mode() ) {
            $this->log( 'Prepared post data: ' . print_r( $post_data, true ), 'debug' );
        }
        
        // Log write-ahead audit entry BEFORE API call
        $tracker->log_api_audit( 'create', $product_id, $post_data );
        
        // Submit to Delcampe API (forced: also log that we bypassed idempotency/SKU check)
        if ($force_publish && function_exists('delcampe_publish_audit')) {
            delcampe_publish_audit(array(
                'phase' => 'create_forced',
                'product_id' => (int)$product_id,
                'note' => 'Bypassed idempotency/SKU checks for debugging'
            ));
        }
        $response = $this->submit_listing( $post_data, $sync_logger, (int)$product_id );
        
        // Log API response to audit trail
        $delcampe_id = $tracker->log_api_audit( 'create_response', $product_id, $post_data, $response );
        
        if ( is_wp_error( $response ) ) {
            $this->log( 'API submission failed: ' . $response->get_error_message(), 'error' );
            
            // Log error with sync logger
            if ( $sync_logger ) {
                $sync_logger->log_sync_error( 
                    $product_id, 
                    $response->get_error_message(),
                    array( 'error_code' => $response->get_error_code() )
                );
            }
            
            // PHASE 1 DUPLICATE PREVENTION - Release the lock on error
            delete_transient( $lock_key );
            
            return $response;
        }
        
        // Extract Delcampe ID from response if not already found
        if ( empty( $delcampe_id ) && ! empty( $response['id'] ) ) {
            $delcampe_id = $response['id'];
        }
        
        // Store listing with verification
        if ( ! empty( $delcampe_id ) ) {
            $additional_meta = array(
                '_delcampe_personal_reference' => $listing_data['personal_reference'],
                '_delcampe_profile_id' => $profile_id
            );
            
            $stored = $tracker->store_listing_with_verification( $product_id, $delcampe_id, $additional_meta );
            
            if ( ! $stored ) {
                $this->log( 'Warning: Failed to verify storage of Delcampe ID ' . $delcampe_id . ' for product ' . $product_id, 'warning' );
            }
        }
        
        // Process the response
        $this->process_listing_response( $response, $product_id, $listing_data, $sync_logger );
        
        // PHASE 1 DUPLICATE PREVENTION - Release the lock before return
        delete_transient( $lock_key );
        
        return $response;
    }
    
    /**
     * Process listing creation response
     * 
     * Updates product metadata and sync status based on API response.
     * 
     * @since  1.8.0.0
     * @param  array $response API response data
     * @param  int $product_id Product ID
     * @param  array $listing_data Original listing data
     * @return void
     */
    private function process_listing_response( $response, $product_id, $listing_data, $sync_logger = null ) {
        // Get product integration instance
        $integration = Delcampe_Product_Integration::get_instance();
        
        // Check if response indicates success
        if ( ! empty( $response['success'] ) && $response['success'] === true ) {
            // Success - update product status
            $integration->update_sync_status( $product_id, 'active' );
            
            // Store personal reference if available
            if ( ! empty( $response['personal_reference'] ) ) {
                update_post_meta( $product_id, '_delcampe_personal_reference', sanitize_text_field( $response['personal_reference'] ) );
            }
            
            // Store listing ID if available
            $listing_id = '';
            if ( ! empty( $response['id'] ) ) {
                $listing_id = sanitize_text_field( $response['id'] );
                update_post_meta( $product_id, '_delcampe_listing_id', $listing_id );
            }
            
            // Log success with sync logger
            if ( $sync_logger ) {
                $sync_logger->log_sync_complete(
                    $product_id, 
                    true, 
                    $listing_id,
                    isset($response['message']) ? $response['message'] : 'Listing created successfully'
                );
            }
            
            // Update stock cache if available (v1.10.20.3)
            if (!empty($listing_id) && !empty($listing_data['quantity'])) {
                if (class_exists('Delcampe_Feature_Flags') && 
                    class_exists('Delcampe_Stock_Cache_Manager')) {
                    if (Delcampe_Feature_Flags::is_enabled('stock_caching')) {
                        $cache_manager = Delcampe_Stock_Cache_Manager::get_instance();
                        $cache_manager->write_cache($listing_id, intval($listing_data['quantity']), 'listing_created');
                    }
                }
            }
            
            // Log success
            $this->log( sprintf( 'Successfully created listing for product: %d', $product_id ), 'info' );
            
            // Trigger action for other plugins
            do_action( 'delcampe_listing_created', $product_id, $response );
            
        } elseif ( ! empty( $response['queued'] ) && $response['queued'] === true ) {
            // Item is queued for processing - this is a successful submission
            $integration->update_sync_status( $product_id, 'active' );
            
            // Store personal reference if available
            if ( ! empty( $response['personal_reference'] ) ) {
                update_post_meta( $product_id, '_delcampe_personal_reference', sanitize_text_field( $response['personal_reference'] ) );
            }
            
            // Log success with sync logger
            if ( $sync_logger ) {
                $sync_logger->log_sync_complete(
                    $product_id, 
                    true, 
                    '',
                    isset($response['message']) ? $response['message'] : 'Item queued for processing'
                );
            }
            
            // Log success
            $this->log( sprintf( 'Successfully queued listing for product: %d', $product_id ), 'info' );
            
            // Trigger action for other plugins
            do_action( 'delcampe_listing_queued', $product_id, $response );
            
        } else {
            // The API accepted the request but we're not sure of the status
            $message = isset($response['message']) ? $response['message'] : 'Unknown response status';
            
            // Log with sync logger
            if ( $sync_logger ) {
                $sync_logger->log_sync_complete(
                    $product_id,
                    false,
                    '',
                    $message
                );
            }
            
            $this->log( 
                sprintf( 'Unclear response for product %d. Response: %s', 
                    $product_id, 
                    print_r( $response, true ) 
                ), 
                'warning' 
            );
            
            // Mark as pending since we're unsure
            $integration->update_sync_status( $product_id, 'pending' );
            
            // Store personal reference for tracking
            if ( ! empty( $listing_data['personal_reference'] ) ) {
                update_post_meta( $product_id, '_delcampe_personal_reference', sanitize_text_field( $listing_data['personal_reference'] ) );
            }
        }
        
        // PHASE 1 DUPLICATE PREVENTION - Release the lock
        delete_transient( $lock_key );
    }
    
    /**
     * Validate image URLs are accessible
     * 
     * Checks if image URLs can be accessed by external services.
     * Detects common issues like localhost URLs or private IP addresses.
     * 
     * @since  1.2.2.4
     * @version 1.8.0.0
     * @param  array $images Array of image URLs to validate
     * @return true|WP_Error True if valid, error with details otherwise
     */
    private function validate_image_urls( $images ) {
        if ( empty( $images ) || ! is_array( $images ) ) {
            return new WP_Error( 
                'no_images', 
                __( 'No images provided for validation.', 'wc-delcampe-integration' ) 
            );
        }
        
        // Get the site URL
        $site_url = get_site_url();
        $parsed_url = parse_url( $site_url );
        $host = isset( $parsed_url['host'] ) ? $parsed_url['host'] : '';
        
        // Check if the site is using a private/local address
        $is_private = false;
        
        // Check for private IP ranges (RFC 1918)
        if ( preg_match( '/^(?:192\.168|10\.|172\.(?:1[6-9]|2[0-9]|3[01]))\.|^127\.|^::1$|^fc00:/i', $host ) ) {
            $is_private = true;
        }
        
        // Check for localhost variations
        if ( in_array( strtolower( $host ), array( 'localhost', 'localhost.localdomain', 'local', 'dev', 'test' ), true ) ) {
            $is_private = true;
        }
        
        // Check for development TLDs
        if ( preg_match( '/\.(local|dev|test|localhost)$/i', $host ) ) {
            $is_private = true;
        }
        
        // If site is private, provide helpful error message
        if ( $is_private ) {
            return new WP_Error(
                'private_site_images',
                sprintf(
                    __( 'Your site (%s) appears to be on a private network. Delcampe cannot access images from private networks. Please ensure your site is publicly accessible or use a service like ngrok or Cloudflare Tunnel for testing.', 'wc-delcampe-integration' ),
                    esc_url( $site_url )
                )
            );
        }
        
        // Check individual image URLs for accessibility issues
        foreach ( $images as $index => $image ) {
            // Handle both string URLs and array format from Image Manager
            if ( is_array( $image ) ) {
                // Image Manager format: array with 'url', 'is_main', 'position' keys
                $url = isset( $image['url'] ) ? $image['url'] : '';
            } else {
                // Simple string URL
                $url = $image;
            }
            
            // Validate URL format
            if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
                return new WP_Error(
                    'invalid_image_url',
                    sprintf( 
                        __( 'Invalid image URL at position %d: %s', 'wc-delcampe-integration' ), 
                        $index + 1, 
                        esc_url( $url ) 
                    )
                );
            }
            
            $image_host = parse_url( $url, PHP_URL_HOST );
            
            // Check for private IP in image URL
            if ( preg_match( '/^(?:192\.168|10\.|172\.(?:1[6-9]|2[0-9]|3[01]))\.|^127\./i', $image_host ) ) {
                return new WP_Error(
                    'local_ip_images',
                    __( 'Images are using local IP addresses that Delcampe cannot access. Please use publicly accessible URLs.', 'wc-delcampe-integration' )
                );
            }
            
            // Check for localhost in image URL
            if ( in_array( strtolower( $image_host ), array( 'localhost', 'localhost.localdomain' ), true ) ) {
                return new WP_Error(
                    'localhost_images',
                    __( 'Images are using localhost URLs that Delcampe cannot access. Please use publicly accessible URLs.', 'wc-delcampe-integration' )
                );
            }
        }
        
        return true;
    }
    
    /**
     * Build listing data from product and profile
     * 
     * Prepares all data needed for creating a Delcampe listing.
     * Combines product information with profile settings.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  WC_Product $product WooCommerce product object
     * @param  array $profile Profile data array
     * @return array Prepared listing data
     */
    private function build_listing_data( $product, $profile ) {
        // Get description builder for generating listing description
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/profiles/class-delcampe-description-builder.php';
        
        // Generate description using description builder
        $generated_description = $this->generate_description( $product, $profile );
        
        // Determine listing type (auction or fixed price)
        $listing_type = isset( $profile['listing_details']['listing_type'] ) ? $profile['listing_details']['listing_type'] : 'FixedPrice';
        $is_auction = ( strtolower( $listing_type ) === 'auction' );
        
        // Get and validate duration - must be one of: 7, 10, 14, 21, 28
        $duration = isset( $profile['listing_details']['duration'] ) ? intval( $profile['listing_details']['duration'] ) : 10;
        $allowed_durations = array( 7, 10, 14, 21, 28 );
        if ( ! in_array( $duration, $allowed_durations, true ) ) {
            $duration = 10; // Default to 10 days
            $this->log( 
                sprintf( 'Duration adjusted to 10 days (was %d)', $profile['listing_details']['duration'] ), 
                'warning' 
            );
        }
        
        // Build listing data structure
        $listing_data = array(
            // Type (determines other required fields)
            'type' => $is_auction ? 'auction' : 'fixedPrice',
            
            // Basic Information (MANDATORY)
            'title' => $this->prepare_title( $product, $profile ),
            'description' => $generated_description,
            'id_category' => $profile['delcampe_category_id'],
            'personal_reference' => $product->get_sku() ?: 'WC-' . $product->get_id(),
            
            // Currency (MANDATORY) - Use profile currency, or WooCommerce currency, or EUR as last resort
            'currency' => isset( $profile['listing_details']['currency'] ) ? $profile['listing_details']['currency'] : get_woocommerce_currency(),
            
            // Duration (OPTIONAL but recommended)
            'duration' => $duration,
            
            // Renew (OPTIONAL)
            'renew' => $this->get_renew_value( $profile ),
            
            // Images (REQUIRED - 1 to 99 images)
            'images' => $this->prepare_images( $product, $profile ),
            
            // Shipping (MANDATORY for most categories)
            'id_shipping_model' => $this->get_shipping_model_id( $profile ),
            
            // Weight (OPTIONAL - in grams)
            'weight' => $this->get_product_weight( $product ),
            
            // Options (all OPTIONAL)
            'option_boldtitle' => ! empty( $profile['listing_details']['bold_title'] ),
            'option_highlight' => ! empty( $profile['listing_details']['highlight'] ),
            'option_coloredborder' => ! empty( $profile['listing_details']['colored_border'] ),
            'option_toplisting' => ! empty( $profile['listing_details']['top_listing'] ),
            'option_topmain' => ! empty( $profile['listing_details']['top_main'] ),
            
            // Preferred end time (OPTIONAL)
            'prefered_end_day' => isset( $profile['listing_details']['preferred_end_day'] ) ? $profile['listing_details']['preferred_end_day'] : null,
            'prefered_end_hour' => isset( $profile['listing_details']['preferred_end_hour'] ) ? $profile['listing_details']['preferred_end_hour'] : null,
        );
        
        // Add type-specific fields
        if ( $is_auction ) {
            // Auction fields (MANDATORY for auctions)
            $listing_data['price_starting'] = $this->calculate_price( $product, $profile );
            $listing_data['price_increment'] = isset( $profile['listing_details']['price_increment'] ) ? $profile['listing_details']['price_increment'] : '0.01';
            // Ensure shipping model ID is string for auctions
            $listing_data['id_shipping_model'] = strval( $listing_data['id_shipping_model'] );
        } else {
            // Fixed price fields (MANDATORY for fixed price)
            $listing_data['fixed_price'] = $this->calculate_price( $product, $profile );
            $listing_data['qty'] = $this->calculate_quantity( $product, $profile );
        }
        
        // Allow filtering of listing data
        $listing_data = apply_filters( 'delcampe_listing_data', $listing_data, $product, $profile );
        
        return $listing_data;
    }
    
    /**
     * Generate description using the description builder
     * 
     * Creates a formatted description based on product data and profile settings.
     * 
     * @since  1.2.2.1
     * @version 1.8.0.0
     * @param  WC_Product $product Product object
     * @param  array $profile Profile data
     * @return string Generated description HTML
     */
    private function generate_description( $product, $profile ) {
        try {
            // Create description builder with product and profile
            $description_builder = new Delcampe_Description_Builder( $product, $profile );
            
            // Generate the description
            $description = $description_builder->build();
            
            // Ensure description is not empty
            if ( empty( $description ) ) {
                $description = $product->get_description();
                if ( empty( $description ) ) {
                    $description = $product->get_short_description();
                }
            }
            
            return $description;
            
        } catch ( Exception $e ) {
            $this->log( 'Description builder error: ' . $e->getMessage(), 'error' );
            // Fallback to basic description
            return $product->get_description() ?: $product->get_short_description();
        }
    }
    
    /**
     * Format listing data for API submission
     * 
     * Converts the internal listing data structure to the format
     * expected by the Delcampe API.
     * 
     * @since  1.2.2.0
     * @version 1.8.0.0
     * @param  array $listing_data Listing data
     * @return array Formatted post data for API
     */
    private function format_for_api( $listing_data ) {
        // Extract type and remove from data
        $type = $listing_data['type'];
        unset( $listing_data['type'] );
        
        // Structure for API submission
        $post_data = array(
            'type' => $type,
            'item' => array()
        );
        
        // Map fields to item array
        foreach ( $listing_data as $key => $value ) {
            // Skip null or empty values except for boolean fields
            if ( $value === null || ( $value === '' && ! is_bool( $value ) ) ) {
                continue;
            }
            
            // Handle special fields
            if ( $key === 'images' && is_array( $value ) ) {
                // Images need special handling - convert to simple URL array
                $image_urls = array();
                foreach ( $value as $image ) {
                    if ( is_array( $image ) && isset( $image['url'] ) ) {
                        // Image Manager format with metadata
                        $image_urls[] = $image['url'];
                    } elseif ( is_string( $image ) ) {
                        // Simple URL string
                        $image_urls[] = $image;
                    }
                }
                $post_data['item']['images'] = $image_urls;
            } else {
                // Add to item array
                $post_data['item'][$key] = $value;
            }
        }
        
        return $post_data;
    }
    
    /**
     * Submit listing to Delcampe API
     * 
     * Sends the prepared listing data to Delcampe's API endpoint
     * and handles the response.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  array $post_data Formatted post data
     * @return array|WP_Error Response array or error
     */
    private function submit_listing( $post_data, $sync_logger = null, $product_id = null ) {
        // Get authentication token
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if ( is_wp_error( $token ) ) {
            return new WP_Error(
                'auth_failed',
                sprintf(
                    __( 'Failed to authenticate with Delcampe: %s', 'wc-delcampe-integration' ),
                    $token->get_error_message()
                )
            );
        }
        
        // Prepare API request URL (do not log token)
        $api_url = DELCAMPE_API_BASE_URL . $this->endpoints['add_item'] . '?token=' . $token;
        
        // Prepare request arguments
        $args = array(
            'method'  => 'POST',
            'timeout' => 30,
            'headers' => array(
                'User-Agent' => DELCAMPE_USER_AGENT
            ),
            'body'    => $post_data
        );
        
        // Log API request details
        $this->log( 'Submitting listing to API endpoint: ' . $this->endpoints['add_item'], 'info' );
        // Publish audit: request snapshot (mask token, trim body)
        if (function_exists('delcampe_publish_audit')) {
            $snapshot = $post_data;
            // Reduce noise in audit by summarizing images
            if (isset($snapshot['item']['images']) && is_array($snapshot['item']['images'])) {
                $snapshot['item']['images_count'] = count($snapshot['item']['images']);
                unset($snapshot['item']['images']);
            }
            delcampe_publish_audit(array(
                'phase' => 'create_request',
                'endpoint' => '/item',
                'product_id' => $product_id,
                'request_body' => print_r($snapshot, true)
            ));
        }
        
        // Log to sync logger
        if (isset($sync_logger)) {
            $sync_logger->log_api_request($this->endpoints['add_item'], $post_data);
        }
        
        // Log timing and make API request
        $start_time = microtime(true);
        $this->log( 'Starting POST to /item endpoint', 'debug' );
        
        $start_time = microtime(true);
        $response = wp_remote_post( $api_url, $args );
        $elapsed = round((microtime(true) - $start_time) * 1000);
        
        $elapsed = round(microtime(true) - $start_time, 2);
        $this->log( 'POST to /item completed in ' . $elapsed . 's', 'debug' );
        
        // Check for WordPress errors
        if ( is_wp_error( $response ) ) {
            $this->log( 'API request failed after ' . $elapsed . 's: ' . $response->get_error_message(), 'error' );
            return new WP_Error(
                'api_request_failed',
                sprintf(
                    __( 'Failed to connect to Delcampe API: %s', 'wc-delcampe-integration' ),
                    $response->get_error_message()
                )
            );
        }
        
        // Get response details
        $body = wp_remote_retrieve_body( $response );
        $status_code = wp_remote_retrieve_response_code( $response );
        
        // Log response details
        $this->log( sprintf( 'API response code: %d (%d ms)', $status_code, $elapsed ), 'info' );
        if (function_exists('delcampe_publish_audit')) {
            delcampe_publish_audit(array(
                'phase' => 'create_response',
                'endpoint' => '/item',
                'product_id' => $product_id,
                'status' => $status_code,
                'elapsed_ms' => $elapsed,
                'response_body' => $body
            ));
        }
        
        if ( delcampe_is_debug_mode() ) {
            $this->log( 'API response body: ' . $body, 'debug' );
        }
        
        // Parse XML response
        $parsed = $this->parse_api_response( $body );
        
        // Log to sync logger with parsed response
        if (isset($sync_logger)) {
            $sync_logger->log_api_response($status_code, $body, $parsed);
        }
        
        if ( is_wp_error( $parsed ) ) {
            return $parsed;
        }
        
        // Check for API errors in parsed response
        if ( ! empty( $parsed['error'] ) ) {
            $error_message = $parsed['error'];
            $this->log( 'API returned error: ' . $error_message, 'error' );
            
            return new WP_Error( 
                'api_error', 
                sprintf( __( 'Delcampe API error: %s', 'wc-delcampe-integration' ), $error_message ),
                $parsed 
            );
        }
        
        return $parsed;
    }
    
    /**
     * Parse API response
     * 
     * Parses the XML response from Delcampe API and extracts
     * relevant information. Handles both 200 and 201 Created responses.
     * 
     * @since  1.2.0.0
     * @version 1.8.1.0
     * @param  string $xml_response XML response string
     * @return array|WP_Error Parsed data array or error
     */
    private function parse_api_response( $xml_response ) {
        if ( empty( $xml_response ) ) {
            return new WP_Error( 
                'empty_response', 
                __( 'Empty response received from Delcampe API.', 'wc-delcampe-integration' ) 
            );
        }
        
        // Suppress XML errors and use internal error handling
        libxml_use_internal_errors( true );
        
        try {
            // Attempt to parse XML
            $xml = simplexml_load_string( $xml_response );
            
            if ( $xml === false ) {
                // Collect parsing errors
                $errors = libxml_get_errors();
                $error_messages = array();
                foreach ( $errors as $error ) {
                    $error_messages[] = trim( $error->message );
                }
                libxml_clear_errors();
                
                $error_msg = 'XML Parse Error: ' . implode( '; ', $error_messages );
                $this->log( $error_msg, 'error' );
                
                return new WP_Error( 
                    'xml_parse_error', 
                    __( 'Failed to parse API response. The response may not be valid XML.', 'wc-delcampe-integration' ) 
                );
            }
            
            // Initialize response data array
            $data = array();
            
            // Check if this is a full notification envelope
            if ( isset( $xml->Notification_Type ) ) {
                // This is a callback notification with full envelope
                $data['notification_token'] = isset( $xml->Notification_Token ) ? (string) $xml->Notification_Token : '';
                $data['notification_datetime'] = isset( $xml->Notification_Datetime ) ? (string) $xml->Notification_Datetime : '';
                $data['notification_type'] = (string) $xml->Notification_Type;
                
                // For notifications, assume success unless proven otherwise
                $data['success'] = true;
                $data['status'] = 201; // Notifications typically indicate success
                
                // Extract notification data based on type
                if ( isset( $xml->Notification_Data ) ) {
                    $notification_data = $xml->Notification_Data;
                    
                    switch ( $data['notification_type'] ) {
                        case 'Seller_Item_Add':
                            if ( isset( $notification_data->id_item ) ) {
                                $data['id'] = (string) $notification_data->id_item;
                            }
                            if ( isset( $notification_data->personal_reference ) ) {
                                $data['personal_reference'] = (string) $notification_data->personal_reference;
                            }
                            $data['message'] = 'Your item has been taken into account';
                            $data['queued'] = true;
                            break;
                            
                        case 'Seller_Item_Add_Error_Image':
                            // Item was added but with image errors
                            if ( isset( $notification_data->id_item ) ) {
                                $data['id'] = (string) $notification_data->id_item;
                            }
                            if ( isset( $notification_data->errorImages ) ) {
                                $data['warning'] = 'Some images could not be processed';
                                $data['image_errors'] = array();
                                foreach ( $notification_data->errorImages->imageUrl as $url ) {
                                    $data['image_errors'][] = (string) $url;
                                }
                            }
                            break;
                            
                        case 'Seller_Item_Update':
                        case 'Seller_Item_Close_Sold':
                        case 'Seller_Item_Close_Unsold':
                            // Handle other notification types as needed
                            if ( isset( $notification_data->id_item ) ) {
                                $data['id'] = (string) $notification_data->id_item;
                            }
                            break;
                    }
                }
                
                return $data;
            }
            
            // Otherwise, check for the API response structure we've been using
            // Check for notification type in headers (old structure)
            $notification_type = isset( $xml->Notification_Data->Headers->Notification_type ) ?
                                (string) $xml->Notification_Data->Headers->Notification_type : '';
            
            // Check the status in headers
            $status = isset( $xml->Notification_Data->Headers->Status ) ? 
                      (int) $xml->Notification_Data->Headers->Status : 200;
            
            // Handle response based on status code
            if ( $status === 200 || $status === 201 ) {
                // Success response
                $data['success'] = true;
                $data['status'] = $status;
                $data['notification_type'] = $notification_type;
                
                // Extract data from body
                if ( isset( $xml->Notification_Data->body ) ) {
                    $body = $xml->Notification_Data->body;
                    
                    // Handle 201 Created with Seller_Item_Add notification
                    if ( $status === 201 && $notification_type === 'Seller_Item_Add' ) {
                        // Check for status message
                        if ( isset( $body->Status ) ) {
                            $data['message'] = (string) $body->Status;
                            // Check if message indicates queued status
                            if ( strpos( $data['message'], 'taken into account' ) !== false ) {
                                $data['queued'] = true;
                            }
                        }
                        
                        // Check for personal reference
                        if ( isset( $body->Personal_reference ) ) {
                            $data['personal_reference'] = (string) $body->Personal_reference;
                        }
                        
                        // Check for warnings
                        if ( isset( $body->Warning ) ) {
                            $data['warning'] = (string) $body->Warning;
                            $this->log( 'API Warning: ' . $data['warning'], 'warning' );
                        }
                        
                        // Item ID may not be present for queued items
                        if ( isset( $body->Item_ID ) ) {
                            $data['id'] = (string) $body->Item_ID;
                        }
                    }
                    // Handle standard 200 response (old format)
                    elseif ( isset( $body->item ) ) {
                        // Get status message
                        if ( isset( $body->item->status ) ) {
                            $data['message'] = (string) $body->item->status;
                            // Check if message indicates queued status
                            if ( strpos( $data['message'], 'taken into account' ) !== false ) {
                                $data['queued'] = true;
                            }
                        }
                        
                        // Get personal reference
                        if ( isset( $body->item->personal_reference ) ) {
                            $data['personal_reference'] = (string) $body->item->personal_reference;
                        }
                        
                        // Get item ID (may not be present if queued)
                        if ( isset( $body->item->id ) ) {
                            $data['id'] = (string) $body->item->id;
                        }
                    }
                    // Also check direct body elements
                    elseif ( isset( $body->status ) ) {
                        $data['message'] = (string) $body->status;
                    }
                    elseif ( isset( $body->Status ) ) {
                        $data['message'] = (string) $body->Status;
                    }
                }
            } else {
                // Error response
                $data['success'] = false;
                $data['status'] = $status;
                $data['notification_type'] = $notification_type;
                
                // Extract error from body
                if ( isset( $xml->Notification_Data->body->error ) ) {
                    $data['error'] = (string) $xml->Notification_Data->body->error;
                } elseif ( isset( $xml->Notification_Data->body->e ) ) {
                    $data['error'] = (string) $xml->Notification_Data->body->e;
                } elseif ( isset( $xml->Notification_Data->body->Error ) ) {
                    $data['error'] = (string) $xml->Notification_Data->body->Error;
                } else {
                    $data['error'] = sprintf( __( 'Unknown error (status code: %d)', 'wc-delcampe-integration' ), $status );
                }
            }
            
            return $data;
            
        } catch ( Exception $e ) {
            $this->log( 'Exception parsing API response: ' . $e->getMessage(), 'error' );
            return new WP_Error( 
                'parse_exception', 
                __( 'An error occurred while processing the API response.', 'wc-delcampe-integration' ) 
            );
        } finally {
            // Always restore error handling
            libxml_use_internal_errors( false );
        }
    }
    
    /**
     * Validate listing data before submission
     * 
     * Ensures all required fields are present and valid
     * according to Delcampe API requirements.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  array $listing_data Listing data to validate
     * @return true|WP_Error True if valid, error with details otherwise
     */
    private function validate_listing_data( $listing_data ) {
        // Define required fields for all listing types
        $required_fields = array( 'title', 'id_category', 'currency', 'personal_reference' );
        
        // Add type-specific required fields
        if ( $listing_data['type'] === 'auction' ) {
            $required_fields[] = 'price_starting';
            $required_fields[] = 'price_increment';
            $required_fields[] = 'id_shipping_model'; // Mandatory for auctions
        } else {
            $required_fields[] = 'fixed_price';
            // qty is optional with default of 1
        }
        
        // Check each required field
        foreach ( $required_fields as $field ) {
            if ( empty( $listing_data[$field] ) && $listing_data[$field] !== '0' ) {
                return new WP_Error(
                    'missing_field',
                    sprintf( 
                        __( 'Required field missing: %s', 'wc-delcampe-integration' ), 
                        $field 
                    )
                );
            }
        }
        
        // Validate category ID is numeric
        if ( ! is_numeric( $listing_data['id_category'] ) ) {
            return new WP_Error(
                'invalid_category',
                __( 'Category ID must be numeric.', 'wc-delcampe-integration' )
            );
        }
        
        // Validate title length (max 100 characters)
        if ( strlen( $listing_data['title'] ) > 100 ) {
            return new WP_Error(
                'title_too_long',
                sprintf( 
                    __( 'Title exceeds maximum length of 100 characters (current: %d).', 'wc-delcampe-integration' ), 
                    strlen( $listing_data['title'] ) 
                )
            );
        }
        
        // Validate title is not empty after trimming
        if ( empty( trim( $listing_data['title'] ) ) ) {
            return new WP_Error(
                'empty_title',
                __( 'Title cannot be empty.', 'wc-delcampe-integration' )
            );
        }
        
        // Validate price fields
        if ( isset( $listing_data['fixed_price'] ) ) {
            if ( ! is_numeric( $listing_data['fixed_price'] ) || $listing_data['fixed_price'] <= 0 ) {
                return new WP_Error(
                    'invalid_price',
                    __( 'Price must be a positive number.', 'wc-delcampe-integration' )
                );
            }
        }
        
        if ( isset( $listing_data['price_starting'] ) ) {
            if ( ! is_numeric( $listing_data['price_starting'] ) || $listing_data['price_starting'] <= 0 ) {
                return new WP_Error(
                    'invalid_starting_price',
                    __( 'Starting price must be a positive number.', 'wc-delcampe-integration' )
                );
            }
        }
        
        // Validate images (REQUIRED field)
        if ( empty( $listing_data['images'] ) || ! is_array( $listing_data['images'] ) ) {
            return new WP_Error(
                'no_images',
                __( 'At least one image is required for the listing.', 'wc-delcampe-integration' )
            );
        }
        
        // Validate image count (max 99)
        if ( count( $listing_data['images'] ) > 99 ) {
            return new WP_Error(
                'too_many_images',
                sprintf( 
                    __( 'Maximum 99 images allowed (current: %d).', 'wc-delcampe-integration' ), 
                    count( $listing_data['images'] ) 
                )
            );
        }
        
        // Validate duration if present
        if ( isset( $listing_data['duration'] ) ) {
            $allowed_durations = array( 7, 10, 14, 21, 28 );
            if ( ! in_array( $listing_data['duration'], $allowed_durations, true ) ) {
                return new WP_Error(
                    'invalid_duration',
                    sprintf(
                        __( 'Duration must be one of: %s days.', 'wc-delcampe-integration' ),
                        implode( ', ', $allowed_durations )
                    )
                );
            }
        }
        
        // Validate shipping model ID
        if ( empty( $listing_data['id_shipping_model'] ) ) {
            return new WP_Error(
                'missing_shipping_model',
                __( 'A shipping model must be selected in the profile.', 'wc-delcampe-integration' )
            );
        }
        
        // Validate currency code (3 letters)
        if ( ! preg_match( '/^[A-Z]{3}$/', $listing_data['currency'] ) ) {
            return new WP_Error(
                'invalid_currency',
                __( 'Currency must be a valid 3-letter ISO code (e.g., EUR, USD).', 'wc-delcampe-integration' )
            );
        }
        
        return true;
    }
    
    /**
     * Prepare product title for listing
     * 
     * Creates an optimized title for the Delcampe listing based on
     * product name and profile settings.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  WC_Product $product Product object
     * @param  array $profile Profile data
     * @return string Prepared title (max 100 characters)
     */
    private function prepare_title( $product, $profile ) {
        $title = $product->get_name();
        
        // Add catalog number prefix if configured
        if ( ! empty( $profile['stamp_settings']['add_catalog_number'] ) && 
             ! empty( $profile['stamp_settings']['catalog_prefix'] ) ) {
            
            $catalog_number = $this->get_catalog_number( $product, $profile );
            if ( $catalog_number ) {
                $prefix = $profile['stamp_settings']['catalog_prefix'] . ' ' . $catalog_number . ' - ';
                $title = $prefix . $title;
            }
        }
        
        // Apply title template if set
        if ( ! empty( $profile['listing_details']['title_template'] ) ) {
            $template = $profile['listing_details']['title_template'];
            $title = str_replace( 
                array( '{title}', '{sku}', '{catalog}' ),
                array( 
                    $product->get_name(), 
                    $product->get_sku(), 
                    $this->get_catalog_number( $product, $profile ) 
                ),
                $template
            );
        }
        
        // Ensure title doesn't exceed 100 characters
        if ( strlen( $title ) > 100 ) {
            // Try to cut at word boundary
            $title = substr( $title, 0, 97 );
            $last_space = strrpos( $title, ' ' );
            if ( $last_space !== false && $last_space > 50 ) {
                $title = substr( $title, 0, $last_space );
            }
            $title .= '...';
        }
        
        // Sanitize title
        $title = wp_strip_all_tags( $title );
        $title = trim( $title );
        
        return $title;
    }
    
    /**
     * Calculate listing price
     * 
     * Calculates the final listing price based on product price
     * and profile price adjustments.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  WC_Product $product Product object
     * @param  array $profile Profile data
     * @return string Formatted price
     */
    private function calculate_price( $product, $profile ) {
        // Get base price
        $price = $product->get_price();
        
        // Ensure we have a valid price
        if ( empty( $price ) || ! is_numeric( $price ) ) {
            $price = 0;
        }
        
        // Apply price adjustment if set
        if ( ! empty( $profile['listing_details']['price_adjustment'] ) ) {
            $adjustment = $profile['listing_details']['price_adjustment'];
            
            // Check if percentage adjustment
            if ( strpos( $adjustment, '%' ) !== false ) {
                $percentage = floatval( str_replace( '%', '', $adjustment ) );
                $price = $price * ( 1 + ( $percentage / 100 ) );
            } else {
                // Fixed amount adjustment
                $price = $price + floatval( $adjustment );
            }
        }
        
        // Apply minimum price if set
        if ( ! empty( $profile['listing_details']['minimum_price'] ) ) {
            $min_price = floatval( $profile['listing_details']['minimum_price'] );
            if ( $price < $min_price ) {
                $price = $min_price;
            }
        }
        
        // Apply maximum price if set
        if ( ! empty( $profile['listing_details']['maximum_price'] ) ) {
            $max_price = floatval( $profile['listing_details']['maximum_price'] );
            if ( $price > $max_price ) {
                $price = $max_price;
            }
        }
        
        // Ensure price is not negative
        if ( $price < 0 ) {
            $price = 0;
        }
        
        // Format price to 2 decimal places
        return number_format( $price, 2, '.', '' );
    }
    
    /**
     * Calculate listing quantity
     * 
     * Determines the quantity to list based on profile settings
     * and product stock levels.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  WC_Product $product Product object
     * @param  array $profile Profile data
     * @return int Quantity to list
     */
    private function calculate_quantity( $product, $profile ) {
        $quantity_mode = isset( $profile['listing_details']['quantity_mode'] ) ? 
                        $profile['listing_details']['quantity_mode'] : 'stock';
        
        if ( $quantity_mode === 'fixed' ) {
            // Use fixed quantity from profile
            $quantity = isset( $profile['listing_details']['fixed_quantity'] ) ? 
                       intval( $profile['listing_details']['fixed_quantity'] ) : 1;
        } else {
            // Use stock quantity
            $stock = $product->get_stock_quantity();
            $quantity = ( $stock !== null && $stock > 0 ) ? intval( $stock ) : 1;
            
            // Apply maximum quantity limit if set
            if ( ! empty( $profile['listing_details']['max_quantity'] ) ) {
                $max_qty = intval( $profile['listing_details']['max_quantity'] );
                if ( $quantity > $max_qty ) {
                    $quantity = $max_qty;
                }
            }
        }
        
        // Ensure quantity is at least 1
        return max( 1, $quantity );
    }
    
    /**
     * Get renew value for listing
     * 
     * Determines the renewal setting based on profile configuration.
     * 
     * @since  1.2.2.0
     * @version 1.8.0.0
     * @param  array $profile Profile data
     * @return mixed Renew value (string 'null' for unlimited, or integer)
     */
    private function get_renew_value( $profile ) {
        $auto_renew = isset( $profile['listing_details']['auto_renew'] ) ? 
                     $profile['listing_details']['auto_renew'] : '0';
        
        if ( $auto_renew == '1' || $auto_renew === true ) {
            return 'null'; // Renew until sold (as string per API docs)
        }
        
        // Get specific renew count
        $renew_count = isset( $profile['listing_details']['renew_count'] ) ? 
                      intval( $profile['listing_details']['renew_count'] ) : 0;
        
        return $renew_count;
    }
    
    /**
     * Prepare product images
     * 
     * Collects and prepares product images according to profile settings.
     * Ensures images meet Delcampe requirements.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  WC_Product $product Product object
     * @param  array $profile Profile data
     * @return array Array of image URLs
     */
    private function prepare_images( $product, $profile ) {
        $images = array();
        $image_selection = isset( $profile['listing_details']['image_selection'] ) ? 
                          $profile['listing_details']['image_selection'] : 'all';
        
        // Get main/featured image first
        $main_image_id = $product->get_image_id();
        if ( $main_image_id ) {
            $url = wp_get_attachment_url( $main_image_id );
            if ( $url ) {
                $images[] = esc_url_raw( $url );
            }
        }
        
        // Get gallery images if needed
        if ( $image_selection === 'all' || $image_selection === 'gallery' ) {
            $gallery_ids = $product->get_gallery_image_ids();
            foreach ( $gallery_ids as $image_id ) {
                $url = wp_get_attachment_url( $image_id );
                if ( $url ) {
                    $images[] = esc_url_raw( $url );
                }
            }
        }
        
        // Apply selected images filter if specified
        if ( $image_selection === 'selected' && ! empty( $profile['listing_details']['selected_images'] ) ) {
            $selected_ids = $profile['listing_details']['selected_images'];
            if ( is_array( $selected_ids ) ) {
                $filtered_images = array();
                
                foreach ( $images as $url ) {
                    $attachment_id = attachment_url_to_postid( $url );
                    if ( in_array( $attachment_id, $selected_ids ) ) {
                        $filtered_images[] = $url;
                    }
                }
                
                $images = $filtered_images;
            }
        }
        
        // Ensure we have at least one image
        if ( empty( $images ) ) {
            // Try to get a placeholder or default image
            $placeholder = wc_placeholder_img_src();
            if ( $placeholder ) {
                $images[] = esc_url_raw( $placeholder );
            }
        }
        
        // Remove duplicates
        $images = array_unique( $images );
        
        // Limit to 99 images (Delcampe limit)
        $images = array_slice( $images, 0, 99 );
        
        // Allow filtering of images
        $images = apply_filters( 'delcampe_product_images', $images, $product, $profile );
        
        return $images;
    }
    
    /**
     * Get shipping model ID from profile
     * 
     * Retrieves the configured shipping model ID for the listing.
     * 
     * @since  1.2.2.0
     * @version 1.8.0.0
     * @param  array $profile Profile data
     * @return string Shipping model ID as string
     */
    private function get_shipping_model_id( $profile ) {
        // Check if we have a shipping_model_id in the profile (v1.3.0.0+)
        if ( ! empty( $profile['shipping_model_id'] ) ) {
            return strval( $profile['shipping_model_id'] );
        }
        
        // Fallback to old location for backward compatibility
        if ( ! empty( $profile['shipping_settings']['id_shipping_model'] ) ) {
            return strval( $profile['shipping_settings']['id_shipping_model'] );
        }
        
        // Log warning if no shipping model is set
        $this->log( 'Warning: No shipping model ID found in profile', 'warning' );
        
        // Return default shipping model as string
        return '1';
    }
    
    /**
     * Get product weight in grams
     * 
     * Converts product weight to grams for Delcampe API.
     * 
     * @since  1.2.2.0
     * @version 1.8.0.0
     * @param  WC_Product $product Product object
     * @return int|null Weight in grams or null if not set
     */
    private function get_product_weight( $product ) {
        $weight = $product->get_weight();
        
        if ( ! $weight || ! is_numeric( $weight ) ) {
            return null;
        }
        
        // Convert to grams based on WooCommerce weight unit
        $weight_unit = get_option( 'woocommerce_weight_unit', 'kg' );
        
        switch ( strtolower( $weight_unit ) ) {
            case 'kg':
                return intval( $weight * 1000 );
            case 'g':
                return intval( $weight );
            case 'lbs':
                return intval( $weight * 453.592 );
            case 'oz':
                return intval( $weight * 28.3495 );
            default:
                // Assume grams if unknown unit
                return intval( $weight );
        }
    }
    
    /**
     * Get catalog number
     * 
     * Retrieves catalog number from product SKU or attributes.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  WC_Product $product Product object
     * @param  array $profile Profile data
     * @return string Catalog number or empty string
     */
    private function get_catalog_number( $product, $profile ) {
        // First check SKU
        $sku = $product->get_sku();
        if ( $sku ) {
            return $sku;
        }
        
        // Check for catalog number in attributes
        $attributes = $product->get_attributes();
        $catalog_prefix = isset( $profile['stamp_settings']['catalog_prefix'] ) ? 
                         strtolower( $profile['stamp_settings']['catalog_prefix'] ) : 'michel';
        
        foreach ( $attributes as $attribute ) {
            $name = strtolower( $attribute->get_name() );
            
            // Look for catalog-related attribute names
            if ( strpos( $name, 'catalog' ) !== false || 
                 strpos( $name, $catalog_prefix ) !== false ||
                 strpos( $name, 'number' ) !== false ||
                 strpos( $name, 'cat#' ) !== false ) {
                
                $values = $attribute->get_options();
                if ( ! empty( $values ) ) {
                    return is_array( $values ) ? $values[0] : $values;
                }
            }
        }
        
        // Check meta fields
        $meta_keys = array( '_catalog_number', 'catalog_number', 'cat_number' );
        foreach ( $meta_keys as $key ) {
            $value = get_post_meta( $product->get_id(), $key, true );
            if ( $value ) {
                return $value;
            }
        }
        
        return '';
    }
    
    /**
     * Update an existing listing
     * 
     * Updates an existing Delcampe listing with new product data.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  int $product_id Product ID
     * @param  string $listing_id Delcampe listing ID
     * @return array|WP_Error Result or error
     */
    public function update_listing( $product_id, $listing_id ) {
        // Validate inputs
        if ( empty( $listing_id ) ) {
            return new WP_Error( 'invalid_listing_id', __( 'A valid Delcampe listing ID is required.', 'wc-delcampe-integration' ) );
        }
        if ( ! is_numeric( $product_id ) || $product_id <= 0 ) {
            $product_id = $this->get_product_id_by_listing( $listing_id );
            if ( ! $product_id ) {
                return new WP_Error( 'invalid_product_id', __( 'Could not resolve product for this listing.', 'wc-delcampe-integration' ) );
            }
        }

        // Load product
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            return new WP_Error( 'invalid_product', __( 'Product not found.', 'wc-delcampe-integration' ) );
        }

        // Load listing record (for profile and sold qty checks)
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-listings-model.php';
        $listing_record = Delcampe_Listings_Model::get_listing_by_delcampe_id( $listing_id );

        // Load profile if available
        $profile = null;
        if ( $listing_record && ! empty( $listing_record->profile_id ) ) {
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/profiles/class-delcampe-profiles-model.php';
            $profiles_model = Delcampe_Profiles_Model::get_instance();
            $profile = $profiles_model->get_profile( (int) $listing_record->profile_id );
        }

        // Build full listing data then prune to allowed update fields
        $full_data = $this->build_listing_data( $product, is_array( $profile ) ? $profile : array() );
        
        // Remove fields not applicable to PUT (no 'type')
        unset( $full_data['type'] );

        // Prepare update fields (conservative set known to be accepted)
        $update_fields = array();
        $allow_keys = array(
            'title', 'description', 'fixed_price', 'price_starting', 'images',
            'id_shipping_model', 'weight', 'renew', 'prefered_end_day', 'prefered_end_hour',
            'currency'
        );
        foreach ( $allow_keys as $key ) {
            if ( array_key_exists( $key, $full_data ) ) {
                $update_fields[ $key ] = $full_data[ $key ];
            }
        }

        // Quantity updates are special: only allow when no units sold
        if ( isset( $full_data['qty'] ) && $full_data['qty'] !== null ) {
            $can_update_qty = true;
            if ( $listing_record && isset( $listing_record->quantity_sold ) && (int) $listing_record->quantity_sold > 0 ) {
                $can_update_qty = false; // Delcampe restriction
            }
            if ( $can_update_qty ) {
                $update_fields['qty'] = (int) $full_data['qty'];
            }
        }

        if ( empty( $update_fields ) ) {
            return new WP_Error( 'no_changes', __( 'No updatable fields were found for this listing.', 'wc-delcampe-integration' ) );
        }

        // Authenticate
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-auth.php';
        $auth  = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        if ( is_wp_error( $token ) ) {
            return $token;
        }

        // Endpoint and request
        $api_url   = DELCAMPE_API_BASE_URL . '/item/' . rawurlencode( $listing_id ) . '?token=' . $token;
        $post_data = array( 'item' => $update_fields );

        $response = wp_remote_request( $api_url, array(
            'method'  => 'PUT',
            'timeout' => 30,
            'body'    => http_build_query( $post_data ),
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept'       => 'application/xml',
                'User-Agent'   => DELCAMPE_USER_AGENT,
            ),
        ) );

        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'api_request_failed', sprintf( __( 'Failed to update listing: %s', 'wc-delcampe-integration' ), $response->get_error_message() ) );
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        $body        = wp_remote_retrieve_body( $response );
        $xml         = @simplexml_load_string( $body );
        if (function_exists('delcampe_publish_audit')) {
            delcampe_publish_audit(array(
                'phase' => 'update_response',
                'endpoint' => '/item/{id}',
                'product_id' => (int)$product_id,
                'listing_id' => (string)$listing_id,
                'status' => $status_code,
                'elapsed_ms' => isset($elapsed) ? $elapsed : null,
                'response_body' => $body
            ));
        }

        // Embedded status checking (Delcampe sometimes returns 200 with internal errors)
        $api_status = null;
        $error_msg  = null;
        if ( $xml ) {
            if ( isset( $xml->Notification_Data->Headers->Status ) ) {
                $api_status = (int) $xml->Notification_Data->Headers->Status;
            }
            if ( isset( $xml->Notification_Data->body->error ) ) {
                $error_msg = (string) $xml->Notification_Data->body->error;
            } elseif ( isset( $xml->error ) ) {
                $error_msg = (string) $xml->error;
            }
        }

        $is_success = ( $status_code == 200 || $status_code == 201 ) && ( $api_status === null || $api_status == 200 || $api_status == 201 ) && empty( $error_msg );
        if ( ! $is_success ) {
            if ( ! $error_msg ) {
                $error_msg = sprintf( __( 'Unknown API error (HTTP %d).', 'wc-delcampe-integration' ), $status_code );
            }
            // Special case: quantity update rejection
            if ( $error_msg === 'err_item_incorrectQty' ) {
                $error_msg = __( 'Delcampe rejected quantity update. Listing may have existing sales; create a replacement listing for additional stock.', 'wc-delcampe-integration' );
            }
            return new WP_Error( 'api_error', $error_msg );
        }

        // Update local listing record minimal fields
        if ( $listing_record ) {
            $update_db = array( 'date_updated' => current_time( 'mysql' ) );
            if ( isset( $update_fields['title'] ) ) {
                $update_db['listing_title'] = sanitize_text_field( $update_fields['title'] );
            }
            if ( isset( $update_fields['fixed_price'] ) && is_numeric( $update_fields['fixed_price'] ) ) {
                $update_db['price'] = number_format( (float) $update_fields['fixed_price'], 2, '.', '' );
            }
            if ( isset( $update_fields['currency'] ) ) {
                $update_db['currency'] = sanitize_text_field( $update_fields['currency'] );
            }
            if ( isset( $update_fields['qty'] ) ) {
                $update_db['quantity'] = (int) $update_fields['qty'];
            }
            global $wpdb;
            $wpdb->update( $wpdb->prefix . 'delcampe_listings', $update_db, array( 'id' => (int) $listing_record->id ) );
        }

        return array(
            'success'    => true,
            'listing_id' => (string) $listing_id,
            'message'    => __( 'Listing update submitted successfully.', 'wc-delcampe-integration' ),
        );
    }
    
    /**
     * Close a listing
     * 
     * Closes an active listing on Delcampe.
     * 
     * @since  1.2.0.0
     * @version 1.8.0.0
     * @param  string $listing_id Delcampe listing ID
     * @param  string $reason Reason for closing (optional)
     * @return array|WP_Error Result or error
     */
    public function close_listing( $listing_id, $reason = '' ) {
        // First check if we should use local-only closure
        $use_local_only = apply_filters( 'delcampe_use_local_close_only', false );
        
        if ( $use_local_only ) {
            // Use local-only implementation
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-listing-api-simplified.php';
            $product_id = $this->get_product_id_by_listing( $listing_id );
            if ( $product_id ) {
                return delcampe_mark_listing_ended( $product_id, $listing_id, $reason );
            }
            return array(
                'success' => false,
                'error' => __( 'Could not find product for this listing.', 'wc-delcampe-integration' )
            );
        }
        
        // Get authentication token
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if ( is_wp_error( $token ) ) {
            return array(
                'success' => false,
                'error' => $token->get_error_message()
            );
        }
        
        // Prepare API endpoint with query parameters
        $api_url = DELCAMPE_API_BASE_URL . '/item/' . $listing_id . '?token=' . $token;
        
        // Add reason if provided
        if ( !empty( $reason ) ) {
            $api_url .= '&reason=' . urlencode( $reason );
        }
        
        // Add priority for immediate closure
        $api_url .= '&priority=1';
        
        // Make API request using DELETE method
        $response = wp_remote_request( $api_url, array(
            'method' => 'DELETE',
            'timeout' => 30,
            'headers' => array(
                'User-Agent' => DELCAMPE_USER_AGENT
            )
        ) );
        
        if ( is_wp_error( $response ) ) {
            delcampe_log( 'Failed to close listing: ' . $response->get_error_message(), 'error' );
            return array(
                'success' => false,
                'error' => $response->get_error_message()
            );
        }
        
        $status_code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        
        // Log the response
        delcampe_log( sprintf( 'Close listing response for %s: Status %d', $listing_id, $status_code ), 'info' );
        
        // Check for 404 - listing not found
        if ( $status_code === 404 ) {
            delcampe_log( sprintf( 'Listing %s not found on Delcampe (404)', $listing_id ), 'warning' );
            
            // Fallback to local-only closure for 404
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-listing-api-simplified.php';
            $product_id = $this->get_product_id_by_listing( $listing_id );
            if ( $product_id ) {
                $result = delcampe_mark_listing_ended( $product_id, $listing_id, $reason );
                $result['message'] = __( 'Listing not found on Delcampe (may have been already closed). Marked as ended locally.', 'wc-delcampe-integration' );
                return $result;
            }
            
            return array(
                'success' => false,
                'error' => __( 'Listing not found on Delcampe. It may have already been closed or removed.', 'wc-delcampe-integration' )
            );
        }
        
        // Check if response is HTML (API endpoint might not exist)
        if ( strpos( $body, '<!DOCTYPE' ) !== false || strpos( $body, '<html' ) !== false ) {
            delcampe_log( 'Delcampe API returned HTML instead of XML - endpoint may not exist', 'warning' );
            
            // Fallback to local-only closure
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-listing-api-simplified.php';
            $product_id = $this->get_product_id_by_listing( $listing_id );
            if ( $product_id ) {
                return delcampe_mark_listing_ended( $product_id, $listing_id, $reason );
            }
            
            return array(
                'success' => false,
                'error' => __( 'The Delcampe API does not support closing listings. Please close the listing manually on Delcampe.', 'wc-delcampe-integration' )
            );
        }
        
        // Check for success (status 200 or 204 indicates request accepted for DELETE)
        if ( $status_code === 200 || $status_code === 204 ) {
            // DELETE requests might return minimal or no body
            return array(
                'success' => true,
                'listing_id' => $listing_id,
                'message' => __( 'Listing closure request submitted successfully.', 'wc-delcampe-integration' ),
                'note' => __( 'The listing will be closed shortly. Confirmation comes via Seller_Item_Close_Manually notification.', 'wc-delcampe-integration' )
            );
        }
        
        // Parse XML response for error details
        libxml_use_internal_errors( true );
        $xml = simplexml_load_string( $body );
        
        if ( $xml === false && !empty( $body ) ) {
            delcampe_log( 'Failed to parse close listing response: ' . substr( $body, 0, 200 ), 'error' );
            return array(
                'success' => false,
                'error' => __( 'Failed to parse API response. The response may not be valid XML.', 'wc-delcampe-integration' )
            );
        }
        
        // Check for other known error codes
        if ( $status_code === 409 ) {
            return array(
                'success' => false,
                'error' => __( 'Listing is already closed or cannot be closed at this time.', 'wc-delcampe-integration' )
            );
        }
        
        if ( $status_code === 400 ) {
            return array(
                'success' => false,
                'error' => __( 'Invalid request format. Please check the listing ID.', 'wc-delcampe-integration' )
            );
        }
        
        // Check for error in XML
        $error_message = '';
        if ( isset( $xml->Notification_Data->body->error ) ) {
            $error_message = (string) $xml->Notification_Data->body->error;
        } elseif ( isset( $xml->error ) ) {
            $error_message = (string) $xml->error;
        }
        
        return array(
            'success' => false,
            'error' => $error_message ?: sprintf( __( 'Unknown error occurred (HTTP %d) while closing listing.', 'wc-delcampe-integration' ), $status_code )
        );
    }
    
    /**
     * Log messages for debugging
     * 
     * Logs messages to WordPress debug log if logging is enabled.
     * 
     * @since  1.8.0.0
     * @param  string $message Message to log
     * @param  string $level Log level (debug, info, warning, error)
     * @return void
     */
    private function log( $message, $level = 'info' ) {
        // Check if logging is enabled (errors are always logged)
        $logging_enabled = get_option( DELCAMPE_LOGGING_OPTION, false );
        
        if ( ! $logging_enabled && $level !== 'error' ) {
            return;
        }
        
        // Format the log message
        $formatted_message = sprintf(
            '[%s] [Delcampe Listing API v%s] [%s] %s',
            current_time( 'Y-m-d H:i:s' ),
            DELCAMPE_PLUGIN_VERSION,
            strtoupper( $level ),
            $message
        );
        
        // Only log to custom log file if enabled, not to WordPress debug log
        if ( defined( 'DELCAMPE_LOG_FILE' ) ) {
            $log_dir = dirname( DELCAMPE_LOG_FILE );
            if ( ! file_exists( $log_dir ) ) {
                wp_mkdir_p( $log_dir );
            }
            @file_put_contents( 
                DELCAMPE_LOG_FILE, 
                $formatted_message . PHP_EOL, 
                FILE_APPEND | LOCK_EX 
            );
        }
    }
    
    /**
     * Get webhook URL for callbacks
     * 
     * @since 1.10.4.0
     * @return string Webhook URL
     */
    private function get_webhook_url() {
        // Get site URL
        $site_url = get_site_url();
        
        // Build webhook URL
        $webhook_url = $site_url . '/wp-content/plugins/delcampe-sync/webhook/delcampe-callback.php';
        
        // Allow filtering for custom webhook URLs
        $webhook_url = apply_filters('delcampe_webhook_url', $webhook_url);
        
        return $webhook_url;
    }
    
    /**
     * Check if a SKU already exists on Delcampe
     * 
     * v1.10.23.0 - Added to prevent duplicate listings
     * Searches Delcampe for items with matching personal_reference (SKU)
     * 
     * @param string $sku The SKU/personal_reference to search for
     * @return array|false Returns item data if found, false if not found
     */
    public function check_sku_exists_on_delcampe($sku) {
        if (empty($sku)) {
            return false;
        }
        
        // Get authentication instance
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            $this->log('Cannot check SKU - authentication failed: ' . $token->get_error_message(), 'warning');
            return false;
        }
        
        // Use correct endpoint per Delcampe wiki: GET /item/reference/{PERSONAL_REFERENCE}?token=TOKEN
        // Cache result to avoid repeated calls
        $cache_key = 'del_ref_' . md5($sku);
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            if (is_array($cached)) {
                return $cached; // cached hit
            }
            // cached negative
            return false;
        }
        $api_url = DELCAMPE_API_BASE_URL . '/item/reference/' . urlencode($sku) . '?token=' . $token;
        
        // Log timing start
        $start_time = microtime(true);
        $this->log('SKU check starting for: ' . $sku . ' at ' . $api_url, 'debug');
        
        $args = array(
            'timeout' => 10, // Reduced timeout to fail faster if endpoint doesn't exist
            'headers' => array(
                'User-Agent' => DELCAMPE_USER_AGENT
            )
        );
        
        // Make request to the correct endpoint
        $response = wp_remote_get($api_url, $args);
        
        // Log timing
        $elapsed = round(microtime(true) - $start_time, 2);
        
        if (is_wp_error($response)) {
            $this->log('SKU check request failed after ' . $elapsed . 's: ' . $response->get_error_message(), 'error');
            return false;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        $this->log('SKU check completed in ' . $elapsed . 's with status ' . $status_code, 'debug');
        
        // 404 means SKU not found - this is normal and expected
        if ($status_code === 404) {
            $this->log('No existing Delcampe item found for SKU: ' . $sku, 'debug');
            // Negative cache for 1 hour
            set_transient($cache_key, 'nf', HOUR_IN_SECONDS);
            return false;
        }
        
        // Any other non-200 status is an error
        if ($status_code !== 200) {
            $this->log('SKU check returned unexpected status ' . $status_code . ': ' . substr($body, 0, 200), 'warning');
            return false;
        }
        
        // Parse XML response (Delcampe API returns XML)
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($body);
        
        if ($xml === false) {
            $this->log('Failed to parse XML response for SKU check', 'error');
            return false;
        }
        
        // Check if item exists in response
        if (isset($xml->id) || isset($xml->Item_ID)) {
            $item_id = isset($xml->id) ? (string)$xml->id : (string)$xml->Item_ID;
            $this->log('Found existing Delcampe item ' . $item_id . ' for SKU: ' . $sku, 'info');
            $result = array(
                'id' => $item_id,
                'personal_reference' => $sku,
                'exists' => true
            );
            // Cache positive result for 1 day
            set_transient($cache_key, $result, DAY_IN_SECONDS);
            return $result;
        }
        
        $this->log('No existing Delcampe item found for SKU: ' . $sku, 'debug');
        set_transient($cache_key, 'nf', HOUR_IN_SECONDS);
        return false;
    }
}
