<?php
/**
 * Delcampe Listing Closer
 * 
 * Handles closing/ending listings on Delcampe
 * 
 * @package Delcampe_Sync
 * @since 1.10.5.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Listing Closer Class
 */
class Delcampe_Listing_Closer {
    
    /**
     * Instance
     */
    private static $instance = null;
    
    /**
     * Auth instance
     */
    private $auth;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->auth = Delcampe_Auth::get_instance();
        
        // Add AJAX handlers
        add_action('wp_ajax_delcampe_close_listing', array($this, 'ajax_close_listing'));
    }
    
    /**
     * Close a listing on Delcampe
     * 
     * @param int $product_id Product ID
     * @param string $reason Reason for closure (optional)
     * @return array|WP_Error Result array or error
     */
    public function close_listing($product_id, $reason = '') {
        // Get listing ID
        $listing_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        
        if (empty($listing_id)) {
            return new WP_Error(
                'no_listing_id',
                __('No Delcampe listing ID found for this product.', 'wc-delcampe-integration')
            );
        }
        
        // Get auth token
        $token = $this->auth->get_auth_token();
        if (is_wp_error($token)) {
            return $token;
        }
        
        // Close on Delcampe API
        $api_url = DELCAMPE_API_BASE_URL . '/item/' . $listing_id . '/close?token=' . $token;
        
        // Prepare request data
        $body = array();
        if (!empty($reason)) {
            $body['reason'] = $reason;
        }
        
        $response = wp_remote_post($api_url, array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'User-Agent' => DELCAMPE_USER_AGENT
            ),
            'body' => $body
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Log the response
        delcampe_log(sprintf('Close listing response for %s: Status %d', $listing_id, $status_code), 'info');
        
        // Parse response
        $result = $this->parse_close_response($body, $status_code);
        
        if ($result['success']) {
            // Update local status
            $this->update_local_status($product_id, $listing_id, 'ended');
            
            // Trigger action
            do_action('delcampe_listing_closed', $product_id, $listing_id);
        }
        
        return $result;
    }
    
    /**
     * Parse close response
     */
    private function parse_close_response($xml_response, $status_code) {
        if (empty($xml_response)) {
            // Some APIs return empty response on success
            if ($status_code === 200 || $status_code === 204) {
                return array(
                    'success' => true,
                    'message' => __('Listing closed successfully.', 'wc-delcampe-integration')
                );
            }
        }
        
        // Parse XML
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($xml_response);
        
        if ($xml === false) {
            return array(
                'success' => false,
                'error' => __('Failed to parse API response.', 'wc-delcampe-integration')
            );
        }
        
        // Check for success indicators
        if ($status_code === 200 || $status_code === 201) {
            return array(
                'success' => true,
                'message' => __('Listing closed successfully.', 'wc-delcampe-integration')
            );
        }
        
        // Check for errors
        $error_message = '';
        if (isset($xml->Notification_Data->body->error)) {
            $error_message = (string) $xml->Notification_Data->body->error;
        } elseif (isset($xml->error)) {
            $error_message = (string) $xml->error;
        }
        
        return array(
            'success' => false,
            'error' => $error_message ?: __('Unknown error occurred.', 'wc-delcampe-integration')
        );
    }
    
    /**
     * Update local status
     */
    private function update_local_status($product_id, $listing_id, $status) {
        // Update product meta
        update_post_meta($product_id, '_delcampe_sync_status', $status);
        
        // If listing is ended/closed, remove the Delcampe item ID since it's no longer relevant
        if (in_array($status, array('ended', 'closed', 'closed_unsold', 'closed_manually'))) {
            delete_post_meta($product_id, '_delcampe_item_id');
            delete_post_meta($product_id, '_delcampe_listing_id');
            
            // Log the cleanup
            delcampe_log(sprintf(
                '[Listing Closer] Cleaned up Delcampe IDs for product #%d (listing %s) - Status: %s',
                $product_id,
                $listing_id,
                $status
            ));
        }
        
        // Update listings table if it exists
        global $wpdb;
        $table_name = $wpdb->prefix . DELCAMPE_TABLE_LISTINGS;
        
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") === $table_name) {
            $wpdb->update(
                $table_name,
                array(
                    'status' => $status,
                    'end_date' => current_time('mysql')
                ),
                array('delcampe_id' => $listing_id),
                array('%s', '%s'),
                array('%s')
            );
        }
    }
    
    /**
     * AJAX handler for closing listing
     */
    public function ajax_close_listing() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Permission denied');
        }
        
        $product_id = intval($_POST['product_id']);
        $listing_id = sanitize_text_field($_POST['listing_id'] ?? '');
        $reason = sanitize_text_field($_POST['reason'] ?? '');
        
        // If we have listing_id directly, use the API method
        if (!empty($listing_id)) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
            $api = Delcampe_Listing_API::get_instance();
            $result = $api->close_listing($listing_id, $reason);
            
            if (!is_wp_error($result) && $result['success']) {
                // Update local status
                $this->update_local_status($product_id, $listing_id, 'ended');
            }
        } else {
            // Use the class method that looks up listing_id
            $result = $this->close_listing($product_id, $reason);
        }
        
        if (is_wp_error($result)) {
            wp_send_json_error(array(
                'message' => $result->get_error_message()
            ));
        }
        
        wp_send_json_success($result);
    }
}