<?php
/**
 * Delcampe Listings Cleanup
 * 
 * Checks all listings against Delcampe and marks any that are no longer active as ended
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.2.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Delcampe_Listings_Cleanup {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Clean up all listings by checking their status on Delcampe
     * 
     * @return array Results of the cleanup
     */
    public function cleanup_all_listings() {
        error_log('[Delcampe Listings Cleanup] Starting cleanup of all listings');
        
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        // Only check listings that should already be live on Delcampe.
        // Skip 'processing' to avoid flipping freshly-queued items back to ended.
        $active_listings = $wpdb->get_results(
            "SELECT * FROM {$listings_table} 
             WHERE status IN ('published','verified','changed')"
        );
        
        error_log('[Delcampe Listings Cleanup] Found ' . count($active_listings) . ' non-ended listings to check');
        
        $results = array(
            'total' => count($active_listings),
            'ended' => 0,
            'still_active' => 0,
            'errors' => 0
        );
        
        // Get authentication token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            error_log('[Delcampe Listings Cleanup] Authentication failed: ' . $token->get_error_message());
            return $token;
        }
        
        // First, get all currently open items from Delcampe
        $open_items = $this->get_all_open_items($token);
        if (is_wp_error($open_items)) {
            return $open_items;
        }
        
        error_log('[Delcampe Listings Cleanup] Found ' . count($open_items) . ' open items on Delcampe');
        
        // Create a lookup array for faster checking
        $open_delcampe_ids = array();
        foreach ($open_items as $item) {
            $open_delcampe_ids[$item] = true;
        }
        
        // Check each listing
        foreach ($active_listings as $listing) {
            if (empty($listing->delcampe_id)) {
                // No Delcampe ID - mark as error
                $wpdb->update(
                    $listings_table,
                    array('status' => 'error'),
                    array('id' => $listing->id)
                );
                $results['errors']++;
                error_log('[Delcampe Listings Cleanup] Listing ' . $listing->id . ' has no Delcampe ID - marked as error');
                continue;
            }
            
            // Check if this item is in the open items list
            if (!isset($open_delcampe_ids[$listing->delcampe_id])) {
                // Not in open items - mark as ended
                $wpdb->update(
                    $listings_table,
                    array('status' => 'ended'),
                    array('id' => $listing->id)
                );
                $results['ended']++;
                error_log('[Delcampe Listings Cleanup] Listing ' . $listing->delcampe_id . ' not found in open items - marked as ended');
            } else {
                $results['still_active']++;
            }
        }
        
        error_log('[Delcampe Listings Cleanup] Cleanup complete - Ended: ' . $results['ended'] . ', Still active: ' . $results['still_active'] . ', Errors: ' . $results['errors']);
        
        return $results;
    }
    
    /**
     * Get all open items from Delcampe
     * 
     * @param string $token Authentication token
     * @return array|WP_Error Array of Delcampe IDs or error
     */
    private function get_all_open_items($token) {
        $all_items = array();
        $start = 0;
        $limit = 100;
        $has_more = true;
        
        while ($has_more) {
            $items = $this->fetch_open_items($token, $start, $limit);
            if (is_wp_error($items)) {
                return $items;
            }
            
            if (empty($items)) {
                $has_more = false;
            } else {
                $all_items = array_merge($all_items, $items);
                $start += $limit;
                
                // Safety limit - don't fetch more than 1000 items
                if ($start >= 1000) {
                    error_log('[Delcampe Listings Cleanup] Reached safety limit of 1000 items');
                    $has_more = false;
                }
            }
        }
        
        return $all_items;
    }
    
    /**
     * Fetch open items from Delcampe API
     * 
     * @param string $token Authentication token
     * @param int $start Starting index
     * @param int $limit Number of items to fetch
     * @return array|WP_Error Array of Delcampe IDs or error
     */
    private function fetch_open_items($token, $start, $limit) {
        $api_url = DELCAMPE_API_URL . '/item/opened';
        $params = array(
            'token' => $token,
            'startingItem' => $start,
            'numberOfItems' => $limit
        );
        
        $url = $api_url . '?' . http_build_query($params);
        error_log('[Delcampe Listings Cleanup] Fetching open items from: ' . $url);
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        if ($xml === false) {
            return new WP_Error('xml_parse_error', 'Failed to parse XML response');
        }
        
        $items = array();
        
        // Navigate to the correct XML path for items
        if (isset($xml->Notification_Data->body->item)) {
            foreach ($xml->Notification_Data->body->item as $item) {
                $items[] = (string)$item->id_item;
            }
        }
        
        return $items;
    }
}
