<?php
/**
 * Delcampe Listings Sync
 * 
 * Handles synchronization of listing status between Delcampe and WooCommerce
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.2.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Delcampe_Listings_Sync {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Sync closed/ended listings from Delcampe
     * 
     * Fetches ALL closed listings using pagination and updates their status in the local database
     * Tracks processed items to avoid re-processing in subsequent runs
     * 
     * @param int $batch_size Number of items to fetch per batch (default 500)
     * @param int $max_pages Maximum pages to process (default 20 = 10,000 items)
     * @return array|WP_Error Array of processed listings or error
     * @since 1.10.18.2
     */
    public function sync_closed_listings($batch_size = 500, $max_pages = 20) {
        delcampe_log('[Delcampe Listings Sync] Starting comprehensive sync_closed_listings with batch_size=' . $batch_size);
        
        // Ensure listings table exists
        $this->ensure_listings_table_exists();
        
        // Get list of already processed closed items (last 30 days)
        $processed_items = get_option('delcampe_processed_closed_items', array());
        
        // Clean up old entries (older than 30 days)
        $thirty_days_ago = time() - (30 * 24 * 60 * 60);
        foreach ($processed_items as $item_id => $timestamp) {
            if ($timestamp < $thirty_days_ago) {
                unset($processed_items[$item_id]);
            }
        }
        
        $total_processed = array();
        $total_skipped = 0;
        $total_already_processed = 0;
        $current_page = 0;
        $has_more_items = true;
        
        // Get authentication token once
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            delcampe_log('[Delcampe Listings Sync] Authentication failed: ' . $token->get_error_message());
            return $token;
        }
        
        // Process in batches until no more items or max pages reached
        while ($has_more_items && $current_page < $max_pages) {
            $start_index = $current_page * $batch_size;
            
            // Build API URL for closed items
            $api_url = DELCAMPE_API_URL . '/item/closed';
            $params = array(
                'token' => $token,
                'startingItem' => $start_index,
                'numberOfItems' => $batch_size,
                'order' => 'desc'
            );
            
            $url = $api_url . '?' . http_build_query($params);
            delcampe_log('[Delcampe Listings Sync] Fetching page ' . ($current_page + 1) . ' (items ' . $start_index . '-' . ($start_index + $batch_size) . ')');
            
            // Make API request
            $response = wp_remote_get($url, array(
                'timeout' => 30,
                'headers' => array(
                    'Accept' => 'application/xml'
                )
            ));
            
            if (is_wp_error($response)) {
                delcampe_log('[Delcampe Listings Sync] API request failed on page ' . ($current_page + 1) . ': ' . $response->get_error_message());
                break; // Stop processing on error
            }
            
            $response_code = wp_remote_retrieve_response_code($response);
            
            if ($response_code !== 200) {
                delcampe_log('[Delcampe Listings Sync] API returned non-200 status on page ' . ($current_page + 1) . ': ' . $response_code);
                break;
            }
            
            $body = wp_remote_retrieve_body($response);
            
            // Parse XML response
            $xml = simplexml_load_string($body);
            if ($xml === false) {
                delcampe_log('[Delcampe Listings Sync] Failed to parse XML response on page ' . ($current_page + 1));
                break;
            }
            
            // Check for API errors
            if (isset($xml->Notification_Data->Headers->Status) && (string)$xml->Notification_Data->Headers->Status !== '200') {
                delcampe_log('[Delcampe Listings Sync] API error status on page ' . ($current_page + 1) . ': ' . (string)$xml->Notification_Data->Headers->Status);
                break;
            }
            
            // Navigate to the correct XML path for items
            $items = null;
            if (isset($xml->Notification_Data->body->item)) {
                $items = $xml->Notification_Data->body->item;
                delcampe_log('[Delcampe Listings Sync] Page ' . ($current_page + 1) . ' has ' . count($items) . ' items');
            } else {
                delcampe_log('[Delcampe Listings Sync] No items found on page ' . ($current_page + 1) . ' - end of results');
                $has_more_items = false;
                break;
            }
            
            // If we got less than batch_size items, we're at the end
            if (count($items) < $batch_size) {
                $has_more_items = false;
            }
            
            $page_processed = 0;
            $page_skipped = 0;
            $page_already_processed = 0;
            
            // Work directly with database since we don't have a listings model yet
            
            // Process each closed item
            foreach ($items as $item) {
                $delcampe_id = (string)$item->id_item;
                $personal_reference = (string)$item->personal_reference;
                $title = (string)$item->title;
                
                // Check if we've already processed this item in a previous run (v1.10.18.2)
                if (isset($processed_items[$delcampe_id])) {
                    $page_already_processed++;
                    $total_already_processed++;
                    continue;
                }
                
                // Check if we have this listing in our database
                global $wpdb;
                $listings_table = $wpdb->prefix . 'delcampe_listings';
                
                $existing_listing = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$listings_table} WHERE delcampe_id = %s OR personal_reference = %s",
                    $delcampe_id,
                    $personal_reference
                ));
                
                if ($existing_listing) {
                    // Check if already ended to avoid redundant updates (v1.10.18.1)
                    if ($existing_listing->status === 'ended') {
                        // Already ended, skip processing
                        $page_skipped++;
                        $total_skipped++;
                        // Mark as processed so we don't check again
                        $processed_items[$delcampe_id] = time();
                        continue;
                    }
                    
                    // Update listing status to ended - use 'status' column only
                    $wpdb->update(
                        $listings_table,
                        array(
                            'status' => 'ended'
                        ),
                        array('id' => $existing_listing->id)
                    );
                    
                    $total_processed[] = array(
                        'delcampe_id' => $delcampe_id,
                        'product_id' => $existing_listing->product_id,
                        'title' => $title,
                        'action' => 'updated_to_ended'
                    );
                    
                    $page_processed++;
                    
                    // Mark as processed
                    $processed_items[$delcampe_id] = time();
                    
                } else {
                    // Try to find product by SKU
                    $product_id = 0;
                    if (!empty($personal_reference)) {
                        $product_id = wc_get_product_id_by_sku($personal_reference);
                    }
                    
                    // Create listing record as ended
                    $wpdb->insert(
                        $listings_table,
                        array(
                            'product_id' => $product_id,
                            'delcampe_id' => $delcampe_id,
                            'personal_reference' => $personal_reference,
                            'listing_title' => $title,
                            'status' => 'ended'
                        )
                    );
                    
                    $total_processed[] = array(
                        'delcampe_id' => $delcampe_id,
                        'product_id' => $product_id,
                        'title' => $title,
                        'action' => 'created_as_ended'
                    );
                    
                    $page_processed++;
                    
                    // Mark as processed
                    $processed_items[$delcampe_id] = time();
                }
            }
            
            // Log page summary
            delcampe_log('[Delcampe Listings Sync] Page ' . ($current_page + 1) . ' summary: ' . 
                     'Processed=' . $page_processed . ', ' .
                     'Already Ended=' . $page_skipped . ', ' .
                     'Previously Processed=' . $page_already_processed);
            
            $current_page++;
            
            // Small delay between pages to avoid hammering the API
            if ($has_more_items && $current_page < $max_pages) {
                usleep(500000); // 0.5 second delay
            }
        }
        
        // Save processed items list for future runs
        update_option('delcampe_processed_closed_items', $processed_items);
        
        // Log comprehensive summary (v1.10.18.2)
        delcampe_log('[Delcampe Listings Sync] COMPLETE SUMMARY:');
        delcampe_log('[Delcampe Listings Sync] - Pages processed: ' . $current_page);
        delcampe_log('[Delcampe Listings Sync] - Total updated: ' . count($total_processed));
        delcampe_log('[Delcampe Listings Sync] - Already ended (skipped): ' . $total_skipped);
        delcampe_log('[Delcampe Listings Sync] - Previously processed: ' . $total_already_processed);
        delcampe_log('[Delcampe Listings Sync] - Total items checked: ' . ($current_page * $batch_size));
        
        return $total_processed;
    }
    
    /**
     * Sync all listing statuses
     * 
     * Checks status of all active listings and updates accordingly
     * 
     * @return array Results of sync operation
     */
    public function sync_all_listing_statuses() {
        // Get all active listings from database
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        $active_listings = $wpdb->get_results(
            "SELECT * FROM {$listings_table} 
             WHERE status IN ('published', 'verified', 'changed')"
        );
        
        $results = array(
            'checked' => 0,
            'updated' => 0,
            'errors' => 0
        );
        
        foreach ($active_listings as $listing) {
            $results['checked']++;
            
            // Check if listing still exists on Delcampe
            $status = $this->check_listing_status($listing->delcampe_id);
            
            if ($status === 'ended' || $status === 'closed') {
                // Update to ended
                $wpdb->update(
                    $listings_table,
                    array(
                        'status' => 'ended'
                    ),
                    array('id' => $listing->id)
                );
                $results['updated']++;
            } elseif ($status === false) {
                $results['errors']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Check individual listing status on Delcampe
     * 
     * @param string $delcampe_id
     * @return string|false Status or false on error
     */
    private function check_listing_status($delcampe_id) {
        // This would need to be implemented based on Delcampe's API
        // For now, return false to indicate not implemented
        return false;
    }
    
    /**
     * Ensure listings table exists
     * 
     * @since 1.10.2.0
     */
    private function ensure_listings_table_exists() {
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$listings_table}'");
        if (!$table_exists) {
            delcampe_log('[Delcampe Listings Sync] Creating listings table...');
            
            $charset_collate = $wpdb->get_charset_collate();
            $sql = "CREATE TABLE {$listings_table} (
                id int(11) NOT NULL AUTO_INCREMENT,
                product_id bigint(20) DEFAULT 0,
                delcampe_id varchar(255) NOT NULL,
                personal_reference varchar(255) DEFAULT '',
                listing_title varchar(255) DEFAULT '',
                listing_status varchar(50) DEFAULT 'pending',
                listing_start datetime DEFAULT NULL,
                listing_end datetime DEFAULT NULL,
                listing_data longtext DEFAULT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY product_id (product_id),
                KEY delcampe_id (delcampe_id),
                KEY listing_status (listing_status)
            ) {$charset_collate}";
            
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
            
            delcampe_log('[Delcampe Listings Sync] Listings table created');
        } else {
            // Table exists, check if status column exists (not listing_status)
            $columns = $wpdb->get_col("SHOW COLUMNS FROM {$listings_table}");
            if (!in_array('status', $columns)) {
                delcampe_log('[Delcampe Listings Sync] Adding missing status column...');
                $wpdb->query("ALTER TABLE {$listings_table} ADD COLUMN status varchar(50) DEFAULT 'active'");
                delcampe_log('[Delcampe Listings Sync] Added status column');
            }
        }
    }
    
    /**
     * Sync open/active listings from Delcampe
     * 
     * Fetches all open listings and ensures they're marked as published in the local database
     * Also updates any missing listing records
     * 
     * @param int $batch_size Number of items to fetch per batch (default 100)
     * @param int $max_pages Maximum pages to process (default 5 = 500 items)
     * @return array|WP_Error Array of processed listings or error
     * @since 1.10.18.8
     */
    public function sync_open_listings($batch_size = 100, $max_pages = 5) {
        delcampe_log('[Delcampe Listings Sync] Starting sync_open_listings with batch_size=' . $batch_size);
        
        // Ensure listings table exists
        $this->ensure_listings_table_exists();
        
        // Get authentication token
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            delcampe_log('[Delcampe Listings Sync] Authentication failed: ' . $token->get_error_message());
            return $token;
        }
        
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        $total_processed = array();
        $current_page = 0;
        
        // Process pages of open items
        while ($current_page < $max_pages) {
            $api_url = DELCAMPE_API_BASE_URL . '/item/opened?token=' . $token . 
                      '&limit=' . $batch_size . 
                      '&offset=' . ($current_page * $batch_size);
            
            delcampe_log('[Delcampe Listings Sync] Fetching page ' . ($current_page + 1) . ' of open items...');
            
            $response = wp_remote_get($api_url, array(
                'timeout' => 30,
                'headers' => array(
                    'User-Agent' => DELCAMPE_USER_AGENT,
                    'Accept' => 'application/xml'
                )
            ));
            
            if (is_wp_error($response)) {
                delcampe_log('[Delcampe Listings Sync] API request failed: ' . $response->get_error_message());
                return $response;
            }
            
            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code !== 200) {
                delcampe_log('[Delcampe Listings Sync] API returned code ' . $response_code);
                return new WP_Error('api_error', 'API returned code ' . $response_code);
            }
            
            $response_body = wp_remote_retrieve_body($response);
            libxml_use_internal_errors(true);
            $xml = simplexml_load_string($response_body);
            
            if ($xml === false) {
                delcampe_log('[Delcampe Listings Sync] Failed to parse XML response');
                return new WP_Error('xml_parse_error', 'Failed to parse API response');
            }
            
            // Check if we have items
            $items = array();
            if (isset($xml->Notification_Data->body->item)) {
                foreach ($xml->Notification_Data->body->item as $item) {
                    $items[] = $item;
                }
            } elseif (isset($xml->body->item)) {
                foreach ($xml->body->item as $item) {
                    $items[] = $item;
                }
            }
            
            $has_more_items = (count($items) == $batch_size);
            
            foreach ($items as $item) {
                $delcampe_id = (string) $item->id;
                $personal_reference = (string) $item->personal_reference;
                $title = (string) $item->title;
                
                // Check if listing exists in database
                $existing_listing = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$listings_table} WHERE delcampe_id = %s",
                    $delcampe_id
                ));
                
                if ($existing_listing) {
                    // Update status to published if not already
                    if ($existing_listing->status !== 'published') {
                        $wpdb->update(
                            $listings_table,
                            array('status' => 'published'),
                            array('id' => $existing_listing->id)
                        );
                        
                        $total_processed[] = array(
                            'delcampe_id' => $delcampe_id,
                            'action' => 'updated_to_published'
                        );
                    }
                } else {
                    // Find product by SKU (direct match - no suffix stripping)
                    $product_id = 0;
                    if (!empty($personal_reference)) {
                        // Direct SKU lookup - SKUs must match exactly
                        $product_id = wc_get_product_id_by_sku($personal_reference);
                    }
                    
                    // Create new listing record
                    $wpdb->insert(
                        $listings_table,
                        array(
                            'product_id' => $product_id,
                            'delcampe_id' => $delcampe_id,
                            'personal_reference' => $personal_reference,
                            'listing_title' => $title,
                            'status' => 'published'
                        )
                    );
                    
                    $total_processed[] = array(
                        'delcampe_id' => $delcampe_id,
                        'product_id' => $product_id,
                        'action' => 'created_as_published'
                    );
                }
            }
            
            delcampe_log('[Delcampe Listings Sync] Page ' . ($current_page + 1) . 
                     ' processed: ' . count($items) . ' items');
            
            $current_page++;
            
            // Don't continue if we got fewer items than requested
            if (!$has_more_items) {
                break;
            }
            
            // Small delay between pages
            if ($current_page < $max_pages) {
                usleep(500000); // 0.5 second delay
            }
        }
        
        delcampe_log('[Delcampe Listings Sync] Open listings sync complete. Total processed: ' . 
                 count($total_processed));
        
        return $total_processed;
    }
}