<?php
/**
 * Delcampe Master Sync Manager
 * 
 * Comprehensive synchronization manager that coordinates all sync operations
 * based on the configured sync interval in Advanced settings
 * 
 * @package Delcampe_Sync
 * @since 1.10.10.4
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Master_Sync {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Logger instance
     */
    private $logger;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Get logger instance
        $this->logger = Delcampe_Sync_Logger::get_instance();
        
        // Hook into WordPress cron system
        add_action('delcampe_master_sync', array($this, 'run_master_sync'));
        
        // Schedule master sync based on settings
        $this->schedule_master_sync();
        
        // Add custom cron schedules
        add_filter('cron_schedules', array($this, 'add_custom_schedules'));
        
        // Hook for manual sync trigger
        add_action('wp_ajax_delcampe_run_master_sync', array($this, 'ajax_run_master_sync'));
    }
    
    /**
     * Helper method to write to log
     */
    private function log($message, $type = 'info') {
        if ($this->logger) {
            // Format log entry as array for write_log method
            $entry = array(
                'timestamp' => current_time('mysql'),
                'event' => 'Master Sync',
                'message' => $message,
                'type' => $type
            );
            $this->logger->write_log($entry);
        }
        
        // Also use error_log for critical messages
        if ($type === 'error') {
            error_log('[Delcampe Master Sync ERROR] ' . $message);
        }
    }
    
    /**
     * Add custom cron schedules
     */
    public function add_custom_schedules($schedules) {
        // Add 15-minute interval
        $schedules['every_15_minutes'] = array(
            'interval' => 900,
            'display' => 'Every 15 minutes'
        );
        
        // Add 30-minute interval
        $schedules['every_30_minutes'] = array(
            'interval' => 1800,
            'display' => 'Every 30 minutes'
        );
        
        return $schedules;
    }
    
    /**
     * Schedule master sync based on settings
     */
    private function schedule_master_sync() {
        // v1.10.35.16: Use separate intervals for orders and listings
        // Clear existing events
        wp_clear_scheduled_hook('delcampe_master_sync');
        
        // Map interval settings to WordPress cron schedules
        $schedule_map = array(
            'every_15_minutes' => 'every_15_minutes',
            'every_30_minutes' => 'every_30_minutes',
            'hourly' => 'hourly',
            'twicedaily' => 'twicedaily',
            'daily' => 'daily'
        );
        
        // Determine which interval to use based on enabled sync types
        $orders_enabled = get_option('delcampe_orders_sync_enabled', 'yes');
        $listings_enabled = get_option('delcampe_listings_sync_enabled', 'no');
        
        $interval = 'hourly'; // Default
        
        if ($orders_enabled === 'yes' && $listings_enabled === 'yes') {
            // Both enabled - use the more frequent interval
            $orders_interval = get_option('delcampe_orders_sync_interval', 'every_15_minutes');
            $listings_interval = get_option('delcampe_listings_sync_interval', 'hourly');
            
            // Use orders interval if it's more frequent
            $interval = $orders_interval;
        } elseif ($orders_enabled === 'yes') {
            // Only orders enabled
            $interval = get_option('delcampe_orders_sync_interval', 'every_15_minutes');
        } elseif ($listings_enabled === 'yes') {
            // Only listings enabled
            $interval = get_option('delcampe_listings_sync_interval', 'hourly');
        } else {
            // Neither enabled - still schedule but it will skip
            $interval = get_option('delcampe_sync_interval', 'hourly'); // Legacy fallback
        }
        
        $schedule = isset($schedule_map[$interval]) ? $schedule_map[$interval] : 'hourly';
        
        // Schedule a single instance
        $result = wp_schedule_event(time() + 60, $schedule, 'delcampe_master_sync');
        
        if ($result === false) {
            $this->log('ERROR: Failed to schedule master sync');
        } else {
            $this->log('Master sync scheduled with interval: ' . $schedule);
            $this->log('Orders sync: ' . $orders_enabled . ', Listings sync: ' . $listings_enabled);
        }
    }
    
    /**
     * Run the master sync process
     * This coordinates all sync operations
     */
    public function run_master_sync() {
        $this->log('=== MASTER SYNC STARTED ===');
        
        // v1.10.35.16: Check separate sync settings for orders and listings
        $orders_sync_enabled = get_option('delcampe_orders_sync_enabled', 'yes'); // Default: enabled
        $listings_sync_enabled = get_option('delcampe_listings_sync_enabled', 'no'); // Default: disabled
        
        // Legacy fallback for compatibility
        $legacy_sync_enabled = get_option('delcampe_sync_enabled', null);
        if ($legacy_sync_enabled !== null && $orders_sync_enabled === 'yes' && $listings_sync_enabled === 'no') {
            // If legacy setting exists and new settings are at defaults, use legacy
            if ($legacy_sync_enabled !== 'yes') {
                $this->log('Master sync is disabled in settings (legacy)');
                return;
            }
        }
        
        // Check if at least one sync type is enabled
        if ($orders_sync_enabled !== 'yes' && $listings_sync_enabled !== 'yes') {
            $this->log('Both order sync and listings sync are disabled');
            return;
        }
        
        // OPTIMIZATION v1.10.35.1: Check if we have any active listings first
        // This prevents unnecessary API calls when there are no listings on Delcampe
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        $active_listings_count = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$listings_table}
            WHERE status IN ('active', 'published', 'verified', 'processing')
            AND delcampe_id IS NOT NULL
            AND delcampe_id != ''
        ");
        
        if ($active_listings_count == 0) {
            $this->log('No active listings found. Skipping sync to avoid unnecessary API calls.');
            // Store skip status
            update_option('delcampe_last_master_sync', array(
                'start_time' => current_time('mysql'),
                'end_time' => current_time('mysql'),
                'status' => 'skipped',
                'reason' => 'no_active_listings',
                'active_listings' => 0
            ));
            return;
        }
        
        $this->log("Found {$active_listings_count} active listings. Proceeding with sync...");
        
        // Prevent concurrent runs using transient lock
        $lock_key = 'delcampe_master_sync_lock';
        if (get_transient($lock_key)) {
            $this->log('Master sync already running, skipping');
            return;
        }
        
        // Set lock for 10 minutes
        set_transient($lock_key, true, 600);
        
        try {
            // Track sync results
            $results = array(
                'start_time' => current_time('mysql'),
                'orders' => array('checked' => 0, 'imported' => 0, 'errors' => 0),
                'listings' => array('checked' => 0, 'updated' => 0, 'errors' => 0),
                'inventory' => array('checked' => 0, 'updated' => 0, 'errors' => 0),
                'prices' => array('checked' => 0, 'updated' => 0, 'errors' => 0),
                'closed' => array('checked' => 0, 'updated' => 0, 'errors' => 0)
            );
            
            // v1.10.35.16: Respect separate sync settings
            
            // 1. Sync Orders (Sold Items) - only if orders sync is enabled
            if ($orders_sync_enabled === 'yes') {
                $this->log('Step 1: Syncing orders from Delcampe...');
                $results['orders'] = $this->sync_orders();
            } else {
                $this->log('Step 1: Order sync disabled - skipping');
            }
            
            // 2-4. Listings related syncs - only if listings sync is enabled
            if ($listings_sync_enabled === 'yes') {
                // 2. Sync Closed/Ended Listings
                $this->log('Step 2: Syncing closed listings...');
                $results['closed'] = $this->sync_closed_listings();
                
                // 3. Update Active Listing Statuses
                $this->log('Step 3: Updating active listing statuses...');
                $results['listings'] = $this->sync_listing_statuses();
                
                // 4. Sync Inventory Levels
                $this->log('Step 4: Syncing inventory levels...');
                $results['inventory'] = $this->sync_inventory();
            } else {
                $this->log('Steps 2-4: Listings/Inventory sync disabled - skipping');
            }
            
            // 5. Check for Price Changes (if enabled)
            if (get_option('delcampe_sync_prices', 'no') === 'yes') {
                $this->log('Step 5: Checking for price changes...');
                $results['prices'] = $this->sync_prices();
            }
            
            // 6. Process Webhook Queue (if any)
            $this->log('Step 6: Processing webhook queue...');
            $this->process_webhook_queue();
            
            // Save sync results
            $results['end_time'] = current_time('mysql');
            $results['status'] = 'completed';
            update_option('delcampe_last_master_sync', $results);
            
            $this->log('=== MASTER SYNC COMPLETED ===');
            $this->log('Results: ' . json_encode($results));
            
        } catch (Exception $e) {
            $this->log('Master sync error: ' . $e->getMessage(), 'error');
            $results['status'] = 'error';
            $results['error'] = $e->getMessage();
            update_option('delcampe_last_master_sync', $results);
        } finally {
            // Release lock
            delete_transient($lock_key);
        }
    }
    
    /**
     * Sync orders from Delcampe
     */
    private function sync_orders() {
        $results = array('checked' => 0, 'imported' => 0, 'errors' => 0);
        
        try {
            // Get order manager instance
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-order-manager.php';
            $order_manager = Delcampe_Order_Manager::get_instance();
            
            // Import sold items from last 7 days
            // Note: import_sold_items was renamed to import_orders in newer versions
            $import_result = $order_manager->import_orders(0, 100, 'master_sync');
            
            if (is_wp_error($import_result)) {
                $this->log('Order sync error: ' . $import_result->get_error_message(), 'error');
                $results['errors']++;
            } else {
                $results['checked'] = count($import_result);
                foreach ($import_result as $item) {
                    // Check if status key exists before accessing it
                    if (isset($item['status']) && ($item['status'] === 'created' || $item['status'] === 'updated')) {
                        $results['imported']++;
                    }
                }
            }
        } catch (Exception $e) {
            $this->log('Order sync exception: ' . $e->getMessage(), 'error');
            $results['errors']++;
        }
        
        return $results;
    }
    
    /**
     * Sync closed listings
     */
    private function sync_closed_listings() {
        $results = array('checked' => 0, 'updated' => 0, 'errors' => 0);
        
        try {
            // Get listings sync instance
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listings-sync.php';
            $listings_sync = Delcampe_Listings_Sync::get_instance();
            
            // Sync closed listings - process up to 10,000 items (20 pages of 500)
            // This will use pagination and tracking to avoid re-processing
            $sync_result = $listings_sync->sync_closed_listings(500, 20);
            
            if (is_wp_error($sync_result)) {
                $this->log('Closed listings sync error: ' . $sync_result->get_error_message(), 'error');
                $results['errors']++;
            } else {
                $results['checked'] = count($sync_result);
                foreach ($sync_result as $item) {
                    if ($item['action'] === 'updated_to_ended') {
                        $results['updated']++;
                    }
                }
            }
        } catch (Exception $e) {
            $this->log('Closed listings sync exception: ' . $e->getMessage(), 'error');
            $results['errors']++;
        }
        
        return $results;
    }
    
    /**
     * Sync listing statuses
     */
    private function sync_listing_statuses() {
        $results = array('checked' => 0, 'updated' => 0, 'errors' => 0);
        
        try {
            global $wpdb;
            $listings_table = $wpdb->prefix . 'delcampe_listings';
            
            // Get all processing listings
            $processing_listings = $wpdb->get_results(
                "SELECT * FROM {$listings_table} 
                 WHERE status = 'processing' 
                 AND date_created > DATE_SUB(NOW(), INTERVAL 24 HOUR)"
            );
            
            $results['checked'] = count($processing_listings);
            
            // Check each listing
            foreach ($processing_listings as $listing) {
                $status = $this->check_listing_on_delcampe($listing->personal_reference);
                if ($status['found']) {
                    $wpdb->update(
                        $listings_table,
                        array(
                            'delcampe_id' => $status['delcampe_id'],
                            'status' => 'published'
                        ),
                        array('id' => $listing->id)
                    );
                    $results['updated']++;
                }
            }
        } catch (Exception $e) {
            $this->log('Listing status sync exception: ' . $e->getMessage(), 'error');
            $results['errors']++;
        }
        
        return $results;
    }
    
    /**
     * Check if listing exists on Delcampe
     */
    private function check_listing_on_delcampe($reference) {
        // Get auth token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return array('found' => false, 'error' => $token->get_error_message());
        }
        
        // Search for item by reference
        $api_url = DELCAMPE_API_URL . '/item/opened';
        $params = array(
            'token' => $token,
            'personal_reference' => $reference
        );
        
        $url = $api_url . '?' . http_build_query($params);
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return array('found' => false, 'error' => $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        if ($xml && isset($xml->Notification_Data->body->item)) {
            foreach ($xml->Notification_Data->body->item as $item) {
                if ((string)$item->personal_reference === $reference) {
                    return array(
                        'found' => true,
                        'delcampe_id' => (string)$item->id_item
                    );
                }
            }
        }
        
        return array('found' => false);
    }
    
    /**
     * Sync inventory levels
     */
    private function sync_inventory() {
        $results = array('checked' => 0, 'updated' => 0, 'errors' => 0);
        
        try {
            // Get inventory manager instance
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-inventory-manager.php';
            $inventory_manager = Delcampe_Inventory_Manager::get_instance();
            
            // Sync inventory for all active listings
            global $wpdb;
            $listings_table = $wpdb->prefix . 'delcampe_listings';
            
            $active_listings = $wpdb->get_results(
                "SELECT * FROM {$listings_table} 
                 WHERE status = 'published' 
                 AND delcampe_id IS NOT NULL"
            );
            
            $results['checked'] = count($active_listings);
            
            foreach ($active_listings as $listing) {
                if ($listing->product_id) {
                    $sync_result = $inventory_manager->sync_product_stock($listing->product_id);
                    if ($sync_result && !is_wp_error($sync_result)) {
                        $results['updated']++;
                    } else {
                        $results['errors']++;
                    }
                }
            }
        } catch (Exception $e) {
            $this->log('Inventory sync exception: ' . $e->getMessage(), 'error');
            $results['errors']++;
        }
        
        return $results;
    }
    
    /**
     * Sync prices
     */
    private function sync_prices() {
        $results = array('checked' => 0, 'updated' => 0, 'errors' => 0);
        
        try {
            // Get price sync instance
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-price-sync.php';
            $price_sync = Delcampe_Price_Sync::get_instance();
            
            // Process any pending price updates - correct method name
            $sync_result = $price_sync->process_price_sync_queue();
            
            if (isset($sync_result['processed'])) {
                $results['checked'] = $sync_result['total'];
                $results['updated'] = $sync_result['processed'];
                $results['errors'] = $sync_result['failed'];
            }
        } catch (Exception $e) {
            $this->log('Price sync exception: ' . $e->getMessage(), 'error');
            $results['errors']++;
        }
        
        return $results;
    }
    
    /**
     * Process webhook queue
     */
    private function process_webhook_queue() {
        global $wpdb;
        $webhooks_table = $wpdb->prefix . 'delcampe_webhooks';
        
        // Check if webhooks table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$webhooks_table}'") !== $webhooks_table) {
            return;
        }
        
        // Process any pending webhooks
        $pending_webhooks = $wpdb->get_results(
            "SELECT * FROM {$webhooks_table} 
             WHERE status = 'pending' 
             ORDER BY id ASC 
             LIMIT 10"
        );
        
        foreach ($pending_webhooks as $webhook) {
            // Process webhook based on type
            $this->log('Processing webhook: ' . $webhook->notification_type);
            
            // Mark as processed
            $wpdb->update(
                $webhooks_table,
                array('status' => 'processed'),
                array('id' => $webhook->id)
            );
        }
    }
    
    /**
     * AJAX handler for manual sync
     */
    public function ajax_run_master_sync() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Run sync
        $this->run_master_sync();
        
        // Get results
        $results = get_option('delcampe_last_master_sync', array());
        
        wp_send_json_success($results);
    }
    
    /**
     * Get sync status
     */
    public static function get_sync_status() {
        $last_sync = get_option('delcampe_last_master_sync', array());
        $next_sync = wp_next_scheduled('delcampe_master_sync');
        
        return array(
            'last_sync' => $last_sync,
            'next_sync' => $next_sync ? date('Y-m-d H:i:s', $next_sync) : 'Not scheduled',
            'interval' => get_option('delcampe_sync_interval', 'hourly'),
            'enabled' => get_option('delcampe_sync_enabled', 'yes') === 'yes'
        );
    }
}