<?php
/**
 * Delcampe Order Consolidation Handler
 * 
 * Handles consolidation of multiple Delcampe items into single WooCommerce orders
 * when they are part of the same payment/bill
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.27.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Order_Consolidation {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Pending orders by buyer (for consolidation)
     * Key: buyer_id, Value: array of order data
     */
    private $pending_consolidation = array();
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Hook to process payment notifications
        add_action('delcampe_payment_received', array($this, 'consolidate_orders_on_payment'), 10, 1);
        
        // Store pending items temporarily
        add_action('delcampe_item_sold', array($this, 'queue_item_for_consolidation'), 10, 2);
    }
    
    /**
     * Queue an item for potential consolidation
     * Instead of creating order immediately, wait for payment notification
     * 
     * @param array $item_data Item sold data
     * @param array $buyer_info Buyer information
     */
    public function queue_item_for_consolidation($item_data, $buyer_info) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        $buyer_id = isset($buyer_info['id']) ? $buyer_info['id'] : 'unknown';
        
        if (!isset($this->pending_consolidation[$buyer_id])) {
            $this->pending_consolidation[$buyer_id] = array();
        }
        
        // Store item data for later consolidation
        $this->pending_consolidation[$buyer_id][] = array(
            'item_data' => $item_data,
            'buyer_info' => $buyer_info,
            'timestamp' => current_time('timestamp')
        );
        
        // Set a transient to persist this data (expires in 1 hour)
        set_transient('delcampe_pending_' . $buyer_id, $this->pending_consolidation[$buyer_id], HOUR_IN_SECONDS);
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'ITEM_QUEUED_FOR_CONSOLIDATION',
            'buyer_id' => $buyer_id,
            'item_id' => $item_data['item_id'],
            'pending_count' => count($this->pending_consolidation[$buyer_id])
        ));
    }
    
    /**
     * Consolidate orders when payment is received
     * This is where we combine multiple items into a single order
     * 
     * @param array $payment_data Payment notification data
     */
    public function consolidate_orders_on_payment($payment_data) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        try {
            // Extract bill information
            $bill_id = isset($payment_data['bill_id']) ? $payment_data['bill_id'] : null;
            $items = isset($payment_data['items']) ? $payment_data['items'] : array();
            $buyer_id = isset($payment_data['buyer_id']) ? $payment_data['buyer_id'] : null;
            
            if (!$bill_id || empty($items)) {
                throw new Exception('Missing bill ID or items in payment notification');
            }
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'CONSOLIDATION_START',
                'bill_id' => $bill_id,
                'item_count' => count($items),
                'total_amount' => $payment_data['total_amount'],
                'seller_amount' => $payment_data['seller_amount']
            ));
            
            // Check if we already have orders for these items
            $existing_orders = $this->find_existing_orders_for_items($items);
            
            if (count($existing_orders) > 1) {
                // Multiple orders exist - consolidate them
                $this->merge_orders($existing_orders, $payment_data);
            } elseif (count($existing_orders) == 1) {
                // Single order exists - update it with payment info
                $this->update_order_with_payment($existing_orders[0], $payment_data);
            } else {
                // No orders exist yet - create consolidated order
                $this->create_consolidated_order($items, $payment_data);
            }
            
        } catch (Exception $e) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'CONSOLIDATION_ERROR',
                'error' => $e->getMessage(),
                'payment_data' => json_encode($payment_data)
            ));
        }
    }
    
    /**
     * Find existing WooCommerce orders for the given items
     * 
     * @param array $items Array of items from payment notification
     * @return array Array of WC_Order objects
     */
    private function find_existing_orders_for_items($items) {
        global $wpdb;
        $orders = array();
        
        foreach ($items as $item) {
            // Try to find order by SKU
            if (isset($item['sku']) && !empty($item['sku'])) {
                $product_id = wc_get_product_id_by_sku($item['sku']);
                
                if ($product_id) {
                    // Find recent orders containing this product
                    $order_ids = $wpdb->get_col($wpdb->prepare("
                        SELECT DISTINCT oi.order_id 
                        FROM {$wpdb->prefix}woocommerce_order_items oi
                        JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim ON oi.order_item_id = oim.order_item_id
                        JOIN {$wpdb->posts} p ON oi.order_id = p.ID
                        WHERE oim.meta_key = '_product_id'
                        AND oim.meta_value = %d
                        AND p.post_status = 'wc-on-hold'
                        AND p.post_date > %s
                        ORDER BY p.ID DESC
                        LIMIT 1
                    ", $product_id, date('Y-m-d H:i:s', strtotime('-24 hours'))));
                    
                    foreach ($order_ids as $order_id) {
                        $order = wc_get_order($order_id);
                        if ($order && !in_array($order, $orders)) {
                            $orders[] = $order;
                        }
                    }
                }
            }
            
            // Also try by Delcampe item ID if stored
            if (isset($item['item_id'])) {
                $order_id = $wpdb->get_var($wpdb->prepare("
                    SELECT post_id FROM {$wpdb->postmeta}
                    WHERE meta_key = '_delcampe_item_id'
                    AND meta_value = %s
                    LIMIT 1
                ", $item['item_id']));
                
                if ($order_id) {
                    $order = wc_get_order($order_id);
                    if ($order && !in_array($order, $orders)) {
                        $orders[] = $order;
                    }
                }
            }
        }
        
        return $orders;
    }
    
    /**
     * Merge multiple orders into one consolidated order
     * 
     * @param array $orders Array of WC_Order objects to merge
     * @param array $payment_data Payment information
     */
    private function merge_orders($orders, $payment_data) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        // Use the first order as the primary
        $primary_order = array_shift($orders);
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'MERGING_ORDERS',
            'primary_order' => $primary_order->get_id(),
            'orders_to_merge' => array_map(function($o) { return $o->get_id(); }, $orders)
        ));
        
        // Move all items from other orders to primary order
        foreach ($orders as $order) {
            foreach ($order->get_items() as $item) {
                // Clone the item to the primary order
                $new_item = new WC_Order_Item_Product();
                $new_item->set_product($item->get_product());
                $new_item->set_quantity($item->get_quantity());
                $new_item->set_subtotal($item->get_subtotal());
                $new_item->set_total($item->get_total());
                $primary_order->add_item($new_item);
            }
            
            // Add note to the cancelled order
            $order->add_order_note(sprintf(
                'Order consolidated into order #%d (Bill ID: %s)',
                $primary_order->get_id(),
                $payment_data['bill_id']
            ));
            
            // Cancel the duplicate order
            $order->update_status('cancelled', 'Consolidated into another order');
        }
        
        // Update the primary order with payment information
        $this->update_order_with_payment($primary_order, $payment_data);
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'ORDERS_MERGED',
            'primary_order' => $primary_order->get_id(),
            'cancelled_orders' => count($orders)
        ));
    }
    
    /**
     * Update order with payment information
     * 
     * @param WC_Order $order The order to update
     * @param array $payment_data Payment information
     */
    private function update_order_with_payment($order, $payment_data) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        // Store bill ID
        $order->update_meta_data('_delcampe_bill_id', $payment_data['bill_id']);
        
        // Calculate and add shipping
        $items_total = 0;
        foreach ($order->get_items() as $item) {
            $items_total += $item->get_total();
        }
        
        // Calculate shipping from the difference
        $shipping_amount = $payment_data['seller_amount'] - $items_total;
        
        if ($shipping_amount > 0) {
            $shipping_item = new WC_Order_Item_Shipping();
            $shipping_item->set_method_title('Delcampe Shipping');
            $shipping_item->set_method_id('delcampe_shipping');
            $shipping_item->set_total($shipping_amount);
            $order->add_item($shipping_item);
        }
        
        // Add financial metadata
        $order->update_meta_data('_delcampe_buyer_paid', $payment_data['total_amount']);
        $order->update_meta_data('_delcampe_platform_fee', $payment_data['total_amount'] - $payment_data['seller_amount']);
        $order->update_meta_data('_delcampe_seller_receives', $payment_data['seller_amount']);
        
        if (isset($payment_data['transaction_id'])) {
            $order->update_meta_data('_transaction_id', $payment_data['transaction_id']);
        }
        
        // Update status to processing
        $order->update_status('processing', 'Payment received via Delcampe (Bill: ' . $payment_data['bill_id'] . ')');
        
        // Recalculate totals
        $order->calculate_totals(false);
        $order->save();
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'ORDER_UPDATED_WITH_PAYMENT',
            'order_id' => $order->get_id(),
            'bill_id' => $payment_data['bill_id'],
            'total' => $order->get_total()
        ));
    }
    
    /**
     * Create a new consolidated order from payment data
     * 
     * @param array $items Items in the payment
     * @param array $payment_data Payment information
     * @return WC_Order|WP_Error
     */
    private function create_consolidated_order($items, $payment_data) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        try {
            // Create the order
            $order = wc_create_order(array(
                'status' => 'processing', // Already paid
                'created_via' => 'delcampe_webhook'
            ));
            
            if (is_wp_error($order)) {
                throw new Exception('Failed to create order: ' . $order->get_error_message());
            }
            
            // Set customer information
            if (isset($payment_data['buyer_info'])) {
                $buyer = $payment_data['buyer_info'];
                $order->set_billing_first_name($buyer['firstname'] ?? 'Delcampe');
                $order->set_billing_last_name($buyer['lastname'] ?? 'Buyer');
                $order->set_billing_email($buyer['email'] ?? '');
                $order->set_billing_address_1($buyer['address'] ?? '');
                $order->set_billing_city($buyer['city'] ?? '');
                $order->set_billing_postcode($buyer['zipcode'] ?? '');
                $order->set_billing_country($buyer['country'] ?? '');
            }
            
            // Add items
            $items_total = 0;
            foreach ($items as $item_data) {
                $product = null;
                
                // Find product by SKU
                if (isset($item_data['sku'])) {
                    $product_id = wc_get_product_id_by_sku($item_data['sku']);
                    if ($product_id) {
                        $product = wc_get_product($product_id);
                    }
                }
                
                $item = new WC_Order_Item_Product();
                
                if ($product) {
                    $item->set_product($product);
                } else {
                    $item->set_name($item_data['title'] ?? 'Delcampe Item');
                }
                
                $item->set_quantity($item_data['qty'] ?? 1);
                $price = floatval(str_replace(array('CA$', '$', ','), '', $item_data['price']));
                $item->set_subtotal($price);
                $item->set_total($price);
                $order->add_item($item);
                
                $items_total += $price;
            }
            
            // Add shipping
            $shipping_amount = $payment_data['seller_amount'] - $items_total;
            if ($shipping_amount > 0) {
                $shipping_item = new WC_Order_Item_Shipping();
                $shipping_item->set_method_title('Delcampe Shipping');
                $shipping_item->set_method_id('delcampe_shipping');
                $shipping_item->set_total($shipping_amount);
                $order->add_item($shipping_item);
            }
            
            // Set metadata
            $order->update_meta_data('_delcampe_bill_id', $payment_data['bill_id']);
            $order->update_meta_data('_delcampe_consolidated', 'yes');
            $order->update_meta_data('_transaction_id', $payment_data['transaction_id'] ?? '');
            
            // Calculate and save
            $order->calculate_totals(false);
            $order->save();
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'CONSOLIDATED_ORDER_CREATED',
                'order_id' => $order->get_id(),
                'bill_id' => $payment_data['bill_id'],
                'item_count' => count($items),
                'total' => $order->get_total()
            ));
            
            return $order;
            
        } catch (Exception $e) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'CONSOLIDATED_ORDER_CREATION_FAILED',
                'error' => $e->getMessage()
            ));
            
            return new WP_Error('creation_failed', $e->getMessage());
        }
    }
}