<?php
/**
 * Delcampe Order Manager
 * 
 * Manages order synchronization between Delcampe and WooCommerce
 * Handles order imports, status updates, customer creation, and inventory management
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Orders
 * @since 1.4.0.0
 * @version 1.10.12.7
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_Order_Manager
 * 
 * Core order management functionality for Delcampe integration
 * Handles bi-directional order synchronization
 * Enhanced in v1.5.0.0 to include automatic inventory reduction
 * 
 * @since 1.4.0.0
 * @version 1.5.0.0
 */
class Delcampe_Order_Manager {
    
    /**
     * Singleton instance
     * @var Delcampe_Order_Manager|null
     */
    private static $instance = null;
    
    /**
     * Database table name for order sync
     * @var string
     */
    private $table_name;
    
    /**
     * Database table for order fees
     * @var string
     */
    private $fees_table_name;
    
    /**
     * Get singleton instance
     * 
     * @return Delcampe_Order_Manager
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     * 
     * Initialize order manager and set up hooks
     * 
     * @version 1.5.0.0
     */
    private function __construct() {
        global $wpdb;
        
        // Set table names
        $this->table_name = $wpdb->prefix . 'delcampe_orders';
        $this->fees_table_name = $wpdb->prefix . 'delcampe_order_fees';
        
        // Initialize hooks
        $this->init_hooks();
        
        // Order manager initialized
    }
    
    /**
     * Generate idempotent order ID for Delcampe sales
     * Based on ChatGPT's analysis of patterns and community best practices
     * 
     * Pattern observed:
     * - First sale: item_id only (e.g., "2353609113")
     * - Subsequent sales same buyer: item_id-buyer_id-counter
     * - Payment received: Can update to BILL-{bill_id} format
     * 
     * @param string $item_id Delcampe item ID
     * @param string $buyer_id Buyer ID or 'unknown'
     * @param string $bill_id Optional bill ID from payment notification
     * @return string Idempotent order ID
     * @since 1.10.13.19
     */
    private function generate_idempotent_order_id($item_id, $buyer_id = null, $bill_id = null) {
        global $wpdb;
        
        // If we have a bill ID, use it for ultimate idempotency
        if ($bill_id) {
            return 'BILL-' . $bill_id;
        }
        
        // Sanitize item ID - extract numeric part only
        $item_id = preg_replace('/[^0-9]/', '', $item_id);
        
        // CRITICAL: Never use 'unknown' for real orders
        // If no buyer ID, we should not create an order at all
        if (!$buyer_id || $buyer_id === 'unknown' || $buyer_id === '') {
            delcampe_log('[Order ID Generation] Skipping order creation - no valid buyer ID for item ' . $item_id);
            return null; // Signal that order should not be created
        }
        
        // Check if an order already exists for this item+buyer
        // This prevents duplicates when same item is processed multiple times
        $pattern = $item_id . '-' . $buyer_id . '-%';
        $existing = $wpdb->get_col($wpdb->prepare(
            "SELECT delcampe_order_id FROM {$this->table_name} 
             WHERE delcampe_order_id LIKE %s
             ORDER BY id ASC",
            $pattern
        ));
        
        if (empty($existing)) {
            // First order for this item+buyer combination
            return sprintf('%s-%s-1', $item_id, $buyer_id);
        }
        
        // Find max counter and increment
        $max_counter = 0;
        foreach ($existing as $order_id) {
            if (preg_match('/-([0-9]+)$/', $order_id, $matches)) {
                $counter = intval($matches[1]);
                if ($counter > $max_counter) {
                    $max_counter = $counter;
                }
            }
        }
        
        return sprintf('%s-%s-%d', $item_id, $buyer_id, $max_counter + 1);
    }
    
    /**
     * Initialize WordPress hooks
     * 
     * @since 1.4.0.0
     */
    private function init_hooks() {
        // Cron hook for automatic order import
        add_action('delcampe_import_orders_cron', array($this, 'import_orders_cron'));
        
        // WooCommerce order status change hook
        add_action('woocommerce_order_status_changed', array($this, 'sync_order_status_to_delcampe'), 10, 3);
        
        // Add order meta box for both legacy and HPOS
        add_action('add_meta_boxes', array($this, 'add_order_meta_box'), 10, 2);
        add_action('woocommerce_process_shop_order_meta', array($this, 'add_hpos_order_meta_box'), 10, 2);
        
        // Schedule cron only if auto-import is enabled and not already scheduled
        $auto_import = get_option('delcampe_auto_import_orders', 'yes');
        if ($auto_import === 'yes' && !wp_next_scheduled('delcampe_import_orders_cron')) {
            wp_schedule_event(time(), 'hourly', 'delcampe_import_orders_cron');
        } elseif ($auto_import !== 'yes' && wp_next_scheduled('delcampe_import_orders_cron')) {
            // Unschedule if auto-import is disabled but cron is still scheduled
            wp_clear_scheduled_hook('delcampe_import_orders_cron');
        }
    }
    
    /**
     * Create database tables
     * 
     * Creates tables for order sync and fee tracking
     * 
     * @since 1.4.0.0
     */
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Orders sync table
        // v1.10.13.14 - Changed delcampe_order_id to VARCHAR to support composite IDs
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            delcampe_order_id varchar(64) NOT NULL,
            delcampe_item_id bigint(20) unsigned NOT NULL,
            wc_order_id bigint(20) unsigned DEFAULT NULL,
            wc_product_id bigint(20) unsigned DEFAULT NULL,
            buyer_id bigint(20) unsigned NOT NULL,
            buyer_nickname varchar(255) NOT NULL,
            buyer_email varchar(255) NOT NULL,
            sale_price decimal(10,2) NOT NULL,
            currency varchar(3) NOT NULL,
            quantity int(11) NOT NULL DEFAULT 1,
            sale_date datetime NOT NULL,
            payment_status varchar(50) DEFAULT 'pending',
            payment_date datetime DEFAULT NULL,
            shipping_status varchar(50) DEFAULT 'pending',
            tracking_number varchar(255) DEFAULT NULL,
            sync_status varchar(50) DEFAULT 'pending',
            last_sync datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY delcampe_order_id (delcampe_order_id),
            KEY wc_order_id (wc_order_id),
            KEY sync_status (sync_status),
            KEY buyer_email (buyer_email)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // v1.10.13.14 - Migrate existing BIGINT to VARCHAR if needed
        $this->migrate_order_id_column();
        
        // Order fees table
        $sql_fees = "CREATE TABLE IF NOT EXISTS {$this->fees_table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            order_id bigint(20) unsigned NOT NULL,
            fee_type varchar(50) NOT NULL,
            fee_amount decimal(10,2) NOT NULL,
            fee_percentage decimal(5,2) DEFAULT NULL,
            currency varchar(3) NOT NULL,
            description text,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id),
            KEY fee_type (fee_type)
        ) $charset_collate;";
        
        dbDelta($sql_fees);
        
        delcampe_log('[Delcampe Order Manager v1.5.0.0] Database tables created');
    }
    
    /**
     * Migrate order ID column from BIGINT to VARCHAR
     * 
     * @since 1.10.13.14
     */
    private function migrate_order_id_column() {
        global $wpdb;
        
        // Check if migration is needed
        $column_type = $wpdb->get_var($wpdb->prepare(
            "SELECT DATA_TYPE FROM INFORMATION_SCHEMA.COLUMNS 
             WHERE TABLE_SCHEMA = %s 
             AND TABLE_NAME = %s 
             AND COLUMN_NAME = 'delcampe_order_id'",
            DB_NAME,
            $this->table_name
        ));
        
        if ($column_type === 'bigint') {
            // Need to migrate
            delcampe_log('[Order Manager] Migrating delcampe_order_id from BIGINT to VARCHAR');
            
            // SECURITY FIX v1.10.21.2: Properly escape table names in schema operations
            $safe_table = esc_sql($this->table_name);
            
            // Drop the unique index first
            $wpdb->query("ALTER TABLE `$safe_table` DROP INDEX delcampe_order_id");
            
            // Change column type
            $wpdb->query("ALTER TABLE `$safe_table` MODIFY delcampe_order_id VARCHAR(64) NOT NULL");
            
            // Re-add unique index
            $wpdb->query("ALTER TABLE `$safe_table` ADD UNIQUE KEY delcampe_order_id (delcampe_order_id)");
            
            delcampe_log('[Order Manager] Migration complete');
        }
    }
    
    /**
     * Import orders from Delcampe
     * 
     * Fetches sold items from Delcampe API and creates WooCommerce orders
     * 
     * @param int $start Starting item index
     * @param int $limit Number of items to fetch
     * @return array|WP_Error Array of imported orders or error
     * 
     * @since 1.4.0.0
     */
    public function import_orders($start = 0, $limit = 100, $source = 'manual') {
        // Get authentication token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            delcampe_log('[Delcampe Order Manager] Authentication failed: ' . $token->get_error_message());
            return $token;
        }
        
        // Try multiple endpoints for sold items (v1.10.13.13)
        // Delcampe has different endpoints for different sale types
        // Items can show as "open" even when sold if qty > 1
        $endpoints = array(
            '/item/sold',                    // General sold items
            '/item/fixedprice/closed/sold',  // Fixed price sold items
            '/item/auction/closed/sold',     // Auction sold items
            '/item/closed',                  // All closed items (check for buyers)
            '/item/opened'                   // Opened items (might have buyers if qty > 1)
        );
        
        $all_items = array();
        
        foreach ($endpoints as $endpoint) {
            $api_url = DELCAMPE_API_URL . $endpoint;
            $params = array(
                'token' => $token,
                'startingItem' => $start,
                'numberOfItems' => $limit,
                'order' => 'desc' // Most recent first
            );
            
            $url = $api_url . '?' . http_build_query($params);
            
            delcampe_log('[Delcampe Order Manager] Checking endpoint: ' . $endpoint);
            
            // Make API request
            $response = wp_remote_get($url, array(
                'timeout' => 30,
                'headers' => array(
                    'Accept' => 'application/xml'
                )
            ));
            
            if (is_wp_error($response)) {
                delcampe_log('[Delcampe Order Manager] API request failed for ' . $endpoint . ': ' . $response->get_error_message());
                continue; // Try next endpoint
            }
            
            $body = wp_remote_retrieve_body($response);
            
            // Parse XML response
            $xml = simplexml_load_string($body);
            if ($xml === false) {
                delcampe_log('[Delcampe Order Manager] Failed to parse XML response for ' . $endpoint);
                continue; // Try next endpoint
            }
            
            // Check for API errors
            if (isset($xml->e)) {
                delcampe_log('[Delcampe Order Manager] API error for ' . $endpoint . ': ' . (string)$xml->e);
                continue; // Try next endpoint
            }
            
            // Find items in various XML structures
            $items = null;
            if (isset($xml->Notification_Data->body->item)) {
                $items = $xml->Notification_Data->body->item;
            } elseif (isset($xml->body->item)) {
                $items = $xml->body->item;
            } elseif (isset($xml->item)) {
                $items = $xml->item;
            }
            
            if ($items) {
                delcampe_log('[Delcampe Order Manager] Found ' . count($items) . ' items in ' . $endpoint);
                foreach ($items as $item) {
                    // Add to collection for processing
                    $all_items[] = $item;
                }
            }
        } // End foreach endpoints
        
        // Now process all collected items
        $imported_orders = array();
        $skipped_items = array();
        
        foreach ($all_items as $item) {
            $item_id = (string)$item->id_item;
            
            // CRITICAL: Always fetch buyer info to ensure we have valid data
            // This prevents creating orders with 'unknown' buyer
            if (!isset($item->buyer) || !isset($item->buyer->buyer_id)) {
                // Try to fetch buyer info from API
                $buyer_url = "https://rest.delcampe.net/item/{$item_id}/buyer?token={$token}";
                delcampe_log('[Order Import] Fetching buyer info for item ' . $item_id);
                
                $buyer_response = wp_remote_get($buyer_url, array(
                    'timeout' => 10,
                    'headers' => array('Accept' => 'application/xml')
                ));
                
                if (!is_wp_error($buyer_response) && wp_remote_retrieve_response_code($buyer_response) == 200) {
                    $buyer_body = wp_remote_retrieve_body($buyer_response);
                    $buyer_xml = simplexml_load_string($buyer_body);
                    
                    // Try to find buyer info
                    if ($buyer_xml && isset($buyer_xml->Notification_Data->body->buyer) && 
                        isset($buyer_xml->Notification_Data->body->buyer->buyer_id)) {
                        $item->buyer = $buyer_xml->Notification_Data->body->buyer;
                        delcampe_log('[Order Import] Found buyer via API for item ' . $item_id . 
                                   ', buyer ID: ' . (string)$item->buyer->buyer_id);
                    } else {
                        delcampe_log('[Order Import] No buyer found for item ' . $item_id . ' - skipping');
                        $skipped_items[] = $item_id;
                        continue; // No buyer found, skip
                    }
                } else {
                    delcampe_log('[Order Import] Failed to fetch buyer info for item ' . $item_id . ' - skipping');
                    $skipped_items[] = $item_id;
                    continue; // Could not fetch buyer info, skip
                }
            }
            
            // Double-check we have valid buyer ID
            if (!isset($item->buyer->buyer_id) || empty($item->buyer->buyer_id)) {
                delcampe_log('[Order Import] Invalid buyer data for item ' . $item_id . ' - skipping');
                $skipped_items[] = $item_id;
                continue;
            }
            
            // Import this order
            $result = $this->import_single_order($item, 'processing', 'manual');
            if (!is_wp_error($result)) {
                $imported_orders[] = $result;
            }
        }
        
        delcampe_log('[Delcampe Order Manager] Imported ' . count($imported_orders) . ' orders, skipped ' . count($skipped_items) . ' items');
        
        // Return in the format expected by the UI
        return array(
            'imported' => $imported_orders,
            'skipped' => $skipped_items
        );
    }
    
    /**
     * Import closed sales from Delcampe
     * 
     * Fetches closed sales from Delcampe and marks corresponding WooCommerce orders as completed
     * 
     * @param int $start Starting item index
     * @param int $limit Number of items to fetch
     * @return array|WP_Error Array of processed orders or error
     * 
     * @since 1.10.2.0
     */
    public function import_closed_sales($start = 0, $limit = 100, $source = 'manual') {
        delcampe_log('[Delcampe Order Manager] Starting import_closed_sales with start=' . $start . ', limit=' . $limit);
        
        // Get authentication token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            delcampe_log('[Delcampe Order Manager] Authentication failed for closed sales: ' . $token->get_error_message());
            return $token;
        }
        
        delcampe_log('[Delcampe Order Manager] Got auth token: ' . substr($token, 0, 10) . '...');
        
        // Build API URL for closed items
        $api_url = DELCAMPE_API_URL . '/item/closed';
        $params = array(
            'token' => $token,
            'startingItem' => $start,
            'numberOfItems' => $limit,
            'order' => 'desc' // Most recent first
        );
        
        $url = $api_url . '?' . http_build_query($params);
        delcampe_log('[Delcampe Order Manager] API URL for closed sales: ' . $url);
        
        // Make API request
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            delcampe_log('[Delcampe Order Manager] API request failed for closed sales: ' . $response->get_error_message());
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        delcampe_log('[Delcampe Order Manager] API response code: ' . $response_code);
        
        $body = wp_remote_retrieve_body($response);
        delcampe_log('[Delcampe Order Manager] API response body (first 500 chars): ' . substr($body, 0, 500));
        
        // Parse XML response
        $xml = simplexml_load_string($body);
        if ($xml === false) {
            delcampe_log('[Delcampe Order Manager] Failed to parse XML response for closed sales');
            delcampe_log('[Delcampe Order Manager] Full response body: ' . $body);
            return new WP_Error('xml_parse_error', 'Failed to parse XML response');
        }
        
        // Check for API errors in the newer format
        if (isset($xml->Notification_Data->Headers->Status) && (string)$xml->Notification_Data->Headers->Status !== '200') {
            delcampe_log('[Delcampe Order Manager] API error status: ' . (string)$xml->Notification_Data->Headers->Status);
            return new WP_Error('api_error', 'API returned error status: ' . (string)$xml->Notification_Data->Headers->Status);
        }
        
        $processed_orders = array();
        
        // Navigate to the correct XML path for items
        $items = null;
        if (isset($xml->Notification_Data->body->item)) {
            // New API format
            $items = $xml->Notification_Data->body->item;
            delcampe_log('[Delcampe Order Manager] Found items in new format. Count: ' . count($items));
        } elseif (isset($xml->item)) {
            // Old API format
            $items = $xml->item;
            delcampe_log('[Delcampe Order Manager] Found items in old format. Count: ' . count($items));
        } else {
            delcampe_log('[Delcampe Order Manager] No items found in response');
            return $processed_orders; // Return empty array if no items
        }
        
        // Process each closed item
        foreach ($items as $item) {
            // Check if this is a sold item (has buyer info)
            if (!isset($item->buyer)) {
                continue; // Not sold, just closed
            }
            
            $delcampe_id = (string)$item->id_item;
            
            // Find the corresponding WooCommerce order
            global $wpdb;
            $order_data = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->table_name} WHERE delcampe_order_id = %s",
                $delcampe_id
            ));
            
            if ($order_data && $order_data->wc_order_id) {
                $order = wc_get_order($order_data->wc_order_id);
                
                if ($order && $order->get_status() !== 'completed') {
                    // Check if we should auto-complete orders on closure
                    if ('yes' === get_option('delcampe_complete_on_closure', 'no')) {
                        // Update order status to completed
                        $order->update_status('completed', __('Order closed on Delcampe', 'wc-delcampe-integration'));
                    } else {
                        // Just add a note but leave in processing
                        $order->add_order_note(__('Order closed on Delcampe. Please ship and mark as completed.', 'wc-delcampe-integration'));
                    }
                    
                    // Update sync status
                    $wpdb->update(
                        $this->table_name,
                        array(
                            'sync_status' => 'completed',
                            'shipping_status' => 'completed',
                            'last_sync' => current_time('mysql')
                        ),
                        array('id' => $order_data->id)
                    );
                    
                    $processed_orders[] = $order_data->wc_order_id;
                    delcampe_log('[Delcampe Order Manager] Marked order ' . $order_data->wc_order_id . ' as completed (Delcampe ID: ' . $delcampe_id . ')');
                }
            } else {
                // This is a closed sale we haven't imported yet - import it as completed
                $result = $this->import_single_order($item, 'completed', $source);
                if (!is_wp_error($result)) {
                    $processed_orders[] = $result;
                }
            }
        }
        
        delcampe_log('[Delcampe Order Manager] Processed ' . count($processed_orders) . ' closed sales');
        
        return $processed_orders;
    }
    
    /**
     * Import a single order from Delcampe item data
     * 
     * Enhanced in v1.5.0.0 to properly handle inventory reduction
     * Enhanced in v1.10.2.0 to support custom order status
     * 
     * @param SimpleXMLElement $item Delcampe item XML data
     * @param string $status Optional order status (default: 'processing')
     * @return int|WP_Error WooCommerce order ID or error
     * 
     * @since 1.4.0.0
     * @version 1.10.12.7
     */
    public function import_single_order($item, $status = 'processing', $source = 'manual') {
        global $wpdb;
        
        // Extract order data from XML
        // v1.10.13.12 - Create unique order ID for each sale
        // Using just id_item causes issues when same item bought multiple times
        $item_id = (string)$item->id_item;
        
        // CRITICAL: Only create orders when we have confirmed buyer info
        if (!isset($item->buyer) || !isset($item->buyer->buyer_id)) {
            delcampe_log('[Order Import] Skipping item ' . $item_id . ' - no buyer information');
            return new WP_Error('no_buyer', 'Item has no buyer information');
        }
        
        $buyer_id = (string)$item->buyer->buyer_id;
        $buyer_nick = isset($item->buyer->buyer_nickname) ? (string)$item->buyer->buyer_nickname : 'buyer';
        
        // Validate buyer ID
        if (empty($buyer_id) || $buyer_id === 'unknown') {
            delcampe_log('[Order Import] Skipping item ' . $item_id . ' - invalid buyer ID');
            return new WP_Error('invalid_buyer', 'Invalid buyer ID');
        }
        
        /**
         * DUPLICATE PREVENTION - Added 2025-08-28 from production fix
         * Build composite order ID early to check for duplicates
         */
        // Generate idempotent order ID (matches observed patterns)
        $delcampe_order_id = $this->generate_idempotent_order_id($item_id, $buyer_id);
        
        if (!$delcampe_order_id) {
            delcampe_log('[Order Import] Failed to generate order ID for item ' . $item_id);
            return new WP_Error('id_generation_failed', 'Could not generate order ID');
        }
        
        // Check transient first (fastest) - Extended to 4 hours per production fix
        $recent_import = get_transient('delcampe_imported_' . $delcampe_order_id);
        if ($recent_import) {
            // Verify the order still exists
            $order_check = wc_get_order($recent_import);
            if ($order_check) {
                delcampe_log('[DUPLICATE PREVENTION] Order ' . $delcampe_order_id . ' was imported recently (WC #' . $recent_import . ')', 'info');
                return new WP_Error('recently_imported', 'Order imported in last 4 hours', array('order_id' => $recent_import));
            }
        }
        
        // Double-check sync table
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT wc_order_id FROM {$this->table_name} WHERE delcampe_order_id = %s",
            $delcampe_order_id
        ));
        
        if ($existing) {
            // Set transient to speed up future checks
            set_transient('delcampe_imported_' . $delcampe_order_id, $existing, 4 * HOUR_IN_SECONDS);
            
            delcampe_log('[DUPLICATE PREVENTION] Order ' . $delcampe_order_id . ' already exists (WC #' . $existing . ')', 'info');
            return new WP_Error('order_exists', 'Order already imported', array('order_id' => $existing));
        }
        
        // Check if ANY order already exists for this item in sync table
        // This prevents creating duplicates with different buyer IDs
        $existing_orders = $wpdb->get_results($wpdb->prepare(
            "SELECT delcampe_order_id, wc_order_id FROM {$this->table_name} 
             WHERE delcampe_order_id LIKE %s OR delcampe_order_id = %s",
            $item_id . '-%',
            $delcampe_order_id
        ));
        
        // Check if we already have this exact order
        foreach ($existing_orders as $existing_order) {
            if ($existing_order->delcampe_order_id === $delcampe_order_id) {
                delcampe_log('[Order Import] Order already exists in sync table: ' . $delcampe_order_id);
                return new WP_Error('order_exists', 'Order already imported', array('order_id' => $existing_order->wc_order_id));
            }
            // Also check if this is the same buyer with a different format
            if (strpos($existing_order->delcampe_order_id, '-' . $buyer_id . '-') !== false) {
                delcampe_log('[Order Import] Order for this buyer already exists: ' . $existing_order->delcampe_order_id);
                return new WP_Error('order_exists', 'Order already imported', array('order_id' => $existing_order->wc_order_id));
            }
        }
        
        // Also check WooCommerce orders directly for this Delcampe order ID (in case sync table is out of sync)
        $wc_check = wc_get_orders(array(
            'limit' => 1,
            'meta_key' => '_delcampe_order_id',
            'meta_value' => $delcampe_order_id,
            'return' => 'ids'
        ));
        
        if (!empty($wc_check)) {
            delcampe_log('[Order Import] Order exists in WooCommerce but not sync table: ' . $delcampe_order_id . ' -> WC #' . $wc_check[0]);
            
            // Try to add to sync table to prevent future duplicates
            $this->add_to_sync_table_retroactively($delcampe_order_id, $wc_check[0], $item, $buyer_data);
            
            return new WP_Error('order_exists', 'Order already exists in WooCommerce', array('order_id' => $wc_check[0]));
        }
        
        // Extract buyer information
        // Handle missing or empty names like WP-Lister does
        $buyer_firstname = !empty($item->buyer->buyer_firstname) ? (string)$item->buyer->buyer_firstname : '';
        $buyer_surname = !empty($item->buyer->buyer_surname) ? (string)$item->buyer->buyer_surname : '';
        $buyer_nickname = (string)$item->buyer->buyer_nickname;
        
        // If names are empty, try to split nickname (like WP-Lister does with buyer name)
        if (empty($buyer_firstname) && empty($buyer_surname) && !empty($buyer_nickname)) {
            @list($buyer_firstname, $buyer_surname) = explode(' ', $buyer_nickname, 2);
            if (empty($buyer_surname)) {
                $buyer_surname = $buyer_nickname;  // Use nickname as surname if no space
            }
        }
        
        $buyer_data = array(
            'id' => (string)$item->buyer->buyer_id,
            'nickname' => $buyer_nickname,
            'email' => (string)$item->buyer->buyer_email,
            'firstname' => $buyer_firstname,
            'surname' => $buyer_surname,
            'address' => (string)$item->buyer->buyer_address,
            'zipcode' => (string)$item->buyer->buyer_zipcode,
            'city' => (string)$item->buyer->buyer_city,
            'state' => (string)$item->buyer->buyer_state,
            'country' => (string)$item->buyer->buyer_country,
            'phone' => (string)$item->buyer->buyer_phone
        );
        
        // Create or get WooCommerce customer
        $customer_id = $this->create_or_get_customer($buyer_data);
        
        // Find corresponding WooCommerce product
        $wc_product_id = $this->find_wc_product_by_delcampe_id($delcampe_order_id);
        
        if (!$wc_product_id) {
            // Try to find by personal reference if available
            $personal_ref = (string)$item->personal_reference;
            if ($personal_ref) {
                $wc_product_id = $this->find_wc_product_by_sku($personal_ref);
            }
        }
        
        // Determine sale price
        // Check multiple possible price fields from Delcampe
        $sale_price = 0;
        
        // Log all available price fields for debugging
        delcampe_log('[Order Import] Price fields for item ' . $delcampe_order_id . ':');
        delcampe_log('  - fixed_price: ' . (isset($item->fixed_price) ? $item->fixed_price : 'not set'));
        delcampe_log('  - price_present: ' . (isset($item->price_present) ? $item->price_present : 'not set'));
        delcampe_log('  - best_bid: ' . (isset($item->best_bid) ? $item->best_bid : 'not set'));
        delcampe_log('  - price: ' . (isset($item->price) ? $item->price : 'not set'));
        delcampe_log('  - price_start: ' . (isset($item->price_start) ? $item->price_start : 'not set'));
        delcampe_log('  - price_sold: ' . (isset($item->price_sold) ? $item->price_sold : 'not set'));
        
        // Priority order for price fields (v1.10.13.19)
        // Added more fields based on Delcampe API variations
        if (!empty($item->price_sold) && (float)$item->price_sold > 0) {
            $sale_price = (float)$item->price_sold;
            delcampe_log('[Order Import] Using price_sold: ' . $sale_price);
        } elseif (!empty($item->fixed_price) && (float)$item->fixed_price > 0) {
            $sale_price = (float)$item->fixed_price;
            delcampe_log('[Order Import] Using fixed_price: ' . $sale_price);
        } elseif (!empty($item->price_present) && (float)$item->price_present > 0) {
            $sale_price = (float)$item->price_present;
            delcampe_log('[Order Import] Using price_present: ' . $sale_price);
        } elseif (!empty($item->best_bid) && (float)$item->best_bid > 0) {
            $sale_price = (float)$item->best_bid;
            delcampe_log('[Order Import] Using best_bid: ' . $sale_price);
        } elseif (!empty($item->price) && (float)$item->price > 0) {
            $sale_price = (float)$item->price;
            delcampe_log('[Order Import] Using price: ' . $sale_price);
        } elseif (!empty($item->price_start) && (float)$item->price_start > 0) {
            // Use starting price as fallback for fixed price items
            $sale_price = (float)$item->price_start;
            delcampe_log('[Order Import] Using price_start as fallback: ' . $sale_price);
        }
        
        // Log if no price found
        if ($sale_price <= 0) {
            delcampe_log('[Order Import] WARNING: No valid price found for item ' . $delcampe_order_id);
            
            // Try to get price from buyer info if available
            if (isset($item->buyer->price_paid) && (float)$item->buyer->price_paid > 0) {
                $sale_price = (float)$item->buyer->price_paid;
                delcampe_log('[Order Import] Using buyer price_paid: ' . $sale_price);
            } else {
                // Check if we have it in the database from a previous webhook
                global $wpdb;
                $db_price = $wpdb->get_var($wpdb->prepare(
                    "SELECT sale_price FROM {$this->table_name} WHERE delcampe_order_id = %s",
                    $delcampe_order_id
                ));
                if ($db_price && (float)$db_price > 0) {
                    $sale_price = (float)$db_price;
                    delcampe_log('[Order Import] Using price from database: ' . $sale_price);
                } else {
                    // Use a default price of 0.01 to prevent $0 orders
                    // This will need to be manually corrected
                    $sale_price = 0.01;
                    delcampe_log('[Order Import] No price found - using 0.01 as placeholder');
                }
            }
        }
        
        // Create WooCommerce order
        $order_data = array(
            'status' => $status, // Use provided status
            'customer_id' => $customer_id,
            'created_via' => 'delcampe', // Changed to match WP-Lister pattern
            'currency' => (string)$item->currency,
        );
        
        $order = wc_create_order($order_data);
        
        if (is_wp_error($order)) {
            delcampe_log('[Delcampe Order Manager] Failed to create WooCommerce order: ' . $order->get_error_message());
            return $order;
        }
        
        // Add product to order and handle inventory
        $product_added = false;
        $quantity_sold = 1; // Default quantity
        
        if ($wc_product_id) {
            $product = wc_get_product($wc_product_id);
            if ($product) {
                // Add product to order
                $order->add_product($product, $quantity_sold, array(
                    'subtotal' => $sale_price,
                    'total' => $sale_price
                ));
                $product_added = true;
                
                // Handle inventory reduction (v1.5.0.0)
                if ($product->managing_stock()) {
                    $old_stock = $product->get_stock_quantity();
                    $new_stock = wc_update_product_stock($product, $quantity_sold, 'decrease');
                    
                    // Log the stock reduction
                    error_log(sprintf(
                        '[Delcampe Order Manager v1.5.0.0] Reduced stock for product #%d (%s) from %d to %d (Delcampe order #%s)',
                        $product->get_id(),
                        $product->get_name(),
                        $old_stock,
                        $new_stock,
                        $delcampe_order_id
                    ));
                    
                    // If inventory manager is available, log the sync
                    if (class_exists('Delcampe_Inventory_Manager')) {
                        do_action('delcampe_order_stock_reduced', $product, $quantity_sold, $order->get_id(), $delcampe_order_id);
                    }
                }
            }
        }
        
        // If product not found, add as custom line item
        if (!$product_added) {
            $item_name = (string)$item->title;
            if (!empty($item->subtitle)) {
                $item_name .= ' - ' . (string)$item->subtitle;
            }
            
            $line_item = new WC_Order_Item_Product();
            $line_item->set_name($item_name);
            $line_item->set_quantity($quantity_sold);
            $line_item->set_subtotal($sale_price);
            $line_item->set_total($sale_price);
            $order->add_item($line_item);
        }
        
        // Set billing address
        $order->set_billing_first_name($buyer_data['firstname']);
        $order->set_billing_last_name($buyer_data['surname']);
        $order->set_billing_email($buyer_data['email']); // Keep Delcampe forwarding emails
        $order->set_billing_phone($buyer_data['phone']);
        $order->set_billing_address_1($buyer_data['address']);
        $order->set_billing_city($buyer_data['city']);
        $order->set_billing_state($buyer_data['state']);
        $order->set_billing_postcode($buyer_data['zipcode']);
        $order->set_billing_country($buyer_data['country']);
        
        // Set shipping address (same as billing)
        $order->set_shipping_first_name($buyer_data['firstname']);
        $order->set_shipping_last_name($buyer_data['surname']);
        $order->set_shipping_address_1($buyer_data['address']);
        $order->set_shipping_city($buyer_data['city']);
        $order->set_shipping_state($buyer_data['state']);
        $order->set_shipping_postcode($buyer_data['zipcode']);
        $order->set_shipping_country($buyer_data['country']);
        
        // Check for shipping information in the item data
        // Note: Delcampe usually provides shipping in payment notification, not item data
        // This is a fallback in case shipping data is available in the item
        if (isset($item->shipping_cost) && (float)$item->shipping_cost > 0) {
            $shipping_cost = (float)$item->shipping_cost;
            
            // Add shipping to order
            $shipping_item = new WC_Order_Item_Shipping();
            $shipping_item->set_method_title(
                isset($item->shipping_method) 
                    ? 'Delcampe: ' . (string)$item->shipping_method
                    : 'Delcampe Shipping'
            );
            $shipping_item->set_method_id('delcampe_shipping');
            $shipping_item->set_total($shipping_cost);
            $order->add_item($shipping_item);
            
            delcampe_log('[Delcampe Order Manager] Added shipping cost: ' . $shipping_cost);
        }
        
        // Add order meta
        $order->add_meta_data('_delcampe_order_id', $delcampe_order_id);
        $order->add_meta_data('_delcampe_item_id', $item_id); // Store actual item ID separately
        $order->add_meta_data('_delcampe_buyer_id', $buyer_data['id']);
        $order->add_meta_data('_delcampe_buyer_nickname', $buyer_data['nickname']);
        
        // Store shipping method if available
        if (isset($item->shipping_method)) {
            $order->add_meta_data('_delcampe_shipping_method', (string)$item->shipping_method);
        }
        
        // Mark that stock was reduced during import (v1.5.0.0)
        $order->add_meta_data('_delcampe_stock_reduced', 'yes');
        
        // Set order source attribution (v1.10.12.7)
        $order->add_meta_data('_wc_order_attribution_source_type', 'utm');
        $order->add_meta_data('_wc_order_attribution_utm_source', 'Delcampe');
        $order->add_meta_data('_wc_order_attribution_utm_medium', 'marketplace');
        $order->add_meta_data('_wc_order_attribution_utm_campaign', 'delcampe_sync');
        
        // Use proper reconciliation for initial sale
        require_once DWC_PLUGIN_DIR . '/includes/class-delcampe-order-reconciliation.php';
        Delcampe_Order_Reconciliation::reconcile_initial_sale($order, $item, $buyer_data);
        
        // Calculate order total (v1.10.13.19)
        // Ensure totals are properly set for email templates
        $total = $sale_price;
        if (isset($shipping_cost) && $shipping_cost > 0) {
            $total += $shipping_cost;
        }
        
        // Set the subtotal and total explicitly
        $order->set_subtotal($sale_price);
        $order->set_total($total);
        
        // Calculate totals WITHOUT tax (Delcampe is tax-exempt)
        $order->calculate_totals(false);
        
        // Double-check the total is not zero
        if ($order->get_total() <= 0 && $total > 0) {
            $order->set_total($total);
            delcampe_log('[Order Import] Manually set order total to: ' . $total);
        }
        
        $order->save();
        
        // Add order notes similar to eBay/WPLister (v1.10.12.7)
        // Note 1: Import notification with source
        $source_text = '';
        if ($source === 'webhook') {
            $source_text = ' ' . __('(via webhook - real-time)', 'wc-delcampe-integration');
        } elseif ($source === 'cron') {
            $source_text = ' ' . __('(via scheduled import)', 'wc-delcampe-integration');
        } else {
            $source_text = ' ' . __('(manual import)', 'wc-delcampe-integration');
        }
        
        $import_note = sprintf(
            __('Delcampe order imported successfully%s', 'wc-delcampe-integration') . "\n" .
            __('Delcampe ID: %s', 'wc-delcampe-integration') . "\n" .
            __('Buyer: %s', 'wc-delcampe-integration') . "\n" .
            __('Amount: %s %s', 'wc-delcampe-integration'),
            $source_text,
            $delcampe_order_id,
            $buyer_data['nickname'],
            number_format($sale_price, 2),
            (string)$item->currency
        );
        $order->add_order_note($import_note, false, false);
        
        // Note 2: Stock reduction (if applicable)
        if ($product_added && $wc_product_id) {
            $product = wc_get_product($wc_product_id);
            if ($product && $product->managing_stock()) {
                $stock_note = sprintf(
                    __('Stock reduced for "%s" (SKU: %s)', 'wc-delcampe-integration') . "\n" .
                    __('Quantity sold: %d', 'wc-delcampe-integration'),
                    $product->get_name(),
                    $product->get_sku() ?: 'N/A',
                    $quantity_sold
                );
                $order->add_order_note($stock_note, false, false);
            }
        }
        
        // Note 3: Payment status
        if ($status === 'processing') {
            $payment_note = __('Awaiting payment confirmation from Delcampe', 'wc-delcampe-integration');
            $order->add_order_note($payment_note, false, false);
        } elseif ($status === 'completed') {
            $payment_note = __('Payment confirmed via Delcampe', 'wc-delcampe-integration');
            $order->add_order_note($payment_note, false, false);
        }
        
        // Save to sync table with proper error handling
        $sync_result = $wpdb->insert($this->table_name, array(
            'delcampe_order_id' => $delcampe_order_id,
            'delcampe_item_id' => $item_id,  // Fixed: Use actual item_id not composite order_id
            'wc_order_id' => $order->get_id(),
            'wc_product_id' => $wc_product_id,
            'buyer_id' => $buyer_data['id'],
            'buyer_nickname' => $buyer_data['nickname'],
            'buyer_email' => $buyer_data['email'],
            'sale_price' => $sale_price > 0 ? $sale_price : $order->get_total(), // Use order total if sale_price is 0
            'currency' => (string)$item->currency,
            'quantity' => $quantity_sold,
            'sale_date' => current_time('mysql'),
            'sync_status' => 'synced',
            'last_sync' => current_time('mysql'),
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        ));
        
        // Check if insert failed and log the error
        if ($sync_result === false) {
            delcampe_log('[ERROR] Failed to save order to sync table: ' . $wpdb->last_error);
            delcampe_log('[ERROR] Order data: Delcampe #' . $delcampe_order_id . ' -> WC #' . $order->get_id());
            
            // Add error note to order
            $order->add_order_note(
                __('WARNING: Failed to save to Delcampe sync table. This may cause duplicate imports.', 'wc-delcampe-integration') . 
                "\nError: " . $wpdb->last_error,
                false,
                false
            );
            
            // Try to prevent duplicate by setting a transient (4 hours per production fix)
            set_transient('delcampe_order_' . $delcampe_order_id, $order->get_id(), 4 * HOUR_IN_SECONDS);
        } else {
            // Set transient to prevent duplicate imports (4 hours per production fix)
            set_transient('delcampe_imported_' . $delcampe_order_id, $order->get_id(), 4 * HOUR_IN_SECONDS);
            delcampe_log('[IMPORT SUCCESS] Created order #' . $order->get_id() . ' for ' . $delcampe_order_id, 'info');
        }
        
        // Set flag to prevent re-sync to Delcampe (v1.10.12.0)
        set_transient('delcampe_importing_order', true, 60);
        
        // Log successful import
        delcampe_log('[Delcampe Order Manager v1.5.0.0] Imported order: Delcampe #' . $delcampe_order_id . ' -> WC #' . $order->get_id());
        
        // Trigger action for other plugins to hook into (v1.5.0.0)
        do_action('delcampe_order_imported', $order->get_id(), $delcampe_order_id, $wc_product_id);
        
        // Trigger business event logging with additional data (v1.10.12.0)
        do_action('delcampe_order_imported_with_details', $order->get_id(), array(
            'id_order' => $delcampe_order_id,
            'item_id' => $delcampe_order_id,
            'buyer_pseudo' => $buyer_data['nickname'],
            'quantity' => $quantity_sold,
            'price' => $sale_price
        ));
        
        return $order->get_id();
    }
    
    /**
     * Add order to sync table retroactively
     * Used when order exists in WC but not in sync table
     * 
     * @param string $delcampe_order_id
     * @param int $wc_order_id
     * @param object $item
     * @param array $buyer_data
     * @since 1.10.13.27
     */
    private function add_to_sync_table_retroactively($delcampe_order_id, $wc_order_id, $item, $buyer_data) {
        global $wpdb;
        
        $order = wc_get_order($wc_order_id);
        if (!$order) {
            return;
        }
        
        // Extract item ID from composite order ID
        $item_id = explode('-', $delcampe_order_id)[0];
        
        // Get product ID if possible
        $items = $order->get_items();
        $wc_product_id = 0;
        if (!empty($items)) {
            $first_item = reset($items);
            $wc_product_id = $first_item->get_product_id();
        }
        
        // Insert into sync table
        $wpdb->insert($this->table_name, array(
            'delcampe_order_id' => $delcampe_order_id,
            'delcampe_item_id' => $item_id,
            'wc_order_id' => $wc_order_id,
            'wc_product_id' => $wc_product_id,
            'buyer_id' => isset($buyer_data['id']) ? $buyer_data['id'] : '0',
            'buyer_nickname' => isset($buyer_data['nickname']) ? $buyer_data['nickname'] : '',
            'buyer_email' => isset($buyer_data['email']) ? $buyer_data['email'] : $order->get_billing_email(),
            'sale_price' => $order->get_total(),
            'currency' => $order->get_currency(),
            'quantity' => 1,
            'sale_date' => $order->get_date_created()->format('Y-m-d H:i:s'),
            'sync_status' => 'synced',
            'last_sync' => current_time('mysql'),
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        ));
        
        if ($wpdb->insert_id) {
            delcampe_log('[Order Import] Retroactively added order to sync table: ' . $delcampe_order_id);
        }
    }
    
    /**
     * Create or get WooCommerce customer
     * 
     * @param array $buyer_data Buyer information from Delcampe
     * @return int Customer ID
     * 
     * @since 1.4.0.0
     */
    private function create_or_get_customer($buyer_data) {
        // Check if customer exists by email
        $user = get_user_by('email', $buyer_data['email']);
        
        if ($user) {
            // Update customer meta with Delcampe info
            update_user_meta($user->ID, 'delcampe_buyer_id', $buyer_data['id']);
            update_user_meta($user->ID, 'delcampe_nickname', $buyer_data['nickname']);
            return $user->ID;
        }
        
        // Create new customer
        $username = sanitize_user($buyer_data['nickname']);
        $counter = 1;
        
        // Ensure unique username
        while (username_exists($username)) {
            $username = sanitize_user($buyer_data['nickname'] . '_' . $counter);
            $counter++;
        }
        
        $customer_data = array(
            'user_login' => $username,
            'user_email' => $buyer_data['email'],
            'first_name' => $buyer_data['firstname'],
            'last_name' => $buyer_data['surname'],
            'display_name' => $buyer_data['firstname'] . ' ' . $buyer_data['surname'],
            'role' => 'customer'
        );
        
        $customer_id = wp_insert_user($customer_data);
        
        if (!is_wp_error($customer_id)) {
            // Add customer meta
            update_user_meta($customer_id, 'delcampe_buyer_id', $buyer_data['id']);
            update_user_meta($customer_id, 'delcampe_nickname', $buyer_data['nickname']);
            
            // Add billing info
            update_user_meta($customer_id, 'billing_first_name', $buyer_data['firstname']);
            update_user_meta($customer_id, 'billing_last_name', $buyer_data['surname']);
            update_user_meta($customer_id, 'billing_email', $buyer_data['email']);
            update_user_meta($customer_id, 'billing_phone', $buyer_data['phone']);
            update_user_meta($customer_id, 'billing_address_1', $buyer_data['address']);
            update_user_meta($customer_id, 'billing_city', $buyer_data['city']);
            update_user_meta($customer_id, 'billing_state', $buyer_data['state']);
            update_user_meta($customer_id, 'billing_postcode', $buyer_data['zipcode']);
            update_user_meta($customer_id, 'billing_country', $buyer_data['country']);
            
            delcampe_log('[Delcampe Order Manager v1.5.0.0] Created new customer: ' . $username);
        }
        
        return $customer_id;
    }
    
    /**
     * Find WooCommerce product by Delcampe ID
     * 
     * @param string $delcampe_id Delcampe item ID
     * @return int|null Product ID or null
     * 
     * @since 1.4.0.0
     */
    private function find_wc_product_by_delcampe_id($delcampe_id) {
        global $wpdb;
        
        $product_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} 
            WHERE meta_key = '_delcampe_item_id' AND meta_value = %s
            LIMIT 1",
            $delcampe_id
        ));
        
        return $product_id ? (int)$product_id : null;
    }
    
    /**
     * Find WooCommerce product by SKU
     * 
     * @param string $sku Product SKU
     * @return int|null Product ID or null
     * 
     * @since 1.4.0.0
     */
    private function find_wc_product_by_sku($sku) {
        global $wpdb;
        
        $product_id = $wpdb->get_var($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} 
            WHERE meta_key = '_sku' AND meta_value = %s
            LIMIT 1",
            $sku
        ));
        
        return $product_id ? (int)$product_id : null;
    }
    
    /**
     * Sync order status to Delcampe
     * 
     * @param int $order_id WooCommerce order ID
     * @param string $old_status Old order status
     * @param string $new_status New order status
     * 
     * @since 1.4.0.0
     */
    public function sync_order_status_to_delcampe($order_id, $old_status, $new_status) {
        // Check if this is a Delcampe order
        $order = wc_get_order($order_id);
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        
        if (!$delcampe_order_id) {
            return; // Not a Delcampe order
        }
        
        // Map WooCommerce status to Delcampe actions
        switch ($new_status) {
            case 'completed':
                // Mark as shipped on Delcampe
                $tracking = $order->get_meta('_tracking_number');
                if ($tracking) {
                    $this->update_delcampe_shipping($delcampe_order_id, $tracking);
                }
                break;
                
            case 'refunded':
                // Handle refund on Delcampe
                $this->process_delcampe_refund($delcampe_order_id, $order);
                break;
        }
        
        // Sync the status to our database table
        $this->sync_order_status_from_wc($order_id);
    }
    
    /**
     * Update shipping information on Delcampe
     * 
     * @param string $delcampe_order_id Delcampe order ID
     * @param string $tracking_number Tracking number
     * @return bool|WP_Error Success or error
     * 
     * @since 1.4.0.0
     */
    private function update_delcampe_shipping($delcampe_order_id, $tracking_number) {
        // This would need to be implemented based on Delcampe's shipping update API
        // Currently placeholder for future implementation
        delcampe_log('[Delcampe Order Manager] Shipping update for order ' . $delcampe_order_id . ' with tracking ' . $tracking_number);
        return true;
    }
    
    /**
     * Process refund on Delcampe
     * 
     * @param string $delcampe_order_id Delcampe order ID
     * @param WC_Order $order WooCommerce order
     * @return bool|WP_Error Success or error
     * 
     * @since 1.4.0.0
     */
    private function process_delcampe_refund($delcampe_order_id, $order) {
        // This would need to be implemented based on Delcampe's refund API
        // Currently placeholder for future implementation
        delcampe_log('[Delcampe Order Manager] Refund processing for order ' . $delcampe_order_id);
        return true;
    }
    
    /**
     * Cron job to import orders
     * 
     * @since 1.4.0.0
     * @version 1.10.2.0 - Added closed sales import
     */
    public function import_orders_cron() {
        // Check if auto-import is enabled
        $auto_import = get_option('delcampe_auto_import_orders', 'yes');
        if ($auto_import !== 'yes') {
            return;
        }
        
        // Import recent sold orders
        $this->import_orders(0, 50, 'cron');
        
        // Import closed sales
        $this->import_closed_sales(0, 50, 'cron');
        
        // Update status of pending orders
        $update_result = $this->update_pending_orders_from_delcampe();
        
        // Log the update results
        $logger = Delcampe_Sync_Logger::get_instance();
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'CRON_PENDING_ORDERS_UPDATE',
            'result' => $update_result
        ));
    }
    
    /**
     * Create order from webhook notification
     * 
     * Special method for webhook-triggered order creation
     * Ensures no duplicate orders and proper error handling
     * 
     * @param string $item_id Delcampe item ID
     * @param SimpleXMLElement $item_data Full item data from API
     * @return int|WP_Error Order ID or error
     * 
     * @since 1.10.5.15
     */
    public function create_order_from_webhook($item_id, $item_data) {
        global $wpdb;
        
        // Double-check if order already exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT wc_order_id FROM {$this->table_name} WHERE delcampe_order_id = %s",
            $item_id
        ));
        
        if ($existing) {
            // Order already exists, just return the ID
            delcampe_log('[Delcampe Order Manager] Order already exists for item ' . $item_id . ', WC Order ID: ' . $existing);
            return (int) $existing;
        }
        
        // Use the existing import_single_order method with webhook source
        $order_id = $this->import_single_order($item_data, 'processing', 'webhook');
        
        return $order_id;
    }
    
    /**
     * Add order meta box
     * 
     * @param string $post_type Post type
     * @param WP_Post $post Post object
     * 
     * @since 1.4.0.0
     */
    public function add_order_meta_box($post_type, $post) {
        // Handle both legacy and HPOS
        if ($post_type === 'shop_order') {
            // Legacy post-based orders
            $order = wc_get_order($post->ID);
        } elseif ($post_type === 'woocommerce_page_wc-orders' && isset($_GET['id'])) {
            // HPOS orders
            $order = wc_get_order($_GET['id']);
        } else {
            return;
        }
        
        // Check if this is a Delcampe order
        if (!$order || !$order->get_meta('_delcampe_order_id')) {
            return;
        }
        
        // Add meta box based on storage type
        $screen = class_exists('\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController') 
            && wc_get_container()->get(\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class)->custom_orders_table_usage_is_enabled()
            ? wc_get_page_screen_id('shop-order')
            : 'shop_order';
            
        add_meta_box(
            'delcampe_order_info',
            __('Delcampe Order Information', 'wc-delcampe-integration'),
            array($this, 'render_order_meta_box'),
            $screen,
            'side',
            'high'
        );
    }
    
    /**
     * Add HPOS order meta box
     * 
     * @param int $order_id Order ID
     * @param WC_Order $order Order object
     * 
     * @since 1.8.0.0
     */
    public function add_hpos_order_meta_box($order_id, $order) {
        // Check if this is a Delcampe order
        if (!$order || !$order->get_meta('_delcampe_order_id')) {
            return;
        }
        
        // For HPOS, we need to render the meta box content directly
        echo '<div class="postbox">';
        echo '<h2 class="hndle">' . __('Delcampe Order Information', 'wc-delcampe-integration') . '</h2>';
        echo '<div class="inside">';
        $this->render_order_meta_box($order);
        echo '</div></div>';
    }
    
    /**
     * Render order meta box
     * 
     * @param WP_Post|WC_Order $post_or_order Post object or Order object
     * 
     * @since 1.4.0.0
     */
    public function render_order_meta_box($post_or_order) {
        // Handle both legacy and HPOS
        if (is_a($post_or_order, 'WP_Post')) {
            $order = wc_get_order($post_or_order->ID);
            $order_id = $post_or_order->ID;
        } else {
            $order = $post_or_order;
            $order_id = $order->get_id();
        }
        
        if (!$order) {
            return;
        }
        
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        $delcampe_buyer_id = $order->get_meta('_delcampe_buyer_id');
        $delcampe_nickname = $order->get_meta('_delcampe_buyer_nickname');
        
        ?>
        <p>
            <strong><?php _e('Delcampe Order ID:', 'wc-delcampe-integration'); ?></strong><br>
            <?php echo esc_html($delcampe_order_id); ?>
        </p>
        <p>
            <strong><?php _e('Buyer ID:', 'wc-delcampe-integration'); ?></strong><br>
            <?php echo esc_html($delcampe_buyer_id); ?>
        </p>
        <p>
            <strong><?php _e('Buyer Nickname:', 'wc-delcampe-integration'); ?></strong><br>
            <?php echo esc_html($delcampe_nickname); ?>
        </p>
        <?php
        
        // Show sync status
        global $wpdb;
        $sync_data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE wc_order_id = %d",
            $order_id
        ));
        
        if ($sync_data) {
            ?>
            <p>
                <strong><?php _e('Sync Status:', 'wc-delcampe-integration'); ?></strong><br>
                <?php echo esc_html(ucfirst($sync_data->sync_status)); ?>
            </p>
            <p>
                <strong><?php _e('Last Sync:', 'wc-delcampe-integration'); ?></strong><br>
                <?php echo esc_html($sync_data->last_sync); ?>
            </p>
            <?php
        }
    }
    
    /**
     * Get or create a WooCommerce customer
     * 
     * @param array $customer_data Customer data with email, names, etc.
     * @return int Customer ID
     * @since 1.10.27.0
     */
    private function get_or_create_customer($customer_data) {
        $email = isset($customer_data['email']) ? sanitize_email($customer_data['email']) : '';
        
        if (empty($email) || !is_email($email)) {
            // Generate a placeholder email if not provided
            $email = 'delcampe_' . uniqid() . '@delcampe.net';
        }
        
        // Check if customer exists
        $user = get_user_by('email', $email);
        
        if ($user) {
            return $user->ID;
        }
        
        // Create new customer
        $username = isset($customer_data['username']) ? $customer_data['username'] : $email;
        $username = sanitize_user($username);
        
        // Ensure unique username
        if (username_exists($username)) {
            $username = $username . '_' . uniqid();
        }
        
        $customer_id = wc_create_new_customer(
            $email,
            $username,
            wp_generate_password()
        );
        
        if (!is_wp_error($customer_id)) {
            // Update customer metadata
            if (isset($customer_data['first_name'])) {
                update_user_meta($customer_id, 'first_name', $customer_data['first_name']);
            }
            if (isset($customer_data['last_name'])) {
                update_user_meta($customer_id, 'last_name', $customer_data['last_name']);
            }
            
            // Mark as Delcampe customer
            update_user_meta($customer_id, '_is_delcampe_customer', 'yes');
            update_user_meta($customer_id, '_delcampe_buyer_id', $customer_data['buyer_id'] ?? '');
            
            return $customer_id;
        }
        
        // If customer creation failed, return 0 (guest)
        return 0;
    }
    
    /**
     * Create order from Seller_Item_Close_Sold webhook notification
     * This is called when an item is sold in real-time
     * 
     * @param string $item_id Delcampe item ID
     * @param array $buyer_info Buyer information from webhook
     * @param array $sale_info Sale information from webhook
     * @return WC_Order|WP_Error
     * @since 1.10.12.7
     */
    public function create_order_from_close_sold($item_id, $buyer_info, $sale_info) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        try {
            // Generate idempotent order ID
            $buyer_id = isset($buyer_info['id']) ? $buyer_info['id'] : null;
            $bill_id = isset($sale_info['bill_id']) ? $sale_info['bill_id'] : null;
            $delcampe_order_id = $this->generate_idempotent_order_id($item_id, $buyer_id, $bill_id);
            
            // Check for duplicate order using the unique ID
            global $wpdb;
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT wc_order_id FROM {$this->table_name} 
                WHERE delcampe_order_id = %s LIMIT 1",
                $delcampe_order_id
            ));
            
            if ($existing) {
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_ORDER_DUPLICATE',
                    'item_id' => $item_id,
                    'order_id' => $delcampe_order_id,
                    'existing_order' => $existing
                ));
                return new WP_Error('duplicate_order', 'Order already exists: ' . $delcampe_order_id);
            }
            
            // Create or get customer
            $customer_data = array(
                'email' => isset($buyer_info['email']) ? $buyer_info['email'] : 'delcampe_buyer_' . $item_id . '@delcampe.net',
                'first_name' => isset($buyer_info['firstname']) ? $buyer_info['firstname'] : '',
                'last_name' => isset($buyer_info['lastname']) ? $buyer_info['lastname'] : '',
                'username' => isset($buyer_info['nickname']) ? $buyer_info['nickname'] : 'delcampe_' . $item_id
            );
            
            $customer_id = $this->get_or_create_customer($customer_data);
            
            // Create the order
            $order = wc_create_order(array(
                'customer_id' => $customer_id,
                'status' => 'on-hold', // Will be updated to processing when payment is confirmed
                'created_via' => 'delcampe_webhook'
            ));
            
            if (is_wp_error($order)) {
                throw new Exception('Failed to create order: ' . $order->get_error_message());
            }
            
            // Set billing information for better email notifications
            // Use nickname as both first and last name if real names aren't available
            $billing_first = isset($buyer_info['firstname']) && !empty($buyer_info['firstname']) 
                ? $buyer_info['firstname'] 
                : (isset($buyer_info['nickname']) ? $buyer_info['nickname'] : 'Delcampe');
            
            $billing_last = isset($buyer_info['lastname']) && !empty($buyer_info['lastname']) 
                ? $buyer_info['lastname'] 
                : 'Buyer';
            
            $order->set_billing_first_name($billing_first);
            $order->set_billing_last_name($billing_last);
            $order->set_billing_email(isset($buyer_info['email']) ? $buyer_info['email'] : $customer_data['email']);
            
            // Set address fields if available
            if (isset($buyer_info['address'])) {
                $order->set_billing_address_1($buyer_info['address']);
            }
            if (isset($buyer_info['city'])) {
                $order->set_billing_city($buyer_info['city']);
            }
            if (isset($buyer_info['state'])) {
                $order->set_billing_state($buyer_info['state']);
            }
            if (isset($buyer_info['zipcode'])) {
                $order->set_billing_postcode($buyer_info['zipcode']);
            }
            if (isset($buyer_info['country'])) {
                $order->set_billing_country($buyer_info['country']);
            }
            if (isset($buyer_info['phone'])) {
                $order->set_billing_phone($buyer_info['phone']);
            }
            
            // Find product by SKU if provided
            $product = null;
            if (isset($sale_info['sku']) && !empty($sale_info['sku'])) {
                $product_id = wc_get_product_id_by_sku($sale_info['sku']);
                if ($product_id) {
                    $product = wc_get_product($product_id);
                }
            }
            
            // Add line item
            if ($product) {
                // Use actual product with proper price
                $item = new WC_Order_Item_Product();
                $item->set_product($product);
                $item->set_quantity(1);
                
                // Use the sale price from Delcampe if available, otherwise use product price
                $price = 0;
                if (isset($sale_info['price'])) {
                    $raw = (string) $sale_info['price'];
                    $num = preg_replace('/[^0-9.,]/', '', $raw);
                    $num = str_replace(',', '.', $num);
                    $price = (float) $num;
                }
                if ($price <= 0) {
                    $price = (float) $product->get_price();
                }
                $used_placeholder_price = false;
                if ($price <= 0) {
                    // Prevent zero-value line items; placeholder until payment webhook corrects totals
                    $price = 0.01;
                    $used_placeholder_price = true;
                }
                
                $item->set_subtotal($price);
                $item->set_total($price);
                $order->add_item($item);
                
                // Add order note about product match
                $order->add_order_note(sprintf(
                    __('Delcampe order imported via webhook (real-time)', 'wc-delcampe-integration') . "\n" .
                    __('Item ID: %s', 'wc-delcampe-integration') . "\n" .
                    __('Matched product: %s (SKU: %s)', 'wc-delcampe-integration') . "\n" .
                    __('Sale price: %s %s', 'wc-delcampe-integration'),
                    $item_id,
                    $product->get_name(),
                    $sale_info['sku'],
                    number_format($price, 2),
                    isset($sale_info['currency']) ? $sale_info['currency'] : get_woocommerce_currency()
                ));
                if ($used_placeholder_price) {
                    $order->add_order_note(__('Price not provided in close_sold webhook; using temporary 0.01 until payment notification updates totals.', 'wc-delcampe-integration'));
                }
            } else {
                // Create placeholder line item
                $item = new WC_Order_Item_Product();
                $item->set_name(isset($sale_info['title']) ? $sale_info['title'] : 'Delcampe Item #' . $item_id);
                $item->set_quantity(1);
                
                // Normalize/derive price
                $price = 0;
                if (isset($sale_info['price'])) {
                    $raw = (string) $sale_info['price'];
                    $num = preg_replace('/[^0-9.,]/', '', $raw);
                    $num = str_replace(',', '.', $num);
                    $price = (float) $num;
                }
                if ($price <= 0 && isset($sale_info['sku']) && !empty($sale_info['sku'])) {
                    // Try product by SKU to get a sensible price
                    $pid = wc_get_product_id_by_sku($sale_info['sku']);
                    if ($pid) {
                        $p = wc_get_product($pid);
                        if ($p) {
                            $price = (float) $p->get_price();
                        }
                    }
                }
                $used_placeholder_price_2 = false;
                if ($price <= 0) {
                    $price = 0.01; // final fallback to avoid $0 orders
                    $used_placeholder_price_2 = true;
                }
                $item->set_subtotal($price);
                $item->set_total($price);
                $order->add_item($item);
                
                // Add order note about missing product
                $order->add_order_note(sprintf(
                    __('Delcampe order imported via webhook (real-time)', 'wc-delcampe-integration') . "\n" .
                    __('Item ID: %s', 'wc-delcampe-integration') . "\n" .
                    __('Note: Product with SKU %s not found in WooCommerce', 'wc-delcampe-integration') . "\n" .
                    __('Sale price: %s %s', 'wc-delcampe-integration'),
                    $item_id,
                    isset($sale_info['sku']) ? $sale_info['sku'] : 'N/A',
                    number_format($price, 2),
                    isset($sale_info['currency']) ? $sale_info['currency'] : get_woocommerce_currency()
                ));
                if ($used_placeholder_price_2) {
                    $order->add_order_note(__('Price not provided in close_sold webhook; using temporary 0.01 until payment notification updates totals.', 'wc-delcampe-integration'));
                }
            }
            
            // Set order metadata
            $order->update_meta_data('_delcampe_order_id', $item_id);
            $order->update_meta_data('_delcampe_buyer_id', isset($buyer_info['id']) ? $buyer_info['id'] : '');
            $order->update_meta_data('_delcampe_buyer_nickname', isset($buyer_info['nickname']) ? $buyer_info['nickname'] : '');
            $order->update_meta_data('_delcampe_import_source', 'webhook_realtime');
            $order->update_meta_data('_delcampe_sale_date', isset($sale_info['date']) ? $sale_info['date'] : current_time('mysql'));
            
            // Set source attribution
            $order->update_meta_data('_wc_order_attribution_utm_source', 'Delcampe');
            
            // Set currency if provided
            if (isset($sale_info['currency'])) {
                $order->set_currency($sale_info['currency']);
            }
            
            // CRITICAL: Disable tax calculation for Delcampe orders
            // Delcampe doesn't collect taxes - platform fees are paid by buyer
            $order->set_prices_include_tax(false);
            $order->set_cart_tax(0);
            $order->set_shipping_tax(0);
            
            // Calculate totals WITHOUT applying tax
            $order->calculate_totals(false); // false = don't recalculate taxes
            
            // Force remove any auto-added taxes
            foreach ($order->get_items('tax') as $tax_item_id => $tax_item) {
                $order->remove_item($tax_item_id);
            }
            
            // Use proper reconciliation system
            require_once DWC_PLUGIN_DIR . '/includes/class-delcampe-order-reconciliation.php';
            
            // For webhook orders, we're at initial sale stage
            $reconciliation_data = new stdClass();
            
            // CRITICAL FIX (v1.10.36.2): Extract price properly from sale_info
            // The price might be 0 or missing in Seller_Item_Close_Sold webhook
            $extracted_price = 0;
            if (isset($sale_info['price']) && $sale_info['price'] > 0) {
                $extracted_price = $sale_info['price'];
            } elseif ($product) {
                // Fallback to product price if webhook didn't provide price
                $extracted_price = (float) $product->get_price();
            }
            
            $reconciliation_data->fixed_price = $extracted_price;
            $reconciliation_data->currency = isset($sale_info['currency']) ? $sale_info['currency'] : 'USD';
            $reconciliation_data->quantity_sold = isset($sale_info['quantity']) ? $sale_info['quantity'] : 1;
            
            // Log if price is still zero after extraction attempts
            if ($extracted_price <= 0) {
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_ORDER_ZERO_PRICE',
                    'item_id' => $item_id,
                    'sale_info' => $sale_info,
                    'message' => 'Price not available in Seller_Item_Close_Sold webhook - awaiting Seller_Payment_Received'
                ));
            }
            
            Delcampe_Order_Reconciliation::reconcile_initial_sale($order, $reconciliation_data, $buyer_info);
            
            // Recalculate WITHOUT taxes (Delcampe is tax-exempt)
            $order->calculate_totals(false);

            // Safety: ensure total reflects at least the item total if still zero
            if ($order->get_total() <= 0) {
                $sum = 0;
                foreach ($order->get_items('line_item') as $li) { $sum += (float) $li->get_total(); }
                if ($sum > 0) {
                    $order->set_total($sum);
                }
            }
            
            // Save order
            $order->save();
            
            // Store unique order ID on WooCommerce order
            // v1.10.36.4: Use update_meta_data() for HPOS compatibility
            $order->update_meta_data('_delcampe_order_id', $delcampe_order_id);
            $order->update_meta_data('_delcampe_item_id', $item_id);
            $order->save();
            
            // Double-check meta was saved (HPOS compatibility)
            if (!$order->get_meta('_delcampe_item_id')) {
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_ORDER_META_SAVE_FAILED',
                    'order_id' => $order->get_id(),
                    'item_id' => $item_id,
                    'warning' => 'Failed to save _delcampe_item_id meta - will use table fallback'
                ));
            }
            
            // Record in sync table with both unique order ID and item ID
            $wpdb->insert(
                $this->table_name,
                array(
                    'delcampe_order_id' => $delcampe_order_id,  // Unique composite ID
                    'delcampe_item_id' => $item_id,              // Actual item ID
                    'wc_order_id' => $order->get_id(),
                    'sale_price' => isset($sale_info['price']) ? $sale_info['price'] : 0,
                    'buyer_nickname' => isset($buyer_info['nickname']) ? $buyer_info['nickname'] : '',
                    'buyer_email' => isset($buyer_info['email']) ? $buyer_info['email'] : $customer_data['email'],
                    'payment_status' => 'pending', // Will be updated when payment notification arrives
                    'sync_status' => 'synced',
                    'last_sync' => current_time('mysql')
                ),
                array('%s', '%d', '%d', '%f', '%s', '%s', '%s', '%s', '%s')
            );
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_ORDER_CREATED',
                'item_id' => $item_id,
                'wc_order_id' => $order->get_id(),
                'buyer' => isset($buyer_info['nickname']) ? $buyer_info['nickname'] : 'Unknown',
                'amount' => isset($sale_info['price']) ? $sale_info['price'] : 0
            ));
            
            return $order;
            
        } catch (Exception $e) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_ORDER_CREATE_ERROR',
                'item_id' => $item_id,
                'error' => $e->getMessage()
            ));
            
            return new WP_Error('order_creation_failed', $e->getMessage());
        }
    }
    
    /**
     * Finalize order when payment is received
     * Updates order with final totals, shipping, and payment details
     * 
     * @param int $order_id WooCommerce order ID
     * @param array $payment_info Payment information from webhook
     * @param array $paypal_info Optional PayPal transaction details
     * @return WC_Order|WP_Error
     * @since 1.10.13.3
     */
    public function finalize_order_on_payment($order_id, $payment_info, $paypal_info = array()) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        try {
            $order = wc_get_order($order_id);
            if (!$order) {
                throw new Exception('Order not found: ' . $order_id);
            }
            
            // Check for duplicate payment processing using bill_id
            if (isset($payment_info['bill_id'])) {
                $existing_bill = $order->get_meta('_delcampe_bill_id');
                if ($existing_bill && $existing_bill === $payment_info['bill_id']) {
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'PAYMENT_ALREADY_PROCESSED',
                        'order_id' => $order_id,
                        'bill_id' => $payment_info['bill_id']
                    ));
                    return $order; // Already processed
                }
                
                // Store bill_id as unique payment identifier
                $order->update_meta_data('_delcampe_bill_id', $payment_info['bill_id']);
            }
            
            // Get current items total
            $items_total = 0;
            foreach ($order->get_items() as $item) {
                $items_total += $item->get_total();
            }
            
            // Calculate fees and shipping from actual payment data
            $buyer_paid_total = isset($paypal_info['gross']) ? $paypal_info['gross'] : $payment_info['total_amount'];
            $seller_receives = isset($paypal_info['net']) ? $paypal_info['net'] : $payment_info['seller_amount'];
            $shipping_amount = isset($paypal_info['shipping']) ? $paypal_info['shipping'] : 0;
            
            // If no explicit shipping from PayPal, calculate it
            if (!$shipping_amount && isset($payment_info['seller_amount'])) {
                // Seller receives = items + shipping (no platform fee)
                // So shipping = seller_amount - items_total
                $shipping_amount = max(0, $payment_info['seller_amount'] - $items_total);
            }
            
            // Calculate all fees
            $platform_fee = 0;
            $paypal_fee = 0;
            
            if (isset($payment_info['total_amount']) && isset($payment_info['seller_amount'])) {
                // Platform fee = what buyer pays minus what seller gets (before PayPal)
                $platform_fee = $payment_info['total_amount'] - $payment_info['seller_amount'];
            }
            
            if (isset($paypal_info['fee'])) {
                $paypal_fee = $paypal_info['fee'];
            }
            
            // Clear any existing shipping and taxes
            foreach ($order->get_items('shipping') as $ship_item_id => $ship_item) {
                $order->remove_item($ship_item_id);
            }
            foreach ($order->get_items('tax') as $tax_item_id => $tax_item) {
                $order->remove_item($tax_item_id);
            }
            
            // Add proper shipping if exists
            if ($shipping_amount > 0) {
                $shipping_item = new WC_Order_Item_Shipping();
                $shipping_title = 'Delcampe Shipping';
                if (isset($payment_info['shipping_method']) && !empty($payment_info['shipping_method'])) {
                    $shipping_title .= ' (' . $payment_info['shipping_method'] . ')';
                }
                $shipping_item->set_method_title($shipping_title);
                $shipping_item->set_method_id('delcampe_shipping');
                $shipping_item->set_total($shipping_amount);
                $order->add_item($shipping_item);
            }
            
            // Store all financial details as meta for reporting
            $order->update_meta_data('_delcampe_platform_fee', number_format($platform_fee, 2, '.', ''));
            $order->update_meta_data('_delcampe_buyer_paid_total', number_format($buyer_paid_total, 2, '.', ''));
            $order->update_meta_data('_delcampe_seller_receives', number_format($seller_receives, 2, '.', ''));
            
            if ($paypal_fee > 0) {
                $order->update_meta_data('_paypal_transaction_fee', number_format($paypal_fee, 2, '.', ''));
            }
            
            if (isset($payment_info['transaction_id'])) {
                $order->update_meta_data('_transaction_id', $payment_info['transaction_id']);
            }
            
            if (isset($payment_info['payment_method'])) {
                $order->update_meta_data('_delcampe_payment_method', $payment_info['payment_method']);
            }
            
            // CRITICAL: Ensure NO taxes
            $order->set_cart_tax(0);
            $order->set_shipping_tax(0);
            
            // Recalculate totals (should be items + shipping only)
            $order->calculate_totals(false); // false = don't add taxes
            
            // Add comprehensive financial note
            $financial_note = sprintf(
                "Payment Reconciliation:\n" .
                "=====================================\n" .
                "Bill ID: %s\n" .
                "Items Total: %s %s\n" .
                "Shipping: %s %s\n" .
                "Order Total: %s %s (what you receive)\n" .
                "-------------------------------------\n" .
                "Buyer Paid: %s %s\n" .
                "Platform Fee (to Delcampe): %s %s\n" .
                "PayPal Fee: %s %s\n" .
                "Net to Seller: %s %s\n" .
                "=====================================",
                isset($payment_info['bill_id']) ? $payment_info['bill_id'] : 'N/A',
                number_format($items_total, 2),
                $order->get_currency(),
                number_format($shipping_amount, 2),
                $order->get_currency(),
                number_format($items_total + $shipping_amount, 2),
                $order->get_currency(),
                number_format($buyer_paid_total, 2),
                $order->get_currency(),
                number_format($platform_fee, 2),
                $order->get_currency(),
                number_format($paypal_fee, 2),
                $order->get_currency(),
                number_format($seller_receives, 2),
                $order->get_currency()
            );
            
            $order->add_order_note($financial_note);
            
            // Update order status to processing (paid)
            $order->update_status('processing', __('Payment confirmed via Delcampe webhook', 'wc-delcampe-integration'));
            
            // Save all changes
            $order->save();
            
            // Update payment status in sync table
            global $wpdb;
            $wpdb->update(
                $this->table_name,
                array('payment_status' => 'paid'),
                array('wc_order_id' => $order_id),
                array('%s'),
                array('%d')
            );
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'ORDER_FINALIZED_ON_PAYMENT',
                'order_id' => $order_id,
                'bill_id' => isset($payment_info['bill_id']) ? $payment_info['bill_id'] : 'N/A',
                'items_total' => $items_total,
                'shipping' => $shipping_amount,
                'order_total' => $items_total + $shipping_amount,
                'platform_fee' => $platform_fee,
                'paypal_fee' => $paypal_fee,
                'net_to_seller' => $seller_receives
            ));
            
            return $order;
            
        } catch (Exception $e) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'ORDER_FINALIZE_ERROR',
                'order_id' => $order_id,
                'error' => $e->getMessage()
            ));
            
            return new WP_Error('finalize_failed', $e->getMessage());
        }
    }
    
    /**
     * Sync order status from WooCommerce to Delcampe table
     * Updates payment_status and shipping_status based on WC order status
     * 
     * @param int $wc_order_id WooCommerce order ID
     * @return bool Success status
     * @since 1.10.13.4
     */
    public function sync_order_status_from_wc($wc_order_id) {
        global $wpdb;
        
        $order = wc_get_order($wc_order_id);
        if (!$order) {
            return false;
        }
        
        // Map WooCommerce status to our statuses
        $wc_status = $order->get_status();
        $payment_status = 'pending';
        $shipping_status = 'pending';
        
        switch ($wc_status) {
            case 'pending':
            case 'on-hold':
                $payment_status = 'pending';
                $shipping_status = 'pending';
                break;
                
            case 'processing':
                $payment_status = 'paid';
                $shipping_status = 'pending';
                break;
                
            case 'completed':
                $payment_status = 'paid';
                $shipping_status = 'shipped';
                break;
                
            case 'cancelled':
            case 'refunded':
                $payment_status = 'refunded';
                $shipping_status = 'cancelled';
                break;
                
            case 'failed':
                $payment_status = 'failed';
                $shipping_status = 'failed';
                break;
        }
        
        // Update the delcampe_orders table
        $result = $wpdb->update(
            $this->table_name,
            array(
                'payment_status' => $payment_status,
                'shipping_status' => $shipping_status,
                'last_sync' => current_time('mysql')
            ),
            array('wc_order_id' => $wc_order_id),
            array('%s', '%s', '%s'),
            array('%d')
        );
        
        // Log the sync
        $logger = Delcampe_Sync_Logger::get_instance();
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'ORDER_STATUS_SYNCED',
            'wc_order_id' => $wc_order_id,
            'wc_status' => $wc_status,
            'payment_status' => $payment_status,
            'shipping_status' => $shipping_status
        ));
        
        return $result !== false;
    }
    
    /**
     * Sync all order statuses
     * Updates all Delcampe order records with current WC status
     * 
     * @return array Results array
     * @since 1.10.13.4
     */
    public function sync_all_order_statuses() {
        global $wpdb;
        
        // Get all orders that have WC order IDs
        $orders = $wpdb->get_results(
            "SELECT DISTINCT wc_order_id FROM {$this->table_name} 
            WHERE wc_order_id IS NOT NULL AND wc_order_id > 0"
        );
        
        $synced = 0;
        $failed = 0;
        
        foreach ($orders as $order_row) {
            if ($this->sync_order_status_from_wc($order_row->wc_order_id)) {
                $synced++;
            } else {
                $failed++;
            }
        }
        
        return array(
            'synced' => $synced,
            'failed' => $failed,
            'total' => count($orders)
        );
    }
    
    /**
     * Update pending orders from Delcampe
     * Checks Delcampe for status changes on all pending orders
     * 
     * @return array Results with updated count
     * @since 1.10.13.5
     */
    public function update_pending_orders_from_delcampe() {
        global $wpdb;
        $logger = Delcampe_Sync_Logger::get_instance();
        $auth = Delcampe_Auth::get_instance();
        
        // Get all pending orders
        $pending_orders = $wpdb->get_results(
            "SELECT * FROM {$this->table_name} 
            WHERE (payment_status = 'pending' OR shipping_status = 'pending')
            AND delcampe_order_id IS NOT NULL
            AND delcampe_order_id != '0'"
        );
        
        if (empty($pending_orders)) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'NO_PENDING_ORDERS_TO_UPDATE'
            ));
            return array(
                'checked' => 0,
                'updated' => 0,
                'message' => __('No pending orders to update', 'wc-delcampe-integration')
            );
        }
        
        $checked = 0;
        $updated = 0;
        $errors = 0;
        
        foreach ($pending_orders as $order_record) {
            $checked++;
            
            try {
                // Get auth token
                $token = $auth->get_auth_token();
                if (is_wp_error($token)) {
                    $errors++;
                    continue;
                }
                
                // v1.10.35.17: Use delcampe_item_id not delcampe_order_id for API calls
                // delcampe_order_id is a composite ID, delcampe_item_id is the actual Delcampe item
                $item_id_to_check = $order_record->delcampe_item_id ? $order_record->delcampe_item_id : $order_record->delcampe_order_id;
                $item_url = "https://rest.delcampe.net/item/{$item_id_to_check}?token={$token}";
                $item_response = wp_remote_get($item_url, array(
                    'timeout' => 30,
                    'headers' => array('Accept' => 'application/xml')
                ));
                
                if (is_wp_error($item_response)) {
                    $errors++;
                    // Log the specific error
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'PENDING_ORDER_API_ERROR',
                        'order_id' => $order_record->delcampe_order_id,
                        'error_message' => $item_response->get_error_message(),
                        'error_code' => $item_response->get_error_code(),
                        'error_data' => $item_response->get_error_data()
                    ));
                    continue;
                }
                
                // Check for rate limiting
                $response_code = wp_remote_retrieve_response_code($item_response);
                if ($response_code === 429) {
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'RATE_LIMIT_HIT',
                        'order_id' => $order_record->delcampe_order_id,
                        'retry_after' => wp_remote_retrieve_header($item_response, 'retry-after')
                    ));
                    // Back off exponentially
                    sleep(min(30, pow(2, $errors)));
                    $errors++;
                    continue;
                }
                
                // Log non-200 responses
                if ($response_code !== 200) {
                    $errors++;
                    $body = wp_remote_retrieve_body($item_response);
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'PENDING_ORDER_HTTP_ERROR',
                        'order_id' => $order_record->delcampe_order_id,
                        'http_code' => $response_code,
                        'response_preview' => substr($body, 0, 500)
                    ));
                    continue;
                }
                
                // Parse XML response
                $body = wp_remote_retrieve_body($item_response);
                $xml = simplexml_load_string($body);
                if ($xml === false) {
                    $errors++;
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'PENDING_ORDER_XML_PARSE_ERROR',
                        'order_id' => $order_record->delcampe_order_id,
                        'body_preview' => substr($body, 0, 500)
                    ));
                    continue;
                }
                
                // Navigate to item data
                $item_data = null;
                if (isset($xml->Notification_Data->body->item)) {
                    $item_data = $xml->Notification_Data->body->item;
                } elseif (isset($xml->body->item)) {
                    $item_data = $xml->body->item;
                }
                
                if (!$item_data) {
                    $errors++;
                    continue;
                }
                
                $status_changed = false;
                $new_payment_status = $order_record->payment_status;
                $new_shipping_status = $order_record->shipping_status;
                
                // Check item status
                $item_status = isset($item_data['status']) ? $item_data['status'] : 'open';
                
                // Check in sold items endpoint for payment confirmation
                $sold_url = "https://rest.delcampe.net/item/fixedprice/closed/sold?token={$token}";
                $sold_response = wp_remote_get($sold_url, array(
                    'timeout' => 30,
                    'headers' => array('Accept' => 'application/xml')
                ));
                
                if (!is_wp_error($sold_response)) {
                    $sold_body = wp_remote_retrieve_body($sold_response);
                    $sold_xml = simplexml_load_string($sold_body);
                    
                    $items_array = array();
                    if ($sold_xml && isset($sold_xml->Notification_Data->body->items)) {
                        $items_array = $sold_xml->Notification_Data->body->items;
                    } elseif ($sold_xml && isset($sold_xml->body->items)) {
                        $items_array = $sold_xml->body->items;
                    }
                    
                    foreach ($items_array as $sold_item) {
                        if (isset($sold_item['id_item']) && $sold_item['id_item'] == $order_record->delcampe_order_id) {
                            // Found in sold items - payment is confirmed
                            if ($new_payment_status !== 'paid') {
                                $new_payment_status = 'paid';
                                $status_changed = true;
                            }
                            
                            // Check if marked as sent
                            if (isset($sold_item['sent']) && $sold_item['sent'] == true) {
                                if ($new_shipping_status !== 'shipped') {
                                    $new_shipping_status = 'shipped';
                                    $status_changed = true;
                                }
                            }
                            break;
                        }
                    }
                }
                
                // Also check buyer endpoint for additional status info
                // v1.10.35.17: Use the actual item ID for API calls
                $buyer_url = "https://rest.delcampe.net/item/{$item_id_to_check}/buyer?token={$token}";
                $buyer_response = wp_remote_get($buyer_url, array(
                    'timeout' => 30,
                    'headers' => array('Accept' => 'application/xml')
                ));
                
                if (!is_wp_error($buyer_response)) {
                    $buyer_body = wp_remote_retrieve_body($buyer_response);
                    $buyer_xml = simplexml_load_string($buyer_body);
                    
                    // CRITICAL FIX (v1.10.36.2): Buyer info does NOT mean payment is confirmed!
                    // Buyer info is available immediately when item sells, BEFORE payment.
                    // Payment status should ONLY be set to 'paid' when:
                    // 1. Seller_Payment_Received webhook is received (via reconciliation), OR
                    // 2. bill_id or transaction_id exists in the item data (explicit payment proof)
                    
                    // Check for explicit payment confirmation (bill_id indicates payment processed)
                    $has_payment_proof = false;
                    if ($buyer_xml) {
                        // Check for bill_id in various possible locations
                        if (isset($buyer_xml->bill_id) && !empty((string)$buyer_xml->bill_id)) {
                            $has_payment_proof = true;
                        } elseif (isset($buyer_xml->Notification_Data->body->bill_id) && !empty((string)$buyer_xml->Notification_Data->body->bill_id)) {
                            $has_payment_proof = true;
                        } elseif (isset($buyer_xml->body->bill_id) && !empty((string)$buyer_xml->body->bill_id)) {
                            $has_payment_proof = true;
                        }
                    }
                    
                    // Only mark as paid if we have explicit payment proof
                    if ($has_payment_proof && $new_payment_status !== 'paid') {
                        $new_payment_status = 'paid';
                        $status_changed = true;
                    }
                }
                
                // Update if status changed
                if ($status_changed) {
                    // Update our database
                    $wpdb->update(
                        $this->table_name,
                        array(
                            'payment_status' => $new_payment_status,
                            'shipping_status' => $new_shipping_status,
                            'last_sync' => current_time('mysql')
                        ),
                        array('id' => $order_record->id),
                        array('%s', '%s', '%s'),
                        array('%d')
                    );
                    
                    // Update WooCommerce order if exists
                    if ($order_record->wc_order_id) {
                        $wc_order = wc_get_order($order_record->wc_order_id);
                        if ($wc_order) {
                            // Update WC order status based on new statuses
                            if ($new_shipping_status === 'shipped') {
                                // Check if we should auto-complete when shipped
                                if ('yes' === get_option('delcampe_complete_on_closure', 'no')) {
                                    $wc_order->update_status('completed', __('Marked as shipped on Delcampe', 'wc-delcampe-integration'));
                                } else {
                                    $wc_order->add_order_note(__('Marked as shipped on Delcampe. Please verify shipment and mark as completed.', 'wc-delcampe-integration'));
                                }
                            } elseif ($new_payment_status === 'paid') {
                                $wc_order->update_status('processing', __('Payment confirmed on Delcampe', 'wc-delcampe-integration'));
                            }
                            
                            // Add order note
                            $wc_order->add_order_note(sprintf(
                                __('Status updated from Delcampe sync: Payment %s, Shipping %s', 'wc-delcampe-integration'),
                                $new_payment_status,
                                $new_shipping_status
                            ));
                        }
                    }
                    
                    $updated++;
                    
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'DELCAMPE_ORDER_STATUS_UPDATED',
                        'order_id' => $order_record->delcampe_order_id,
                        'payment_status' => $new_payment_status,
                        'shipping_status' => $new_shipping_status
                    ));
                }
                
            } catch (Exception $e) {
                $errors++;
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'DELCAMPE_ORDER_UPDATE_ERROR',
                    'order_id' => $order_record->delcampe_order_id,
                    'error' => $e->getMessage()
                ));
            }
        }
        
        $result = array(
            'checked' => $checked,
            'updated' => $updated,
            'errors' => $errors,
            'message' => sprintf(
                __('Checked %d orders, updated %d, errors %d', 'wc-delcampe-integration'),
                $checked,
                $updated,
                $errors
            )
        );
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'PENDING_ORDERS_UPDATE_COMPLETE',
            'result' => $result
        ));
        
        return $result;
    }
    
    /**
     * Get order statistics
     * 
     * @return array Order statistics
     * 
     * @since 1.4.0.0
     */
    public function get_order_statistics() {
        global $wpdb;
        
        $stats = array();
        
        // Total orders
        $stats['total'] = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
        
        // Orders by sync status
        $status_counts = $wpdb->get_results(
            "SELECT sync_status, COUNT(*) as count 
            FROM {$this->table_name} 
            GROUP BY sync_status"
        );
        
        foreach ($status_counts as $status) {
            $stats['by_status'][$status->sync_status] = $status->count;
        }
        
        // Orders by payment status
        $payment_counts = $wpdb->get_results(
            "SELECT payment_status, COUNT(*) as count 
            FROM {$this->table_name} 
            GROUP BY payment_status"
        );
        
        foreach ($payment_counts as $payment) {
            $stats['by_payment'][$payment->payment_status] = $payment->count;
        }
        
        // Total revenue
        $stats['total_revenue'] = $wpdb->get_var(
            "SELECT SUM(sale_price) FROM {$this->table_name}"
        );
        
        // Orders today
        $stats['today'] = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name} 
            WHERE DATE(created_at) = CURDATE()"
        );
        
        // Orders this month
        $stats['this_month'] = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name} 
            WHERE MONTH(created_at) = MONTH(CURRENT_DATE()) 
            AND YEAR(created_at) = YEAR(CURRENT_DATE())"
        );
        
        return $stats;
    }
}
