<?php
/**
 * Delcampe Price Sync
 * 
 * Handles automatic price synchronization between WooCommerce and Delcampe
 * 
 * @package WC_Delcampe_Integration
 * @version 1.10.10.0
 * @since   1.10.10.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Delcampe_Price_Sync class
 *
 * @since 1.10.10.0
 */
class Delcampe_Price_Sync {
    
    /**
     * Singleton instance
     * @var Delcampe_Price_Sync
     */
    private static $instance = null;
    
    /**
     * Auth instance
     * @var Delcampe_Auth
     */
    private $auth;
    
    /**
     * Listings model
     * @var Delcampe_Listings_Model
     */
    private $listings_model;
    
    /**
     * Logger instance
     * @var Delcampe_Sync_Logger
     */
    private $logger;
    
    /**
     * Get singleton instance
     *
     * @return Delcampe_Price_Sync
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init();
    }
    
    /**
     * Ensure the price sync queue table exists
     */
    private function ensure_table_exists() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_price_sync_queue';
        
        // Check if table exists
        if($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
            $charset_collate = $wpdb->get_charset_collate();
            
            $sql = "CREATE TABLE $table_name (
                id bigint(20) NOT NULL AUTO_INCREMENT,
                product_id bigint(20) NOT NULL,
                delcampe_id varchar(50) NOT NULL,
                old_price decimal(10,2),
                new_price decimal(10,2) NOT NULL,
                status varchar(20) DEFAULT 'pending',
                attempts int DEFAULT 0,
                error_message text,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY product_id (product_id),
                KEY delcampe_id (delcampe_id),
                KEY status (status)
            ) $charset_collate;";
            
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        }
    }
    
    /**
     * Initialize price sync
     */
    private function init() {
        // Ensure database table exists
        $this->ensure_table_exists();
        
        // Initialize dependencies
        $this->auth = Delcampe_Auth::get_instance();
        $this->listings_model = new Delcampe_Listings_Model();
        $this->logger = Delcampe_Sync_Logger::get_instance();
        
        // Hook into WooCommerce price updates
        add_action('woocommerce_product_set_price', array($this, 'sync_price_to_delcampe'), 10, 2);
        add_action('woocommerce_product_set_regular_price', array($this, 'sync_price_to_delcampe'), 10, 2);
        add_action('woocommerce_product_set_sale_price', array($this, 'sync_price_to_delcampe'), 10, 2);
        
        // Hook into product save
        add_action('woocommerce_update_product', array($this, 'handle_product_price_update'), 10, 1);
        
        // Add cron for syncing prices from Delcampe
        add_action('delcampe_sync_prices_from_delcampe', array($this, 'sync_prices_from_delcampe'));
        
        // v1.10.35.17: Disabled automatic price sync - should be controlled by settings
        // Schedule cron if not already scheduled - DISABLED
        /*
        if (!wp_next_scheduled('delcampe_sync_prices_from_delcampe')) {
            wp_schedule_event(time(), 'hourly', 'delcampe_sync_prices_from_delcampe');
        }
        */
        
        // Add admin settings
        add_filter('delcampe_sync_settings', array($this, 'add_price_sync_settings'));
        
        // Add webhook handler for price changes
        add_action('delcampe_webhook_price_change', array($this, 'handle_webhook_price_change'), 10, 2);
    }
    
    /**
     * Handle product price update
     *
     * @param int $product_id Product ID
     */
    public function handle_product_price_update($product_id) {
        // Check if price sync is enabled
        if (!get_option('delcampe_enable_price_sync', 'yes') === 'yes') {
            return;
        }
        
        // Get the product
        $product = wc_get_product($product_id);
        if (!$product) {
            return;
        }
        
        // Check if product is linked to Delcampe
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (empty($delcampe_id)) {
            return;
        }
        
        // Get the current price
        $price = $product->get_price();
        if (empty($price)) {
            return;
        }
        
        // Queue price update for Delcampe
        $this->queue_price_update($product_id, $delcampe_id, $price);
    }
    
    /**
     * Queue price update for Delcampe
     *
     * @param int $product_id WooCommerce product ID
     * @param string $delcampe_id Delcampe listing ID
     * @param float $price New price
     */
    private function queue_price_update($product_id, $delcampe_id, $price) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_price_sync_queue';
        
        // Create table if it doesn't exist
        $this->create_price_sync_table();
        
        // Check if update already queued
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table_name} 
             WHERE product_id = %d AND status = 'pending'",
            $product_id
        ));
        
        if ($existing) {
            // Update existing queue entry
            $wpdb->update(
                $table_name,
                array(
                    'new_price' => $price,
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $existing),
                array('%f', '%s'),
                array('%d')
            );
        } else {
            // Insert new queue entry
            $wpdb->insert(
                $table_name,
                array(
                    'product_id' => $product_id,
                    'delcampe_id' => $delcampe_id,
                    'new_price' => $price,
                    'status' => 'pending',
                    'created_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql')
                ),
                array('%d', '%s', '%f', '%s', '%s', '%s')
            );
        }
        
        // Log the queued update
        $this->logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'PRICE_UPDATE_QUEUED',
            'product_id' => $product_id,
            'delcampe_id' => $delcampe_id,
            'new_price' => $price
        ));
    }
    
    /**
     * Process price sync queue
     */
    public function process_price_sync_queue() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_price_sync_queue';
        
        // Initialize result array
        $result = array(
            'total' => 0,
            'processed' => 0,
            'failed' => 0
        );
        
        // Get pending updates
        $pending_updates = $wpdb->get_results(
            "SELECT * FROM {$table_name} 
             WHERE status = 'pending' 
             ORDER BY created_at ASC 
             LIMIT 10"
        );
        
        if (empty($pending_updates)) {
            return $result;
        }
        
        $result['total'] = count($pending_updates);
        
        foreach ($pending_updates as $update) {
            $process_result = $this->process_single_price_update($update);
            if ($process_result) {
                $result['processed']++;
            } else {
                $result['failed']++;
            }
        }
        
        return $result;
    }
    
    /**
     * Process single price update
     *
     * @param object $update Queue entry
     */
    private function process_single_price_update($update) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_price_sync_queue';
        
        try {
            // Get auth token
            $token = $this->auth->get_auth_token();
            if (!$token) {
                throw new Exception('Failed to get auth token');
            }
            
            // v1.10.35.5: Check rate limit before request
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-rate-limiter.php';
            $rate_limiter = Delcampe_Rate_Limiter::get_instance();
            
            $can_request = $rate_limiter->can_make_request();
            if ( is_wp_error( $can_request ) ) {
                throw new Exception('Rate limited: ' . $can_request->get_error_message());
            }
            
            // v1.10.35.15: Fixed - Delcampe API requires token as URL parameter, not Bearer auth
            // Prepare API request
            $api_url = 'https://rest.delcampe.net/item/' . $update->delcampe_id . '?token=' . $token;
            
            $response = wp_remote_request($api_url, array(
                'method' => 'PUT',
                'headers' => array(
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode(array(
                    'price' => floatval($update->new_price)
                )),
                'timeout' => 30
            ));
            
            // Record the request
            $status_code = is_wp_error($response) ? 0 : wp_remote_retrieve_response_code($response);
            $rate_limiter->record_request('/item/' . $update->delcampe_id, $status_code);
            
            if (is_wp_error($response)) {
                throw new Exception($response->get_error_message());
            }
            
            // Handle specific error codes
            if ($status_code === 429) {
                throw new Exception('Rate limited - will retry later');
            } elseif ($status_code === 401) {
                delete_transient('delcampe_access_token');
                throw new Exception('Authentication failed - token expired');
            } elseif ($status_code !== 200 && $status_code !== 204) {
                $body = wp_remote_retrieve_body($response);
                throw new Exception('API error: ' . $body);
            }
            
            // Update queue status
            $wpdb->update(
                $table_name,
                array(
                    'status' => 'completed',
                    'processed_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $update->id),
                array('%s', '%s', '%s'),
                array('%d')
            );
            
            // Log success
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'PRICE_UPDATE_SUCCESS',
                'product_id' => $update->product_id,
                'delcampe_id' => $update->delcampe_id,
                'new_price' => $update->new_price
            ));
            
            return true;
            
        } catch (Exception $e) {
            // Update queue status
            $wpdb->update(
                $table_name,
                array(
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $update->id),
                array('%s', '%s', '%s'),
                array('%d')
            );
            
            // Log error
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'PRICE_UPDATE_FAILED',
                'product_id' => $update->product_id,
                'delcampe_id' => $update->delcampe_id,
                'error' => $e->getMessage()
            ));
            
            return false;
        }
    }
    
    /**
     * Sync prices from Delcampe to WooCommerce
     * v1.10.35.5: Added rate limiting and batch processing
     */
    public function sync_prices_from_delcampe() {
        // Check if price sync is enabled
        if (get_option('delcampe_enable_price_sync', 'yes') !== 'yes') {
            return;
        }
        
        // Load rate limiter
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-rate-limiter.php';
        $rate_limiter = Delcampe_Rate_Limiter::get_instance();
        
        // Check if we're rate limited
        $can_request = $rate_limiter->can_make_request();
        if ( is_wp_error( $can_request ) ) {
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'PRICE_SYNC_RATE_LIMITED',
                'error' => $can_request->get_error_message()
            ));
            return;
        }
        
        // Get all active listings (reduced batch size to prevent rate limiting)
        $listings = $this->listings_model->get_listings(array(
            'status' => 'published',
            'limit' => 10  // Reduced from 100 to prevent rate limiting
        ));
        
        if (empty($listings)) {
            return;
        }
        
        $processed = 0;
        $max_per_run = 5; // Process max 5 items per run to prevent rate limiting
        
        foreach ($listings as $listing) {
            if ($processed >= $max_per_run) {
                // Schedule next batch for later
                wp_schedule_single_event(time() + 60, 'delcampe_sync_prices_from_delcampe');
                break;
            }
            
            // Check rate limit before each request
            $can_request = $rate_limiter->can_make_request();
            if ( is_wp_error( $can_request ) ) {
                // We hit the rate limit, stop processing and schedule retry
                wp_schedule_single_event(time() + 60, 'delcampe_sync_prices_from_delcampe');
                break;
            }
            
            $this->sync_single_price_from_delcampe($listing);
            $processed++;
            
            // Add small delay between requests
            usleep(500000); // 0.5 second delay
        }
    }
    
    /**
     * Sync single price from Delcampe
     *
     * @param object $listing Listing object
     */
    private function sync_single_price_from_delcampe($listing) {
        try {
            // Get auth token
            $token = $this->auth->get_auth_token();
            if (!$token) {
                throw new Exception('Failed to get auth token');
            }
            
            // v1.10.35.5: Record API request for rate limiting
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-rate-limiter.php';
            $rate_limiter = Delcampe_Rate_Limiter::get_instance();
            
            // v1.10.35.15: Fixed - Delcampe API requires token as URL parameter, not Bearer auth
            // Get item details from Delcampe
            $api_url = 'https://rest.delcampe.net/item/' . $listing->delcampe_id . '?token=' . $token;
            
            $response = wp_remote_get($api_url, array(
                'timeout' => 30
            ));
            
            // Record the request
            $status_code = is_wp_error($response) ? 0 : wp_remote_retrieve_response_code($response);
            $rate_limiter->record_request('/item/' . $listing->delcampe_id, $status_code);
            
            if (is_wp_error($response)) {
                throw new Exception($response->get_error_message());
            }
            
            // Handle rate limiting and auth errors
            if ($status_code === 429) {
                throw new Exception('Rate limited - will retry later');
            } elseif ($status_code === 401) {
                // Clear token cache to force renewal
                delete_transient('delcampe_access_token');
                throw new Exception('Authentication failed - token expired');
            } elseif ($status_code !== 200) {
                throw new Exception('API error: HTTP ' . $status_code);
            }
            
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (!isset($data['price'])) {
                return;
            }
            
            $delcampe_price = floatval($data['price']);
            
            // Get WooCommerce product
            $product = wc_get_product($listing->product_id);
            if (!$product) {
                return;
            }
            
            $current_price = floatval($product->get_price());
            
            // Check if prices are different
            if (abs($current_price - $delcampe_price) < 0.01) {
                return; // Prices are the same
            }
            
            // Update WooCommerce price
            $product->set_price($delcampe_price);
            $product->set_regular_price($delcampe_price);
            $product->save();
            
            // Log the sync
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'PRICE_SYNC_FROM_DELCAMPE',
                'product_id' => $listing->product_id,
                'delcampe_id' => $listing->delcampe_id,
                'old_price' => $current_price,
                'new_price' => $delcampe_price
            ));
            
        } catch (Exception $e) {
            $this->logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'PRICE_SYNC_ERROR',
                'product_id' => $listing->product_id,
                'delcampe_id' => $listing->delcampe_id,
                'error' => $e->getMessage()
            ));
        }
    }
    
    /**
     * Handle webhook price change notification
     *
     * @param string $delcampe_id Delcampe listing ID
     * @param float $new_price New price
     */
    public function handle_webhook_price_change($delcampe_id, $new_price) {
        // Find the linked product
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        $listing = $wpdb->get_row($wpdb->prepare(
            "SELECT product_id FROM {$table_name} WHERE delcampe_id = %s",
            $delcampe_id
        ));
        
        if (!$listing) {
            return;
        }
        
        // Update WooCommerce price
        $product = wc_get_product($listing->product_id);
        if (!$product) {
            return;
        }
        
        $product->set_price($new_price);
        $product->set_regular_price($new_price);
        $product->save();
        
        // Log the webhook update
        $this->logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_PRICE_UPDATE',
            'product_id' => $listing->product_id,
            'delcampe_id' => $delcampe_id,
            'new_price' => $new_price
        ));
    }
    
    /**
     * Add price sync settings
     *
     * @param array $settings Current settings
     * @return array Modified settings
     */
    public function add_price_sync_settings($settings) {
        $settings['price_sync'] = array(
            'title' => __('Price Synchronization', 'wc-delcampe-integration'),
            'fields' => array(
                'delcampe_enable_price_sync' => array(
                    'title' => __('Enable Price Sync', 'wc-delcampe-integration'),
                    'type' => 'checkbox',
                    'description' => __('Automatically sync prices between WooCommerce and Delcampe', 'wc-delcampe-integration'),
                    'default' => 'yes'
                ),
                'delcampe_price_sync_direction' => array(
                    'title' => __('Sync Direction', 'wc-delcampe-integration'),
                    'type' => 'select',
                    'options' => array(
                        'both' => __('Both ways', 'wc-delcampe-integration'),
                        'to_delcampe' => __('WooCommerce to Delcampe only', 'wc-delcampe-integration'),
                        'from_delcampe' => __('Delcampe to WooCommerce only', 'wc-delcampe-integration')
                    ),
                    'default' => 'both',
                    'description' => __('Choose how prices should be synchronized', 'wc-delcampe-integration')
                ),
                'delcampe_price_sync_frequency' => array(
                    'title' => __('Sync Frequency', 'wc-delcampe-integration'),
                    'type' => 'select',
                    'options' => array(
                        'realtime' => __('Real-time (via webhooks)', 'wc-delcampe-integration'),
                        'hourly' => __('Every hour', 'wc-delcampe-integration'),
                        'twice_daily' => __('Twice daily', 'wc-delcampe-integration'),
                        'daily' => __('Once daily', 'wc-delcampe-integration')
                    ),
                    'default' => 'hourly',
                    'description' => __('How often to sync prices from Delcampe', 'wc-delcampe-integration')
                )
            )
        );
        
        return $settings;
    }
    
    /**
     * Create price sync queue table
     */
    private function create_price_sync_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_price_sync_queue';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            product_id bigint(20) NOT NULL,
            delcampe_id varchar(50) NOT NULL,
            new_price decimal(10,2) NOT NULL,
            status varchar(20) DEFAULT 'pending',
            error_message text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP,
            processed_at datetime,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY delcampe_id (delcampe_id),
            KEY status (status)
        ) {$charset_collate};";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Sync price to Delcampe
     *
     * @param float $price New price
     * @param WC_Product $product Product object
     */
    public function sync_price_to_delcampe($price, $product) {
        // Check if price sync is enabled
        if (get_option('delcampe_enable_price_sync', 'yes') !== 'yes') {
            return;
        }
        
        // Check sync direction
        $direction = get_option('delcampe_price_sync_direction', 'both');
        if ($direction === 'from_delcampe') {
            return; // Only sync from Delcampe, not to
        }
        
        $this->handle_product_price_update($product->get_id());
    }
}

// Initialize the class only after all plugins are loaded
add_action('plugins_loaded', function() {
    if (class_exists('Delcampe_Sync_Logger') && class_exists('Delcampe_Auth') && class_exists('Delcampe_Listings_Model')) {
        Delcampe_Price_Sync::get_instance();
    }
}, 20);

// Add cron processor after class is initialized
add_action('plugins_loaded', function() {
    if (class_exists('Delcampe_Price_Sync')) {
        add_action('delcampe_process_price_sync_queue', array(Delcampe_Price_Sync::get_instance(), 'process_price_sync_queue'));
    }
}, 25);

// Schedule price sync queue processor if not already scheduled
add_action('init', function() {
    if (!wp_next_scheduled('delcampe_process_price_sync_queue')) {
        wp_schedule_event(time(), 'delcampe_five_minutes', 'delcampe_process_price_sync_queue');
    }
});