<?php
/**
 * Product Integration for Delcampe Plugin
 * Version: 1.2.0.2
 *
 * Handles integration between WooCommerce products and Delcampe profiles
 * Adds metabox to product edit page for profile selection
 * Manages product-profile associations and sync status
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Product_Integration
 * @since 1.2.0.0
 * @version 1.2.0.2
 * 
 * Changelog:
 * 1.2.0.2 - Integrated with sync handler for queue-based processing
 * 1.2.0.1 - Added AJAX handlers for sync functionality
 *         - Fixed singleton pattern usage for Delcampe_Profiles_Model
 * 1.2.0.0 - Initial implementation of product-profile integration
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Delcampe_Product_Integration
 * 
 * Manages the integration between WooCommerce products and Delcampe profiles
 * 
 * @since 1.2.0.0
 * @version 1.2.0.2
 */
class Delcampe_Product_Integration {
    
    /**
     * Singleton instance
     * @var Delcampe_Product_Integration|null
     * @since 1.2.0.0
     */
    private static $instance = null;
    
    /**
     * Meta key for storing profile ID
     * @var string
     * @since 1.2.0.0
     */
    const PROFILE_META_KEY = '_delcampe_profile_id';
    
    /**
     * Meta key for storing Delcampe listing ID
     * @var string
     * @since 1.2.0.0
     */
    const LISTING_ID_META_KEY = '_delcampe_listing_id';
    
    /**
     * Meta key for storing sync status
     * @var string
     * @since 1.2.0.0
     */
    const SYNC_STATUS_META_KEY = '_delcampe_sync_status';
    
    /**
     * Meta key for storing last sync timestamp
     * @var string
     * @since 1.2.0.0
     */
    const LAST_SYNC_META_KEY = '_delcampe_last_sync';
    
    /**
     * Get singleton instance
     * 
     * @since 1.2.0.0
     * @return Delcampe_Product_Integration
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     * 
     * Sets up hooks for product integration
     * 
     * @since 1.2.0.0
     * @version 1.2.0.2
     */
    private function __construct() {
        // Add metabox to product edit page
        add_action('add_meta_boxes', array($this, 'add_product_metabox'));
        
        // Save metabox data
        add_action('save_post_product', array($this, 'save_product_meta'), 10, 3);
        
        // Add column to product list
        add_filter('manage_product_posts_columns', array($this, 'add_product_columns'));
        add_action('manage_product_posts_custom_column', array($this, 'render_product_columns'), 10, 2);
        
        // Add bulk actions
        add_filter('bulk_actions-edit-product', array($this, 'add_bulk_actions'));
        add_filter('handle_bulk_actions-edit-product', array($this, 'handle_bulk_actions'), 10, 3);
        
        // Add admin notices for bulk actions
        add_action('admin_notices', array($this, 'bulk_action_admin_notices'));
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // Register AJAX handlers
        add_action('wp_ajax_delcampe_sync_product', array($this, 'ajax_sync_product'));
        
        // Product integration initialized
    }
    
    /**
     * AJAX handler for syncing a product
     * 
     * @since 1.2.0.1
     * @version 1.2.0.2 - Now uses sync handler
     */
    public function ajax_sync_product() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_product_ajax')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'wc-delcampe-integration')));
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
        }
        
        // Get product ID
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        if (!$product_id) {
            wp_send_json_error(array('message' => __('Invalid product ID.', 'wc-delcampe-integration')));
        }
        
        // Verify product exists
        $product = wc_get_product($product_id);
        if (!$product) {
            wp_send_json_error(array('message' => __('Product not found.', 'wc-delcampe-integration')));
        }
        
        // Check if product has a profile
        $profile_id = get_post_meta($product_id, self::PROFILE_META_KEY, true);
        if (!$profile_id) {
            wp_send_json_error(array('message' => __('No Delcampe profile assigned to this product.', 'wc-delcampe-integration')));
        }
        
        // Get sync handler
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-sync-handler.php';
        $sync_handler = Delcampe_Sync_Handler::get_instance();
        
        // Determine action (create or update)
        $listing_id = get_post_meta($product_id, self::LISTING_ID_META_KEY, true);
        $action = $listing_id ? 'update' : 'create';
        
        // Add to sync queue
        $sync_handler->add_to_queue($product_id, $profile_id, $action);
        
        // Log the sync request
        delcampe_debug('[Delcampe Product Integration v1.2.0.2] Product ' . $product_id . ' added to sync queue with action: ' . $action);
        
        // Prepare response data
        $response_data = array(
            'message' => __('Product queued for synchronization.', 'wc-delcampe-integration'),
            'sync_status' => 'pending',
            'last_sync' => time(),
            'last_sync_formatted' => date_i18n(get_option('date_format') . ' ' . get_option('time_format'), time())
        );
        
        // Send success response
        wp_send_json_success($response_data);
    }
    
    /**
     * Add metabox to product edit page
     * 
     * @since 1.2.0.0
     */
    public function add_product_metabox() {
        add_meta_box(
            'delcampe_product_settings',
            __('Delcampe Integration', 'wc-delcampe-integration'),
            array($this, 'render_product_metabox'),
            'product',
            'side',
            'default'
        );
    }
    
    /**
     * Render product metabox content
     * 
     * @since 1.2.0.0
     * @version 1.2.0.2 - Added error display
     * @param WP_Post $post The current post object
     */
    public function render_product_metabox($post) {
        // Add nonce for security
        wp_nonce_field('delcampe_product_meta', 'delcampe_product_meta_nonce');
        
        // Get current values
        $profile_id = get_post_meta($post->ID, self::PROFILE_META_KEY, true);
        $listing_id = get_post_meta($post->ID, self::LISTING_ID_META_KEY, true);
        $sync_status = get_post_meta($post->ID, self::SYNC_STATUS_META_KEY, true);
        $last_sync = get_post_meta($post->ID, self::LAST_SYNC_META_KEY, true);
        $last_error = get_post_meta($post->ID, '_delcampe_last_error', true);
        
        // Get available profiles using singleton pattern
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/profiles/class-delcampe-profiles-model.php';
        $profiles_model = Delcampe_Profiles_Model::get_instance();
        $profiles = $profiles_model->get_all();
        
        ?>
        <div class="delcampe-product-meta">
            <!-- Profile Selection -->
            <p>
                <label for="delcampe_profile_id">
                    <strong><?php _e('Delcampe Profile:', 'wc-delcampe-integration'); ?></strong>
                </label>
                <select name="delcampe_profile_id" id="delcampe_profile_id" class="widefat">
                    <option value=""><?php _e('— No Profile —', 'wc-delcampe-integration'); ?></option>
                    <?php foreach ($profiles as $profile): ?>
                        <option value="<?php echo esc_attr($profile['profile_id']); ?>" 
                                <?php selected($profile_id, $profile['profile_id']); ?>>
                            <?php echo esc_html($profile['profile_name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <span class="description">
                    <?php _e('Select a profile to use for Delcampe listing', 'wc-delcampe-integration'); ?>
                </span>
            </p>
            
            <?php if ($profile_id): ?>
                <!-- Profile Details Preview -->
                <div class="profile-details" style="margin-top: 10px; padding: 10px; background: #f0f0f0; border-radius: 4px;">
                    <?php
                    $selected_profile = null;
                    foreach ($profiles as $profile) {
                        if ($profile['profile_id'] == $profile_id) {
                            $selected_profile = $profile;
                            break;
                        }
                    }
                    
                    if ($selected_profile):
                        ?>
                        <p style="margin: 0 0 5px 0;">
                            <strong><?php _e('Selected Profile:', 'wc-delcampe-integration'); ?></strong><br>
                            <?php echo esc_html($selected_profile['profile_name']); ?>
                        </p>
                        <?php if (!empty($selected_profile['delcampe_category_name'])): ?>
                            <p style="margin: 0 0 5px 0;">
                                <strong><?php _e('Category:', 'wc-delcampe-integration'); ?></strong><br>
                                <?php echo esc_html($selected_profile['delcampe_category_name']); ?>
                            </p>
                        <?php endif; ?>
                        <?php if (!empty($selected_profile['listing_details']['duration'])): ?>
                            <p style="margin: 0;">
                                <strong><?php _e('Duration:', 'wc-delcampe-integration'); ?></strong> 
                                <?php echo esc_html($selected_profile['listing_details']['duration']); ?> <?php _e('days', 'wc-delcampe-integration'); ?>
                            </p>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            
            <!-- Sync Status -->
            <?php if ($listing_id || $sync_status): ?>
                <hr style="margin: 15px 0;">
                <div class="sync-status">
                    <?php if ($listing_id): ?>
                        <p>
                            <strong><?php _e('Delcampe Listing ID:', 'wc-delcampe-integration'); ?></strong><br>
                            <code><?php echo esc_html($listing_id); ?></code>
                        </p>
                    <?php endif; ?>
                    
                    <?php if ($sync_status): ?>
                        <p>
                            <strong><?php _e('Sync Status:', 'wc-delcampe-integration'); ?></strong><br>
                            <span class="sync-status-<?php echo esc_attr($sync_status); ?>">
                                <?php echo esc_html(ucfirst($sync_status)); ?>
                            </span>
                        </p>
                    <?php endif; ?>
                    
                    <?php if ($last_sync): ?>
                        <p>
                            <strong><?php _e('Last Sync:', 'wc-delcampe-integration'); ?></strong><br>
                            <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_sync)); ?>
                        </p>
                    <?php endif; ?>
                    
                    <?php if ($sync_status === 'error' && $last_error): ?>
                        <p style="color: #dc3232;">
                            <strong><?php _e('Last Error:', 'wc-delcampe-integration'); ?></strong><br>
                            <?php echo esc_html($last_error); ?>
                        </p>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            
            <!-- Quick Actions -->
            <?php if ($profile_id): ?>
                <hr style="margin: 15px 0;">
                <div class="quick-actions">
                    <button type="button" class="button button-small" id="delcampe-sync-now" data-product-id="<?php echo esc_attr($post->ID); ?>">
                        <?php _e('Sync Now', 'wc-delcampe-integration'); ?>
                    </button>
                    <?php if ($listing_id): ?>
                        <button type="button" class="button button-small" id="delcampe-view-listing" data-listing-id="<?php echo esc_attr($listing_id); ?>">
                            <?php _e('View on Delcampe', 'wc-delcampe-integration'); ?>
                        </button>
                        <?php if ($sync_status === 'active'): ?>
                            <button type="button" class="button button-small" id="delcampe-close-listing" 
                                    data-product-id="<?php echo esc_attr($post->ID); ?>"
                                    data-listing-id="<?php echo esc_attr($listing_id); ?>"
                                    style="color: #a00;">
                                <?php _e('Close Listing', 'wc-delcampe-integration'); ?>
                            </button>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
        
        <style>
            .delcampe-product-meta .sync-status-active { color: #46b450; }
            .delcampe-product-meta .sync-status-pending { color: #f56e28; }
            .delcampe-product-meta .sync-status-error { color: #dc3232; }
            .delcampe-product-meta .sync-status-inactive { color: #666; }
        </style>
        <?php
    }
    
    /**
     * Save product meta data
     * 
     * @since 1.2.0.0
     * @param int $post_id The post ID
     * @param WP_Post $post The post object
     * @param bool $update Whether this is an update
     */
    public function save_product_meta($post_id, $post, $update) {
        // Check nonce
        if (!isset($_POST['delcampe_product_meta_nonce']) || 
            !wp_verify_nonce($_POST['delcampe_product_meta_nonce'], 'delcampe_product_meta')) {
            return;
        }
        
        // Check autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Save profile ID
        if (isset($_POST['delcampe_profile_id'])) {
            $profile_id = sanitize_text_field($_POST['delcampe_profile_id']);
            if ($profile_id) {
                update_post_meta($post_id, self::PROFILE_META_KEY, $profile_id);
                
                // Set initial sync status if not already set
                if (!get_post_meta($post_id, self::SYNC_STATUS_META_KEY, true)) {
                    update_post_meta($post_id, self::SYNC_STATUS_META_KEY, 'pending');
                }
            } else {
                delete_post_meta($post_id, self::PROFILE_META_KEY);
                // Clear sync status if no profile selected
                delete_post_meta($post_id, self::SYNC_STATUS_META_KEY);
            }
        }
        
        delcampe_debug('[Delcampe Product Integration v1.2.0.2] Saved product meta for product ID: ' . $post_id);
    }
    
    /**
     * Add columns to product list
     * 
     * @since 1.2.0.0
     * @param array $columns Existing columns
     * @return array Modified columns
     */
    public function add_product_columns($columns) {
        // Add after price column
        $new_columns = array();
        foreach ($columns as $key => $value) {
            $new_columns[$key] = $value;
            if ($key === 'price') {
                $new_columns['delcampe_status'] = __('Delcampe', 'wc-delcampe-integration');
            }
        }
        
        return $new_columns;
    }
    
    /**
     * Render product column content
     * 
     * @since 1.2.0.0
     * @version 1.2.0.1 - Fixed singleton pattern usage
     * @param string $column The column name
     * @param int $post_id The post ID
     */
    public function render_product_columns($column, $post_id) {
        if ($column === 'delcampe_status') {
            $profile_id = get_post_meta($post_id, self::PROFILE_META_KEY, true);
            $listing_id = get_post_meta($post_id, self::LISTING_ID_META_KEY, true);
            $sync_status = get_post_meta($post_id, self::SYNC_STATUS_META_KEY, true);
            
            if (!$profile_id) {
                echo '<span style="color: #999;">—</span>';
                return;
            }
            
            // Get profile name using singleton pattern
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/profiles/class-delcampe-profiles-model.php';
            $profiles_model = Delcampe_Profiles_Model::get_instance();
            $profile = $profiles_model->get_profile($profile_id);
            
            echo '<div class="delcampe-status-column">';
            
            if ($profile) {
                echo '<small style="display: block; margin-bottom: 3px;">' . esc_html($profile['profile_name']) . '</small>';
            }
            
            if ($listing_id) {
                echo '<span class="dashicons dashicons-yes" style="color: #46b450;" title="' . 
                     esc_attr__('Listed on Delcampe', 'wc-delcampe-integration') . '"></span> ';
            }
            
            if ($sync_status) {
                $status_colors = array(
                    'active' => '#46b450',
                    'pending' => '#f56e28',
                    'error' => '#dc3232',
                    'inactive' => '#666'
                );
                $color = isset($status_colors[$sync_status]) ? $status_colors[$sync_status] : '#999';
                echo '<span style="color: ' . esc_attr($color) . ';">' . esc_html(ucfirst($sync_status)) . '</span>';
            }
            
            echo '</div>';
        }
    }
    
    /**
     * Add bulk actions to product list
     * 
     * @since 1.2.0.0
     * @param array $actions Existing actions
     * @return array Modified actions
     */
    public function add_bulk_actions($actions) {
        $actions['delcampe_assign_profile'] = __('Assign Delcampe Profile', 'wc-delcampe-integration');
        $actions['delcampe_remove_profile'] = __('Remove Delcampe Profile', 'wc-delcampe-integration');
        $actions['delcampe_sync_now'] = __('Sync to Delcampe', 'wc-delcampe-integration');
        
        return $actions;
    }
    
    /**
     * Handle bulk actions
     * 
     * @since 1.2.0.0
     * @version 1.2.0.2 - Now uses sync handler
     * @param string $redirect_to The redirect URL
     * @param string $action The action being performed
     * @param array $post_ids The selected post IDs
     * @return string The redirect URL
     */
    public function handle_bulk_actions($redirect_to, $action, $post_ids) {
        if (!in_array($action, array('delcampe_assign_profile', 'delcampe_remove_profile', 'delcampe_sync_now'))) {
            return $redirect_to;
        }
        
        $processed = 0;
        
        switch ($action) {
            case 'delcampe_assign_profile':
                // This would need a profile selection interface
                // For now, we'll add a notice
                $redirect_to = add_query_arg('delcampe_bulk_profile_needed', count($post_ids), $redirect_to);
                break;
                
            case 'delcampe_remove_profile':
                foreach ($post_ids as $post_id) {
                    delete_post_meta($post_id, self::PROFILE_META_KEY);
                    delete_post_meta($post_id, self::SYNC_STATUS_META_KEY);
                    $processed++;
                }
                $redirect_to = add_query_arg('delcampe_profiles_removed', $processed, $redirect_to);
                break;
                
            case 'delcampe_sync_now':
                // Load sync handler
                require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-sync-handler.php';
                $sync_handler = Delcampe_Sync_Handler::get_instance();
                
                foreach ($post_ids as $post_id) {
                    $profile_id = get_post_meta($post_id, self::PROFILE_META_KEY, true);
                    if ($profile_id) {
                        $listing_id = get_post_meta($post_id, self::LISTING_ID_META_KEY, true);
                        $action = $listing_id ? 'update' : 'create';
                        
                        $sync_handler->add_to_queue($post_id, $profile_id, $action);
                        $processed++;
                    }
                }
                $redirect_to = add_query_arg('delcampe_sync_queued', $processed, $redirect_to);
                break;
        }
        
        return $redirect_to;
    }
    
    /**
     * Display admin notices for bulk actions
     * 
     * @since 1.2.0.0
     */
    public function bulk_action_admin_notices() {
        if (!empty($_GET['delcampe_profiles_removed'])) {
            $count = intval($_GET['delcampe_profiles_removed']);
            printf(
                '<div class="notice notice-success is-dismissible"><p>' .
                _n('Removed Delcampe profile from %s product.',
                   'Removed Delcampe profiles from %s products.',
                   $count,
                   'wc-delcampe-integration') .
                '</p></div>',
                $count
            );
        }
        
        if (!empty($_GET['delcampe_sync_queued'])) {
            $count = intval($_GET['delcampe_sync_queued']);
            printf(
                '<div class="notice notice-success is-dismissible"><p>' .
                _n('Queued %s product for Delcampe sync.',
                   'Queued %s products for Delcampe sync.',
                   $count,
                   'wc-delcampe-integration') .
                '</p></div>',
                $count
            );
        }
        
        if (!empty($_GET['delcampe_bulk_profile_needed'])) {
            $count = intval($_GET['delcampe_bulk_profile_needed']);
            echo '<div class="notice notice-info is-dismissible"><p>';
            printf(
                __('To assign a profile to %s products, please use the individual product edit pages or implement a bulk profile selector.', 'wc-delcampe-integration'),
                $count
            );
            echo '</p></div>';
        }
    }
    
    /**
     * Enqueue admin scripts and styles
     * 
     * @since 1.2.0.0
     * @param string $hook The current admin page hook
     */
    public function enqueue_admin_assets($hook) {
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }
        
        global $post;
        if (!$post || $post->post_type !== 'product') {
            return;
        }
        
        // Enqueue product integration JavaScript
        wp_enqueue_script(
            'delcampe-product-integration',
            plugin_dir_url(dirname(__FILE__)) . 'assets/js/delcampe-product-integration.js',
            array('jquery'),
            '1.2.0.2',
            true
        );
        
        // Localize script
        wp_localize_script('delcampe-product-integration', 'delcampe_product', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('delcampe_product_ajax'),
            'strings' => array(
                'sync_confirm' => __('Are you sure you want to sync this product now?', 'wc-delcampe-integration'),
                'syncing' => __('Syncing...', 'wc-delcampe-integration'),
                'sync_now' => __('Sync Now', 'wc-delcampe-integration'),
                'sync_complete' => __('Sync complete!', 'wc-delcampe-integration'),
                'sync_error' => __('Sync failed. Please try again.', 'wc-delcampe-integration')
            )
        ));
    }
    
    /**
     * Get products by profile ID
     * 
     * @since 1.2.0.0
     * @param int $profile_id The profile ID
     * @return array Array of product IDs
     */
    public function get_products_by_profile($profile_id) {
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'meta_query' => array(
                array(
                    'key' => self::PROFILE_META_KEY,
                    'value' => $profile_id,
                    'compare' => '='
                )
            )
        );
        
        $query = new WP_Query($args);
        return $query->posts;
    }
    
    /**
     * Get product sync status
     * 
     * @since 1.2.0.0
     * @param int $product_id The product ID
     * @return array Array with sync information
     */
    public function get_product_sync_status($product_id) {
        return array(
            'profile_id' => get_post_meta($product_id, self::PROFILE_META_KEY, true),
            'listing_id' => get_post_meta($product_id, self::LISTING_ID_META_KEY, true),
            'sync_status' => get_post_meta($product_id, self::SYNC_STATUS_META_KEY, true),
            'last_sync' => get_post_meta($product_id, self::LAST_SYNC_META_KEY, true)
        );
    }
    
    /**
     * Update product sync status
     * 
     * @since 1.2.0.0
     * @param int $product_id The product ID
     * @param string $status The sync status
     * @param string $listing_id Optional listing ID
     */
    public function update_sync_status($product_id, $status, $listing_id = null) {
        update_post_meta($product_id, self::SYNC_STATUS_META_KEY, $status);
        update_post_meta($product_id, self::LAST_SYNC_META_KEY, time());
        
        if ($listing_id !== null) {
            if ($listing_id) {
                update_post_meta($product_id, self::LISTING_ID_META_KEY, $listing_id);
            } else {
                delete_post_meta($product_id, self::LISTING_ID_META_KEY);
            }
        }
        
        delcampe_debug('[Delcampe Product Integration v1.2.0.2] Updated sync status for product ' . $product_id . ': ' . $status);
    }
}

// Initialize the class
Delcampe_Product_Integration::get_instance();
