<?php
/**
 * Delcampe Queue Admin Interface
 * 
 * Provides a simple admin interface for monitoring queue status and managing the queue.
 * Replaces the complex batch monitoring system with a clean, reliable interface.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Queue_Admin {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_delcampe_queue_stats', array($this, 'ajax_queue_stats'));
        add_action('wp_ajax_delcampe_queue_pause', array($this, 'ajax_queue_pause'));
        add_action('wp_ajax_delcampe_queue_resume', array($this, 'ajax_queue_resume'));
        add_action('wp_ajax_delcampe_queue_clear', array($this, 'ajax_queue_clear'));
        add_action('wp_ajax_delcampe_queue_bulk_action', array($this, 'ajax_queue_bulk_action'));
    }
    
    /**
     * Add admin menu page
     */
    public function add_admin_menu() {
        add_submenu_page(
            'delcampe-main',
            __('Queue Monitor', 'wc-delcampe-integration'),
            __('Queue', 'wc-delcampe-integration'),
            'manage_woocommerce',
            'delcampe-queue',
            array($this, 'render_admin_page')
        );
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_scripts($hook) {
        if ($hook !== 'delcampe_page_delcampe-queue') {
            return;
        }
        
        wp_enqueue_script('jquery');
        wp_enqueue_style('delcampe-queue-admin', DWC_PLUGIN_URL . 'assets/css/queue-admin.css', array(), DWC_VERSION);
        wp_enqueue_script('delcampe-queue-admin', DWC_PLUGIN_URL . 'assets/js/queue-admin.js', array('jquery'), DWC_VERSION, true);
        
        // Localize script with nonce
        wp_localize_script('delcampe-queue-admin', 'delcampe_queue', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('delcampe_queue_action')
        ));
    }
    
    /**
     * Render admin page
     */
    public function render_admin_page() {
        $queue = Delcampe_Queue::get_instance();
        $stats = $queue->get_stats();
        $is_paused = get_transient('delcampe_queue_paused');
        
        // Check if migration is needed
        $migration_needed = false;
        $migration_status = get_option('delcampe_queue_migration_status', '');
        if (class_exists('Delcampe_Queue_Migration')) {
            $migration_needed = Delcampe_Queue_Migration::is_migration_needed();
        }
        
        ?>
        <div class="wrap">
            <h1><?php _e('Delcampe Queue Monitor', 'wc-delcampe-integration'); ?></h1>
            
            <?php if ($migration_needed): ?>
            <div class="notice notice-warning">
                <h2><?php _e('Migration Required', 'wc-delcampe-integration'); ?></h2>
                <p><?php _e('The new queue system requires migration of existing queue data. This process will:', 'wc-delcampe-integration'); ?></p>
                <ul>
                    <li><?php _e('Create optimized queue tables', 'wc-delcampe-integration'); ?></li>
                    <li><?php _e('Migrate existing queue items', 'wc-delcampe-integration'); ?></li>
                    <li><?php _e('Preserve all data with backup', 'wc-delcampe-integration'); ?></li>
                </ul>
                <p>
                    <input type="hidden" id="delcampe_queue_nonce" value="<?php echo wp_create_nonce('delcampe_queue_action'); ?>" />
                    <button type="button" id="migrate-btn" class="button button-primary">
                        <?php _e('Run Migration', 'wc-delcampe-integration'); ?>
                    </button>
                    <span class="spinner" style="float: none;"></span>
                </p>
                <div id="migration-result"></div>
            </div>
            <?php elseif ($migration_status === 'completed'): ?>
            <div class="notice notice-success is-dismissible">
                <p><?php _e('Migration completed successfully! The new queue system is active.', 'wc-delcampe-integration'); ?></p>
            </div>
            <?php endif; ?>
            
            <div id="delcampe-queue-status" class="delcampe-queue-dashboard">
                
                <!-- Queue Statistics -->
                <div class="delcampe-stats-grid">
                    <div class="stat-box pending">
                        <h3><?php _e('Pending', 'wc-delcampe-integration'); ?></h3>
                        <span class="stat-number" id="stat-pending"><?php echo intval($stats['pending']); ?></span>
                        <p><?php _e('Items waiting to be processed', 'wc-delcampe-integration'); ?></p>
                    </div>
                    
                    <div class="stat-box processing">
                        <h3><?php _e('Processing', 'wc-delcampe-integration'); ?></h3>
                        <span class="stat-number" id="stat-processing"><?php echo intval($stats['processing']); ?></span>
                        <p><?php _e('Items currently being processed', 'wc-delcampe-integration'); ?></p>
                    </div>
                    
                    <div class="stat-box published">
                        <h3><?php _e('Published', 'wc-delcampe-integration'); ?></h3>
                        <span class="stat-number" id="stat-published"><?php echo intval($stats['published']); ?></span>
                        <p><?php _e('Items successfully published', 'wc-delcampe-integration'); ?></p>
                    </div>
                    
                    <div class="stat-box errored">
                        <h3><?php _e('Errors', 'wc-delcampe-integration'); ?></h3>
                        <span class="stat-number" id="stat-errored"><?php echo intval($stats['errored']); ?></span>
                        <p><?php _e('Items that failed to process', 'wc-delcampe-integration'); ?></p>
                    </div>
                </div>
                
                <!-- Queue Controls -->
                <div class="delcampe-controls">
                    <div class="control-group">
                        <h3><?php _e('Queue Controls', 'wc-delcampe-integration'); ?></h3>
                        
                        <?php if ($is_paused): ?>
                        <button type="button" id="resume-queue" class="button button-primary">
                            <?php _e('Resume Queue', 'wc-delcampe-integration'); ?>
                        </button>
                        <p class="description"><?php _e('Queue processing is currently paused', 'wc-delcampe-integration'); ?></p>
                        <?php else: ?>
                        <button type="button" id="pause-queue" class="button">
                            <?php _e('Pause Queue', 'wc-delcampe-integration'); ?>
                        </button>
                        <p class="description"><?php _e('Queue processing is running', 'wc-delcampe-integration'); ?></p>
                        <?php endif; ?>
                        
                        <button type="button" id="clear-completed" class="button">
                            <?php _e('Clear Completed', 'wc-delcampe-integration'); ?>
                        </button>
                        <button type="button" id="refresh-stats" class="button">
                            <?php _e('Refresh', 'wc-delcampe-integration'); ?>
                        </button>
                    </div>
                </div>
                
                <!-- Performance Metrics -->
                <div class="delcampe-metrics">
                    <h3><?php _e('Performance Metrics', 'wc-delcampe-integration'); ?></h3>
                    
                    <table class="widefat fixed striped">
                        <tbody>
                            <tr>
                                <td><strong><?php _e('Average Processing Time', 'wc-delcampe-integration'); ?></strong></td>
                                <td id="avg-processing-time">
                                    <?php 
                                    $avg_time = $stats['avg_processing_time'];
                                    echo $avg_time ? number_format($avg_time, 1) . ' seconds' : __('No data', 'wc-delcampe-integration');
                                    ?>
                                </td>
                            </tr>
                            <tr>
                                <td><strong><?php _e('Total Items in Queue', 'wc-delcampe-integration'); ?></strong></td>
                                <td id="total-items"><?php echo intval($stats['total']); ?></td>
                            </tr>
                            <tr>
                                <td><strong><?php _e('Queue Status', 'wc-delcampe-integration'); ?></strong></td>
                                <td id="queue-status">
                                    <span class="status-indicator <?php echo $is_paused ? 'paused' : 'running'; ?>">
                                        <?php echo $is_paused ? __('Paused', 'wc-delcampe-integration') : __('Running', 'wc-delcampe-integration'); ?>
                                    </span>
                                </td>
                            </tr>
                            <tr>
                                <td><strong><?php _e('Next Cron Run', 'wc-delcampe-integration'); ?></strong></td>
                                <td id="next-cron">
                                    <?php 
                                    $next_run = wp_next_scheduled('delcampe_process_queue');
                                    echo $next_run ? date('Y-m-d H:i:s', $next_run) : __('Not scheduled', 'wc-delcampe-integration');
                                    ?>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                
                <!-- Recent Activity Log -->
                <div class="delcampe-activity">
                    <h3><?php _e('Recent Activity', 'wc-delcampe-integration'); ?></h3>
                    <div id="activity-log">
                        <?php $this->render_activity_log(); ?>
                    </div>
                </div>
                
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            
            // Auto-refresh stats every 30 seconds
            setInterval(function() {
                refreshStats();
            }, 30000);
            
            // Manual refresh button
            $('#refresh-stats').on('click', function() {
                refreshStats();
            });
            
            // Pause queue
            $('#pause-queue').on('click', function() {
                $.post(ajaxurl, {
                    action: 'delcampe_queue_pause',
                    nonce: '<?php echo wp_create_nonce('delcampe_queue_admin'); ?>'
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Error: ' + response.data.message);
                    }
                });
            });
            
            // Resume queue
            $('#resume-queue').on('click', function() {
                $.post(ajaxurl, {
                    action: 'delcampe_queue_resume',
                    nonce: '<?php echo wp_create_nonce('delcampe_queue_admin'); ?>'
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Error: ' + response.data.message);
                    }
                });
            });
            
            // Clear completed items
            $('#clear-completed').on('click', function() {
                if (confirm('<?php _e('Clear all completed and errored items?', 'wc-delcampe-integration'); ?>')) {
                    $.post(ajaxurl, {
                        action: 'delcampe_queue_clear',
                        nonce: '<?php echo wp_create_nonce('delcampe_queue_admin'); ?>'
                    }, function(response) {
                        if (response.success) {
                            refreshStats();
                            alert(response.data.message);
                        } else {
                            alert('Error: ' + response.data.message);
                        }
                    });
                }
            });
            
            function refreshStats() {
                $.post(ajaxurl, {
                    action: 'delcampe_queue_stats'
                }, function(response) {
                    if (response.success) {
                        var stats = response.data;
                        $('#stat-pending').text(stats.pending);
                        $('#stat-processing').text(stats.processing);
                        $('#stat-published').text(stats.published);
                        $('#stat-errored').text(stats.errored);
                        $('#total-items').text(stats.total);
                        
                        if (stats.avg_processing_time) {
                            $('#avg-processing-time').text(parseFloat(stats.avg_processing_time).toFixed(1) + ' seconds');
                        }
                    }
                });
            }
        });
        </script>
        
        <style>
        .delcampe-queue-dashboard {
            max-width: 1200px;
        }
        
        .delcampe-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        
        .stat-box {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 20px;
            text-align: center;
            border-left: 4px solid #ccc;
        }
        
        .stat-box.pending { border-left-color: #ffb900; }
        .stat-box.processing { border-left-color: #00a0d2; }
        .stat-box.published { border-left-color: #46b450; }
        .stat-box.errored { border-left-color: #dc3232; }
        
        .stat-box h3 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #666;
        }
        
        .stat-number {
            display: block;
            font-size: 36px;
            font-weight: bold;
            line-height: 1;
            margin: 10px 0;
        }
        
        .stat-box p {
            margin: 10px 0 0 0;
            font-size: 13px;
            color: #666;
        }
        
        .delcampe-controls, .delcampe-metrics, .delcampe-activity {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 20px;
            margin: 20px 0;
        }
        
        .delcampe-controls h3, .delcampe-metrics h3, .delcampe-activity h3 {
            margin-top: 0;
        }
        
        .control-group button {
            margin-right: 10px;
        }
        
        .status-indicator {
            padding: 4px 8px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 12px;
            text-transform: uppercase;
        }
        
        .status-indicator.running {
            background: #46b450;
            color: white;
        }
        
        .status-indicator.paused {
            background: #ffb900;
            color: white;
        }
        
        #activity-log {
            max-height: 300px;
            overflow-y: auto;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 10px;
            background: #f9f9f9;
            font-family: monospace;
            font-size: 12px;
        }
        </style>
        <?php
    }

    /**
     * AJAX: Bulk queue actions (retry, cancel, move_top, delete)
     */
    public function ajax_queue_bulk_action() {
        if (!current_user_can('manage_woocommerce')) { wp_die(__('Insufficient permissions', 'wc-delcampe-integration')); }
        check_ajax_referer('delcampe_queue_bulk');
        global $wpdb; $table = $wpdb->prefix . 'delcampe_queue';
        $ids = isset($_POST['ids']) && is_array($_POST['ids']) ? array_map('intval', $_POST['ids']) : array();
        $act = isset($_POST['bulk_action']) ? sanitize_key($_POST['bulk_action']) : '';
        if (empty($ids) || !$act) { wp_send_json_error(array('message'=>'No items selected')); }
        $in = '(' . implode(',', $ids) . ')';
        switch ($act) {
            case 'retry':
                $wpdb->query("UPDATE {$table} SET state='pending', attempts=0, next_attempt_at=NULL, worker_id=NULL, locked_at=NULL WHERE id IN {$in}");
                break;
            case 'cancel':
                $wpdb->query("UPDATE {$table} SET state='errored', error_message='Cancelled by admin', worker_id=NULL, locked_at=NULL WHERE id IN {$in} AND state<>'processing'");
                break;
            case 'move_top':
                $wpdb->query("UPDATE {$table} SET priority = priority + 10, next_attempt_at=NULL WHERE id IN {$in}");
                break;
            case 'delete':
                $wpdb->query("DELETE FROM {$table} WHERE id IN {$in} AND state<>'processing'");
                break;
            default:
                wp_send_json_error(array('message'=>'Unknown action'));
        }
        wp_send_json_success(array('message'=>'Bulk action applied'));
    }
    
    /**
     * Render activity log
     */
    private function render_activity_log() {
        // Get recent log entries
        $log_file = WP_CONTENT_DIR . '/uploads/delcampe-debug.log';
        
        if (!file_exists($log_file)) {
            echo __('No recent activity', 'wc-delcampe-integration');
            return;
        }
        
        $lines = file($log_file);
        $recent_lines = array_slice($lines, -20); // Last 20 lines
        
        foreach ($recent_lines as $line) {
            if (strpos($line, '[Queue') !== false || strpos($line, '[Delcampe Queue]') !== false) {
                echo esc_html($line) . "\n";
            }
        }
    }

    /**
     * Render embedded queue monitor (for use inside other admin tabs).
     */
    public function render_embedded() {
        $queue = Delcampe_Queue::get_instance();
        $stats = $queue->get_stats();
        $is_paused = get_transient('delcampe_queue_paused');
        ?>
        <input type="hidden" id="delcampe_queue_nonce" value="<?php echo esc_attr( wp_create_nonce('delcampe_queue_action') ); ?>" />
        <div id="delcampe-queue-status" class="delcampe-queue-dashboard">
            <div class="delcampe-stats-grid">
                <div class="stat-box pending">
                    <h3><?php _e('Pending', 'wc-delcampe-integration'); ?></h3>
                    <span class="stat-number" id="stat-pending"><?php echo intval($stats['pending']); ?></span>
                    <p><?php _e('Items waiting to be processed', 'wc-delcampe-integration'); ?></p>
                </div>
                <div class="stat-box processing">
                    <h3><?php _e('Processing', 'wc-delcampe-integration'); ?></h3>
                    <span class="stat-number" id="stat-processing"><?php echo intval($stats['processing']); ?></span>
                    <p><?php _e('Items currently being processed', 'wc-delcampe-integration'); ?></p>
                </div>
                <div class="stat-box published">
                    <h3><?php _e('Published', 'wc-delcampe-integration'); ?></h3>
                    <span class="stat-number" id="stat-published"><?php echo intval($stats['published']); ?></span>
                    <p><?php _e('Items successfully published', 'wc-delcampe-integration'); ?></p>
                </div>
                <div class="stat-box errored">
                    <h3><?php _e('Errored', 'wc-delcampe-integration'); ?></h3>
                    <span class="stat-number" id="stat-errored"><?php echo intval($stats['errored']); ?></span>
                    <p><?php _e('Items that need attention', 'wc-delcampe-integration'); ?></p>
                </div>
            </div>
            <div class="delcampe-controls">
                <h3><?php _e('Controls', 'wc-delcampe-integration'); ?></h3>
                <div class="control-group">
                    <?php if ($is_paused): ?>
                        <button type="button" id="resume-queue" class="button button-primary"><?php _e('Resume Queue', 'wc-delcampe-integration'); ?></button>
                        <p class="description"><?php _e('Queue processing is currently paused', 'wc-delcampe-integration'); ?></p>
                    <?php else: ?>
                        <button type="button" id="pause-queue" class="button"><?php _e('Pause Queue', 'wc-delcampe-integration'); ?></button>
                        <p class="description"><?php _e('Queue processing is running', 'wc-delcampe-integration'); ?></p>
                    <?php endif; ?>
                    <button type="button" id="clear-completed" class="button"><?php _e('Clear Completed', 'wc-delcampe-integration'); ?></button>
                    <button type="button" id="refresh-stats" class="button"><?php _e('Refresh', 'wc-delcampe-integration'); ?></button>
                </div>
            </div>
            <div class="delcampe-metrics">
                <h3><?php _e('Performance Metrics', 'wc-delcampe-integration'); ?></h3>
                <table class="widefat fixed striped"><tbody>
                    <tr>
                        <td><strong><?php _e('Average Processing Time', 'wc-delcampe-integration'); ?></strong></td>
                        <td id="avg-processing-time"><?php echo $stats['avg_processing_time'] ? number_format($stats['avg_processing_time'], 1) . ' seconds' : __('No data', 'wc-delcampe-integration'); ?></td>
                    </tr>
                    <tr>
                        <td><strong><?php _e('Total Items in Queue', 'wc-delcampe-integration'); ?></strong></td>
                        <td id="total-items"><?php echo intval($stats['total']); ?></td>
                    </tr>
                    <tr>
                        <td><strong><?php _e('Queue Status', 'wc-delcampe-integration'); ?></strong></td>
                        <td id="queue-status"><span class="status-indicator <?php echo $is_paused ? 'paused' : 'running'; ?>"><?php echo $is_paused ? __('Paused', 'wc-delcampe-integration') : __('Running', 'wc-delcampe-integration'); ?></span></td>
                    </tr>
                </tbody></table>
            </div>
            <div class="delcampe-activity">
                <h3><?php _e('Recent Activity', 'wc-delcampe-integration'); ?></h3>
                <div id="activity-log"><?php $this->render_activity_log(); ?></div>
            </div>
            
            <!-- Queue Items -->
            <div class="delcampe-activity">
                <h3><?php _e('Queue Items', 'wc-delcampe-integration'); ?></h3>
                <?php $this->render_items_table(); ?>
            </div>
        </div>
        <script>
        jQuery(function($){
            function refreshStats(){
                $.post(ajaxurl,{action:'delcampe_queue_stats'},function(resp){
                    if(resp && resp.success){
                        var s=resp.data; $('#stat-pending').text(s.pending); $('#stat-processing').text(s.processing); $('#stat-published').text(s.published); $('#stat-errored').text(s.errored); $('#total-items').text(s.total); if(s.avg_processing_time){ $('#avg-processing-time').text(parseFloat(s.avg_processing_time).toFixed(1)+' seconds'); }
                    }
                });
            }
            setInterval(refreshStats,30000);
            $('#refresh-stats').on('click', refreshStats);
            $('#pause-queue').on('click', function(){ $.post(ajaxurl,{action:'delcampe_queue_pause',nonce:'<?php echo wp_create_nonce('delcampe_queue_admin'); ?>'}, function(){ refreshStats(); }); });
            $('#resume-queue').on('click', function(){ $.post(ajaxurl,{action:'delcampe_queue_resume',nonce:'<?php echo wp_create_nonce('delcampe_queue_admin'); ?>'}, function(){ refreshStats(); }); });
            $('#clear-completed').on('click', function(){ if(!confirm('<?php echo esc_js(__('Clear all completed and errored items?', 'wc-delcampe-integration')); ?>')) return; $.post(ajaxurl,{action:'delcampe_queue_clear',nonce:'<?php echo wp_create_nonce('delcampe_queue_admin'); ?>'}, function(){ refreshStats(); }); });
        });
        </script>
        <?php
    }

    /**
     * Render a simple table of current queue items with bulk actions.
     */
    private function render_items_table() {
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_queue';
        $rows = $wpdb->get_results("SELECT id, product_id, action, state, priority, attempts, next_attempt_at, created_at FROM {$table} ORDER BY id ASC LIMIT 200");
        if (empty($rows)) {
            echo '<p>' . esc_html__('No items in the queue.', 'wc-delcampe-integration') . '</p>';
            return;
        }
        $nonce = wp_create_nonce('delcampe_queue_bulk');
        echo '<form id="delcampe-queue-bulk" method="post" action="' . esc_url(admin_url('admin-ajax.php')) . '">';
        echo '<input type="hidden" name="action" value="delcampe_queue_bulk_action" />';
        echo '<input type="hidden" name="_wpnonce" value="' . esc_attr($nonce) . '" />';
        echo '<div style="margin-bottom:10px">';
        echo '<select name="bulk_action">';
        echo '<option value="retry">' . esc_html__('Retry (set to pending)', 'wc-delcampe-integration') . '</option>';
        echo '<option value="cancel">' . esc_html__('Cancel (mark errored)', 'wc-delcampe-integration') . '</option>';
        echo '<option value="move_top">' . esc_html__('Move to Top (priority++)', 'wc-delcampe-integration') . '</option>';
        echo '<option value="delete">' . esc_html__('Delete', 'wc-delcampe-integration') . '</option>';
        echo '</select> ';
        echo '<button class="button">' . esc_html__('Apply', 'wc-delcampe-integration') . '</button>';
        echo '</div>';
        echo '<table class="widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th style="width:24px"><input type="checkbox" id="qsel_all" /></th>';
        echo '<th>' . esc_html__('ID', 'wc-delcampe-integration') . '</th>';
        echo '<th>' . esc_html__('Product', 'wc-delcampe-integration') . '</th>';
        echo '<th>' . esc_html__('Action', 'wc-delcampe-integration') . '</th>';
        echo '<th>' . esc_html__('State', 'wc-delcampe-integration') . '</th>';
        echo '<th>' . esc_html__('Priority', 'wc-delcampe-integration') . '</th>';
        echo '<th>' . esc_html__('Attempts', 'wc-delcampe-integration') . '</th>';
        echo '<th>' . esc_html__('Next Attempt', 'wc-delcampe-integration') . '</th>';
        echo '<th>' . esc_html__('Created', 'wc-delcampe-integration') . '</th>';
        echo '</tr></thead><tbody>';
        foreach ($rows as $r) {
            $title = get_the_title($r->product_id);
            echo '<tr>';
            echo '<td><input type="checkbox" name="ids[]" value="' . (int)$r->id . '" /></td>';
            echo '<td>#' . (int)$r->id . '</td>';
            echo '<td>#' . (int)$r->product_id . ' ' . esc_html($title ?: '') . '</td>';
            echo '<td>' . esc_html($r->action) . '</td>';
            echo '<td>' . esc_html($r->state) . '</td>';
            echo '<td>' . (int)$r->priority . '</td>';
            echo '<td>' . (int)$r->attempts . '</td>';
            echo '<td>' . esc_html($r->next_attempt_at ?: '-') . '</td>';
            echo '<td>' . esc_html($r->created_at) . '</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        echo '</form>';
        ?>
        <script>
        jQuery(function($){
            // Select all toggle
            $('#qsel_all').on('change', function(){
                $('#delcampe-queue-bulk input[type=checkbox][name="ids[]"]').prop('checked', $(this).prop('checked'));
            });

            // AJAX bulk action submit
            $('#delcampe-queue-bulk').on('submit', function(e){
                e.preventDefault();
                var $f = $(this);
                var data = $f.serialize();
                $.post($f.attr('action'), data)
                 .done(function(resp){
                    try { if (typeof resp === 'string') resp = JSON.parse(resp); } catch(e){}
                    if (resp && resp.success) {
                        alert(resp.data && resp.data.message ? resp.data.message : 'Done');
                        // Simple refresh to reflect changes
                        location.reload();
                    } else {
                        alert('Error: ' + (resp && resp.data && resp.data.message ? resp.data.message : 'Bulk action failed'));
                    }
                 })
                 .fail(function(){ alert('Network error'); });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX handler for queue stats
     */
    public function ajax_queue_stats() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        $queue = Delcampe_Queue::get_instance();
        $stats = $queue->get_stats();
        
        wp_send_json_success($stats);
    }
    
    /**
     * AJAX handler for pausing queue
     */
    public function ajax_queue_pause() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        check_ajax_referer('delcampe_queue_admin', 'nonce');
        
        set_transient('delcampe_queue_paused', true, HOUR_IN_SECONDS);
        
        wp_send_json_success(array(
            'message' => __('Queue processing paused', 'wc-delcampe-integration')
        ));
    }
    
    /**
     * AJAX handler for resuming queue
     */
    public function ajax_queue_resume() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        check_ajax_referer('delcampe_queue_admin', 'nonce');
        
        delete_transient('delcampe_queue_paused');
        
        wp_send_json_success(array(
            'message' => __('Queue processing resumed', 'wc-delcampe-integration')
        ));
    }
    
    /**
     * AJAX handler for clearing completed items
     */
    public function ajax_queue_clear() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        check_ajax_referer('delcampe_queue_admin', 'nonce');
        
        $queue = Delcampe_Queue::get_instance();
        $cleared = $queue->cleanup(0); // Clear all completed/errored items
        
        wp_send_json_success(array(
            'message' => sprintf(__('Cleared %d completed items', 'wc-delcampe-integration'), $cleared)
        ));
    }
}
