<?php
/**
 * Delcampe Queue Worker
 * 
 * Handles the actual processing of queue items, API calls, and state management.
 * Includes rate limiting, error handling, and worker lease management.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Queue_Worker {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Worker ID for this instance
     */
    private $worker_id;
    
    /**
     * Rate limiting
     */
    private $rate_limit_key = 'delcampe_api_rate_limit';
    private $requests_per_minute = 150;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->worker_id = 'worker_' . getmypid() . '_' . time();
        $this->requests_per_minute = apply_filters('delcampe_rate_limit_per_minute', 150);
    }
    
    /**
     * Process a batch of queue items
     * 
     * @param int $batch_size Maximum number of items to process (default: unlimited)
     * @return array Processing results
     */
    public function process_batch($batch_size = 9999) {
        $queue = Delcampe_Queue::get_instance();
        $processed = 0;
        $errors = 0;
        $start_time = time();
        
        delcampe_log("[Queue Worker {$this->worker_id}] Starting batch processing (max: {$batch_size})");
        
        while ($processed < $batch_size) {
            // Check if we should continue processing
            if (!$this->should_continue_processing($start_time)) {
                break;
            }
            
            // Get next item from queue
            $item = $queue->dequeue($this->worker_id);
            if (!$item) {
                break; // No more items
            }
            
            // Check rate limit before processing
            if (!$this->check_rate_limit()) {
                // Requeue item and wait
                $this->requeue_for_rate_limit($item);
                delcampe_log("[Queue Worker {$this->worker_id}] Rate limit reached, stopping batch");
                break;
            }
            
            // Process the item
            $result = $this->process_item($item);
            
            if ($result === true) {
                $processed++;
            } else {
                $errors++;
            }
            
            // Small delay between items to be gentle on the server
            usleep(100000); // 0.1 seconds
        }
        
        $duration = time() - $start_time;
        delcampe_log("[Queue Worker {$this->worker_id}] Batch complete: {$processed} processed, {$errors} errors in {$duration}s");
        
        // Clear the processing status
        delete_transient('delcampe_current_processing');
        
        return array(
            'processed' => $processed,
            'errors' => $errors,
            'duration' => $duration
        );
    }
    
    /**
     * Process a single queue item
     * 
     * @param object $item Queue item
     * @return bool Success
     */
    public function process_item($item) {
        try {
            delcampe_log("[Queue Worker {$this->worker_id}] Processing item {$item->id}: product {$item->product_id}, action {$item->action}");
            
            // Get the product
            $product = wc_get_product($item->product_id);
            if (!$product) {
                $this->fail_item($item, 'Product not found', true);
                return false;
            }
            
            // Store current processing status for real-time display
            $sku = $product->get_sku();
            $title = $product->get_name();
            set_transient('delcampe_current_processing', array(
                'product_id' => $item->product_id,
                'sku' => $sku ?: 'No SKU',
                'title' => substr($title, 0, 50) . (strlen($title) > 50 ? '...' : ''),
                'action' => $item->action,
                'timestamp' => time()
            ), 60);
            
            // Check if product is in valid state for publishing
            if ($item->action === Delcampe_Queue::ACTION_CREATE) {
                if ($product->get_status() !== 'publish') {
                    $this->fail_item($item, 'Product is not published', true);
                    return false;
                }
                
                // Check if already has Delcampe listing
                $existing_listing_id = get_post_meta($item->product_id, '_delcampe_listing_id', true);
                if ($existing_listing_id) {
                    $this->complete_item($item, $existing_listing_id);
                    return true;
                }
            }
            
            // Record API request for rate limiting
            $this->record_api_request();
            
            // Perform the actual API call based on action
            switch ($item->action) {
                case Delcampe_Queue::ACTION_CREATE:
                    return $this->create_listing($item, $product);
                    
                case Delcampe_Queue::ACTION_UPDATE:
                    return $this->update_listing($item, $product);
                    
                case Delcampe_Queue::ACTION_DELETE:
                    return $this->delete_listing($item, $product);
                    
                default:
                    $this->fail_item($item, 'Unknown action: ' . $item->action, true);
                    return false;
            }
            
        } catch (Exception $e) {
            delcampe_log("[Queue Worker {$this->worker_id}] Exception processing item {$item->id}: " . $e->getMessage());
            $this->fail_item($item, 'Exception: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create a new Delcampe listing
     * 
     * @param object $item Queue item
     * @param WC_Product $product WooCommerce product
     * @return bool Success
     */
    private function create_listing($item, $product) {
        if (function_exists('delcampe_publish_audit')) {
            delcampe_publish_audit(array(
                'phase' => 'worker_start',
                'endpoint' => '/item',
                'product_id' => (int)$item->product_id,
                'action' => 'create'
            ));
        }
        // Get profile for this product
        $profile_id = get_post_meta($item->product_id, '_delcampe_profile_id', true);
        if (!$profile_id) {
            $this->fail_item($item, 'No Delcampe profile assigned to product', true);
            return false;
        }
        
        // Use existing listing API
        $listing_api = Delcampe_Listing_API::get_instance();
        
        // Create listing via API with correct parameters
        $result = $listing_api->create_listing($item->product_id, $profile_id);
        
        if (is_wp_error($result)) {
            $error_code = $result->get_error_code();
            $error_message = $result->get_error_message();
            
            // Determine if this is a permanent error
            $permanent_errors = ['invalid_profile', 'invalid_product', 'validation_error'];
            $is_permanent = in_array($error_code, $permanent_errors);
            
            $this->fail_item($item, "API Error ({$error_code}): {$error_message}", $is_permanent);
            return false;
        }
        
        // Check if item was queued by Delcampe (common case)
        if (is_array($result) && isset($result['queued']) && $result['queued'] === true) {
            // Item is queued on Delcampe side - this is a success
            // Store personal reference if available for later tracking
            $personal_reference = isset($result['personal_reference']) ? $result['personal_reference'] : null;
            if ($personal_reference) {
                update_post_meta($item->product_id, '_delcampe_personal_reference', $personal_reference);
            }
            
            delcampe_log("[Queue Worker {$this->worker_id}] Product {$item->product_id} queued on Delcampe - waiting 2 seconds to check for ID");
            
            // Wait 2 seconds for Delcampe to process and potentially assign an ID
            sleep(2);
            
            // Try to get the listing ID using the personal reference
            $listing_id = $this->check_for_delcampe_id($item->product_id, $personal_reference);
            
            if ($listing_id) {
                // Got an ID! Mark as completed
                $this->complete_item($item, $listing_id);
                delcampe_log("[Queue Worker {$this->worker_id}] Got listing ID {$listing_id} after 2-second wait for product {$item->product_id}");
            } else {
                // No ID yet - mark as verifying for later resolution
                $queue = Delcampe_Queue::get_instance();
                $queue->mark_verifying($item->id, (int)$item->product_id, $personal_reference);
                delcampe_log("[Queue Worker {$this->worker_id}] No ID after 2-second wait - marked as verifying for product {$item->product_id}");
            }
        } else if (is_array($result)) {
            // Treat any explicit success response as completion even if ID is not present
            if (!empty($result['success'])) {
                $listing_id = $this->extract_listing_id($result);
                if ($listing_id) {
                    $this->complete_item($item, $listing_id);
                    delcampe_log("[Queue Worker {$this->worker_id}] Created listing {$listing_id} for product {$item->product_id}");
                } else {
                    // No immediate ID - wait 2 seconds and check
                    $personal_reference = isset($result['personal_reference']) ? $result['personal_reference'] : null;
                    delcampe_log("[Queue Worker {$this->worker_id}] Success but no ID - waiting 2 seconds to check for product {$item->product_id}");
                    
                    sleep(2);
                    
                    // Try to get the listing ID using the personal reference
                    $listing_id = $this->check_for_delcampe_id($item->product_id, $personal_reference);
                    
                    if ($listing_id) {
                        // Got an ID after waiting!
                        $this->complete_item($item, $listing_id);
                        delcampe_log("[Queue Worker {$this->worker_id}] Got listing ID {$listing_id} after 2-second wait for product {$item->product_id}");
                        if (function_exists('delcampe_publish_audit')) {
                            delcampe_publish_audit(array(
                                'phase' => 'worker_completed_after_wait',
                                'product_id' => (int)$item->product_id,
                                'action' => 'create',
                                'listing_id' => $listing_id
                            ));
                        }
                    } else {
                        // Still no ID - move to verifying
                        $queue = Delcampe_Queue::get_instance();
                        $queue->mark_verifying($item->id, (int)$item->product_id, $personal_reference);
                        delcampe_log("[Queue Worker {$this->worker_id}] No ID after 2-second wait - marked as verifying for product {$item->product_id}");
                        if (function_exists('delcampe_publish_audit')) {
                            delcampe_publish_audit(array(
                                'phase' => 'worker_verifying',
                                'product_id' => (int)$item->product_id,
                                'action' => 'create',
                                'reason' => 'no_id_after_wait'
                            ));
                        }
                    }
                }
                return true;
            }
            
            // Extract listing ID from response (immediate creation case)
            $listing_id = $this->extract_listing_id($result);
            if ($listing_id) {
                $this->complete_item($item, $listing_id);
                delcampe_log("[Queue Worker {$this->worker_id}] Created listing {$listing_id} for product {$item->product_id}");
                return true;
            }
            
            // Unknown shape: fail with context
            $this->fail_item($item, 'No listing ID returned from API and item not queued');
            return false;
        } else {
            // Non-array unexpected result
            $this->fail_item($item, 'Unexpected API result type for create');
            return false;
        }

        return true;
    }
    
    /**
     * Update an existing Delcampe listing
     * 
     * @param object $item Queue item
     * @param WC_Product $product WooCommerce product
     * @return bool Success
     */
    private function update_listing($item, $product) {
        if (function_exists('delcampe_publish_audit')) {
            delcampe_publish_audit(array(
                'phase' => 'worker_start',
                'endpoint' => '/item/{id}',
                'product_id' => (int)$item->product_id,
                'action' => 'update',
                'listing_id' => (string) get_post_meta($item->product_id, '_delcampe_listing_id', true)
            ));
        }
        // Get existing listing ID
        $listing_id = get_post_meta($item->product_id, '_delcampe_listing_id', true);
        if (!$listing_id) {
            // Convert to create action
            $item->action = Delcampe_Queue::ACTION_CREATE;
            return $this->create_listing($item, $product);
        }
        
        // Get profile for this product
        $profile_id = get_post_meta($item->product_id, '_delcampe_profile_id', true);
        if (!$profile_id) {
            $this->fail_item($item, 'No Delcampe profile assigned to product', true);
            return false;
        }
        
        // Use existing listing API
        $listing_api = Delcampe_Listing_API::get_instance();
        
        // Update listing via API with correct parameters
        $result = $listing_api->update_listing($item->product_id, $listing_id);
        
        if (is_wp_error($result)) {
            $error_code = $result->get_error_code();
            $error_message = $result->get_error_message();
            
            // Handle 404 - listing doesn't exist anymore
            if ($error_code === 'not_found') {
                // Convert to create action
                $item->action = Delcampe_Queue::ACTION_CREATE;
                return $this->create_listing($item, $product);
            }
            
            $permanent_errors = ['invalid_profile', 'invalid_product', 'validation_error'];
            $is_permanent = in_array($error_code, $permanent_errors);
            
            $this->fail_item($item, "API Error ({$error_code}): {$error_message}", $is_permanent);
            return false;
        }
        
        // Mark as completed
        $this->complete_item($item, $listing_id);
        
        delcampe_log("[Queue Worker {$this->worker_id}] Updated listing {$listing_id} for product {$item->product_id}");
        
        return true;
    }
    
    /**
     * Delete a Delcampe listing
     * 
     * @param object $item Queue item
     * @param WC_Product $product WooCommerce product
     * @return bool Success
     */
    private function delete_listing($item, $product) {
        // Get existing listing ID
        $listing_id = get_post_meta($item->product_id, '_delcampe_listing_id', true);
        if (!$listing_id) {
            // Already deleted or never existed
            $this->complete_item($item);
            return true;
        }
        
        // Use existing listing closer
        $listing_closer = Delcampe_Listing_Closer::get_instance();
        
        // Close/delete listing via API
        $result = $listing_closer->close_listing($listing_id);
        
        if (is_wp_error($result)) {
            $error_code = $result->get_error_code();
            $error_message = $result->get_error_message();
            
            // 404 means already deleted - that's fine
            if ($error_code === 'not_found') {
                $this->complete_item($item);
                return true;
            }
            
            $this->fail_item($item, "API Error ({$error_code}): {$error_message}");
            return false;
        }
        
        // Clean up local data
        delete_post_meta($item->product_id, '_delcampe_listing_id');
        delete_post_meta($item->product_id, '_delcampe_sync_status');
        
        // Mark as completed
        $this->complete_item($item);
        
        delcampe_log("[Queue Worker {$this->worker_id}] Deleted listing {$listing_id} for product {$item->product_id}");
        
        return true;
    }
    
    /**
     * Build listing data from product and profile
     * 
     * @param WC_Product $product WooCommerce product
     * @param int $profile_id Profile ID
     * @return array Listing data
     */
    
    /**
     * Extract listing ID from API response
     * 
     * @param mixed $response API response
     * @return string|null Listing ID
     */
    private function extract_listing_id($response) {
        // Extract listing ID from the API response
        if (is_array($response)) {
            if (isset($response['listing_id'])) {
                return $response['listing_id'];
            }
            if (isset($response['id'])) {
                return $response['id'];
            }
            if (isset($response['delcampe_id'])) {
                return $response['delcampe_id'];
            }
        }
        return null;
    }
    
    /**
     * Check for Delcampe ID using personal reference
     * 
     * @param int $product_id Product ID
     * @param string $personal_reference Personal reference (SKU)
     * @return string|null Delcampe listing ID if found, null otherwise
     */
    private function check_for_delcampe_id($product_id, $personal_reference) {
        if (empty($personal_reference)) {
            return null;
        }
        
        try {
            // Get auth token
            $auth = Delcampe_Auth::get_instance();
            $token = $auth->get_auth_token();
            
            if (is_wp_error($token)) {
                delcampe_log("[Queue Worker] Failed to get auth token for ID check: " . $token->get_error_message());
                return null;
            }
            
            // Query Delcampe API for item by personal reference
            $api_url = DELCAMPE_API_URL . '/item/reference/' . urlencode($personal_reference) . '?token=' . $token;
            
            $response = wp_remote_get($api_url, array(
                'timeout' => 10,
                'headers' => array(
                    'Accept' => 'application/xml'
                )
            ));
            
            if (is_wp_error($response)) {
                delcampe_log("[Queue Worker] API error checking for ID: " . $response->get_error_message());
                return null;
            }
            
            $body = wp_remote_retrieve_body($response);
            $status_code = wp_remote_retrieve_response_code($response);
            
            if ($status_code !== 200) {
                return null;
            }
            
            // Parse XML response
            libxml_use_internal_errors(true);
            $xml = simplexml_load_string($body);
            
            if ($xml === false) {
                return null;
            }
            
            // Look for the item ID in the response
            if (isset($xml->item->id)) {
                $listing_id = (string)$xml->item->id;
                if (!empty($listing_id)) {
                    // Store the ID immediately
                    update_post_meta($product_id, '_delcampe_item_id', $listing_id);
                    update_post_meta($product_id, '_delcampe_listing_id', $listing_id);
                    
                    // Update listings table if exists
                    global $wpdb;
                    $table_name = $wpdb->prefix . 'delcampe_listings';
                    $wpdb->update(
                        $table_name,
                        array(
                            'delcampe_id' => $listing_id,
                            'status' => 'published'
                        ),
                        array('product_id' => $product_id),
                        array('%s', '%s'),
                        array('%d')
                    );
                    
                    return $listing_id;
                }
            }
            
            return null;
            
        } catch (Exception $e) {
            delcampe_log("[Queue Worker] Exception checking for ID: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Mark item as completed successfully
     * 
     * @param object $item Queue item
     * @param string $listing_id Delcampe listing ID (optional)
     */
    private function complete_item($item, $listing_id = null) {
        $queue = Delcampe_Queue::get_instance();
        $queue->mark_completed($item->id, $listing_id);
        if (function_exists('delcampe_publish_audit')) {
            delcampe_publish_audit(array(
                'phase' => 'worker_completed',
                'product_id' => (int)$item->product_id,
                'action' => $item->action,
                'listing_id' => $listing_id ? (string)$listing_id : null
            ));
        }
    }
    
    /**
     * Mark item as failed
     * 
     * @param object $item Queue item
     * @param string $error_message Error message
     * @param bool $is_permanent Whether this is a permanent failure
     */
    private function fail_item($item, $error_message, $is_permanent = false) {
        $queue = Delcampe_Queue::get_instance();
        $queue->mark_failed($item->id, $error_message, $is_permanent);
        if (function_exists('delcampe_publish_audit')) {
            delcampe_publish_audit(array(
                'phase' => 'worker_failed',
                'product_id' => (int)$item->product_id,
                'action' => $item->action,
                'permanent' => (bool)$is_permanent,
                'error' => (string)$error_message
            ));
        }
    }
    
    /**
     * Check rate limit
     * 
     * @return bool Whether we can make another API request
     */
    private function check_rate_limit() {
        $current_minute = floor(time() / 60);
        $requests = get_transient($this->rate_limit_key . '_' . $current_minute);
        
        if ($requests === false) {
            $requests = 0;
        }
        
        return $requests < $this->requests_per_minute;
    }
    
    /**
     * Record an API request for rate limiting
     */
    private function record_api_request() {
        $current_minute = floor(time() / 60);
        $key = $this->rate_limit_key . '_' . $current_minute;
        $requests = get_transient($key);
        
        if ($requests === false) {
            $requests = 0;
        }
        
        $requests++;
        set_transient($key, $requests, 120); // Keep for 2 minutes
    }
    
    /**
     * Requeue item due to rate limit
     * 
     * @param object $item Queue item
     */
    private function requeue_for_rate_limit($item) {
        global $wpdb;
        
        $queue = Delcampe_Queue::get_instance();
        $next_minute = ceil(time() / 60) * 60; // Next minute boundary
        
        $wpdb->update(
            $queue->get_table_name(),
            array(
                'state' => Delcampe_Queue::STATE_PENDING,
                'next_attempt_at' => date('Y-m-d H:i:s', $next_minute),
                'worker_id' => null,
                'locked_at' => null,
                'updated_at' => current_time('mysql')
            ),
            array('id' => $item->id)
        );
    }
    
    /**
     * Check if we should continue processing
     * 
     * @param int $start_time Processing start time
     * @return bool Whether to continue
     */
    private function should_continue_processing($start_time) {
        // Stop if we've been running too long (5 minutes)
        if (time() - $start_time > 300) {
            return false;
        }
        
        // Stop if memory usage is too high
        if (memory_get_usage(true) > (256 * 1024 * 1024)) { // 256MB
            return false;
        }
        
        // Check for pause flag
        if (get_transient('delcampe_queue_paused')) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Get worker statistics
     * 
     * @return array Worker stats
     */
    public function get_stats() {
        return array(
            'worker_id' => $this->worker_id,
            'pid' => getmypid(),
            'memory_usage' => memory_get_usage(true),
            'memory_peak' => memory_get_peak_usage(true),
            'rate_limit_remaining' => $this->get_rate_limit_remaining()
        );
    }
    
    /**
     * Get remaining API requests for current minute
     * 
     * @return int Remaining requests
     */
    private function get_rate_limit_remaining() {
        $current_minute = floor(time() / 60);
        $requests = get_transient($this->rate_limit_key . '_' . $current_minute);
        
        if ($requests === false) {
            $requests = 0;
        }
        
        return max(0, $this->requests_per_minute - $requests);
    }
}
