<?php
/**
 * Delcampe API Rate Limit Monitor
 * 
 * Monitors API rate limits and provides warnings when approaching limits
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Core
 * @since 1.10.12.4
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Delcampe_Rate_Limit
 * 
 * Handles API rate limit monitoring and management
 * 
 * @since 1.10.12.4
 */
class Delcampe_Rate_Limit {
    
    /**
     * Singleton instance
     * @var Delcampe_Rate_Limit
     */
    private static $instance = null;
    
    /**
     * Cache key for rate limit data
     * @var string
     */
    const CACHE_KEY = 'delcampe_rate_limit';
    
    /**
     * Cache duration (1 minute for more frequent updates)
     * @var int
     */
    const CACHE_DURATION = 60;
    
    /**
     * Get singleton instance
     * 
     * @return Delcampe_Rate_Limit
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Add AJAX handlers
        add_action('wp_ajax_delcampe_get_rate_limit', array($this, 'ajax_get_rate_limit'));
        
        // Add admin bar indicator
        add_action('admin_bar_menu', array($this, 'add_admin_bar_indicator'), 100);
        
        // Add admin notices for warnings
        add_action('admin_notices', array($this, 'display_rate_limit_warning'));
        
        // Add rate limit display to all Delcampe admin pages
        add_action('admin_notices', array($this, 'display_rate_limit_on_delcampe_pages'), 5);
    }
    
    /**
     * Get current rate limit status
     * 
     * @param bool $force_refresh Force API call instead of using cache
     * @return array|WP_Error Rate limit data or error
     */
    public function get_rate_limit($force_refresh = false) {
        // Check cache first
        if (!$force_refresh) {
            $cached = get_transient(self::CACHE_KEY);
            if ($cached !== false) {
                return $cached;
            }
        }
        
        // Get authentication token
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        // Make API request
        $api_url = DELCAMPE_API_BASE_URL . '/item/rate-limit?token=' . $token;
        
        $response = wp_remote_get($api_url, array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => DELCAMPE_USER_AGENT,
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            // Check response headers for rate limit info
            $headers = wp_remote_retrieve_headers($response);
            if (isset($headers['x-ratelimit-remaining'])) {
                $data = array(
                    'remaining' => intval($headers['x-ratelimit-remaining']),
                    'limit' => intval($headers['x-ratelimit-limit'] ?? 10000),
                    'reset' => $headers['x-ratelimit-reset'] ?? '',
                    'source' => 'headers'
                );
                set_transient(self::CACHE_KEY, $data, self::CACHE_DURATION);
                return $data;
            }
            
            return new WP_Error('api_error', sprintf(__('API returned error code: %d', 'wc-delcampe-integration'), $response_code));
        }
        
        // Parse XML response
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($response_body);
        
        if ($xml === false) {
            return new WP_Error('xml_parse_error', __('Failed to parse XML response', 'wc-delcampe-integration'));
        }
        
        // Check if we have the raw count value from API
        $remaining_count = null;
        $reset_date = '';
        
        // The API wraps the response in Notification_Data
        if (isset($xml->Notification_Data->body->rate_limit)) {
            $rate_limit = $xml->Notification_Data->body->rate_limit;
            $remaining_count = intval($rate_limit->count ?? 0);
            $reset_date = (string) ($rate_limit->renew_date ?? '');
        } elseif (isset($xml->body->rate_limit)) {
            $rate_limit = $xml->body->rate_limit;
            $remaining_count = intval($rate_limit->count ?? 0);
            $reset_date = (string) ($rate_limit->renew_date ?? '');
        } elseif (isset($xml->rate_limit)) {
            $rate_limit = $xml->rate_limit;
            $remaining_count = intval($rate_limit->count ?? 0);
            $reset_date = (string) ($rate_limit->renew_date ?? '');
        }
        
        // The API returns 10000 as the daily limit, not 1000
        $data = array(
            'remaining' => $remaining_count !== null ? $remaining_count : 10000,
            'limit' => 10000, // Actual daily limit from API
            'reset' => $reset_date,
            'percentage' => 0,
            'source' => 'api'
        );
        
        // Calculate percentage used (not remaining)
        if ($data['limit'] > 0) {
            $used = $data['limit'] - $data['remaining'];
            $data['percentage'] = round(($used / $data['limit']) * 100);
            // Show at least 1% if any calls were used
            if ($used > 0 && $data['percentage'] == 0) {
                $data['percentage'] = 1;
            }
        }
        
        // Determine warning level
        if ($data['percentage'] >= 90) {
            $data['level'] = 'critical';
        } elseif ($data['percentage'] >= 75) {
            $data['level'] = 'warning';
        } else {
            $data['level'] = 'normal';
        }
        
        // Cache the result
        set_transient(self::CACHE_KEY, $data, self::CACHE_DURATION);
        
        return $data;
    }
    
    /**
     * AJAX handler to get rate limit
     */
    public function ajax_get_rate_limit() {
        check_ajax_referer('delcampe_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Permission denied', 'wc-delcampe-integration'));
        }
        
        $rate_limit = $this->get_rate_limit(true);
        
        if (is_wp_error($rate_limit)) {
            wp_send_json_error($rate_limit->get_error_message());
        }
        
        wp_send_json_success($rate_limit);
    }
    
    /**
     * Add rate limit indicator to admin bar
     * 
     * @param WP_Admin_Bar $wp_admin_bar
     */
    public function add_admin_bar_indicator($wp_admin_bar) {
        if (!current_user_can('manage_woocommerce')) {
            return;
        }
        
        // Only show on admin pages
        if (!is_admin()) {
            return;
        }
        
        $rate_limit = $this->get_rate_limit();
        
        if (is_wp_error($rate_limit)) {
            return;
        }
        
        $color = '#46b450'; // Green
        if ($rate_limit['level'] === 'warning') {
            $color = '#f0ad4e'; // Orange
        } elseif ($rate_limit['level'] === 'critical') {
            $color = '#dc3545'; // Red
        }
        
        $wp_admin_bar->add_node(array(
            'id' => 'delcampe-rate-limit',
            'title' => sprintf(
                '<span style="color: %s;">Delcampe API: %d/%d</span>',
                $color,
                $rate_limit['remaining'],
                $rate_limit['limit']
            ),
            'href' => admin_url('admin.php?page=delcampe&tab=advanced'),
            'meta' => array(
                'title' => sprintf(
                    __('API calls remaining: %d of %d (%d%% used)', 'wc-delcampe-integration'),
                    $rate_limit['remaining'],
                    $rate_limit['limit'],
                    $rate_limit['percentage']
                )
            )
        ));
    }
    
    /**
     * Display rate limit on all Delcampe admin pages
     */
    public function display_rate_limit_on_delcampe_pages() {
        // Check if we're on a Delcampe admin page using multiple methods
        $is_delcampe_page = false;
        
        // Method 1: Check screen ID
        $screen = get_current_screen();
        if ($screen && strpos($screen->id, 'delcampe') !== false) {
            $is_delcampe_page = true;
        }
        
        // Method 2: Check page parameter
        if (!$is_delcampe_page && isset($_GET['page'])) {
            $page = $_GET['page'];
            // Check if page contains 'delcampe' anywhere in the string
            if (stripos($page, 'delcampe') !== false) {
                $is_delcampe_page = true;
            }
        }
        
        if (!$is_delcampe_page) {
            return;
        }
        
        // Don't show on bulk publish page as it has its own display
        if (isset($_GET['page']) && $_GET['page'] === 'delcampe-bulk-publish') {
            return;
        }
        
        // Display the rate limit status
        echo $this->get_rate_limit_display(true);
    }
    
    /**
     * Display rate limit warning in admin notices
     */
    public function display_rate_limit_warning() {
        // Only show on Delcampe pages or when publishing
        $screen = get_current_screen();
        if (!$screen || (strpos($screen->id, 'delcampe') === false && $screen->id !== 'edit-product')) {
            return;
        }
        
        $rate_limit = $this->get_rate_limit();
        
        if (is_wp_error($rate_limit)) {
            return;
        }
        
        if ($rate_limit['level'] === 'critical') {
            ?>
            <div class="notice notice-error">
                <p>
                    <strong><?php _e('Delcampe API Rate Limit Critical', 'wc-delcampe-integration'); ?></strong>
                    <?php printf(
                        __('Only %d API calls remaining (resets %s). Consider pausing bulk operations.', 'wc-delcampe-integration'),
                        $rate_limit['remaining'],
                        $rate_limit['reset'] ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime(get_date_from_gmt($rate_limit['reset'], 'Y-m-d H:i:s'))) : __('in 24 hours', 'wc-delcampe-integration')
                    ); ?>
                </p>
            </div>
            <?php
        } elseif ($rate_limit['level'] === 'warning') {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php _e('Delcampe API Rate Limit Warning', 'wc-delcampe-integration'); ?></strong>
                    <?php printf(
                        __('%d API calls remaining (%d%% used).', 'wc-delcampe-integration'),
                        $rate_limit['remaining'],
                        $rate_limit['percentage']
                    ); ?>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Check if we should pause operations
     * 
     * @return bool True if operations should be paused
     */
    public function should_pause_operations() {
        $rate_limit = $this->get_rate_limit();
        
        if (is_wp_error($rate_limit)) {
            return false; // Don't pause if we can't check
        }
        
        // Pause if less than 100 calls remaining (1% of 10000 limit)
        // This gives enough buffer for bulk operations
        return $rate_limit['remaining'] < 100;
    }
    
    /**
     * Get formatted rate limit display
     * 
     * @param bool $show_refresh Whether to show refresh button
     * @return string HTML formatted rate limit display
     */
    public function get_rate_limit_display($show_refresh = true) {
        $rate_limit = $this->get_rate_limit();
        
        if (is_wp_error($rate_limit)) {
            return '<span style="color: #999;">' . __('Rate limit unavailable', 'wc-delcampe-integration') . '</span>';
        }
        
        $color = '#46b450'; // Green
        if ($rate_limit['level'] === 'warning') {
            $color = '#f0ad4e'; // Orange
        } elseif ($rate_limit['level'] === 'critical') {
            $color = '#dc3545'; // Red
        }
        
        $html = sprintf(
            '<div class="delcampe-rate-limit-display" id="rate-limit-display" style="padding: 10px; background: #f9f9f9; border-left: 4px solid %s; margin: 10px 0; position: relative;">',
            $color
        );
        
        $html .= '<strong>' . __('Listing Creation Limit', 'wc-delcampe-integration') . '</strong>';
        
        if ($show_refresh) {
            $html .= '<button type="button" class="button button-small" onclick="refreshRateLimit()" style="float: right; margin-top: -3px;">' .
                     '<span class="dashicons dashicons-update" style="vertical-align: middle;"></span> ' .
                     __('Refresh', 'wc-delcampe-integration') . '</button>';
        }
        
        $html .= '<br>';
        
        $html .= sprintf(
            __('New listings remaining: %d / %d (%d%% used)', 'wc-delcampe-integration'),
            $rate_limit['remaining'],
            $rate_limit['limit'],
            $rate_limit['percentage']
        );
        
        if ($rate_limit['reset']) {
            // Convert UTC time to local WordPress timezone
            $reset_local = get_date_from_gmt($rate_limit['reset'], 'Y-m-d H:i:s');
            $reset_time = strtotime($reset_local);
            $now = current_time('timestamp');
            $hours_until_reset = round(($reset_time - $now) / 3600, 1);
            
            $html .= '<br><small>' . sprintf(
                __('Daily limit resets: %s (%s hours)', 'wc-delcampe-integration'),
                date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $reset_time),
                $hours_until_reset
            ) . '</small>';
        } else {
            $html .= '<br><small>' . __('Daily limit (resets every 24 hours)', 'wc-delcampe-integration') . '</small>';
        }
        
        $html .= '</div>';
        
        // Add JavaScript and CSS for refresh button
        if ($show_refresh) {
            $html .= '<style>
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            .dashicons.spin {
                animation: spin 1s linear infinite;
                display: inline-block;
            }
            </style>
            <script>
            function refreshRateLimit() {
                var button = event.target.closest("button");
                button.disabled = true;
                button.innerHTML = \'<span class="dashicons dashicons-update spin"></span> Refreshing...\';
                
                jQuery.post(ajaxurl, {
                    action: "delcampe_get_rate_limit",
                    nonce: "' . wp_create_nonce('delcampe_admin_nonce') . '"
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert("Error refreshing rate limit");
                        button.disabled = false;
                        button.innerHTML = \'<span class="dashicons dashicons-update"></span> Refresh\';
                    }
                });
            }
            </script>';
        }
        
        return $html;
    }
}

// Initialize the class
add_action('init', function() {
    if (is_admin()) {
        Delcampe_Rate_Limit::get_instance();
    }
});