<?php
/**
 * Delcampe API Rate Limiter
 * 
 * Manages API rate limiting to prevent 429 errors and handles authentication failures.
 * Implements exponential backoff and token renewal strategies.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.35.5
 */

class Delcampe_Rate_Limiter {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Rate limit settings
     */
    const MAX_REQUESTS_PER_SECOND = 2;      // Max 2 requests per second
    const MAX_REQUESTS_PER_MINUTE = 60;     // Max 60 requests per minute  
    const MAX_REQUESTS_PER_HOUR = 1000;     // Max 1000 requests per hour
    const BACKOFF_429_SECONDS = 60;         // Wait 60 seconds after 429
    const BACKOFF_401_SECONDS = 5;          // Wait 5 seconds for token renewal
    
    /**
     * Transient keys
     */
    const TRANSIENT_REQUEST_COUNT = 'delcampe_api_request_count';
    const TRANSIENT_RATE_LIMIT = 'delcampe_api_rate_limited';
    const TRANSIENT_LAST_REQUEST = 'delcampe_api_last_request';
    const TRANSIENT_HOURLY_COUNT = 'delcampe_api_hourly_count';
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Initialize rate limiting
    }
    
    /**
     * Check if we can make an API request
     * 
     * @return bool|WP_Error True if allowed, WP_Error if rate limited
     */
    public function can_make_request() {
        // Check if we're in a rate limit backoff period
        $rate_limit = get_transient(self::TRANSIENT_RATE_LIMIT);
        if ($rate_limit) {
            $wait_time = $rate_limit - time();
            if ($wait_time > 0) {
                return new WP_Error(
                    'rate_limited',
                    sprintf('Rate limited. Please wait %d seconds.', $wait_time),
                    array('wait_seconds' => $wait_time)
                );
            }
        }
        
        // Check requests per second
        $last_request = get_transient(self::TRANSIENT_LAST_REQUEST);
        if ($last_request) {
            $time_since_last = microtime(true) - $last_request;
            if ($time_since_last < (1.0 / self::MAX_REQUESTS_PER_SECOND)) {
                $wait_ms = (int)((1.0 / self::MAX_REQUESTS_PER_SECOND - $time_since_last) * 1000);
                usleep($wait_ms * 1000); // Sleep for the required time
            }
        }
        
        // Check requests per minute
        $minute_count = $this->get_minute_count();
        if ($minute_count >= self::MAX_REQUESTS_PER_MINUTE) {
            return new WP_Error(
                'minute_limit',
                sprintf('Reached limit of %d requests per minute', self::MAX_REQUESTS_PER_MINUTE)
            );
        }
        
        // Check requests per hour
        $hourly_count = (int) get_transient(self::TRANSIENT_HOURLY_COUNT);
        if ($hourly_count >= self::MAX_REQUESTS_PER_HOUR) {
            return new WP_Error(
                'hourly_limit', 
                sprintf('Reached limit of %d requests per hour', self::MAX_REQUESTS_PER_HOUR)
            );
        }
        
        return true;
    }
    
    /**
     * Record an API request
     * 
     * @param string $endpoint The API endpoint called
     * @param int $status_code The HTTP status code received
     */
    public function record_request($endpoint, $status_code = 200) {
        // Update last request time
        set_transient(self::TRANSIENT_LAST_REQUEST, microtime(true), 60);
        
        // Update minute counter
        $this->increment_minute_count();
        
        // Update hourly counter
        $hourly_count = (int) get_transient(self::TRANSIENT_HOURLY_COUNT);
        set_transient(self::TRANSIENT_HOURLY_COUNT, $hourly_count + 1, HOUR_IN_SECONDS);
        
        // Handle rate limit responses
        if ($status_code === 429) {
            $this->handle_rate_limit();
        } elseif ($status_code === 401) {
            $this->handle_auth_failure();
        }
        
        // Log the request
        $this->log_request($endpoint, $status_code);
    }
    
    /**
     * Handle 429 rate limit response
     */
    private function handle_rate_limit() {
        // Set backoff period
        set_transient(self::TRANSIENT_RATE_LIMIT, time() + self::BACKOFF_429_SECONDS, self::BACKOFF_429_SECONDS);
        
        // Log the rate limit
        delcampe_log('[Rate Limiter] API rate limit hit (429). Backing off for ' . self::BACKOFF_429_SECONDS . ' seconds.');
        
        // Trigger token renewal after backoff
        wp_schedule_single_event(time() + self::BACKOFF_429_SECONDS, 'delcampe_renew_token_after_rate_limit');
    }
    
    /**
     * Handle 401 authentication failure
     */
    private function handle_auth_failure() {
        // Clear cached token
        delete_transient('delcampe_access_token');
        
        // Try to renew token
        if (class_exists('Delcampe_Auth')) {
            $auth = Delcampe_Auth::get_instance();
            $token = $auth->get_auth_token(); // Fixed method name - was get_access_token
            
            if (is_wp_error($token)) {
                delcampe_log('[Rate Limiter] Failed to renew token after 401: ' . $token->get_error_message());
            } else {
                delcampe_log('[Rate Limiter] Successfully renewed token after 401');
            }
        }
    }
    
    /**
     * Get minute request count
     */
    private function get_minute_count() {
        $counts = get_transient(self::TRANSIENT_REQUEST_COUNT);
        if (!is_array($counts)) {
            return 0;
        }
        
        // Clean old entries (older than 60 seconds)
        $now = time();
        $counts = array_filter($counts, function($timestamp) use ($now) {
            return ($now - $timestamp) < 60;
        });
        
        // Update transient with cleaned data
        set_transient(self::TRANSIENT_REQUEST_COUNT, $counts, 60);
        
        return count($counts);
    }
    
    /**
     * Increment minute counter
     */
    private function increment_minute_count() {
        $counts = get_transient(self::TRANSIENT_REQUEST_COUNT);
        if (!is_array($counts)) {
            $counts = array();
        }
        
        // Add current timestamp
        $counts[] = time();
        
        // Keep only last 60 seconds
        $now = time();
        $counts = array_filter($counts, function($timestamp) use ($now) {
            return ($now - $timestamp) < 60;
        });
        
        set_transient(self::TRANSIENT_REQUEST_COUNT, $counts, 60);
    }
    
    /**
     * Log API request for monitoring
     */
    private function log_request($endpoint, $status_code) {
        if ($status_code >= 400) {
            delcampe_log(sprintf(
                '[Rate Limiter] API request failed: %s (Status: %d)',
                $endpoint,
                $status_code
            ));
        }
    }
    
    /**
     * Reset all rate limits (for testing/recovery)
     */
    public function reset_limits() {
        delete_transient(self::TRANSIENT_REQUEST_COUNT);
        delete_transient(self::TRANSIENT_RATE_LIMIT);
        delete_transient(self::TRANSIENT_LAST_REQUEST);
        delete_transient(self::TRANSIENT_HOURLY_COUNT);
        
        delcampe_log('[Rate Limiter] All rate limits have been reset');
    }
    
    /**
     * Get current rate limit status
     */
    public function get_status() {
        return array(
            'minute_count' => $this->get_minute_count(),
            'minute_limit' => self::MAX_REQUESTS_PER_MINUTE,
            'hourly_count' => (int) get_transient(self::TRANSIENT_HOURLY_COUNT),
            'hourly_limit' => self::MAX_REQUESTS_PER_HOUR,
            'rate_limited' => get_transient(self::TRANSIENT_RATE_LIMIT) ? true : false,
            'rate_limit_expires' => get_transient(self::TRANSIENT_RATE_LIMIT)
        );
    }
}

// Hook for token renewal after rate limit
add_action('delcampe_renew_token_after_rate_limit', function() {
    if (class_exists('Delcampe_Auth')) {
        $auth = Delcampe_Auth::get_instance();
        $auth->get_access_token(true); // Force renewal
    }
});