<?php
/**
 * Delcampe Real-time Sync Handler
 * 
 * Handles immediate synchronization of inventory changes to Delcampe
 * when stock changes occur from any source (website, eBay, admin).
 * 
 * @package Delcampe_Sync
 * @since 1.10.12.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Realtime_Sync {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Business logger instance
     */
    private $logger;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Get logger instance
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-business-logger.php';
        $this->logger = Delcampe_Business_Logger::get_instance();
        
        // Hook into stock changes
        add_action('woocommerce_product_set_stock', array($this, 'handle_stock_change'), 20, 1);
        add_action('woocommerce_variation_set_stock', array($this, 'handle_stock_change'), 20, 1);
        
        // Hook into order creation
        add_action('woocommerce_new_order', array($this, 'handle_new_order'), 20, 2);
        add_action('woocommerce_order_status_processing', array($this, 'handle_order_processing'), 10, 2);
        add_action('woocommerce_order_status_completed', array($this, 'handle_order_completed'), 10, 2);
        
        // Action Scheduler hook for single product sync
        add_action('delcampe_sync_single_product_inventory', array($this, 'sync_single_product_inventory'), 10, 1);
    }
    
    /**
     * Handle stock change from any source
     */
    public function handle_stock_change($product) {
        // Skip if this is from a Delcampe import
        if (get_transient('delcampe_importing_order')) {
            return;
        }
        
        $product_id = $product->get_id();
        $new_stock = $product->get_stock_quantity();
        
        // Check if this product has active Delcampe listings
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        $active_listings = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$listings_table} 
             WHERE product_id = %d 
             AND status IN ('published', 'active')
             AND delcampe_id IS NOT NULL",
            $product_id
        ));
        
        if (empty($active_listings)) {
            return;
        }
        
        // Update each active listing on Delcampe
        foreach ($active_listings as $listing) {
            $this->update_delcampe_listing_quantity($listing, $new_stock);
        }
        
        // Log the sync
        do_action('delcampe_stock_synced', $product_id, $new_stock, true);
    }
    
    /**
     * Update quantity on Delcampe
     * IMPORTANT: Delcampe API only allows qty updates if NO units have been sold
     * Once any quantity is sold, we must create a new listing instead
     */
    private function update_delcampe_listing_quantity($listing, $new_quantity) {
        // Get auth token
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            $this->logger->log_sync_error('Failed to get auth token', array(
                'listing_id' => $listing->id,
                'error' => $token->get_error_message()
            ));
            return false;
        }
        
        // If quantity is 0, end the listing
        if ($new_quantity <= 0) {
            return $this->end_delcampe_listing($listing, $token);
        }
        
        // Check if any quantity has been sold
        // Delcampe API restriction: Cannot update qty if any units have been sold
        if (isset($listing->quantity_sold) && $listing->quantity_sold > 0) {
            delcampe_log('[Stock Sync] Cannot update listing ' . $listing->delcampe_id . ' - has sold units (' . $listing->quantity_sold . ' sold). Creating replacement listing.');
            
            // Calculate the actual new quantity available
            // We need to account for what's already listed minus what's sold
            $current_available = $listing->quantity - $listing->quantity_sold;
            $additional_stock = $new_quantity - $current_available;
            
            if ($additional_stock <= 0) {
                // No need to create new listing, current listing has enough
                delcampe_log('[Stock Sync] No additional stock needed. Current listing has sufficient quantity.');
                return true;
            }
            
            // Create replacement listing with additional stock
            return $this->create_replacement_listing($listing, $additional_stock, $token);
        }
        
        // Update the quantity using correct endpoint
        // According to API docs: PUT /item/{id} with qty field
        $api_url = DELCAMPE_API_URL . '/item/' . $listing->delcampe_id . '?token=' . $token;
        
        // For quantity updates, we only send the qty field (not quantity!)
        // API docs show: 'item' => array('qty' => 3)
        $post_data = array(
            'item' => array(
                'qty' => $new_quantity  // Must be 'qty' not 'quantity'!
            )
        );
        
        $response = wp_remote_request($api_url, array(
            'method' => 'PUT',
            'timeout' => 30,
            'body' => http_build_query($post_data),
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            $this->logger->log_sync_error('Failed to update Delcampe quantity', array(
                'listing_id' => $listing->id,
                'delcampe_id' => $listing->delcampe_id,
                'error' => $response->get_error_message()
            ));
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        // Log full response for debugging
        delcampe_log('[Stock Sync] API Request URL: ' . $api_url);
        delcampe_log('[Stock Sync] API Request Data: ' . http_build_query($post_data));
        delcampe_log('[Stock Sync] API Response Code: ' . $response_code);
        delcampe_log('[Stock Sync] API Response Body: ' . substr($body, 0, 500));
        
        // Check for embedded status in Delcampe response
        // Delcampe returns HTTP 200 but includes actual status in XML
        $api_status = null;
        $error_msg = null;
        
        if ($xml) {
            // Check for status in notification structure
            if (isset($xml->Notification_Data->Headers->Status)) {
                $api_status = (int)$xml->Notification_Data->Headers->Status;
                delcampe_log('[Stock Sync] API Internal Status: ' . $api_status);
            }
            
            // Check for error message
            if (isset($xml->Notification_Data->body->error)) {
                $error_msg = (string)$xml->Notification_Data->body->error;
                delcampe_log('[Stock Sync] API Error: ' . $error_msg);
            }
        }
        
        // Check for success - must have HTTP 200/201 AND internal status 200/201 (or no internal status)
        $is_success = ($response_code == 200 || $response_code == 201) && 
                      ($api_status === null || $api_status == 200 || $api_status == 201);
        
        if ($is_success && !$error_msg) {
            // Update local database
            global $wpdb;
            $wpdb->update(
                $wpdb->prefix . 'delcampe_listings',
                array('quantity' => $new_quantity),
                array('id' => $listing->id)
            );
            
            // Log success
            $this->logger->log_event('listing_quantity_updated', array(
                'product_id' => $listing->product_id,
                'delcampe_id' => $listing->delcampe_id,
                'quantity_after' => $new_quantity,
                'details' => array(
                    'action' => 'quantity_update',
                    'success' => true
                )
            ));
            
            delcampe_log('[Stock Sync] Successfully updated quantity for listing ' . $listing->delcampe_id . ' to ' . $new_quantity);
            return true;
        } else {
            // Prepare error message
            if (!$error_msg) {
                $error_msg = 'Unknown error';
                if ($xml && isset($xml->error)) {
                    $error_msg = (string)$xml->error;
                } elseif ($xml && isset($xml->message)) {
                    $error_msg = (string)$xml->message;
                }
            }
            
            // Handle specific error cases
            if ($error_msg == 'err_item_incorrectQty') {
                delcampe_log('[Stock Sync] Quantity update rejected - err_item_incorrectQty. This may indicate the listing has a different quantity on Delcampe.');
                
                // Try to sync FROM Delcampe instead
                $this->sync_quantity_from_delcampe($listing);
            }
            
            $this->logger->log_sync_error('Failed to update Delcampe quantity', array(
                'listing_id' => $listing->id,
                'delcampe_id' => $listing->delcampe_id,
                'response_code' => $response_code,
                'error' => $error_msg,
                'response' => substr($body, 0, 200)
            ));
            
            delcampe_log('[Stock Sync] Failed to update quantity for listing ' . $listing->delcampe_id . ': ' . $error_msg);
            return false;
        }
    }
    
    /**
     * Sync quantity FROM Delcampe to WooCommerce
     * Used when Delcampe rejects our update due to mismatch
     */
    private function sync_quantity_from_delcampe($listing) {
        delcampe_log('[Stock Sync] Syncing quantity FROM Delcampe for listing ' . $listing->delcampe_id);
        
        // Get auth token
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return false;
        }
        
        // Get current listing details from Delcampe
        $api_url = DELCAMPE_API_URL . '/item/' . $listing->delcampe_id . '?token=' . $token;
        $response = wp_remote_get($api_url);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        if ($xml && isset($xml->Notification_Data->body->item->qty)) {
            $delcampe_qty = (int)$xml->Notification_Data->body->item->qty;
            
            delcampe_log('[Stock Sync] Delcampe shows quantity: ' . $delcampe_qty);
            
            // Update local database to match Delcampe
            global $wpdb;
            $wpdb->update(
                $wpdb->prefix . 'delcampe_listings',
                array('quantity' => $delcampe_qty),
                array('id' => $listing->id)
            );
            
            // Update WooCommerce product stock
            if ($listing->product_id) {
                $product = wc_get_product($listing->product_id);
                if ($product && $product->managing_stock()) {
                    $product->set_stock_quantity($delcampe_qty);
                    $product->save();
                    delcampe_log('[Stock Sync] Updated WooCommerce product #' . $listing->product_id . ' to quantity ' . $delcampe_qty);
                }
            }
            
            return true;
        }
        
        return false;
    }
    
    /**
     * End a Delcampe listing when stock reaches 0
     * Updated to use correct API endpoint: DELETE /item/{id}
     */
    private function end_delcampe_listing($listing, $token) {
        // According to API docs: DELETE /item/{id} with reason
        $api_url = DELCAMPE_API_URL . '/item/' . $listing->delcampe_id . '?token=' . $token;
        
        $post_data = array(
            'reason' => 'NotAvailableItem',  // Valid API reason for zero stock
            'priority' => 1 // Immediate closure
        );
        
        $response = wp_remote_request($api_url, array(
            'method' => 'DELETE',
            'timeout' => 30,
            'body' => http_build_query($post_data),
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            $this->logger->log_sync_error('Failed to end Delcampe listing', array(
                'listing_id' => $listing->id,
                'delcampe_id' => $listing->delcampe_id,
                'error' => $response->get_error_message()
            ));
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Log response for debugging
        delcampe_log('[Stock Sync] End listing response code: ' . $response_code);
        delcampe_log('[Stock Sync] End listing response: ' . substr($body, 0, 200));
        
        // Check for success (200 or 204 for DELETE)
        if ($response_code == 200 || $response_code == 204) {
            // Update local database
            global $wpdb;
            $wpdb->update(
                $wpdb->prefix . 'delcampe_listings',
                array(
                    'status' => 'ended',
                    'quantity' => 0,
                    'date_finished' => current_time('mysql')
                ),
                array('id' => $listing->id)
            );
            
            // Log the event
            $this->logger->log_event('listing_ended', array(
                'product_id' => $listing->product_id,
                'delcampe_id' => $listing->delcampe_id,
                'details' => array(
                    'reason' => 'out_of_stock',
                    'action' => 'auto_ended'
                )
            ));
            
            delcampe_log('[Stock Sync] Successfully ended listing ' . $listing->delcampe_id . ' due to zero stock');
            return true;
        } else {
            // Log error
            $xml = simplexml_load_string($body);
            $error_msg = 'Unknown error';
            if ($xml && isset($xml->error)) {
                $error_msg = (string)$xml->error;
            }
            
            $this->logger->log_sync_error('Failed to end Delcampe listing', array(
                'listing_id' => $listing->id,
                'delcampe_id' => $listing->delcampe_id,
                'response_code' => $response_code,
                'error' => $error_msg
            ));
            
            return false;
        }
    }
    
    /**
     * Create a replacement listing when original has sold units
     * This is needed because Delcampe doesn't allow qty updates after sales
     */
    private function create_replacement_listing($original_listing, $additional_quantity, $token) {
        delcampe_log('[Stock Sync] Creating replacement listing with ' . $additional_quantity . ' additional units');
        
        // Get the listing API instance
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        $listing_api = new Delcampe_Listing_API();
        
        // Get the original product
        $product = wc_get_product($original_listing->product_id);
        if (!$product) {
            delcampe_log('[Stock Sync] Failed to load product for replacement listing');
            return false;
        }
        
        // Get the profile data if available
        $profile_id = $original_listing->profile_id;
        $profile_data = null;
        
        if ($profile_id) {
            global $wpdb;
            $profile_data = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}delcampe_profiles WHERE id = %d",
                $profile_id
            ), ARRAY_A);
        }
        
        // Prepare listing data based on original
        $listing_data = array(
            'title' => $original_listing->listing_title,
            'personal_reference' => $original_listing->personal_reference . '_ADD_' . time(),
            'qty' => $additional_quantity,
            'fixed_price' => $original_listing->price,
            'currency' => $original_listing->currency ?: 'CAD',
            'id_category' => $original_listing->category_id,
            'description' => $product->get_description() ?: $product->get_short_description(),
            'duration' => 30, // Default 30 days
            'renew' => 'auto',
            'id_shipping_model' => $original_listing->shipping_model_id
        );
        
        // Add images
        $images = array();
        $image_ids = $product->get_gallery_image_ids();
        if ($product->get_image_id()) {
            array_unshift($image_ids, $product->get_image_id());
        }
        
        foreach ($image_ids as $image_id) {
            $image_url = wp_get_attachment_url($image_id);
            if ($image_url) {
                $images[] = $image_url;
            }
        }
        
        if (!empty($images)) {
            $listing_data['images'] = $images;
        }
        
        // Create the new listing
        $api_url = DELCAMPE_API_URL . '/item?token=' . $token;
        
        $post_data = array(
            'type' => 'fixedPrice',
            'item' => $listing_data
        );
        
        $response = wp_remote_post($api_url, array(
            'timeout' => 30,
            'body' => http_build_query($post_data),
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            delcampe_log('[Stock Sync] Failed to create replacement listing: ' . $response->get_error_message());
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        if ($response_code == 200 || $response_code == 201) {
            // Parse response for new listing ID
            $xml = simplexml_load_string($body);
            $new_delcampe_id = null;
            
            if ($xml && isset($xml->id_item)) {
                $new_delcampe_id = (string)$xml->id_item;
            } elseif ($xml && isset($xml->item_id)) {
                $new_delcampe_id = (string)$xml->item_id;
            }
            
            if ($new_delcampe_id) {
                // Create new listing record in database
                global $wpdb;
                $wpdb->insert(
                    $wpdb->prefix . 'delcampe_listings',
                    array(
                        'product_id' => $original_listing->product_id,
                        'delcampe_id' => $new_delcampe_id,
                        'listing_title' => $listing_data['title'],
                        'personal_reference' => $listing_data['personal_reference'],
                        'category_id' => $listing_data['id_category'],
                        'profile_id' => $profile_id,
                        'quantity' => $additional_quantity,
                        'quantity_sold' => 0,
                        'price' => $listing_data['fixed_price'],
                        'currency' => $listing_data['currency'],
                        'shipping_model_id' => $listing_data['id_shipping_model'],
                        'status' => 'processing',
                        'date_created' => current_time('mysql'),
                        'notes' => 'Replacement listing for ' . $original_listing->delcampe_id
                    )
                );
                
                // Log success
                $this->logger->log_event('replacement_listing_created', array(
                    'product_id' => $original_listing->product_id,
                    'original_delcampe_id' => $original_listing->delcampe_id,
                    'new_delcampe_id' => $new_delcampe_id,
                    'additional_quantity' => $additional_quantity,
                    'details' => array(
                        'reason' => 'original_has_sales',
                        'original_sold' => $original_listing->quantity_sold
                    )
                ));
                
                delcampe_log('[Stock Sync] Successfully created replacement listing ' . $new_delcampe_id);
                
                // Optionally close the original listing if all units are accounted for
                // This prevents confusion with multiple active listings
                if ($original_listing->quantity - $original_listing->quantity_sold <= 0) {
                    $this->close_original_listing($original_listing, $token, 'Replaced by ' . $new_delcampe_id);
                }
                
                return true;
            }
        }
        
        delcampe_log('[Stock Sync] Failed to create replacement listing. Response: ' . substr($body, 0, 500));
        return false;
    }
    
    /**
     * Close original listing when creating replacement
     */
    private function close_original_listing($listing, $token, $reason_note = '') {
        $api_url = DELCAMPE_API_URL . '/item/' . $listing->delcampe_id . '?token=' . $token;
        
        $post_data = array(
            'reason' => 'NotAvailableItem',  // Valid API reason for replacement listing
            'priority' => 1 // Immediate closure
        );
        
        $response = wp_remote_request($api_url, array(
            'method' => 'DELETE',
            'timeout' => 30,
            'body' => http_build_query($post_data),
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept' => 'application/xml'
            )
        ));
        
        if (!is_wp_error($response)) {
            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code == 200 || $response_code == 204) {
                // Update local database
                global $wpdb;
                $wpdb->update(
                    $wpdb->prefix . 'delcampe_listings',
                    array(
                        'status' => 'closed',
                        'date_finished' => current_time('mysql'),
                        'notes' => $reason_note
                    ),
                    array('id' => $listing->id)
                );
                
                delcampe_log('[Stock Sync] Closed original listing ' . $listing->delcampe_id);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Handle new order from any source
     */
    public function handle_new_order($order_id, $order) {
        // Set a flag to prevent re-sync if this is from Delcampe
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        if (!empty($delcampe_order_id)) {
            return; // This order came from Delcampe, don't sync back
        }
        
        // Process each line item
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if (!$product) continue;
            
            // The stock should already be reduced by WooCommerce
            // We just need to sync the new quantity to Delcampe
            $new_stock = $product->get_stock_quantity();
            
            // Trigger sync for this product
            as_schedule_single_action(
                time() + 5, // Small delay to ensure stock is fully updated
                'delcampe_sync_single_product_inventory',
                array('product_id' => $product->get_id()),
                'delcampe-sync'
            );
        }
    }
    
    /**
     * Handle order moving to processing
     */
    public function handle_order_processing($order_id, $order) {
        // Additional sync point when order is confirmed
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if ($product) {
                $this->handle_stock_change($product);
            }
        }
    }
    
    /**
     * Handle order completion
     */
    public function handle_order_completed($order_id, $order) {
        // Final sync point when order is completed
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if ($product) {
                $this->handle_stock_change($product);
            }
        }
    }
    
    /**
     * Sync single product inventory (Action Scheduler)
     */
    public function sync_single_product_inventory($product_id) {
        $product = wc_get_product($product_id);
        if (!$product) {
            return;
        }
        
        // Trigger stock change handler
        $this->handle_stock_change($product);
    }
    
    /**
     * Get sync status for a product
     */
    public function get_product_sync_status($product_id) {
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        $listings = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$listings_table} 
             WHERE product_id = %d 
             AND status IN ('published', 'active')",
            $product_id
        ));
        
        return array(
            'synced' => !empty($listings),
            'listings' => $listings,
            'count' => count($listings)
        );
    }
}

// Initialize the real-time sync handler
add_action('init', function() {
    Delcampe_Realtime_Sync::get_instance();
}, 20); // Priority 20 to ensure business logger is initialized first