<?php
/**
 * WooCommerce Delcampe Integration Setup Notice
 * 
 * Displays admin notices for plugin setup and configuration
 * Ensures users must configure the plugin before use
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  Admin
 * @since       1.8.0.0
 * @version     1.8.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Delcampe_Setup_Notice
 * 
 * Handles display of setup notices to ensure proper configuration
 * 
 * @since 1.8.0.0
 */
class Delcampe_Setup_Notice {
    
    /**
     * Initialize setup notices
     * 
     * @since 1.8.0.0
     */
    public static function init() {
        add_action( 'admin_notices', array( __CLASS__, 'display_setup_notice' ) );
        add_action( 'wp_ajax_delcampe_dismiss_setup_notice', array( __CLASS__, 'dismiss_notice' ) );
    }
    
    /**
     * Display setup notice if plugin is not configured
     * 
     * @since 1.8.0.0
     */
    public static function display_setup_notice() {
        // Only show to users who can manage options
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        
        // Check if notice has been dismissed
        if ( get_option( 'delcampe_setup_notice_dismissed', false ) ) {
            return;
        }
        
        // Check if API key is configured
        $api_key = get_option( 'delcampe_api_key', '' );
        if ( ! empty( $api_key ) ) {
            return;
        }
        
        // Check if we're on a Delcampe admin page
        $screen = get_current_screen();
        if ( strpos( $screen->id, 'delcampe' ) !== false ) {
            return; // Don't show on plugin pages
        }
        
        // Display the setup notice
        ?>
        <div class="notice notice-warning is-dismissible" id="delcampe-setup-notice">
            <h3><?php esc_html_e( 'WooCommerce Delcampe Integration Setup Required', 'wc-delcampe-integration' ); ?></h3>
            <p>
                <?php esc_html_e( 'Thank you for installing the WooCommerce Delcampe Integration plugin. To start syncing your products with Delcampe, you need to configure your API key.', 'wc-delcampe-integration' ); ?>
            </p>
            <p>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=delcampe-settings' ) ); ?>" class="button button-primary">
                    <?php esc_html_e( 'Configure Plugin', 'wc-delcampe-integration' ); ?>
                </a>
                <a href="#" class="button button-secondary" id="delcampe-dismiss-setup">
                    <?php esc_html_e( 'Dismiss', 'wc-delcampe-integration' ); ?>
                </a>
            </p>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Handle dismiss button click
            $('#delcampe-dismiss-setup').on('click', function(e) {
                e.preventDefault();
                
                // Send AJAX request to dismiss notice
                $.post(ajaxurl, {
                    action: 'delcampe_dismiss_setup_notice',
                    nonce: '<?php echo wp_create_nonce( 'delcampe_dismiss_setup' ); ?>'
                }, function() {
                    $('#delcampe-setup-notice').fadeOut();
                });
            });
            
            // Handle notice dismiss button (X)
            $('#delcampe-setup-notice').on('click', '.notice-dismiss', function() {
                $.post(ajaxurl, {
                    action: 'delcampe_dismiss_setup_notice',
                    nonce: '<?php echo wp_create_nonce( 'delcampe_dismiss_setup' ); ?>'
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Handle AJAX request to dismiss setup notice
     * 
     * @since 1.8.0.0
     */
    public static function dismiss_notice() {
        // Verify nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'delcampe_dismiss_setup' ) ) {
            wp_die( 'Security check failed' );
        }
        
        // Check user capabilities
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'Insufficient permissions' );
        }
        
        // Update option to dismiss notice
        update_option( 'delcampe_setup_notice_dismissed', true );
        
        wp_die();
    }
    
    /**
     * Reset setup notice (shows it again)
     * Called when plugin is deactivated/reactivated
     * 
     * @since 1.8.0.0
     */
    public static function reset_notice() {
        delete_option( 'delcampe_setup_notice_dismissed' );
    }
}

// Initialize setup notices after init to avoid early translation loading (v1.10.5.12)
add_action( 'init', array( 'Delcampe_Setup_Notice', 'init' ) );
