<?php
/**
 * Delcampe Shipping Models Manager
 * 
 * Manages shipping models from Delcampe and their mapping to WooCommerce shipping methods.
 * Handles retrieval, caching, and mapping of shipping models between platforms.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Shipping
 * @since 1.3.0.0
 * @version 1.3.0.1
 * 
 * This class facilitates:
 * - Fetching available shipping models from Delcampe API
 * - Caching shipping models for performance
 * - Mapping WooCommerce shipping methods to Delcampe models
 * - Storing shipping configurations per profile
 * - Calculating shipping costs based on product weight/destination
 * 
 * All API communication with Delcampe uses XML format.
 * 
 * Changelog:
 * 1.3.0.1 - Fixed XML parsing to match actual API response structure
 *         - Changed shippingmodel to shippingModel (capital M)
 *         - Simplified response structure parsing
 *         - Added better error logging
 * 1.3.0.0 - Initial implementation
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Delcampe_Shipping_Models
 * 
 * Singleton class for managing shipping models between WooCommerce and Delcampe
 * 
 * @version 1.3.0.1
 */
class Delcampe_Shipping_Models {
    
    /**
     * Singleton instance
     * @var Delcampe_Shipping_Models
     */
    private static $instance = null;
    
    /**
     * Cache key for shipping models
     * @var string
     */
    const CACHE_KEY = 'delcampe_shipping_models';
    
    /**
     * Cache duration in seconds (24 hours)
     * @var int
     */
    const CACHE_DURATION = 86400;
    
    /**
     * Get singleton instance
     * 
     * @return Delcampe_Shipping_Models
     * @since 1.3.0.0
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     * 
     * @since 1.3.0.0
     */
    private function __construct() {
        // Initialize hooks for AJAX handlers
        add_action('wp_ajax_delcampe_get_shipping_models', array($this, 'ajax_get_shipping_models'));
        add_action('wp_ajax_delcampe_save_shipping_mapping', array($this, 'ajax_save_shipping_mapping'));
        add_action('wp_ajax_delcampe_refresh_shipping_models', array($this, 'ajax_refresh_shipping_models'));
    }
    
    /**
     * Fetch shipping models from Delcampe API
     * 
     * Makes an API call to retrieve all available shipping models.
     * Results are cached for performance.
     * 
     * @param bool $force_refresh Whether to bypass cache
     * @return array|WP_Error Array of shipping models or error
     * @since 1.3.0.0
     * @version 1.3.0.1 - Fixed API endpoint case sensitivity
     */
    public function fetch_shipping_models($force_refresh = false) {
        // Check cache first unless force refresh
        if (!$force_refresh) {
            $cached = get_transient(self::CACHE_KEY);
            if ($cached !== false) {
                $this->log('Using cached shipping models');
                return $cached;
            }
        }
        
        $this->log('Fetching shipping models from Delcampe API');
        
        // Get authentication token
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            $this->log('Authentication failed: ' . $token->get_error_message());
            return $token;
        }
        
        // Build API URL - note the capital M in shippingModels
        $api_url = DELCAMPE_API_BASE_URL . '/shippingModels?token=' . $token;
        
        $this->log('API URL: ' . str_replace($token, 'TOKEN', $api_url)); // Log URL without token
        
        // Make API request
        $response = wp_remote_get($api_url, array(
            'timeout' => 30,
            'headers' => array(
                'User-Agent' => DELCAMPE_USER_AGENT,
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            $this->log('API request failed: ' . $response->get_error_message());
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        $this->log('API response code: ' . $response_code);
        
        if ($response_code !== 200) {
            $this->log('API response body: ' . substr($response_body, 0, 500)); // Log first 500 chars
            return new WP_Error(
                'api_error',
                sprintf(__('API returned error code: %d', 'wc-delcampe-integration'), $response_code)
            );
        }
        
        // Parse XML response
        $models = $this->parse_shipping_models_response($response_body);
        
        if (is_wp_error($models)) {
            return $models;
        }
        
        // Cache the results
        set_transient(self::CACHE_KEY, $models, self::CACHE_DURATION);
        
        $this->log(sprintf('Successfully fetched %d shipping models', count($models)));
        
        return $models;
    }
    
    /**
     * Parse XML response containing shipping models
     * 
     * Converts Delcampe XML response into structured PHP array
     * 
     * @param string $xml_response Raw XML response from API
     * @return array|WP_Error Parsed shipping models or error
     * @since 1.3.0.0
     * @version 1.3.0.1 - Updated to match actual API response structure
     */
    private function parse_shipping_models_response($xml_response) {
        if (empty($xml_response)) {
            return new WP_Error('empty_response', __('Empty response from API', 'wc-delcampe-integration'));
        }
        
        // Log the response for debugging
        $this->log('Raw XML response: ' . substr($xml_response, 0, 1000));
        
        // Suppress XML errors
        libxml_use_internal_errors(true);
        
        // Parse XML
        $xml = simplexml_load_string($xml_response);
        
        if ($xml === false) {
            $errors = libxml_get_errors();
            $error_message = 'XML parsing failed';
            
            if (!empty($errors)) {
                $error_message .= ': ' . $errors[0]->message;
            }
            
            libxml_clear_errors();
            $this->log($error_message);
            return new WP_Error('xml_parse_error', $error_message);
        }
        
        $models = array();
        
        // Check for error response (v1.3.0.1)
        if (isset($xml->body->e)) {
            $error_message = (string) $xml->body->e;
            $this->log('API error in body: ' . $error_message);
            return new WP_Error('api_error', $error_message);
        }
        
        // Check for error in different locations
        if (isset($xml->e)) {
            $error_message = (string) $xml->e;
            $this->log('API error: ' . $error_message);
            return new WP_Error('api_error', $error_message);
        }
        
        // Parse shipping models from response
        // According to documentation, structure is: <body><shippingModel>...</shippingModel></body>
        if (isset($xml->body->shippingModel)) {
            foreach ($xml->body->shippingModel as $model) {
                $model_data = array(
                    'id' => (int) $model->id,
                    'name' => (string) $model->name,
                    'type' => (string) $model->type,
                    'description' => sprintf(__('Type: %s (ID: %d)', 'wc-delcampe-integration'), (string) $model->type, (int) $model->id),
                    'costs' => array() // Costs must be configured on Delcampe website
                );
                
                $models[] = $model_data;
            }
        }
        // Also check without body wrapper in case response structure varies
        elseif (isset($xml->shippingModel)) {
            foreach ($xml->shippingModel as $model) {
                $model_data = array(
                    'id' => (int) $model->id,
                    'name' => !empty($model->n) ? (string) $model->n : (string) $model->type, // Use type as name if n is empty
                    'type' => (string) $model->type,
                    'description' => '',
                    'costs' => array()
                );
                
                $models[] = $model_data;
            }
        }
        // Check if it's wrapped in Delcampe_Notification structure (like auth responses)
        elseif (isset($xml->Notification_Data->body->shippingModel)) {
            foreach ($xml->Notification_Data->body->shippingModel as $model) {
                $model_data = array(
                    'id' => (int) $model->id,
                    'name' => !empty($model->n) ? (string) $model->n : (string) $model->type, // Use type as name if n is empty
                    'type' => (string) $model->type,
                    'description' => '',
                    'costs' => array()
                );
                
                $models[] = $model_data;
            }
        }
        else {
            $this->log('No shipping models found in expected XML structure');
            $this->log('XML structure: ' . print_r($xml, true));
            return new WP_Error('no_models', __('No shipping models found in response', 'wc-delcampe-integration'));
        }
        
        return $models;
    }
    
    /**
     * Get shipping conditions from Delcampe
     * 
     * Fetches the textual shipping conditions set in Delcampe account
     * 
     * @return string|WP_Error Shipping conditions text or error
     * @since 1.3.0.1
     */
    public function get_shipping_conditions() {
        // Check cache first
        $cached = get_transient('delcampe_shipping_conditions');
        if ($cached !== false) {
            return $cached;
        }
        
        // Get authentication token
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        // Build API URL
        $api_url = DELCAMPE_API_BASE_URL . '/shippingConditions?token=' . $token;
        
        // Make API request
        $response = wp_remote_get($api_url, array(
            'timeout' => 30,
            'headers' => array(
                'User-Agent' => DELCAMPE_USER_AGENT,
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            return new WP_Error('api_error', sprintf(__('API returned error code: %d', 'wc-delcampe-integration'), $response_code));
        }
        
        // Parse XML response
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($response_body);
        
        if ($xml === false) {
            return new WP_Error('xml_parse_error', __('Failed to parse XML response', 'wc-delcampe-integration'));
        }
        
        $conditions = '';
        if (isset($xml->body->{'shipping-conditions'})) {
            $conditions = (string) $xml->body->{'shipping-conditions'};
        } elseif (isset($xml->{'shipping-conditions'})) {
            $conditions = (string) $xml->{'shipping-conditions'};
        }
        
        // Cache for 24 hours
        set_transient('delcampe_shipping_conditions', $conditions, 86400);
        
        return $conditions;
    }
    
    /**
     * Get available shipping models
     * 
     * Wrapper method that returns cached models or fetches new ones
     * 
     * @return array|WP_Error
     * @since 1.3.0.0
     */
    public function get_shipping_models() {
        return $this->fetch_shipping_models();
    }
    
    /**
     * Get shipping model by ID
     * 
     * @param int $model_id Shipping model ID
     * @return array|null Model data or null if not found
     * @since 1.3.0.0
     */
    public function get_shipping_model($model_id) {
        $models = $this->get_shipping_models();
        
        if (is_wp_error($models)) {
            return null;
        }
        
        foreach ($models as $model) {
            if ($model['id'] == $model_id) {
                return $model;
            }
        }
        
        return null;
    }
    
    /**
     * Save shipping model mapping for a profile
     * 
     * @param int $profile_id Profile ID
     * @param array $mapping Shipping model mapping data
     * @return bool Success status
     * @since 1.3.0.0
     */
    public function save_profile_shipping_mapping($profile_id, $mapping) {
        // Validate mapping data
        if (!is_array($mapping)) {
            return false;
        }
        
        // Get profiles model
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/profiles/class-delcampe-profiles-model.php';
        $profiles_model = Delcampe_Profiles_Model::get_instance();
        
        // Get current profile
        $profile = $profiles_model->get($profile_id);
        
        if (!$profile) {
            $this->log('Profile not found: ' . $profile_id);
            return false;
        }
        
        // Update profile with shipping mapping
        $profile['shipping_model_id'] = isset($mapping['model_id']) ? (int) $mapping['model_id'] : 0;
        $profile['shipping_settings'] = isset($mapping['settings']) ? $mapping['settings'] : array();
        
        // Save profile
        $result = $profiles_model->update($profile_id, $profile);
        
        if ($result) {
            $this->log('Saved shipping mapping for profile: ' . $profile_id);
        } else {
            $this->log('Failed to save shipping mapping for profile: ' . $profile_id);
        }
        
        return $result;
    }
    
    /**
     * Get WooCommerce shipping methods
     * 
     * Returns all active shipping methods from WooCommerce
     * 
     * @return array
     * @since 1.3.0.0
     */
    public function get_woocommerce_shipping_methods() {
        $shipping_methods = array();
        
        // Get shipping zones
        $shipping_zones = WC_Shipping_Zones::get_zones();
        
        // Add "Rest of the World" zone
        $shipping_zones[0] = WC_Shipping_Zones::get_zone(0)->get_data();
        
        foreach ($shipping_zones as $zone_id => $zone) {
            $zone_obj = WC_Shipping_Zones::get_zone($zone_id);
            $methods = $zone_obj->get_shipping_methods(true); // Only enabled methods
            
            foreach ($methods as $method) {
                $shipping_methods[] = array(
                    'zone_id' => $zone_id,
                    'zone_name' => $zone['zone_name'],
                    'method_id' => $method->id,
                    'method_instance_id' => $method->get_instance_id(),
                    'method_title' => $method->get_title(),
                    'method_type' => $method->id,
                    'enabled' => $method->is_enabled()
                );
            }
        }
        
        return $shipping_methods;
    }
    
    /**
     * Calculate shipping cost for a product
     * 
     * @param WC_Product $product Product object
     * @param int $shipping_model_id Delcampe shipping model ID
     * @param string $destination_country Destination country code
     * @return float|WP_Error Shipping cost or error
     * @since 1.3.0.0
     */
    public function calculate_shipping_cost($product, $shipping_model_id, $destination_country = '') {
        // Get shipping model
        $model = $this->get_shipping_model($shipping_model_id);
        
        if (!$model) {
            return new WP_Error('invalid_model', __('Invalid shipping model', 'wc-delcampe-integration'));
        }
        
        // Get product weight
        $weight = $product->get_weight();
        
        if (!$weight) {
            $weight = 0;
        }
        
        // Find matching cost entry
        $shipping_cost = 0;
        
        foreach ($model['costs'] as $cost) {
            // Check if country matches or is default
            if ($cost['country'] == $destination_country || empty($cost['country'])) {
                // Check weight limit
                if (empty($cost['max_weight']) || $weight <= $cost['max_weight']) {
                    $shipping_cost = $cost['price'];
                    break;
                }
            }
        }
        
        return $shipping_cost;
    }
    
    /**
     * AJAX handler to get shipping models
     * 
     * @since 1.3.0.0
     */
    public function ajax_get_shipping_models() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_die(__('Security check failed', 'wc-delcampe-integration'));
        }
        
        $models = $this->get_shipping_models();
        
        if (is_wp_error($models)) {
            wp_send_json_error(array(
                'message' => $models->get_error_message()
            ));
        }
        
        wp_send_json_success(array(
            'models' => $models
        ));
    }
    
    /**
     * AJAX handler to save shipping mapping
     * 
     * @since 1.3.0.0
     */
    public function ajax_save_shipping_mapping() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_die(__('Security check failed', 'wc-delcampe-integration'));
        }
        
        $profile_id = isset($_POST['profile_id']) ? (int) $_POST['profile_id'] : 0;
        $mapping = isset($_POST['mapping']) ? $_POST['mapping'] : array();
        
        if (!$profile_id) {
            wp_send_json_error(array(
                'message' => __('Invalid profile ID', 'wc-delcampe-integration')
            ));
        }
        
        $result = $this->save_profile_shipping_mapping($profile_id, $mapping);
        
        if ($result) {
            wp_send_json_success(array(
                'message' => __('Shipping mapping saved successfully', 'wc-delcampe-integration')
            ));
        } else {
            wp_send_json_error(array(
                'message' => __('Failed to save shipping mapping', 'wc-delcampe-integration')
            ));
        }
    }
    
    /**
     * AJAX handler to refresh shipping models
     * 
     * @since 1.3.0.0
     */
    public function ajax_refresh_shipping_models() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_die(__('Security check failed', 'wc-delcampe-integration'));
        }
        
        // Force refresh
        $models = $this->fetch_shipping_models(true);
        
        if (is_wp_error($models)) {
            wp_send_json_error(array(
                'message' => $models->get_error_message()
            ));
        }
        
        wp_send_json_success(array(
            'message' => sprintf(__('Successfully refreshed! Found %d shipping model(s)', 'wc-delcampe-integration'), count($models)),
            'models' => $models,
            'count' => count($models)
        ));
    }
    
    /**
     * Log messages for debugging
     * 
     * @param string $message Message to log
     * @since 1.3.0.0
     * @version 1.3.0.1 - Updated version in log messages
     */
    private function log($message) {
        // Use plugin logging system instead of error_log (v1.10.5.14)
        if (function_exists('delcampe_log')) {
            delcampe_log('[Shipping] ' . $message, 'info');
        }
    }
}
