<?php
/**
 * Delcampe Stock Cache Manager
 * 
 * Manages cached stock values from Delcampe to optimize performance and reduce API calls
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Cache
 * @since 1.10.20.3
 * @version 1.10.20.3
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_Stock_Cache_Manager
 * 
 * Handles all stock caching operations including:
 * - Writing cached values to database
 * - Reading cached values with freshness checks
 * - Batch refresh operations
 * - Cache invalidation
 * - Monitoring and metrics
 */
class Delcampe_Stock_Cache_Manager {
    
    /**
     * Cache TTL in seconds (1 hour default)
     */
    const CACHE_TTL = 3600;
    
    /**
     * Batch size for refresh operations
     */
    const BATCH_SIZE = 50;
    
    /**
     * Maximum batch size to prevent overload
     */
    const MAX_BATCH_SIZE = 100;
    
    /**
     * Instance of the class
     */
    private static $instance = null;
    
    /**
     * Logger instance
     */
    private $logger;
    
    /**
     * Rate limiter instance
     */
    private $rate_limiter;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Initialize logger
        if (class_exists('Delcampe_Business_Logger')) {
            $this->logger = Delcampe_Business_Logger::get_instance();
        }
        
        // Initialize rate limiter
        if (class_exists('Delcampe_Rate_Limit')) {
            $this->rate_limiter = Delcampe_Rate_Limit::get_instance();
        }
        
        // Register hooks
        $this->register_hooks();
    }
    
    /**
     * Register WordPress hooks
     */
    private function register_hooks() {
        // v1.10.35.10: Stock cache refresh is now manual only - no automatic hourly refresh
        // Schedule cron for batch refresh - DISABLED for manual control
        add_action('delcampe_refresh_stock_cache', array($this, 'batch_refresh_stock_cache'));
        
        // DISABLED: Automatic hourly refresh - now manual only from inventory screens
        // Clear any existing scheduled events
        $timestamp = wp_next_scheduled('delcampe_refresh_stock_cache');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'delcampe_refresh_stock_cache');
            delcampe_log('[Stock Cache] Automatic hourly refresh disabled - stock sync is now manual only');
        }
    }
    
    /**
     * Write stock value to cache
     * 
     * @param int $delcampe_id Delcampe listing ID
     * @param int $stock_value Stock quantity
     * @param string $source Source of the update (api, sync, manual)
     * @return bool Success status
     */
    public function write_cache($delcampe_id, $stock_value, $source = 'api') {
        global $wpdb;
        
        if (empty($delcampe_id)) {
            return false;
        }
        
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        // Update cache values
        $result = $wpdb->update(
            $table_name,
            array(
                'delcampe_stock_cached' => intval($stock_value),
                'delcampe_stock_checked_at' => current_time('mysql')
            ),
            array('delcampe_id' => $delcampe_id),
            array('%d', '%s'),
            array('%d')
        );
        
        if ($result !== false) {
            $this->log_cache_update($delcampe_id, $stock_value, $source);
            return true;
        }
        
        return false;
    }
    
    /**
     * Read stock value from cache
     * 
     * @param int $delcampe_id Delcampe listing ID
     * @param bool $check_freshness Whether to check cache freshness
     * @return array|false Array with stock_value and metadata, or false if not found/stale
     */
    public function read_cache($delcampe_id, $check_freshness = true) {
        global $wpdb;
        
        if (empty($delcampe_id)) {
            return false;
        }
        
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        $row = $wpdb->get_row($wpdb->prepare(
            "SELECT delcampe_stock_cached, delcampe_stock_checked_at, quantity 
             FROM $table_name 
             WHERE delcampe_id = %d",
            $delcampe_id
        ), ARRAY_A);
        
        if (!$row || $row['delcampe_stock_cached'] === null) {
            return false;
        }
        
        $cache_data = array(
            'stock_value' => intval($row['delcampe_stock_cached']),
            'checked_at' => $row['delcampe_stock_checked_at'],
            'wc_stock' => intval($row['quantity']),
            'is_fresh' => true,
            'age_seconds' => 0
        );
        
        // Check freshness if requested
        if ($check_freshness && !empty($row['delcampe_stock_checked_at'])) {
            $checked_timestamp = strtotime($row['delcampe_stock_checked_at']);
            $age_seconds = time() - $checked_timestamp;
            $cache_data['age_seconds'] = $age_seconds;
            $cache_data['is_fresh'] = ($age_seconds < self::CACHE_TTL);
        }
        
        return $cache_data;
    }
    
    /**
     * Get cached stock or fetch live if stale
     * 
     * @param int $delcampe_id Delcampe listing ID
     * @param bool $force_refresh Force a live fetch
     * @return array Stock data with source information
     */
    public function get_stock($delcampe_id, $force_refresh = false) {
        // Try cache first unless forced refresh
        if (!$force_refresh) {
            $cached = $this->read_cache($delcampe_id, true);
            if ($cached && $cached['is_fresh']) {
                return array(
                    'stock' => $cached['stock_value'],
                    'source' => 'cache',
                    'checked_at' => $cached['checked_at'],
                    'age_seconds' => $cached['age_seconds']
                );
            }
        }
        
        // Fetch live stock
        $live_stock = $this->fetch_live_stock($delcampe_id);
        if ($live_stock !== false) {
            // Update cache with live value
            $this->write_cache($delcampe_id, $live_stock, 'live_fetch');
            
            return array(
                'stock' => $live_stock,
                'source' => 'live',
                'checked_at' => current_time('mysql'),
                'age_seconds' => 0
            );
        }
        
        // Fall back to stale cache if live fetch failed
        $cached = $this->read_cache($delcampe_id, false);
        if ($cached) {
            // Even if force_refresh was requested, return stale cache if live fetch failed
            // This ensures we always have some data to show
            return array(
                'stock' => $cached['stock_value'],
                'source' => 'stale_cache',
                'checked_at' => $cached['checked_at'],
                'age_seconds' => $cached['age_seconds'],
                'warning' => 'Live fetch failed, using cached value'
            );
        }
        
        return false;
    }
    
    /**
     * Fetch live stock from Delcampe API
     * 
     * @param int $delcampe_id Delcampe listing ID
     * @return int|false Stock value or false on error
     */
    private function fetch_live_stock($delcampe_id) {
        // Check rate limits
        if ($this->rate_limiter && $this->rate_limiter->should_pause_operations()) {
            $this->log_event('Rate limit exceeded for stock fetch', array('delcampe_id' => $delcampe_id));
            return false;
        }
        
        // Use the existing inventory manager's method to fetch stock
        // This avoids duplicating API logic and ensures consistency
        try {
            // Get authentication token
            $auth = Delcampe_Auth::get_instance();
            $token = $auth->get_auth_token();
            
            if (is_wp_error($token)) {
                return false;
            }
            
            // Build API URL - using correct REST endpoint format
            // Correct format is /item/{id} not /item/get
            $url = DELCAMPE_API_URL . '/item/' . $delcampe_id . '?token=' . $token;
            
            // Make API request
            $response = wp_remote_get($url, array(
                'timeout' => 30,
                'headers' => array(
                    'Accept' => 'application/xml'
                )
            ));
            
            if (is_wp_error($response)) {
                throw new Exception($response->get_error_message());
            }
            
            $body = wp_remote_retrieve_body($response);
            if (empty($body)) {
                throw new Exception('Empty response from API');
            }
            
            // Parse XML response
            $xml = simplexml_load_string($body);
            if (!$xml) {
                throw new Exception('Invalid XML response');
            }
            
            // Check for API errors
            if (isset($xml->e)) {
                throw new Exception((string)$xml->e);
            }
            
            // Check for 404 Not Found response (API couldn't retrieve item)
            if (isset($xml->status) && $xml->status == '404') {
                $this->log_event('API returned 404 for item', array(
                    'delcampe_id' => $delcampe_id,
                    'detail' => isset($xml->detail) ? (string)$xml->detail : 'Not Found'
                ));
                // Don't update cache, throw exception to use fallback
                throw new Exception('API returned 404 - item temporarily unavailable');
            }
            
            // Extract quantity from various possible locations
            // This matches the logic in class-delcampe-inventory-manager.php
            $stock = null;
            
            // 1) Direct item node
            if (isset($xml->item)) {
                if (isset($xml->item->qty)) {
                    $stock = (int)$xml->item->qty;
                } elseif (isset($xml->item->quantity)) {
                    $stock = (int)$xml->item->quantity;
                }
            }
            
            // 2) Notification envelope (for some API responses)
            if ($stock === null && isset($xml->Notification_Data->body->item)) {
                $item = $xml->Notification_Data->body->item;
                if (isset($item->qty)) {
                    $stock = (int)$item->qty;
                } elseif (isset($item->quantity)) {
                    $stock = (int)$item->quantity;
                }
            }
            
            // 3) XPath fallback to find any item element
            if ($stock === null) {
                $nodes = $xml->xpath('//item');
                if ($nodes && isset($nodes[0])) {
                    $item = $nodes[0];
                    if (isset($item->qty)) {
                        $stock = (int)$item->qty;
                    } elseif (isset($item->quantity)) {
                        $stock = (int)$item->quantity;
                    }
                }
            }
            
            // 4) Direct qty field at root level
            if ($stock === null && isset($xml->qty)) {
                $stock = (int)$xml->qty;
            }
            
            if ($stock !== null) {
                $this->log_event('Live stock fetched', array(
                    'delcampe_id' => $delcampe_id,
                    'stock' => $stock
                ));
                return $stock;
            }
            
            // Log the XML structure for debugging
            $this->log_error('Quantity not found in XML response', array(
                'delcampe_id' => $delcampe_id,
                'xml_structure' => substr($body, 0, 500) // First 500 chars for debugging
            ));
            
            throw new Exception('Quantity not found in response');
        } catch (Exception $e) {
            $this->log_error('Failed to fetch live stock', array(
                'delcampe_id' => $delcampe_id,
                'error' => $e->getMessage()
            ));
        }
        
        return false;
    }
    
    /**
     * Batch refresh stock cache for stale items
     * 
     * @param int $limit Number of items to refresh
     * @return array Results of the batch operation
     */
    public function batch_refresh_stock_cache($limit = null) {
        global $wpdb;
        
        if ($limit === null) {
            $limit = self::BATCH_SIZE;
        }
        
        // Enforce maximum batch size
        $limit = min($limit, self::MAX_BATCH_SIZE);
        
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        // Get stale or unchecked items
        $stale_threshold = date('Y-m-d H:i:s', time() - self::CACHE_TTL);
        
        $items = $wpdb->get_results($wpdb->prepare("
            SELECT delcampe_id, product_id, delcampe_stock_checked_at 
            FROM $table_name 
            WHERE status IN ('published', 'active')
            AND delcampe_id IS NOT NULL
            AND (
                delcampe_stock_checked_at IS NULL 
                OR delcampe_stock_checked_at < %s
            )
            ORDER BY ISNULL(delcampe_stock_checked_at) DESC, delcampe_stock_checked_at ASC
            LIMIT %d
        ", $stale_threshold, $limit));
        
        $results = array(
            'total' => count($items),
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
            'errors' => array()
        );
        
        foreach ($items as $item) {
            // Check rate limits
            if ($this->rate_limiter && $this->rate_limiter->should_pause_operations()) {
                $results['skipped']++;
                $results['errors'][] = "Rate limit reached after {$results['success']} items";
                break;
            }
            
            // Fetch and update stock
            $stock = $this->fetch_live_stock($item->delcampe_id);
            if ($stock !== false) {
                $this->write_cache($item->delcampe_id, $stock, 'batch_refresh');
                $results['success']++;
            } else {
                $results['failed']++;
                // Still update timestamp to prevent repeated failures
                $this->mark_as_checked($item->delcampe_id);
            }
            
            // Small delay to respect API
            usleep(100000); // 0.1 second
        }
        
        $this->log_event('Batch refresh completed', $results);
        
        return $results;
    }
    
    /**
     * Mark an item as checked without updating stock
     * 
     * @param int $delcampe_id Delcampe listing ID
     */
    private function mark_as_checked($delcampe_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        $wpdb->update(
            $table_name,
            array('delcampe_stock_checked_at' => current_time('mysql')),
            array('delcampe_id' => $delcampe_id),
            array('%s'),
            array('%d')
        );
    }
    
    /**
     * Invalidate cache for specific item
     * 
     * @param int $delcampe_id Delcampe listing ID
     */
    public function invalidate_cache($delcampe_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        $wpdb->update(
            $table_name,
            array(
                'delcampe_stock_cached' => null,
                'delcampe_stock_checked_at' => null
            ),
            array('delcampe_id' => $delcampe_id),
            array('%d', '%s'),
            array('%d')
        );
        
        $this->log_event('Cache invalidated', array('delcampe_id' => $delcampe_id));
    }
    
    /**
     * Get cache statistics
     * 
     * @return array Cache statistics
     */
    public function get_cache_stats() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_listings';
        $stale_threshold = date('Y-m-d H:i:s', time() - self::CACHE_TTL);
        
        $stats = array();
        
        // Total cached items
        $stats['total_cached'] = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM $table_name 
            WHERE delcampe_stock_cached IS NOT NULL
        ");
        
        // Fresh cache items
        $stats['fresh_items'] = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) 
            FROM $table_name 
            WHERE delcampe_stock_cached IS NOT NULL 
            AND delcampe_stock_checked_at >= %s
        ", $stale_threshold));
        
        // Stale cache items
        $stats['stale_items'] = $stats['total_cached'] - $stats['fresh_items'];
        
        // Never checked items
        $stats['never_checked'] = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM $table_name 
            WHERE status IN ('published', 'active')
            AND delcampe_id IS NOT NULL
            AND delcampe_stock_checked_at IS NULL
        ");
        
        // Cache hit rate (if we're tracking this)
        $stats['cache_hit_rate'] = $this->calculate_hit_rate();
        
        // Next refresh time
        $next_refresh = wp_next_scheduled('delcampe_refresh_stock_cache');
        $stats['next_refresh'] = $next_refresh ? date('Y-m-d H:i:s', $next_refresh) : 'Not scheduled';
        
        return $stats;
    }
    
    /**
     * Calculate cache hit rate
     * 
     * @return float Hit rate percentage
     */
    private function calculate_hit_rate() {
        // This would need to track hits/misses in transients or options
        // For now, return a placeholder
        $hits = get_transient('delcampe_cache_hits') ?: 0;
        $misses = get_transient('delcampe_cache_misses') ?: 0;
        
        $total = $hits + $misses;
        if ($total > 0) {
            return round(($hits / $total) * 100, 2);
        }
        
        return 0;
    }
    
    /**
     * Log cache update event
     */
    private function log_cache_update($delcampe_id, $stock_value, $source) {
        $this->log_event('Stock cache updated', array(
            'delcampe_id' => $delcampe_id,
            'stock' => $stock_value,
            'source' => $source
        ));
    }
    
    /**
     * Log event
     */
    private function log_event($message, $data = array()) {
        if ($this->logger) {
            $this->logger->log_event('stock_cache', $message, $data);
        }
        
        // v1.10.35.10: Removed excessive debug logging - only log in debug mode
        if (defined('DELCAMPE_DEBUG_MODE') && DELCAMPE_DEBUG_MODE) {
            error_log('[Delcampe Stock Cache] ' . $message . ' - ' . json_encode($data));
        }
    }
    
    /**
     * Log error
     */
    private function log_error($message, $data = array()) {
        if ($this->logger) {
            $this->logger->log_event('stock_cache_error', $message, $data);
        }
        
        // v1.10.35.10: Keep error logging but reduce verbosity
        if (defined('DELCAMPE_DEBUG_MODE') && DELCAMPE_DEBUG_MODE) {
            error_log('[Delcampe Stock Cache ERROR] ' . $message . ' - ' . json_encode($data));
        } else {
            // In production, only log the message without data dump
            error_log('[Delcampe Stock Cache ERROR] ' . $message);
        }
    }
    
    /**
     * Clean up on deactivation
     */
    public static function deactivate() {
        // Remove cron job
        wp_clear_scheduled_hook('delcampe_refresh_stock_cache');
    }
}

// Initialize the cache manager
add_action('init', array('Delcampe_Stock_Cache_Manager', 'get_instance'));