<?php
/**
 * Delcampe Stuck Order Resolver
 * 
 * Detects and resolves orders stuck in "initial_sale" reconciliation stage
 * Automatically fetches missing payment data from Delcampe API
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.36.0
 */

if (!defined('ABSPATH')) exit;

class Delcampe_Stuck_Order_Resolver {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Schedule daily check for stuck orders
        add_action('init', array($this, 'schedule_stuck_order_check'));
        add_action('delcampe_check_stuck_orders', array($this, 'check_and_resolve_stuck_orders'));
        
        // v1.10.37.0: Schedule daily cleanup of abandoned sales (>7 days old)
        add_action('init', array($this, 'schedule_abandoned_sales_cleanup'));
        add_action('delcampe_cleanup_abandoned_sales', array($this, 'cleanup_abandoned_sales'));
        
        // Manual trigger via admin
        add_action('admin_post_delcampe_resolve_stuck_orders', array($this, 'manual_resolve_stuck_orders'));
    }
    
    /**
     * Schedule stuck order check every 2 hours
     * v1.10.36.3: Changed from daily to 2-hourly for faster payment webhook fallback
     */
    public function schedule_stuck_order_check() {
        if (!wp_next_scheduled('delcampe_check_stuck_orders')) {
            // Run every 2 hours to catch missing payment webhooks quickly
            wp_schedule_event(time(), 'twicedaily', 'delcampe_check_stuck_orders');
        }
    }
    
    /**
     * Find orders stuck in initial_sale stage for more than 2 hours
     * 
     * @param int $hours Hours to wait before considering order stuck (default: 2)
     * @return array Array of WC_Order objects
     */
    public function find_stuck_orders($hours = 2) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        // Calculate cutoff time (default: 2 hours ago for faster resolution)
        // v1.10.36.3: Changed from 24 hours to 2 hours per production feedback
        $cutoff_time = strtotime("-{$hours} hours");
        
        // Query orders with reconciliation stage = initial_sale
        $args = array(
            'limit' => -1,
            'status' => array('on-hold', 'pending'),
            'meta_query' => array(
                array(
                    'key' => '_delcampe_reconciliation_stage',
                    'value' => 'initial_sale',
                    'compare' => '='
                )
            ),
            'date_created' => '<' . $cutoff_time,
            'return' => 'objects'
        );
        
        $orders = wc_get_orders($args);
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'STUCK_ORDER_SCAN',
            'found_count' => count($orders),
            'cutoff_time' => date('Y-m-d H:i:s', $cutoff_time)
        ));
        
        return $orders;
    }
    
    /**
     * Attempt to resolve stuck order
     * 
     * ⚠️ IMPORTANT LIMITATION (v1.10.36.5):
     * Per Delcampe API documentation (verified 2025-10-06), there is NO API endpoint
     * to poll or check payment status. The ONLY way to receive payment confirmation is
     * via the Seller_Payment_Received webhook. If the webhook is missed or fails:
     * 
     * 1. Delcampe will retry up to 5 times
     * 2. After 5 failures, Delcampe sends XML via email fallback
     * 3. NO API method exists to retrieve missed notifications
     * 4. Manual intervention required: check Delcampe dashboard
     * 
     * This method attempts to check /item/{id} and /item/{id}/buyer endpoints,
     * but these do NOT expose payment status according to official API docs.
     * The method is kept for logging and potential future API changes.
     * 
     * @param WC_Order $order Order to resolve
     * @return bool|WP_Error Success or error
     */
    public function resolve_stuck_order($order) {
        $logger = Delcampe_Sync_Logger::get_instance();
        $order_id = $order->get_id();
        
        try {
            // Get Delcampe item ID from order meta
            $item_id = $order->get_meta('_delcampe_item_id');
            
            // v1.10.36.4: Fallback to delcampe_orders table if not in meta
            // (Production bug: meta sometimes not saved properly)
            if (!$item_id) {
                global $wpdb;
                $table_name = $wpdb->prefix . 'delcampe_orders';
                $item_id = $wpdb->get_var($wpdb->prepare(
                    "SELECT delcampe_item_id FROM {$table_name} WHERE wc_order_id = %d LIMIT 1",
                    $order->get_id()
                ));
            }
            
            if (!$item_id) {
                throw new Exception('Missing Delcampe item ID in both order meta and database table');
            }
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'STUCK_ORDER_RESOLUTION_START',
                'order_id' => $order_id,
                'item_id' => $item_id
            ));
            
            // Get auth token
            $auth = Delcampe_Auth::get_instance();
            $token = $auth->get_auth_token();
            
            if (is_wp_error($token)) {
                throw new Exception('Failed to get auth token: ' . $token->get_error_message());
            }
            
            // v1.10.36.3: Try BOTH endpoints for payment data
            // Method 1: /item/{id}/buyer (most reliable for bill_id)
            $buyer_url = DELCAMPE_API_URL . '/item/' . $item_id . '/buyer?token=' . $token;
            $buyer_response = wp_remote_get($buyer_url, array(
                'timeout' => 30,
                'headers' => array('Accept' => 'application/xml')
            ));
            
            $buyer_xml = null;
            if (!is_wp_error($buyer_response)) {
                $buyer_body = wp_remote_retrieve_body($buyer_response);
                $buyer_xml = simplexml_load_string($buyer_body);
            }
            
            // Method 2: /item/{id} (fallback)
            $url = DELCAMPE_API_URL . '/item/' . $item_id . '?token=' . $token;
            $response = wp_remote_get($url, array(
                'timeout' => 30,
                'headers' => array('Accept' => 'application/xml')
            ));
            
            if (is_wp_error($response)) {
                throw new Exception('API request failed: ' . $response->get_error_message());
            }
            
            $body = wp_remote_retrieve_body($response);
            $xml = simplexml_load_string($body);
            
            if (!$xml || isset($xml->e)) {
                $error_msg = isset($xml->e) ? (string)$xml->e : 'Failed to parse response';
                throw new Exception('API error: ' . $error_msg);
            }
            
            // Check if item has payment information
            if (!isset($xml->item)) {
                throw new Exception('No item data in response');
            }
            
            $item = $xml->item;
            
            // Extract payment information if available
            $has_payment_data = false;
            $payment_data = new stdClass();
            
            // v1.10.36.3: Check buyer endpoint FIRST for bill_id (most reliable)
            if ($buyer_xml) {
                // Check various possible locations for bill_id
                if (isset($buyer_xml->bill_id) && !empty((string)$buyer_xml->bill_id)) {
                    $has_payment_data = true;
                    $payment_data->bill_id = (string)$buyer_xml->bill_id;
                } elseif (isset($buyer_xml->Notification_Data->body->bill_id) && !empty((string)$buyer_xml->Notification_Data->body->bill_id)) {
                    $has_payment_data = true;
                    $payment_data->bill_id = (string)$buyer_xml->Notification_Data->body->bill_id;
                } elseif (isset($buyer_xml->body->bill_id) && !empty((string)$buyer_xml->body->bill_id)) {
                    $has_payment_data = true;
                    $payment_data->bill_id = (string)$buyer_xml->body->bill_id;
                }
                
                // Also check for total_amount in buyer endpoint
                if (isset($buyer_xml->total_amount)) {
                    $has_payment_data = true;
                    $payment_data->total_amount = $this->parse_amount((string)$buyer_xml->total_amount);
                }
                if (isset($buyer_xml->seller_amount)) {
                    $payment_data->seller_amount = $this->parse_amount((string)$buyer_xml->seller_amount);
                }
            }
            
            // Fallback: Check item endpoint for bill_id or payment status
            if (!$has_payment_data && isset($item->bill_id) && !empty((string)$item->bill_id)) {
                $has_payment_data = true;
                $payment_data->bill_id = (string)$item->bill_id;
            }
            
            // Extract amounts
            if (isset($item->total_amount)) {
                $has_payment_data = true;
                $payment_data->total_amount = $this->parse_amount((string)$item->total_amount);
            }
            
            if (isset($item->seller_amount)) {
                $payment_data->seller_amount = $this->parse_amount((string)$item->seller_amount);
            }
            
            // Extract shipping cost
            if (isset($item->shipping_cost)) {
                $payment_data->shipping_cost = $this->parse_amount((string)$item->shipping_cost);
            }
            
            // If we have payment data, reconcile it
            if ($has_payment_data) {
                // Create item list for reconciliation
                $payment_data->item_list = new stdClass();
                $payment_data->item_list->item = array();
                
                $item_obj = new stdClass();
                $item_obj->price = $order->get_meta('_delcampe_item_price');
                $item_obj->qty = $order->get_meta('_delcampe_qty_sold') ?: 1;
                $payment_data->item_list->item[] = $item_obj;
                
                // Use reconciliation system to update order
                require_once DWC_PLUGIN_DIR . '/includes/class-delcampe-order-reconciliation.php';
                Delcampe_Order_Reconciliation::reconcile_payment_received($order, $payment_data);
                
                // Add order note
                $order->add_order_note(sprintf(
                    __('Stuck order automatically resolved via API fetch. Missing payment data retrieved from Delcampe.', 'wc-delcampe-integration')
                ));
                
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'STUCK_ORDER_RESOLVED',
                    'order_id' => $order_id,
                    'item_id' => $item_id,
                    'total_amount' => $payment_data->total_amount
                ));
                
                return true;
            } else {
                // No payment data available yet - order is legitimately waiting for payment
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'STUCK_ORDER_NO_PAYMENT_YET',
                    'order_id' => $order_id,
                    'item_id' => $item_id,
                    'message' => 'Order is correctly waiting for payment - no action needed'
                ));
                
                return new WP_Error('no_payment_data', 'Order is correctly waiting for payment');
            }
            
        } catch (Exception $e) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'STUCK_ORDER_RESOLUTION_FAILED',
                'order_id' => $order_id,
                'error' => $e->getMessage()
            ));
            
            return new WP_Error('resolution_failed', $e->getMessage());
        }
    }
    
    /**
     * Parse amount string (removes currency symbols, converts to float)
     * 
     * @param string $amount Amount string
     * @return float Parsed amount
     */
    private function parse_amount($amount) {
        // Remove non-numeric characters except dots and commas
        $amount = preg_replace('/[^0-9.,]/', '', $amount);
        // Replace comma with dot for decimal
        $amount = str_replace(',', '.', $amount);
        return floatval($amount);
    }
    
    /**
     * Check and resolve all stuck orders
     */
    public function check_and_resolve_stuck_orders() {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'STUCK_ORDER_CHECK_START'
        ));
        
        $stuck_orders = $this->find_stuck_orders();
        
        $resolved_count = 0;
        $failed_count = 0;
        $waiting_count = 0;
        
        foreach ($stuck_orders as $order) {
            $result = $this->resolve_stuck_order($order);
            
            if ($result === true) {
                $resolved_count++;
            } elseif (is_wp_error($result)) {
                if ($result->get_error_code() === 'no_payment_data') {
                    $waiting_count++;
                } else {
                    $failed_count++;
                }
            }
        }
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'STUCK_ORDER_CHECK_COMPLETE',
            'total_found' => count($stuck_orders),
            'resolved' => $resolved_count,
            'still_waiting' => $waiting_count,
            'failed' => $failed_count
        ));
    }
    
    /**
     * Manual trigger for admin
     */
    public function manual_resolve_stuck_orders() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Unauthorized');
        }
        
        check_admin_referer('delcampe_resolve_stuck_orders');
        
        $this->check_and_resolve_stuck_orders();
        
        wp_redirect(add_query_arg(array(
            'page' => 'delcampe-orders',
            'stuck_orders_resolved' => '1'
        ), admin_url('admin.php')));
        exit;
    }
    
    /**
     * Schedule abandoned sales cleanup (daily)
     * v1.10.37.0: Removes pending sales >7 days old with no payment
     */
    public function schedule_abandoned_sales_cleanup() {
        if (!wp_next_scheduled('delcampe_cleanup_abandoned_sales')) {
            wp_schedule_event(time(), 'daily', 'delcampe_cleanup_abandoned_sales');
        }
    }
    
    /**
     * Cleanup abandoned sales older than 7 days
     * Removes pending sale data from delcampe_orders table if no payment received
     * 
     * v1.10.37.0: New eBay-style order creation requires cleanup
     * 
     * @param int $days Days to wait before cleanup (default: 7)
     */
    public function cleanup_abandoned_sales($days = 7) {
        global $wpdb;
        $logger = Delcampe_Sync_Logger::get_instance();
        
        $table_name = $wpdb->prefix . 'delcampe_orders';
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'ABANDONED_SALES_CLEANUP_START',
            'cutoff_date' => $cutoff_date,
            'days' => $days
        ));
        
        // Find abandoned sales
        $abandoned_sales = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table_name} 
             WHERE payment_status = 'awaiting_payment'
             AND wc_order_id IS NULL
             AND created_at < %s",
            $cutoff_date
        ));
        
        if (!$abandoned_sales) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'ABANDONED_SALES_CLEANUP_COMPLETE',
                'message' => 'No abandoned sales found'
            ));
            return;
        }
        
        $cleaned_count = 0;
        $failed_count = 0;
        
        foreach ($abandoned_sales as $sale) {
            // Double-check no order was created in meantime
            if ($sale->wc_order_id) {
                continue;
            }
            
            // Delete the abandoned sale
            $deleted = $wpdb->delete(
                $table_name,
                array('id' => $sale->id),
                array('%d')
            );
            
            if ($deleted) {
                $cleaned_count++;
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'ABANDONED_SALE_DELETED',
                    'delcampe_item_id' => $sale->delcampe_item_id,
                    'buyer_nickname' => $sale->buyer_nickname,
                    'sale_date' => $sale->sale_date,
                    'days_old' => round((time() - strtotime($sale->created_at)) / 86400, 1)
                ));
            } else {
                $failed_count++;
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'ABANDONED_SALE_DELETE_FAILED',
                    'delcampe_item_id' => $sale->delcampe_item_id,
                    'error' => $wpdb->last_error
                ));
            }
        }
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'ABANDONED_SALES_CLEANUP_COMPLETE',
            'total_found' => count($abandoned_sales),
            'cleaned' => $cleaned_count,
            'failed' => $failed_count
        ));
    }
}
