<?php
/**
 * Delcampe Sync Logger
 * 
 * Dedicated logging for item sync operations with detailed request/response tracking
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.3.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Sync_Logger {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Log file path
     */
    private $log_file;
    
    /**
     * Constructor
     */
    private function __construct() {
        // Create dedicated sync log file
        $this->log_file = DELCAMPE_LOG_DIR . '/delcampe-item-sync.log';
        
        // Ensure log directory exists
        if (!file_exists(DELCAMPE_LOG_DIR)) {
            wp_mkdir_p(DELCAMPE_LOG_DIR);
        }
    }
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Log sync operation start
     * 
     * @param int $product_id Product ID
     * @param array $product_data Product details being synced
     */
    public function log_sync_start($product_id, $product_data) {
        $entry = array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'SYNC_START',
            'product_id' => $product_id,
            'sku' => isset($product_data['sku']) ? $product_data['sku'] : '',
            'title' => isset($product_data['title']) ? $product_data['title'] : '',
            'price' => isset($product_data['price']) ? $product_data['price'] : '',
            'personal_reference' => isset($product_data['personal_reference']) ? $product_data['personal_reference'] : ''
        );
        
        $this->write_log($entry);
    }
    
    /**
     * Log API request
     * 
     * @param string $endpoint API endpoint
     * @param array $request_data Data being sent
     */
    public function log_api_request($endpoint, $request_data) {
        // Remove sensitive data
        $safe_data = $request_data;
        if (isset($safe_data['token'])) {
            $safe_data['token'] = substr($safe_data['token'], 0, 10) . '...';
        }
        
        $entry = array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'API_REQUEST',
            'endpoint' => $endpoint,
            'method' => 'POST',
            'fields_sent' => array_keys($request_data),
            'personal_reference' => isset($request_data['personal_reference']) ? $request_data['personal_reference'] : '',
            'title' => isset($request_data['title']) ? substr($request_data['title'], 0, 50) . '...' : '',
            'price' => isset($request_data['fixed_price']) ? $request_data['fixed_price'] : '',
            'category' => isset($request_data['id_category']) ? $request_data['id_category'] : ''
        );
        
        $this->write_log($entry);
    }
    
    /**
     * Log API response
     * 
     * @param int $status_code HTTP status code
     * @param string $response_body Raw response body
     * @param array $parsed_response Parsed response data
     */
    public function log_api_response($status_code, $response_body, $parsed_response = null) {
        $entry = array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'API_RESPONSE',
            'status_code' => $status_code,
            'response_length' => strlen($response_body)
        );
        
        // Parse response if not already parsed
        if ($parsed_response === null && !empty($response_body)) {
            // Try to parse XML
            libxml_use_internal_errors(true);
            $xml = simplexml_load_string($response_body);
            
            if ($xml !== false) {
                // Check for full notification envelope
                if (isset($xml->Notification_Type)) {
                    $entry['notification_type'] = (string)$xml->Notification_Type;
                    $entry['notification_token'] = isset($xml->Notification_Token) ? (string)$xml->Notification_Token : '';
                    
                    // Handle notification data
                    if (isset($xml->Notification_Data)) {
                        $notification_data = $xml->Notification_Data;
                        
                        switch ($entry['notification_type']) {
                            case 'Seller_Item_Add':
                                if (isset($notification_data->id_item)) {
                                    $entry['item_id'] = (string)$notification_data->id_item;
                                } else {
                                    $entry['item_id'] = 'NOT_PROVIDED';
                                }
                                if (isset($notification_data->personal_reference)) {
                                    $entry['personal_reference'] = (string)$notification_data->personal_reference;
                                }
                                $entry['item_status'] = 'Your item has been taken into account';
                                $entry['queued_for_processing'] = true;
                                $entry['success'] = true;
                                break;
                                
                            case 'Seller_Item_Add_Error_Image':
                                if (isset($notification_data->id_item)) {
                                    $entry['item_id'] = (string)$notification_data->id_item;
                                }
                                if (isset($notification_data->errorImages)) {
                                    $entry['warning'] = 'Some images could not be processed';
                                    $image_errors = array();
                                    foreach ($notification_data->errorImages->imageUrl as $url) {
                                        $image_errors[] = (string)$url;
                                    }
                                    if (!empty($image_errors)) {
                                        $entry['image_errors'] = json_encode($image_errors);
                                    }
                                }
                                $entry['success'] = true; // Item was still created
                                break;
                        }
                    }
                }
                // Check for API response structure
                elseif (isset($xml->Notification_Data)) {
                    // Check notification type in headers
                    if (isset($xml->Notification_Data->Headers->Notification_type)) {
                        $entry['notification_type'] = (string)$xml->Notification_Data->Headers->Notification_type;
                    }
                    
                    $body = $xml->Notification_Data->body;
                    
                    // Handle 201 Created with Seller_Item_Add notification
                    if ($status_code === 201 && isset($entry['notification_type']) && 
                        $entry['notification_type'] === 'Seller_Item_Add') {
                        
                        if (isset($body->Status)) {
                            $entry['item_status'] = (string)$body->Status;
                            if (strpos($entry['item_status'], 'taken into account') !== false) {
                                $entry['queued_for_processing'] = true;
                            }
                        }
                        
                        if (isset($body->Personal_reference)) {
                            $entry['personal_reference'] = (string)$body->Personal_reference;
                        }
                        
                        if (isset($body->Item_ID)) {
                            $entry['item_id'] = (string)$body->Item_ID;
                        } else {
                            $entry['item_id'] = 'NOT_PROVIDED';
                        }
                        
                        if (isset($body->Warning)) {
                            $entry['warning'] = (string)$body->Warning;
                        }
                        
                        $entry['success'] = true;
                    }
                    // Handle standard 200 response (old format)
                    elseif (isset($body->item)) {
                        $entry['has_item_element'] = true;
                        
                        if (isset($body->item->id)) {
                            $entry['item_id'] = (string)$body->item->id;
                        } else {
                            $entry['item_id'] = 'NOT_PROVIDED';
                        }
                        
                        if (isset($body->item->status)) {
                            $entry['item_status'] = (string)$body->item->status;
                            if (strpos($entry['item_status'], 'taken into account') !== false) {
                                $entry['queued_for_processing'] = true;
                            }
                        }
                    }
                    
                    // Check if response indicates success
                    if (isset($xml->Notification_Data->Headers->Status)) {
                        $api_status = (int)$xml->Notification_Data->Headers->Status;
                        if ($api_status === 200 || $api_status === 201) {
                            $entry['success'] = true;
                        } else {
                            $entry['success'] = false;
                        }
                    }
                }
            }
            
            libxml_clear_errors();
        } elseif (is_array($parsed_response)) {
            // Use provided parsed response
            if (isset($parsed_response['notification_type'])) {
                $entry['notification_type'] = $parsed_response['notification_type'];
            }
            
            if (isset($parsed_response['id'])) {
                $entry['item_id'] = $parsed_response['id'];
            }
            
            if (isset($parsed_response['message'])) {
                $entry['item_status'] = $parsed_response['message'];
            }
            
            if (isset($parsed_response['success'])) {
                $entry['success'] = $parsed_response['success'];
            }
            
            if (isset($parsed_response['queued'])) {
                $entry['queued_for_processing'] = true;
            }
            
            if (isset($parsed_response['warning'])) {
                $entry['warning'] = $parsed_response['warning'];
            }
            
            if (isset($parsed_response['personal_reference'])) {
                $entry['personal_reference'] = $parsed_response['personal_reference'];
            }
        }
        
        $this->write_log($entry);
    }
    
    /**
     * Log sync completion
     * 
     * @param int $product_id Product ID
     * @param bool $success Whether sync was successful
     * @param string $listing_id Delcampe listing ID if available
     * @param string $message Additional message
     */
    public function log_sync_complete($product_id, $success, $listing_id = '', $message = '') {
        $entry = array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'SYNC_COMPLETE',
            'product_id' => $product_id,
            'success' => $success,
            'listing_id' => $listing_id,
            'message' => $message
        );
        
        $this->write_log($entry);
    }
    
    /**
     * Log sync error
     * 
     * @param int $product_id Product ID
     * @param string $error_message Error message
     * @param array $error_details Additional error details
     */
    public function log_sync_error($product_id, $error_message, $error_details = array()) {
        $entry = array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'SYNC_ERROR',
            'product_id' => $product_id,
            'error' => $error_message,
            'details' => $error_details
        );
        
        $this->write_log($entry);
    }
    
    /**
     * Write log entry
     * 
     * @param array $entry Log entry data
     */
    public function write_log($entry) {
        $log_line = '[' . $entry['timestamp'] . '] [' . $entry['event'] . '] ';
        unset($entry['timestamp'], $entry['event']);
        
        // Format the rest of the entry
        $parts = array();
        foreach ($entry as $key => $value) {
            if (is_array($value)) {
                $parts[] = $key . '=' . json_encode($value);
            } elseif (is_bool($value)) {
                $parts[] = $key . '=' . ($value ? 'true' : 'false');
            } else {
                $parts[] = $key . '=' . $value;
            }
        }
        
        $log_line .= implode(' | ', $parts) . PHP_EOL;
        
        // Write to file
        @file_put_contents($this->log_file, $log_line, FILE_APPEND | LOCK_EX);
        
        // Rotate log if too large (5MB)
        if (file_exists($this->log_file) && filesize($this->log_file) > 5242880) {
            $backup_file = $this->log_file . '.' . date('Y-m-d-His');
            @rename($this->log_file, $backup_file);
        }
    }
    
    /**
     * Get recent log entries
     * 
     * @param int $limit Number of entries to return
     * @return array Recent log entries
     */
    public function get_recent_entries($limit = 100) {
        if (!file_exists($this->log_file)) {
            return array();
        }
        
        $lines = file($this->log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        return array_slice($lines, -$limit);
    }
    
    /**
     * Clear the sync log
     */
    public function clear_log() {
        if (file_exists($this->log_file)) {
            @unlink($this->log_file);
        }
    }
}