<?php
/**
 * Plugin Update Checker
 *
 * Handles automatic update checks and integrates with WordPress update system
 *
 * @package WC_Delcampe_Integration
 * @version 1.9.0.5
 * @since   1.9.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Delcampe_Updater class
 *
 * @since 1.9.0.0
 */
class Delcampe_Updater {
    
    /**
     * Update server URL
     * @var string
     */
    const UPDATE_SERVER_URL = 'https://www.sos-tech.ca/plugin-updates/delcampe/';
    
    /**
     * Plugin slug
     * @var string
     */
    private $plugin_slug;
    
    /**
     * Plugin basename
     * @var string
     */
    private $plugin_basename;
    
    /**
     * Plugin data
     * @var array
     */
    private $plugin_data;
    
    /**
     * License manager instance
     * @var Delcampe_License_Manager
     */
    private $license_manager;
    
    /**
     * Cache time in seconds (12 hours)
     * @var int
     */
    const CACHE_TIME = 43200;
    
    /**
     * Constructor
     *
     * @since 1.9.0.0
     * @param string $plugin_file Main plugin file path
     */
    public function __construct($plugin_file) {
        $this->plugin_basename = plugin_basename($plugin_file);
        $this->plugin_slug = dirname($this->plugin_basename);
        
        // Get plugin data without translations to avoid early loading (v1.10.5.13)
        if (!function_exists('get_plugin_data')) {
            require_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }
        $this->plugin_data = get_plugin_data($plugin_file, false, false);
        
        // Get license manager instance
        $this->license_manager = Delcampe_License_Manager::get_instance();
        
        // Hook into WordPress update system
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_update'));
        add_filter('plugins_api', array($this, 'plugin_info'), 10, 3);
        add_action('in_plugin_update_message-' . $this->plugin_basename, array($this, 'update_message'), 10, 2);
        
        // Hook to control download based on license
        add_filter('upgrader_pre_download', array($this, 'check_license_before_download'), 10, 3);
        
        // Add action for force check
        add_action('wp_ajax_delcampe_force_update_check', array($this, 'ajax_force_update_check'));
    }
    
    /**
     * Check for plugin updates
     *
     * @since 1.9.0.0
     * @param object $transient Update transient
     * @return object Modified transient
     */
    public function check_for_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        // Get remote version info
        $remote_info = $this->get_remote_info();
        
        if (is_wp_error($remote_info) || empty($remote_info)) {
            return $transient;
        }
        
        // Compare versions
        $current_version = $this->plugin_data['Version'];
        $remote_version = $remote_info->version;
        
        if (version_compare($current_version, $remote_version, '<')) {
            // Check if user has valid license
            $has_license = $this->license_manager->is_license_valid();
            
            // Prepare update object
            $update = new stdClass();
            $update->slug = $this->plugin_slug;
            $update->plugin = $this->plugin_basename;
            $update->new_version = $remote_version;
            $update->url = $remote_info->homepage;
            $update->tested = $remote_info->tested ?? '';
            $update->requires = $remote_info->requires ?? '';
            $update->requires_php = $remote_info->requires_php ?? '';
            $update->icons = (array) ($remote_info->icons ?? array());
            $update->banners = (array) ($remote_info->banners ?? array());
            $update->banners_rtl = (array) ($remote_info->banners_rtl ?? array());
            
            // Always provide the download URL from the server
            // The actual download will be controlled by WordPress filters
            $update->package = !empty($remote_info->package) ? $remote_info->package : '';
            
            // Add license status to the update object for later use
            $update->license_valid = $has_license;
            
            // Always add to response to show update is available
            $transient->response[$this->plugin_basename] = $update;
        } else {
            // Add to no_update to show "up to date" message
            $transient->no_update[$this->plugin_basename] = (object) array(
                'slug' => $this->plugin_slug,
                'plugin' => $this->plugin_basename,
                'new_version' => $current_version,
                'url' => $this->plugin_data['PluginURI'],
                'package' => ''
            );
        }
        
        return $transient;
    }
    
    /**
     * Provide plugin information for update details
     *
     * @since 1.9.0.0
     * @param false|object|array $result Plugin API result
     * @param string $action API action
     * @param object $args API arguments
     * @return false|object Plugin information
     */
    public function plugin_info($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }
        
        if ($args->slug !== $this->plugin_slug) {
            return $result;
        }
        
        // Get remote info
        $remote_info = $this->get_remote_info();
        
        if (is_wp_error($remote_info) || empty($remote_info)) {
            return $result;
        }
        
        // Prepare plugin information
        $plugin_info = new stdClass();
        $plugin_info->name = $remote_info->name;
        $plugin_info->slug = $this->plugin_slug;
        $plugin_info->version = $remote_info->version;
        $plugin_info->author = $remote_info->author;
        $plugin_info->author_profile = $remote_info->author_profile ?? '';
        $plugin_info->requires = $remote_info->requires ?? '';
        $plugin_info->tested = $remote_info->tested ?? '';
        $plugin_info->requires_php = $remote_info->requires_php ?? '';
        
        // Always provide download link - license check happens during actual download
        $plugin_info->download_link = !empty($remote_info->package) ? $remote_info->package : '';
        
        $plugin_info->sections = (array) ($remote_info->sections ?? array());
        $plugin_info->banners = (array) ($remote_info->banners ?? array());
        $plugin_info->icons = (array) ($remote_info->icons ?? array());
        $plugin_info->homepage = $remote_info->homepage ?? '';
        $plugin_info->short_description = $remote_info->short_description ?? '';
        
        // Add changelog if available
        if (!empty($remote_info->changelog)) {
            $plugin_info->sections['changelog'] = $remote_info->changelog;
        }
        
        // Add license notice if no valid license
        if (!$this->license_manager->is_license_valid()) {
            $plugin_info->sections['license_notice'] = '<div style="background: #fff3cd; border: 1px solid #ffeeba; padding: 10px; margin: 10px 0; border-radius: 4px;">' .
                '<strong>License Required:</strong> A valid license is required to download updates. ' .
                '<a href="' . admin_url('admin.php?page=delcampe-license') . '">Activate your license</a> to enable automatic updates.' .
                '</div>';
        }
        
        return $plugin_info;
    }
    
    /**
     * Display custom update message
     *
     * @since 1.9.0.0
     * @param array $plugin_data Plugin data
     * @param object $response Update response
     */
    public function update_message($plugin_data, $response) {
        if (!$this->license_manager->is_license_valid()) {
            echo '<br />' .
                 '<strong style="color: #d63638;">' . 
                 sprintf(
                     __('There is a new version (%s) of Delcampe Sync available. To update directly register your license key now or purchase a new license key to access all premium features.', 'wc-delcampe-integration'),
                     $response->new_version
                 ) .
                 '</strong> ' .
                 '<br />' .
                 '<a href="' . admin_url('admin.php?page=delcampe-license') . '" class="button button-primary" style="margin: 5px 0;">' . 
                 __('Register License Key', 'wc-delcampe-integration') . 
                 '</a> ' .
                 '<a href="https://www.sos-tech.ca/plugins/delcampe-sync" target="_blank" class="button" style="margin: 5px 0 5px 5px;">' . 
                 __('Purchase License', 'wc-delcampe-integration') . 
                 '</a>';
        }
    }
    
    /**
     * Get remote plugin information
     *
     * @since 1.9.0.0
     * @return object|WP_Error Remote info or error
     */
    private function get_remote_info() {
        // Check cache first
        $cache_key = 'delcampe_update_info';
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        // Make API request
        $response = wp_remote_get(
            self::UPDATE_SERVER_URL . 'info.json',
            array(
                'timeout' => 10,
                'headers' => array(
                    'Accept' => 'application/json'
                )
            )
        );
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);
        
        if (empty($data) || !is_object($data)) {
            return new WP_Error('invalid_response', 'Invalid update server response');
        }
        
        // Cache the result
        set_transient($cache_key, $data, self::CACHE_TIME);
        
        return $data;
    }
    
    /**
     * AJAX handler for force update check
     *
     * @since 1.9.0.0
     */
    public function ajax_force_update_check() {
        if (!current_user_can('update_plugins')) {
            wp_die('Unauthorized');
        }
        
        check_ajax_referer('delcampe_admin_nonce', 'nonce');
        
        // Clear cache
        delete_transient('delcampe_update_info');
        delete_site_transient('update_plugins');
        
        // Force WordPress to check for updates
        wp_update_plugins();
        
        wp_send_json_success(array(
            'message' => __('Update check completed', 'wc-delcampe-integration')
        ));
    }
    
    /**
     * Check license before allowing download
     *
     * @since 1.10.5.4
     * @param bool|WP_Error $reply Whether to bail without returning the package
     * @param string $package The package file URL
     * @param WP_Upgrader $upgrader The WP_Upgrader instance
     * @return bool|WP_Error
     */
    public function check_license_before_download($reply, $package, $upgrader) {
        // Only check for our plugin
        if (!isset($upgrader->skin->plugin_info) || $upgrader->skin->plugin_info['Name'] !== $this->plugin_data['Name']) {
            return $reply;
        }
        
        // Check if license is valid
        if (!$this->license_manager->is_license_valid()) {
            return new WP_Error(
                'no_license',
                sprintf(
                    __('A valid license is required to download updates for %s. <a href="%s">Activate your license</a> or <a href="%s" target="_blank">purchase a license</a>.', 'wc-delcampe-integration'),
                    $this->plugin_data['Name'],
                    admin_url('admin.php?page=delcampe-license'),
                    'https://www.sos-tech.ca/plugins/delcampe-sync'
                )
            );
        }
        
        return $reply;
    }
}