<?php
/**
 * Delcampe Plugin Upgrader
 * 
 * Handles safe plugin upgrades to prevent file deletion
 * 
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Upgrader {
    
    /**
     * Hook into WordPress upgrade process
     */
    public static function init() {
        // Pre-upgrade: backup critical files
        add_filter('upgrader_pre_install', array(__CLASS__, 'pre_upgrade_backup'), 10, 2);
        
        // Post-upgrade: restore if needed
        add_filter('upgrader_post_install', array(__CLASS__, 'post_upgrade_restore'), 10, 3);
        
        // Handle the specific error
        add_filter('upgrader_pre_replace_filter', array(__CLASS__, 'fix_backup_directory'), 10, 4);
    }
    
    /**
     * Fix backup directory permissions before upgrade
     */
    public static function fix_backup_directory($response, $hook_extra, $result, $upgrader) {
        if (!isset($hook_extra['plugin']) || $hook_extra['plugin'] !== 'delcampe-sync/main.php') {
            return $response;
        }
        
        // Ensure upgrade directory exists and is writable
        $upgrade_dir = WP_CONTENT_DIR . '/upgrade/';
        $backup_dir = WP_CONTENT_DIR . '/upgrade-temp-backup/';
        
        if (!file_exists($upgrade_dir)) {
            wp_mkdir_p($upgrade_dir);
        }
        
        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
        }
        
        // Try to set permissions
        @chmod($upgrade_dir, 0755);
        @chmod($backup_dir, 0755);
        
        // Clear any file locks
        clearstatcache();
        
        return $response;
    }
    
    /**
     * Backup critical files before upgrade
     */
    public static function pre_upgrade_backup($response, $hook_extra) {
        // Only for our plugin
        if (!isset($hook_extra['plugin']) || $hook_extra['plugin'] !== 'delcampe-sync/main.php') {
            return $response;
        }
        
        $plugin_dir = WP_PLUGIN_DIR . '/delcampe-sync/';
        $backup_dir = WP_CONTENT_DIR . '/delcampe-sync-backup-' . time() . '/';
        
        // Create backup of critical files
        if (file_exists($plugin_dir)) {
            $critical_files = array(
                '.htaccess',
                'wp-config.php', // if exists
                '*.log',
                'data/*',
                'uploads/*'
            );
            
            wp_mkdir_p($backup_dir);
            
            // Save current version for comparison
            update_option('delcampe_sync_pre_upgrade_version', DELCAMPE_PLUGIN_VERSION);
            update_option('delcampe_sync_backup_dir', $backup_dir);
        }
        
        return $response;
    }
    
    /**
     * Restore files after upgrade if needed
     */
    public static function post_upgrade_restore($response, $hook_extra, $result) {
        // Only for our plugin
        if (!isset($hook_extra['plugin']) || $hook_extra['plugin'] !== 'delcampe-sync/main.php') {
            return $response;
        }
        
        // Clean up backup
        delete_option('delcampe_sync_pre_upgrade_version');
        delete_option('delcampe_sync_backup_dir');
        
        return $response;
    }
    
    /**
     * Alternative update method using direct filesystem
     */
    public static function direct_upgrade($package_file) {
        global $wp_filesystem;
        
        // Initialize filesystem
        if (!function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        
        WP_Filesystem();
        
        $plugin_dir = WP_PLUGIN_DIR . '/delcampe-sync/';
        $temp_dir = WP_CONTENT_DIR . '/upgrade/delcampe-sync-temp/';
        
        // Extract to temp directory
        $unzip_result = unzip_file($package_file, $temp_dir);
        
        if (is_wp_error($unzip_result)) {
            return $unzip_result;
        }
        
        // Move files instead of delete/copy
        if ($wp_filesystem->exists($plugin_dir)) {
            // Rename old directory
            $backup_dir = WP_PLUGIN_DIR . '/delcampe-sync-old-' . time() . '/';
            $wp_filesystem->move($plugin_dir, $backup_dir);
        }
        
        // Move new files
        $wp_filesystem->move($temp_dir . 'delcampe-sync/', $plugin_dir);
        
        // Clean up
        $wp_filesystem->delete($temp_dir, true);
        
        return true;
    }
}

// Initialize upgrader
Delcampe_Upgrader::init();