<?php
/**
 * Delcampe Webhook Handler
 * 
 * Processes incoming webhook notifications from Delcampe
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.4.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Webhook_Handler {
    
    /**
     * Process incoming notification
     * 
     * @param SimpleXMLElement $xml Parsed XML notification
     * @param string $raw_xml Raw XML string
     * @return array Result with success status and message
     */
    public function process_notification($xml, $raw_xml) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        try {
            // Extract notification type
            $notification_type = null;
            if (isset($xml->Notification_Type)) {
                $notification_type = (string)$xml->Notification_Type;
            } elseif (isset($xml->notification_type)) {
                $notification_type = (string)$xml->notification_type;
            }
            
            if (empty($notification_type)) {
                throw new Exception('Missing notification type');
            }
            
            // COMPREHENSIVE RAW DATA LOGGING - Log ALL webhook data for debugging
            $this->log_raw_webhook_data($notification_type, $xml, $raw_xml);
            
            // Log notification details
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_PROCESSING',
                'notification_type' => $notification_type
            ));
            
            // Route to appropriate handler
            switch ($notification_type) {
                case 'Seller_Item_Add':
                    return $this->handle_item_add($xml);
                    
                case 'Seller_Item_Add_Error_Image':
                    return $this->handle_item_add_with_image_error($xml);
                    
                case 'Seller_Item_Update':
                    return $this->handle_item_update($xml);
                    
                case 'Seller_Item_Delete':
                    return $this->handle_item_delete($xml);
                    
                case 'Seller_Item_Sold':
                    return $this->handle_item_sold($xml);
                    
                case 'Seller_Item_Close_Sold':
                    // This is the real-time notification when an item is sold
                    return $this->handle_item_close_sold($xml);
                    
                case 'Seller_Payment_Received':
                    // This notification comes when payment is confirmed
                    return $this->handle_payment_received($xml);
                    
                case 'Seller_Item_Status_Change':
                    return $this->handle_item_status_change($xml);
                    
                // Additional webhook types for catalog sync
                case 'Seller_Item_Close_Unsold':
                    return $this->handle_item_close_unsold($xml);
                    
                case 'Seller_Item_Close_Manually':
                    return $this->handle_item_close_manually($xml);
                    
                case 'Seller_Item_Restart':
                    return $this->handle_item_restart($xml);
                    
                case 'Seller_Item_Move':
                    return $this->handle_item_move($xml);
                    
                default:
                    // Log unknown notification type but don't fail
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'WEBHOOK_UNKNOWN_TYPE',
                        'notification_type' => $notification_type,
                        'raw_xml' => $raw_xml
                    ));
                    
                    return array(
                        'success' => true,
                        'message' => 'Unknown notification type received: ' . $notification_type
                    );
            }
            
        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }
    
    /**
     * Handle Seller_Item_Add notification
     * 
     * @param SimpleXMLElement $xml
     * @return array
     */
    private function handle_item_add($xml) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        // Extract item data
        $item_id = null;
        $personal_reference = null;
        $status = null;
        
        // Check for notification data
        if (isset($xml->Notification_Data)) {
            $data = $xml->Notification_Data;
            
            if (isset($data->id_item)) {
                $item_id = (string)$data->id_item;
            }
            if (isset($data->personal_reference)) {
                $personal_reference = (string)$data->personal_reference;
            }
            if (isset($data->status)) {
                $status = (string)$data->status;
            }
        }
        
        if (empty($item_id) || empty($personal_reference)) {
            throw new Exception('Missing required fields: item_id or personal_reference');
        }
        
        // Log successful item creation
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_ITEM_CREATED',
            'item_id' => $item_id,
            'personal_reference' => $personal_reference,
            'status' => $status ?: 'unknown'
        ));
        
        // Update product metadata
        $result = $this->update_product_listing_id($personal_reference, $item_id, $status);
        
        if ($result) {
            return array(
                'success' => true,
                'message' => 'Item successfully created with ID: ' . $item_id
            );
        } else {
            return array(
                'success' => false,
                'message' => 'Failed to update product with listing ID'
            );
        }
    }
    
    /**
     * Handle Seller_Item_Add_Error_Image notification
     * 
     * @param SimpleXMLElement $xml
     * @return array
     */
    private function handle_item_add_with_image_error($xml) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        // Extract data
        $item_id = null;
        $personal_reference = null;
        $image_errors = array();
        
        if (isset($xml->Notification_Data)) {
            $data = $xml->Notification_Data;
            
            if (isset($data->id_item)) {
                $item_id = (string)$data->id_item;
            }
            if (isset($data->personal_reference)) {
                $personal_reference = (string)$data->personal_reference;
            }
            if (isset($data->errorImages)) {
                foreach ($data->errorImages->imageUrl as $url) {
                    $image_errors[] = (string)$url;
                }
            }
        }
        
        // Log the notification
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_ITEM_CREATED_IMAGE_ERROR',
            'item_id' => $item_id,
            'personal_reference' => $personal_reference,
            'image_errors' => json_encode($image_errors)
        ));
        
        // Update product - item was still created despite image errors
        if (!empty($item_id) && !empty($personal_reference)) {
            $this->update_product_listing_id($personal_reference, $item_id, 'active');
            
            // Add note about image errors
            $this->add_product_note($personal_reference, 
                'Delcampe listing created with ID ' . $item_id . 
                ' but some images failed to upload: ' . implode(', ', $image_errors)
            );
        }
        
        return array(
            'success' => true,
            'message' => 'Item created with image errors'
        );
    }
    
    /**
     * Handle item update notification
     * 
     * @param SimpleXMLElement $xml
     * @return array
     */
    private function handle_item_update($xml) {
        $biz = Delcampe_Business_Logger::get_instance();
        
        // Extract identifiers
        $data = isset($xml->Notification_Data->body) ? $xml->Notification_Data->body : (isset($xml->Notification_Data) ? $xml->Notification_Data : null);
        if (!$data) {
            return array('success' => false, 'message' => 'Malformed update payload');
        }

        $item_id = isset($data->id_item) ? (string)$data->id_item : '';
        $personal_reference = isset($data->personal_reference) ? (string)$data->personal_reference : '';

        $biz->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_ITEM_UPDATE',
            'delcampe_id' => $item_id,
            'personal_reference' => $personal_reference
        ));

        // Collect possible updates
        $updates = array('status' => 'changed', 'date_updated' => current_time('mysql'));
        if (isset($data->item_title)) {
            $updates['listing_title'] = (string)$data->item_title;
        } elseif (isset($data->title)) {
            $updates['listing_title'] = (string)$data->title;
        }
        if (isset($data->fixed_price)) {
            $updates['price'] = number_format((float)$data->fixed_price, 2, '.', '');
        } elseif (isset($data->price)) {
            $updates['price'] = number_format((float)$data->price, 2, '.', '');
        }
        if (isset($data->currency)) {
            $updates['currency'] = (string)$data->currency;
        }
        if (isset($data->qty)) {
            $updates['quantity'] = (int)$data->qty;
        }
        if (isset($data->id_category)) {
            $updates['category_id'] = (string)$data->id_category;
        }

        // Update DB if we have an id
        if (!empty($item_id)) {
            global $wpdb;
            $table = $wpdb->prefix . 'delcampe_listings';
            $wpdb->update($table, $updates, array('delcampe_id' => $item_id));
        }

        // Trigger image sync if images were included
        if (isset($data->images)) {
            $images = array();
            // Support both simple arrays and <imageUrl> structure
            if (isset($data->images->imageUrl)) {
                foreach ($data->images->imageUrl as $img) {
                    $images[] = (string)$img;
                }
            } else {
                foreach ($data->images as $img) {
                    $images[] = (string)$img;
                }
            }
            if (!empty($images) && !empty($item_id)) {
                do_action('delcampe_webhook_image_update', $item_id, $images);
            }
        }

        // Optionally store a note on the product
        if (!empty($personal_reference)) {
            $this->add_product_note($personal_reference, 'Listing updated on Delcampe');
        }

        return array('success' => true, 'message' => 'Item update processed');
    }
    
    /**
     * Handle item delete notification
     * 
     * @param SimpleXMLElement $xml
     * @return array
     */
    private function handle_item_delete($xml) {
        $biz = Delcampe_Business_Logger::get_instance();

        $data = isset($xml->Notification_Data->body) ? $xml->Notification_Data->body : (isset($xml->Notification_Data) ? $xml->Notification_Data : null);
        if (!$data) {
            return array('success' => false, 'message' => 'Malformed delete payload');
        }

        $item_id = isset($data->id_item) ? (string)$data->id_item : '';
        $personal_reference = isset($data->personal_reference) ? (string)$data->personal_reference : '';

        $biz->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_ITEM_DELETE',
            'delcampe_id' => $item_id,
            'personal_reference' => $personal_reference
        ));

        if (!empty($item_id)) {
            global $wpdb;
            $table = $wpdb->prefix . 'delcampe_listings';
            $wpdb->update($table, array(
                'status' => 'deleted',
                'date_finished' => current_time('mysql'),
                'date_updated' => current_time('mysql')
            ), array('delcampe_id' => $item_id));
        }

        if (!empty($personal_reference)) {
            $this->update_product_status($personal_reference, 'deleted');
        }

        return array('success' => true, 'message' => 'Item delete processed');
    }
    
    /**
     * Handle Seller_Item_Close_Sold notification
     * This is the REAL-TIME notification when an item is sold
     * Happens immediately when the sale occurs
     * 
     * Based on actual Delcampe XML format from API documentation
     * 
     * @param SimpleXMLElement $xml
     * @return array
     * @since 1.10.12.7
     * @version 1.10.13.0 - Updated to match actual Delcampe format
     */
    private function handle_item_close_sold($xml) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        try {
            // Log the raw notification for debugging
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_ITEM_CLOSE_SOLD',
                'raw_xml' => $xml->asXML()
            ));
            
            // Extract data from the actual Delcampe format
            $notification_data = isset($xml->Notification_Data) ? $xml->Notification_Data : null;
            
            if (!$notification_data) {
                throw new Exception('Missing Notification_Data in Seller_Item_Close_Sold');
            }
            
            // Extract data based on actual Delcampe format
            // Fields from ChatGPT's research of actual API format
            $item_id = isset($notification_data->id_item) ? (string)$notification_data->id_item : null;
            
            // Extract buyer information
            $buyer_info = array(
                'id' => isset($notification_data->buyer_id) ? (string)$notification_data->buyer_id : '',
                'nickname' => isset($notification_data->buyer_nickname) ? (string)$notification_data->buyer_nickname : '',
                'email' => isset($notification_data->buyer_email) ? (string)$notification_data->buyer_email : '',
                'firstname' => isset($notification_data->buyer_firstname) ? (string)$notification_data->buyer_firstname : '',
                'lastname' => isset($notification_data->buyer_surname) ? (string)$notification_data->buyer_surname : '',
                'address' => isset($notification_data->buyer_address) ? (string)$notification_data->buyer_address : '',
                'city' => isset($notification_data->buyer_city) ? (string)$notification_data->buyer_city : '',
                'zipcode' => isset($notification_data->buyer_zipcode) ? (string)$notification_data->buyer_zipcode : '',
                'country' => isset($notification_data->buyer_country) ? (string)$notification_data->buyer_country : ''
            );
            
            // Extract sale information with improved price extraction
            $sale_info = array(
                'item_title' => isset($notification_data->item_title) ? (string)$notification_data->item_title : '',
                'sku' => isset($notification_data->personal_reference) ? (string)$notification_data->personal_reference : '',
                'price' => $this->extract_price_from_notification($notification_data, $item_id),
                'currency' => isset($notification_data->best_bid_currency) ? (string)$notification_data->best_bid_currency : get_woocommerce_currency(),
                'quantity' => isset($notification_data->qty) ? (int)$notification_data->qty : 1,
                'date' => isset($xml->Notification_Datetime) ? (string)$xml->Notification_Datetime : current_time('mysql'),
                'seller_nickname' => isset($notification_data->seller_nickname) ? (string)$notification_data->seller_nickname : ''
            );
            
            // Decode URL-encoded fields if present
            if (isset($notification_data->url_to_item)) {
                $sale_info['item_url'] = urldecode((string)$notification_data->url_to_item);
            }
            if (isset($notification_data->url_to_rating)) {
                $sale_info['rating_url'] = urldecode((string)$notification_data->url_to_rating);
            }
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_CLOSE_SOLD_PARSED',
                'item_id' => $item_id,
                'buyer_info' => json_encode($buyer_info),
                'sale_info' => json_encode($sale_info)
            ));
            
            if (!$item_id) {
                throw new Exception('Missing item ID in Seller_Item_Close_Sold notification');
            }
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_CLOSE_SOLD_DATA',
                'item_id' => $item_id,
                'buyer_nickname' => $buyer_info['nickname'],
                'price' => $sale_info['price'],
                'currency' => $sale_info['currency'],
                'sku' => $sale_info['sku']
            ));
            
            // v1.10.37.0: Store sale data only - DO NOT create order until payment received
            // This prevents $0.00 stuck orders when payment webhook is missed
            global $wpdb;
            $table_name = $wpdb->prefix . 'delcampe_orders';
            
            // Generate order ID using Delcampe format
            $delcampe_order_id = 'DC-' . $item_id . '-' . time();
            
            // Check if sale already stored
            $existing = $wpdb->get_row($wpdb->prepare(
                "SELECT id FROM {$table_name} WHERE delcampe_item_id = %d",
                $item_id
            ));
            
            if ($existing) {
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_SALE_ALREADY_STORED',
                    'item_id' => $item_id,
                    'existing_id' => $existing->id
                ));
                
                return array(
                    'success' => true,
                    'message' => 'Sale already stored, awaiting payment'
                );
            }
            
            // Store sale data in database (awaiting payment)
            $result = $wpdb->insert(
                $table_name,
                array(
                    'delcampe_order_id' => $delcampe_order_id,
                    'delcampe_item_id' => $item_id,
                    'buyer_id' => isset($buyer_info['id']) ? $buyer_info['id'] : 0,
                    'buyer_nickname' => isset($buyer_info['nickname']) ? $buyer_info['nickname'] : '',
                    'buyer_email' => isset($buyer_info['email']) ? $buyer_info['email'] : '',
                    'buyer_xml' => $xml->asXML(),
                    'sale_price' => isset($sale_info['price']) ? $sale_info['price'] : 0,
                    'currency' => isset($sale_info['currency']) ? $sale_info['currency'] : get_woocommerce_currency(),
                    'quantity' => isset($sale_info['quantity']) ? $sale_info['quantity'] : 1,
                    'sale_date' => isset($sale_info['date']) ? $sale_info['date'] : current_time('mysql'),
                    'sale_xml' => json_encode($sale_info),
                    'payment_status' => 'awaiting_payment',
                    'sync_status' => 'pending_payment',
                    'created_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql')
                ),
                array('%s', '%d', '%d', '%s', '%s', '%s', '%f', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s')
            );
            
            if ($result === false) {
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_SALE_STORAGE_FAILED',
                    'item_id' => $item_id,
                    'error' => $wpdb->last_error
                ));
                
                throw new Exception('Failed to store sale data: ' . $wpdb->last_error);
            }
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_SALE_STORED_AWAITING_PAYMENT',
                'item_id' => $item_id,
                'delcampe_order_id' => $delcampe_order_id,
                'buyer_nickname' => $buyer_info['nickname'],
                'price' => $sale_info['price'],
                'currency' => $sale_info['currency'],
                'message' => 'Sale data stored - WooCommerce order will be created when payment webhook arrives'
            ));
            
            return array(
                'success' => true,
                'message' => 'Seller_Item_Close_Sold notification processed'
            );
            
        } catch (Exception $e) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_CLOSE_SOLD_ERROR',
                'error' => $e->getMessage()
            ));
            
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }
    
    /**
     * Handle Seller_Payment_Received notification
     * Updates the order when payment is confirmed
     * Based on actual Delcampe XML format from API documentation
     * 
     * @param SimpleXMLElement $xml
     * @return array
     * @since 1.10.12.7
     * @version 1.10.13.0 - Updated to match actual format with proper shipping calculation
     */
    private function handle_payment_received($xml) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        try {
            // Log the raw notification
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_PAYMENT_RECEIVED',
                'raw_xml' => $xml->asXML()
            ));
            
            // v1.10.37.0: Now creates orders from stored sale data
            $notification_data = isset($xml->Notification_Data) ? $xml->Notification_Data : null;
            
            if (!$notification_data) {
                throw new Exception('Missing Notification_Data in Seller_Payment_Received');
            }
            
            // Extract IDs and payment info based on actual Delcampe format
            $bill_id = isset($notification_data->id_bill) ? (string)$notification_data->id_bill : null;
            $transaction_id = isset($notification_data->transaction_id) ? (string)$notification_data->transaction_id : null;
            
            // Parse total_amount and seller_amount (they come with currency symbol)
            $total_amount_str = isset($notification_data->total_amount) ? (string)$notification_data->total_amount : '';
            $seller_amount_str = isset($notification_data->seller_amount) ? (string)$notification_data->seller_amount : '';
            
            // Extract currency from the payment notification
            $payment_currency = isset($notification_data->currency) ? (string)$notification_data->currency : '';
            
            // Extract numeric values from amounts (remove currency symbols)
            $total_amount = preg_replace('/[^0-9.,]/', '', $total_amount_str);
            $total_amount = str_replace(',', '.', $total_amount); // Handle European format
            $total_amount = (float)$total_amount;
            
            $seller_amount = preg_replace('/[^0-9.,]/', '', $seller_amount_str);
            $seller_amount = str_replace(',', '.', $seller_amount);
            $seller_amount = (float)$seller_amount;
            
            // Normalize with WooCommerce helpers
            $total_amount = wc_format_decimal($total_amount, wc_get_price_decimals());
            $seller_amount = wc_format_decimal($seller_amount, wc_get_price_decimals());
            
            $payment_info = array(
                'bill_id' => $bill_id,
                'transaction_id' => $transaction_id,
                'total_amount' => $total_amount,
                'seller_amount' => $seller_amount,
                'platform_fees' => $total_amount - $seller_amount,
                'currency' => $payment_currency,
                'shipping_method' => isset($notification_data->shipping_method) ? (string)$notification_data->shipping_method : '',
                'payment_method' => isset($notification_data->payment_method) ? (string)$notification_data->payment_method : '',
                'buyer_comment' => isset($notification_data->buyer_comment) ? (string)$notification_data->buyer_comment : '',
                'date' => isset($xml->Notification_Datetime) ? (string)$xml->Notification_Datetime : current_time('mysql')
            );
            
            // Extract item(s) from item_list
            $items = array();
            if (isset($notification_data->item_list) && isset($notification_data->item_list->item)) {
                foreach ($notification_data->item_list->item as $item) {
                    // Parse the price if present
                    $price_str = isset($item->price) ? (string)$item->price : '';
                    $price_num = preg_replace('/[^0-9.,]/', '', $price_str);
                    $price_num = (float) str_replace(',', '.', $price_num);
                    
                    $items[] = array(
                        'title' => isset($item->item_title) ? (string)$item->item_title : '',
                        'qty' => isset($item->qty) ? (int)$item->qty : 1,
                        'sku' => isset($item->personal_reference) ? (string)$item->personal_reference : '',
                        'url' => isset($item->url_to_item) ? (string)$item->url_to_item : '',
                        'price' => $price_num
                    );
                }
            }
            
            // Extract item ID from first item's URL
            $item_id = null;
            if (!empty($items)) {
                $first_item = $items[0];
                if (!empty($first_item['url']) && preg_match('/item\/(\d+)/', $first_item['url'], $matches)) {
                    $item_id = $matches[1];
                }
            }
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_PAYMENT_PARSED',
                'bill_id' => $bill_id,
                'item_id' => $item_id,
                'total_amount' => $total_amount,
                'seller_amount' => $seller_amount,
                'items' => json_encode($items)
            ));
            
            if (!$bill_id) {
                throw new Exception('Missing bill ID in payment notification');
            }
            
            // v1.10.37.0: Find stored sale data and create WooCommerce order
            global $wpdb;
            $table_name = $wpdb->prefix . 'delcampe_orders';
            
            // Find stored sale data by item_id
            $sale_data = null;
            if ($item_id) {
                $sale_data = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$table_name} 
                     WHERE delcampe_item_id = %d 
                     AND payment_status = 'awaiting_payment'
                     LIMIT 1",
                    $item_id
                ));
            }
            
            // If not found by item_id, try by bill_id (may already have WC order)
            if (!$sale_data) {
                $sale_data = $wpdb->get_row($wpdb->prepare(
                    "SELECT * FROM {$table_name} WHERE delcampe_order_id LIKE %s LIMIT 1",
                    '%' . $bill_id . '%'
                ));
            }
            
            if (!$sale_data) {
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_PAYMENT_NO_SALE_DATA',
                    'item_id' => $item_id,
                    'bill_id' => $bill_id,
                    'message' => 'Payment received but no stored sale data found'
                ));
                
                throw new Exception('Payment received but no stored sale data found for item_id: ' . $item_id);
            }
            
            // Check if order already created
            if ($sale_data->wc_order_id) {
                $order = wc_get_order($sale_data->wc_order_id);
                if ($order) {
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'WEBHOOK_PAYMENT_ORDER_ALREADY_EXISTS',
                        'item_id' => $item_id,
                        'wc_order_id' => $sale_data->wc_order_id,
                        'message' => 'Order already created for this payment'
                    ));
                    
                    return array(
                        'success' => true,
                        'message' => 'Payment already processed for order #' . $sale_data->wc_order_id
                    );
                }
            }
            
            // Create WooCommerce order with complete payment data
            if (class_exists('Delcampe_Order_Manager')) {
                $order_manager = Delcampe_Order_Manager::get_instance();
                
                // Reconstruct buyer_info and sale_info from stored data
                $buyer_xml = null;
                if ($sale_data->buyer_xml) {
                    $buyer_xml = simplexml_load_string($sale_data->buyer_xml);
                }
                
                $sale_info = json_decode($sale_data->sale_xml, true);
                if (!$sale_info) {
                    $sale_info = array();
                }
                
                // Add payment details to sale info
                $sale_info['total_amount'] = $payment_info['total_amount'];
                $sale_info['seller_amount'] = $payment_info['seller_amount'];
                $sale_info['bill_id'] = $payment_info['bill_id'];
                $sale_info['transaction_id'] = $payment_info['transaction_id'];
                $sale_info['platform_fees'] = $payment_info['platform_fees'];
                $sale_info['payment_method'] = $payment_info['payment_method'];
                $sale_info['shipping_method'] = $payment_info['shipping_method'];
                $sale_info['buyer_comment'] = $payment_info['buyer_comment'];
                
                // Extract buyer_info from stored buyer_xml
                $buyer_info = array(
                    'id' => $sale_data->buyer_id,
                    'nickname' => $sale_data->buyer_nickname,
                    'email' => $sale_data->buyer_email
                );
                
                if ($buyer_xml) {
                    $notification_data_buyer = isset($buyer_xml->Notification_Data) ? $buyer_xml->Notification_Data : $buyer_xml;
                    
                    $buyer_info['firstname'] = isset($notification_data_buyer->buyer_firstname) ? (string)$notification_data_buyer->buyer_firstname : '';
                    $buyer_info['lastname'] = isset($notification_data_buyer->buyer_surname) ? (string)$notification_data_buyer->buyer_surname : '';
                    $buyer_info['address'] = isset($notification_data_buyer->buyer_address) ? (string)$notification_data_buyer->buyer_address : '';
                    $buyer_info['city'] = isset($notification_data_buyer->buyer_city) ? (string)$notification_data_buyer->buyer_city : '';
                    $buyer_info['zipcode'] = isset($notification_data_buyer->buyer_zipcode) ? (string)$notification_data_buyer->buyer_zipcode : '';
                    $buyer_info['country'] = isset($notification_data_buyer->buyer_country) ? (string)$notification_data_buyer->buyer_country : '';
                    $buyer_info['phone'] = isset($notification_data_buyer->buyer_phone) ? (string)$notification_data_buyer->buyer_phone : '';
                }
                
                // Create the order NOW with complete data
                $order = $order_manager->create_order_from_close_sold($item_id, $buyer_info, $sale_info);
                
                if (is_wp_error($order)) {
                    throw new Exception('Order creation failed: ' . $order->get_error_message());
                }
                
                // Update sale data with WC order ID
                $wpdb->update(
                    $table_name,
                    array(
                        'wc_order_id' => $order->get_id(),
                        'payment_status' => 'paid',
                        'payment_date' => current_time('mysql'),
                        'sync_status' => 'completed',
                        'updated_at' => current_time('mysql')
                    ),
                    array('id' => $sale_data->id),
                    array('%d', '%s', '%s', '%s', '%s'),
                    array('%d')
                );
                
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_ORDER_CREATED_FROM_PAYMENT',
                    'item_id' => $item_id,
                    'wc_order_id' => $order->get_id(),
                    'bill_id' => $bill_id,
                    'total_amount' => $payment_info['total_amount'],
                    'message' => 'WooCommerce order created from payment webhook'
                ));
                
                return array(
                    'success' => true,
                    'message' => 'Order #' . $order->get_id() . ' created and payment recorded'
                );
            }
            
            throw new Exception('Delcampe_Order_Manager class not found');
            
        } catch (Exception $e) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_PAYMENT_ERROR',
                'error' => $e->getMessage()
            ));
            
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }
    
    /**
     * Handle item sold notification
     * 
     * Creates a WooCommerce order when an item is sold on Delcampe
     * Automatically reduces inventory as configured
     * 
     * @param SimpleXMLElement $xml
     * @return array
     * @since 1.10.4.0
     * @version 1.10.5.15 - Implemented real-time order processing
     */
    private function handle_item_sold($xml) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        try {
            // Extract notification data
            $notification_data = null;
            if (isset($xml->Notification_Data)) {
                $notification_data = $xml->Notification_Data;
            } elseif (isset($xml->notification_data)) {
                $notification_data = $xml->notification_data;
            }
            
            if (!$notification_data) {
                throw new Exception('Missing notification data in Seller_Item_Sold webhook');
            }
            
            // Extract essential data
            $item_id = isset($notification_data->id_item) ? (string)$notification_data->id_item : null;
            $buyer_id = isset($notification_data->buyer_id) ? (string)$notification_data->buyer_id : null;
            $sale_date = isset($notification_data->sale_date) ? (string)$notification_data->sale_date : null;
            $personal_reference = isset($notification_data->personal_reference) ? (string)$notification_data->personal_reference : null;
            
            // Log the sale notification
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_ITEM_SOLD',
                'item_id' => $item_id,
                'buyer_id' => $buyer_id,
                'sale_date' => $sale_date,
                'personal_reference' => $personal_reference
            ));
            
            if (!$item_id) {
                throw new Exception('Missing item ID in sold notification');
            }
            
            // Check if order manager is available
            if (!class_exists('Delcampe_Order_Manager')) {
                throw new Exception('Order Manager not available');
            }
            
            // Get order manager instance
            $order_manager = Delcampe_Order_Manager::get_instance();
            
            // Fetch full order details from Delcampe API
            $auth = Delcampe_Auth::get_instance();
            $token = $auth->get_auth_token();
            
            if (is_wp_error($token)) {
                throw new Exception('Failed to authenticate with Delcampe: ' . $token->get_error_message());
            }
            
            // Get item details from API - using correct REST endpoint format
            // Correct format is /item/{id} not /item/get
            $url = DELCAMPE_API_URL . '/item/' . $item_id . '?token=' . $token;
            
            $response = wp_remote_get($url, array(
                'timeout' => 30,
                'headers' => array('Accept' => 'application/xml')
            ));
            
            if (is_wp_error($response)) {
                throw new Exception('Failed to fetch item details: ' . $response->get_error_message());
            }
            
            $body = wp_remote_retrieve_body($response);
            $item_xml = simplexml_load_string($body);
            
            if (!$item_xml || isset($item_xml->e)) {
                throw new Exception('Failed to get item details from Delcampe');
            }
            
            // Import the order using the webhook-specific method
            $result = $order_manager->create_order_from_webhook($item_id, $item_xml->item);
            
            if (is_wp_error($result)) {
                throw new Exception('Failed to create order: ' . $result->get_error_message());
            }
            
            // Log successful order creation
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_ORDER_CREATED',
                'item_id' => $item_id,
                'wc_order_id' => is_object($result) ? $result->get_id() : $result,
                'message' => 'Order created successfully from webhook'
            ));
            
            // Update listing status based on remaining quantity
            global $wpdb;
            $listings_table = $wpdb->prefix . 'delcampe_listings';
            
            // Check if item has remaining quantity
            $remaining_qty = 0;
            if (isset($item_xml->item->qty)) {
                $remaining_qty = (int)$item_xml->item->qty;
            }
            
            // Only mark as 'sold' if no quantity remains, otherwise keep as 'published'
            $new_status = ($remaining_qty <= 0) ? 'sold' : 'published';
            
            $wpdb->update(
                $listings_table,
                array(
                    'status' => $new_status,
                    'quantity' => $remaining_qty,
                    'last_sync' => current_time('mysql')
                ),
                array('id_item' => $item_id),
                array('%s', '%d', '%s'),
                array('%d')
            );
            
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_LISTING_STATUS_UPDATE',
                'item_id' => $item_id,
                'remaining_qty' => $remaining_qty,
                'new_status' => $new_status
            ));
            
            return array(
                'success' => true,
                'message' => sprintf('Order created successfully for item #%s', $item_id),
                'order_id' => is_object($result) ? $result->get_id() : $result
            );
            
        } catch (Exception $e) {
            $logger->write_log(array(
                'timestamp' => current_time('Y-m-d H:i:s'),
                'event' => 'WEBHOOK_ITEM_SOLD_ERROR',
                'error' => $e->getMessage(),
                'xml' => isset($xml) ? $xml->asXML() : 'No XML data'
            ));
            
            return array(
                'success' => false,
                'message' => 'Failed to process sold item: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Handle item status change notification
     * 
     * @param SimpleXMLElement $xml
     * @return array
     */
    private function handle_item_status_change($xml) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        // Extract data
        $item_id = null;
        $new_status = null;
        $personal_reference = null;
        
        if (isset($xml->Notification_Data)) {
            $data = $xml->Notification_Data;
            
            if (isset($data->id_item)) {
                $item_id = (string)$data->id_item;
            }
            if (isset($data->status)) {
                $new_status = (string)$data->status;
            }
            if (isset($data->personal_reference)) {
                $personal_reference = (string)$data->personal_reference;
            }
        }
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_STATUS_CHANGE',
            'item_id' => $item_id,
            'new_status' => $new_status,
            'personal_reference' => $personal_reference
        ));
        
        // Update product status
        if (!empty($personal_reference) && !empty($new_status)) {
            $this->update_product_status($personal_reference, $new_status);
        }
        
        return array(
            'success' => true,
            'message' => 'Status change processed'
        );
    }
    
    /**
     * Update product with Delcampe listing ID
     * 
     * @param string $personal_reference SKU or personal reference
     * @param string $listing_id Delcampe item ID
     * @param string $status Item status
     * @return bool Success
     */
    private function update_product_listing_id($personal_reference, $listing_id, $status = null) {
        global $wpdb;
        
        // Find product by SKU
        $product_id = wc_get_product_id_by_sku($personal_reference);
        
        if (!$product_id) {
            // Try to find by personal reference in meta
            $product_id = $wpdb->get_var($wpdb->prepare(
                "SELECT post_id FROM {$wpdb->postmeta} 
                WHERE meta_key = '_delcampe_personal_reference' 
                AND meta_value = %s 
                LIMIT 1",
                $personal_reference
            ));
        }
        
        if (!$product_id) {
            error_log('[Delcampe Webhook] Product not found for reference: ' . $personal_reference);
            return false;
        }
        
        // Update listing ID
        update_post_meta($product_id, '_delcampe_listing_id', $listing_id);
        
        // Update sync status
        if ($status) {
            update_post_meta($product_id, '_delcampe_sync_status', $status);
        } else {
            update_post_meta($product_id, '_delcampe_sync_status', 'active');
        }
        
        // Update last sync time
        update_post_meta($product_id, '_delcampe_last_sync', current_time('mysql'));
        
        // Log the update
        error_log('[Delcampe Webhook] Updated product ' . $product_id . ' with listing ID: ' . $listing_id);
        
        return true;
    }
    
    /**
     * Update product status
     * 
     * @param string $personal_reference
     * @param string $status
     * @return bool
     */
    private function update_product_status($personal_reference, $status) {
        global $wpdb;
        
        // Find product
        $product_id = wc_get_product_id_by_sku($personal_reference);
        
        if (!$product_id) {
            $product_id = $wpdb->get_var($wpdb->prepare(
                "SELECT post_id FROM {$wpdb->postmeta} 
                WHERE meta_key = '_delcampe_personal_reference' 
                AND meta_value = %s 
                LIMIT 1",
                $personal_reference
            ));
        }
        
        if (!$product_id) {
            return false;
        }
        
        // Map Delcampe status to our status
        $status_map = array(
            'Draft' => 'draft',
            'InModeration' => 'moderation',
            'Ready' => 'ready',
            'Live' => 'active',
            'Sold' => 'sold',
            'Ended' => 'ended'
        );
        
        $mapped_status = isset($status_map[$status]) ? $status_map[$status] : $status;
        
        update_post_meta($product_id, '_delcampe_sync_status', $mapped_status);
        
        // Clean up Delcampe IDs when listing is closed/ended
        $cleanup_statuses = array('ended', 'closed', 'closed_unsold', 'closed_manually', 'sold');
        if (in_array($mapped_status, $cleanup_statuses) || in_array($status, $cleanup_statuses)) {
            delete_post_meta($product_id, '_delcampe_item_id');
            delete_post_meta($product_id, '_delcampe_listing_id');
            
            // Log the cleanup
            delcampe_log(sprintf(
                '[Webhook Handler] Cleaned up Delcampe IDs for product #%d (SKU: %s) - Status: %s',
                $product_id,
                $personal_reference,
                $mapped_status
            ));
        }
        
        return true;
    }
    
    /**
     * Add note to product
     * 
     * @param string $personal_reference
     * @param string $note
     */
    private function add_product_note($personal_reference, $note) {
        global $wpdb;
        
        // Find product
        $product_id = wc_get_product_id_by_sku($personal_reference);
        
        if (!$product_id) {
            $product_id = $wpdb->get_var($wpdb->prepare(
                "SELECT post_id FROM {$wpdb->postmeta} 
                WHERE meta_key = '_delcampe_personal_reference' 
                AND meta_value = %s 
                LIMIT 1",
                $personal_reference
            ));
        }
        
        if ($product_id) {
            // Add WooCommerce order note (works for products too)
            $product = wc_get_product($product_id);
            if ($product) {
                // Store as a meta note
                $notes = get_post_meta($product_id, '_delcampe_sync_notes', true);
                if (!is_array($notes)) {
                    $notes = array();
                }
                $notes[] = array(
                    'timestamp' => current_time('mysql'),
                    'note' => $note
                );
                update_post_meta($product_id, '_delcampe_sync_notes', $notes);
            }
        }
    }
    
    /**
     * Handle item closed unsold notification
     */
    private function handle_item_close_unsold($xml) {
        $logger = Delcampe_Business_Logger::get_instance();
        
        $item_id = isset($xml->Notification_Data->body->id_item) ? 
                   (string)$xml->Notification_Data->body->id_item : '';
        $personal_reference = isset($xml->Notification_Data->body->personal_reference) ?
                             (string)$xml->Notification_Data->body->personal_reference : '';
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_ITEM_CLOSE_UNSOLD',
            'delcampe_id' => $item_id,
            'personal_reference' => $personal_reference
        ));
        
        // Update listing status in database
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        $wpdb->update(
            $table_name,
            array(
                'status' => 'closed_unsold',
                'date_updated' => current_time('mysql')
            ),
            array('delcampe_id' => $item_id),
            array('%s', '%s'),
            array('%s')
        );
        
        // Update product status if we have a personal reference
        if (!empty($personal_reference)) {
            $this->update_product_status($personal_reference, 'closed_unsold');
            
            // Also find product by listing ID and clean up
            $product_id = $wpdb->get_var($wpdb->prepare(
                "SELECT product_id FROM {$wpdb->prefix}delcampe_listings WHERE delcampe_id = %s LIMIT 1",
                $item_id
            ));
            
            if ($product_id) {
                delete_post_meta($product_id, '_delcampe_item_id');
                delete_post_meta($product_id, '_delcampe_listing_id');
                delcampe_log(sprintf('[Webhook] Cleaned up IDs for product #%d - Item closed unsold', $product_id));
            }
        }
        
        return array(
            'success' => true,
            'message' => 'Item closed unsold processed'
        );
    }
    
    /**
     * Handle item closed manually notification
     */
    private function handle_item_close_manually($xml) {
        $logger = Delcampe_Business_Logger::get_instance();
        
        $item_id = isset($xml->Notification_Data->body->id_item) ? 
                   (string)$xml->Notification_Data->body->id_item : '';
        $personal_reference = isset($xml->Notification_Data->body->personal_reference) ?
                             (string)$xml->Notification_Data->body->personal_reference : '';
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_ITEM_CLOSE_MANUALLY',
            'delcampe_id' => $item_id,
            'personal_reference' => $personal_reference
        ));
        
        // Update listing status in database
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_listings';
        
        $wpdb->update(
            $table_name,
            array(
                'status' => 'closed_manually',
                'date_updated' => current_time('mysql')
            ),
            array('delcampe_id' => $item_id),
            array('%s', '%s'),
            array('%s')
        );
        
        // Update product status if we have a personal reference
        if (!empty($personal_reference)) {
            $this->update_product_status($personal_reference, 'closed_manually');
            
            // Also find product by listing ID and clean up
            $product_id = $wpdb->get_var($wpdb->prepare(
                "SELECT product_id FROM {$wpdb->prefix}delcampe_listings WHERE delcampe_id = %s LIMIT 1",
                $item_id
            ));
            
            if ($product_id) {
                delete_post_meta($product_id, '_delcampe_item_id');
                delete_post_meta($product_id, '_delcampe_listing_id');
                delcampe_log(sprintf('[Webhook] Cleaned up IDs for product #%d - Item closed manually', $product_id));
            }
        }
        
        return array(
            'success' => true,
            'message' => 'Item closed manually processed'
        );
    }
    
    /**
     * Handle item restart notification
     */
    private function handle_item_restart($xml) {
        $logger = Delcampe_Business_Logger::get_instance();
        
        $item_id = isset($xml->Notification_Data->body->id_item) ? 
                   (string)$xml->Notification_Data->body->id_item : '';
        $new_item_id = isset($xml->Notification_Data->body->new_id_item) ? 
                       (string)$xml->Notification_Data->body->new_id_item : '';
        $personal_reference = isset($xml->Notification_Data->body->personal_reference) ?
                             (string)$xml->Notification_Data->body->personal_reference : '';
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_ITEM_RESTART',
            'old_delcampe_id' => $item_id,
            'new_delcampe_id' => $new_item_id,
            'personal_reference' => $personal_reference
        ));
        
        // Update listing with new ID if provided
        if (!empty($new_item_id) && !empty($item_id)) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'delcampe_listings';
            
            $wpdb->update(
                $table_name,
                array(
                    'delcampe_id' => $new_item_id,
                    'status' => 'published',
                    'date_updated' => current_time('mysql')
                ),
                array('delcampe_id' => $item_id),
                array('%s', '%s', '%s'),
                array('%s')
            );
            
            // Update product meta if we have a personal reference
            if (!empty($personal_reference)) {
                $product_id = wc_get_product_id_by_sku($personal_reference);
                if ($product_id) {
                    update_post_meta($product_id, '_delcampe_listing_id', $new_item_id);
                    update_post_meta($product_id, '_delcampe_status', 'published');
                }
            }
        }
        
        return array(
            'success' => true,
            'message' => 'Item restart processed'
        );
    }
    
    /**
     * Handle item move (category change) notification
     */
    private function handle_item_move($xml) {
        $logger = Delcampe_Business_Logger::get_instance();
        
        $item_id = isset($xml->Notification_Data->body->id_item) ? 
                   (string)$xml->Notification_Data->body->id_item : '';
        $new_category = isset($xml->Notification_Data->body->new_category) ? 
                        (string)$xml->Notification_Data->body->new_category : '';
        $personal_reference = isset($xml->Notification_Data->body->personal_reference) ?
                             (string)$xml->Notification_Data->body->personal_reference : '';
        
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_ITEM_MOVE',
            'delcampe_id' => $item_id,
            'new_category' => $new_category,
            'personal_reference' => $personal_reference
        ));
        
        // Update category mapping if we track that
        if (!empty($new_category) && !empty($personal_reference)) {
            $product_id = wc_get_product_id_by_sku($personal_reference);
            if ($product_id) {
                update_post_meta($product_id, '_delcampe_category', $new_category);
            }
        }
        
        return array(
            'success' => true,
            'message' => 'Item move processed'
        );
    }
    
    /**
     * Extract price from notification data with multiple fallbacks
     * 
     * @param SimpleXMLElement $notification_data
     * @param string $item_id
     * @return float
     */
    private function extract_price_from_notification($notification_data, $item_id = null) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        // Try multiple price fields in order of preference (normalize strings like "CAD 12.50")
        $price_fields = array('best_bid', 'fixed_price', 'price', 'sale_price', 'item_price');

        foreach ($price_fields as $field) {
            if (isset($notification_data->$field)) {
                $raw = (string) $notification_data->$field;
                $num = preg_replace('/[^0-9.,]/', '', $raw);
                $num = str_replace(',', '.', $num);
                $price = (float) $num;
                if ($price > 0) {
                    $logger->write_log(array(
                        'timestamp' => current_time('Y-m-d H:i:s'),
                        'event' => 'PRICE_EXTRACTED',
                        'field' => $field,
                        'raw' => $raw,
                        'price' => $price,
                        'item_id' => $item_id
                    ));
                    return $price;
                }
            }
        }
        
        // If we have an item ID, try to fetch from API
        if ($item_id) {
            $auth = Delcampe_Auth::get_instance();
            $token = $auth->get_auth_token();
            
            if (!is_wp_error($token)) {
                $url = DELCAMPE_API_URL . '/item/' . $item_id . '?token=' . $token;
                $response = wp_remote_get($url, array(
                    'timeout' => 15,
                    'headers' => array('Accept' => 'application/xml')
                ));
                
                if (!is_wp_error($response)) {
                    $body = wp_remote_retrieve_body($response);
                    $item_xml = simplexml_load_string($body);
                    
                    if ($item_xml && isset($item_xml->item)) {
                        // Try to get price from API response
                        $api_price_fields = array('fixed_price', 'best_bid', 'price', 'current_price');
                        foreach ($api_price_fields as $field) {
                            if (isset($item_xml->item->$field)) {
                                $raw = (string) $item_xml->item->$field;
                                $num = preg_replace('/[^0-9.,]/', '', $raw);
                                $num = str_replace(',', '.', $num);
                                $price = (float) $num;
                                if ($price > 0) {
                                    $logger->write_log(array(
                                        'timestamp' => current_time('Y-m-d H:i:s'),
                                        'event' => 'PRICE_FROM_API',
                                        'field' => $field,
                                        'raw' => $raw,
                                        'price' => $price,
                                        'item_id' => $item_id
                                    ));
                                    return $price;
                                }
                            }
                        }
                    }
                }
            }
        }
        
        // Log failure to find price
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'PRICE_EXTRACTION_FAILED',
            'item_id' => $item_id,
            'notification_data' => substr(json_encode($notification_data), 0, 500)
        ));
        
        return 0;
    }
    
    /**
     * Log comprehensive raw webhook data for debugging
     * Captures ALL data received from Delcampe webhooks
     * 
     * @param string $notification_type Type of notification
     * @param SimpleXMLElement $xml Parsed XML
     * @param string $raw_xml Raw XML string
     * @return void
     * @since 1.10.36.0
     */
    private function log_raw_webhook_data($notification_type, $xml, $raw_xml) {
        $logger = Delcampe_Sync_Logger::get_instance();
        
        // Determine if this is an order-related webhook
        $order_types = array(
            'Seller_Item_Sold',
            'Seller_Item_Close_Sold',
            'Seller_Payment_Received'
        );
        
        $is_order_webhook = in_array($notification_type, $order_types);
        
        // Extract all possible data fields
        $extracted_data = array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'notification_type' => $notification_type,
            'is_order_related' => $is_order_webhook,
            'notification_datetime' => isset($xml->Notification_Datetime) ? (string)$xml->Notification_Datetime : null,
        );
        
        // Extract Notification_Data contents
        if (isset($xml->Notification_Data)) {
            $data = $xml->Notification_Data;
            
            // Convert SimpleXMLElement to array for comprehensive logging
            $data_array = json_decode(json_encode($data), true);
            $extracted_data['notification_data'] = $data_array;
            
            // Extract common fields
            $fields_to_extract = array(
                'id_item', 'id_bill', 'transaction_id',
                'buyer_id', 'buyer_nickname', 'buyer_email', 'buyer_firstname', 'buyer_surname',
                'buyer_address', 'buyer_city', 'buyer_zipcode', 'buyer_country', 'buyer_phone',
                'item_title', 'personal_reference', 'qty', 'quantity_sold',
                'best_bid', 'fixed_price', 'price', 'sale_price', 'item_price',
                'best_bid_currency', 'currency',
                'total_amount', 'seller_amount', 'platform_fee',
                'shipping_method', 'payment_method', 'buyer_comment',
                'url_to_item', 'url_to_rating',
                'sale_date', 'status'
            );
            
            foreach ($fields_to_extract as $field) {
                if (isset($data->$field)) {
                    $extracted_data[$field] = (string)$data->$field;
                }
            }
            
            // Extract item list if present (for payment notifications)
            if (isset($data->item_list) && isset($data->item_list->item)) {
                $extracted_data['items'] = array();
                foreach ($data->item_list->item as $item) {
                    $extracted_data['items'][] = json_decode(json_encode($item), true);
                }
            }
        }
        
        // Log to dedicated raw webhook log file
        $log_entry = array(
            'event' => 'WEBHOOK_RAW_DATA',
            'data' => $extracted_data,
            'raw_xml' => $raw_xml
        );
        
        // Log with standard logger
        $logger->write_log($log_entry);
        
        // Also log to dedicated webhook raw data file for easy access
        $this->write_webhook_raw_log($notification_type, $extracted_data, $raw_xml);
    }
    
    /**
     * Write to dedicated webhook raw data log file
     * 
     * @param string $notification_type Type of notification
     * @param array $extracted_data Extracted data fields
     * @param string $raw_xml Raw XML
     * @return void
     */
    private function write_webhook_raw_log($notification_type, $extracted_data, $raw_xml) {
        $log_dir = WP_CONTENT_DIR . '/delcampe-logs';
        if (!file_exists($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
        
        $log_file = $log_dir . '/webhook-raw-data.log';
        
        // Create readable log entry
        $log_lines = array(
            str_repeat('=', 80),
            'WEBHOOK RECEIVED: ' . date('Y-m-d H:i:s'),
            'Type: ' . $notification_type,
            str_repeat('-', 80),
            'EXTRACTED DATA:',
            json_encode($extracted_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
            str_repeat('-', 80),
            'RAW XML:',
            $raw_xml,
            str_repeat('=', 80),
            ''
        );
        
        $log_content = implode("\n", $log_lines);
        
        // Append to log file
        file_put_contents($log_file, $log_content, FILE_APPEND);
        
        // Keep log file manageable (max 10MB)
        if (file_exists($log_file) && filesize($log_file) > 10 * 1024 * 1024) {
            // Rotate log file
            $backup_file = $log_dir . '/webhook-raw-data-' . date('Y-m-d') . '.log';
            rename($log_file, $backup_file);
        }
    }
}
