<?php
/**
 * License Manager for WooCommerce Delcampe Integration
 *
 * Handles license validation, activation, and feature access control
 *
 * @package WC_Delcampe_Integration
 * @version 1.8.0.0
 * @since   1.8.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Delcampe_License_Manager class
 *
 * @since 1.8.0.0
 */
class Delcampe_License_Manager {
    
    /**
     * License server URL
     * @var string
     */
    const LICENSE_SERVER_URL = 'https://www.sos-tech.ca/license-api/';
    
    /**
     * Plugin slug for license server
     * @var string
     */
    const PLUGIN_SLUG = 'woocommerce-delcampe-integration';
    
    /**
     * Cache duration in seconds (24 hours)
     * @var int
     */
    const CACHE_DURATION = 86400;
    
    /**
     * Grace period in seconds (7 days)
     * @var int
     */
    const GRACE_PERIOD = 604800;
    
    /**
     * Singleton instance
     * @var Delcampe_License_Manager
     */
    private static $instance = null;
    
    /**
     * License data cache
     * @var object|null
     */
    private $license_data = null;
    
    /**
     * Encryption handler
     * @var Delcampe_Encryption|null
     * @since 1.10.22.0
     */
    private $encryption = null;
    
    /**
     * Feature definitions by license type
     * @var array
     */
    private $feature_matrix = array(
        'trial' => array(
            'listings' => 5,  // Limited to 5 listings in trial
            'profiles' => 2,
            'sync' => true,
            'orders' => true,
            'inventory' => true,
            'templates' => false,
            'bulk_operations' => false,
            'multi_account' => false,
            'automation' => false,
            'api_access' => false,
            'priority_support' => false,
            'import_listings' => false  // Cannot import from Delcampe in trial
        ),
        'basic' => array(
            'listings' => 100,
            'profiles' => 5,
            'sync' => true,
            'orders' => true,
            'inventory' => true,
            'templates' => false,
            'bulk_operations' => true,
            'multi_account' => false,
            'automation' => false,
            'api_access' => false,
            'priority_support' => false,
            'import_listings' => false,  // Cannot import from Delcampe
            'sites' => 1  // Single site license
        ),
        'standard' => array(
            'listings' => 1000,
            'profiles' => 20,
            'sync' => true,
            'orders' => true,
            'inventory' => true,
            'templates' => true,
            'bulk_operations' => true,
            'multi_account' => false,
            'automation' => true,
            'api_access' => false,
            'priority_support' => false,
            'import_listings' => false,  // Cannot import from Delcampe
            'sites' => 1  // Single site license
        ),
        'professional' => array(
            'listings' => 'unlimited',
            'profiles' => 'unlimited',
            'sync' => true,
            'orders' => true,
            'inventory' => true,
            'templates' => true,
            'bulk_operations' => true,
            'multi_account' => true,
            'automation' => true,
            'api_access' => true,
            'priority_support' => true,
            'import_listings' => true,  // Can import from Delcampe
            'sites' => 3  // 3 site licenses
        ),
        'enterprise' => array(
            'listings' => 'unlimited',
            'profiles' => 'unlimited',
            'sync' => true,
            'orders' => true,
            'inventory' => true,
            'templates' => true,
            'bulk_operations' => true,
            'multi_account' => true,
            'automation' => true,
            'api_access' => true,
            'priority_support' => true,
            'white_label' => true,
            'custom_features' => true,
            'import_listings' => true,  // Can import from Delcampe
            'sites' => 10  // 10 site licenses (limited from unlimited)
        )
    );
    
    /**
     * Get singleton instance
     *
     * @since 1.8.0.0
     * @return Delcampe_License_Manager
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     *
     * @since 1.8.0.0
     */
    private function __construct() {
        // Initialize encryption handler (v1.10.22.0)
        require_once plugin_dir_path(dirname(__FILE__)) . 'class-delcampe-encryption.php';
        $this->encryption = Delcampe_Encryption::get_instance();
        
        $this->init_hooks();
        $this->maybe_check_license();
    }
    
    /**
     * Initialize hooks
     *
     * @since 1.8.0.0
     * @version 1.10.5.10 - Defer admin hooks to avoid early translation loading
     */
    private function init_hooks() {
        // Defer admin-related hooks until after init to avoid early translation loading
        add_action('init', array($this, 'init_admin_hooks'));
        
        // Daily license check
        add_action('delcampe_daily_license_check', array($this, 'check_license_status'));
        
        // Schedule daily check if not already scheduled
        if (!wp_next_scheduled('delcampe_daily_license_check')) {
            wp_schedule_event(time(), 'daily', 'delcampe_daily_license_check');
        }
        
        // AJAX handlers
        add_action('wp_ajax_delcampe_activate_license', array($this, 'ajax_activate_license'));
        add_action('wp_ajax_delcampe_deactivate_license', array($this, 'ajax_deactivate_license'));
        add_action('wp_ajax_delcampe_check_license', array($this, 'ajax_check_license'));
        
        // License page is now a tab in Settings - removed in v1.10.5.5
        // add_action('admin_menu', array($this, 'add_license_menu'), 99);
    }
    
    /**
     * Initialize admin hooks after init
     *
     * @since 1.10.5.10
     */
    public function init_admin_hooks() {
        // Admin notices
        add_action('admin_notices', array($this, 'display_license_notices'));
    }
    
    /**
     * Add license menu item
     *
     * @since 1.8.0.0
     */
    public function add_license_menu() {
        add_submenu_page(
            'delcampe',  // Use the correct parent menu slug
            __('License', 'wc-delcampe-integration'),
            __('License', 'wc-delcampe-integration'),
            'manage_options',
            'delcampe-license',
            array($this, 'render_license_page')
        );
    }
    
    /**
     * Activate license
     *
     * @since 1.8.0.0
     * @param string $license_key License key to activate
     * @return array Result array with success status and message
     */
    public function activate_license($license_key) {
        global $wpdb;
        
        // Validate key format
        if (!$this->validate_key_format($license_key)) {
            return array(
                'success' => false,
                'message' => __('Invalid license key format. Please use format: XXXX-XXXX-XXXX-XXXX', 'wc-delcampe-integration')
            );
        }
        
        // Get instance ID
        $instance_id = $this->get_instance_id();
        
        // Make API request
        $response = $this->api_request('activate', array(
            'license_key' => $license_key,
            'instance_id' => $instance_id,
            'domain' => $this->get_domain(),
            'environment' => $this->get_environment_info()
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        // Process response
        if (isset($response['success']) && $response['success']) {
            // Store license data
            $table = $wpdb->prefix . 'delcampe_license';
            
            // Create table if it doesn't exist
            $this->create_license_table();
            
            // SECURITY ENHANCEMENT v1.10.22.0: Encrypt license key before storage
            $encrypted_key = $this->encryption->encrypt($license_key);
            
            $wpdb->replace($table, array(
                'license_key' => $encrypted_key,
                'license_status' => 'active',
                'license_type' => isset($response['license_type']) ? $response['license_type'] : 'basic',
                'activation_date' => current_time('mysql'),
                'expiration_date' => isset($response['expiration_date']) ? $response['expiration_date'] : null,
                'activation_limit' => isset($response['activation_limit']) ? $response['activation_limit'] : 1,
                'activation_count' => isset($response['activation_count']) ? $response['activation_count'] : 1,
                'instance_id' => $instance_id,
                'last_check' => current_time('mysql'),
                'cached_response' => json_encode($response)
            ));
            
            // Clear cache
            delete_transient('delcampe_license_data');
            $this->license_data = null;
            
            // Clear trial status when real license is activated
            delete_option('delcampe_trial_start_date');
            delete_option('delcampe_trial_active');
            
            // Log activation
            $this->log_activation($license_key, 'activated');
            
            return array(
                'success' => true,
                'message' => __('License activated successfully!', 'wc-delcampe-integration'),
                'data' => $response
            );
        }
        
        return array(
            'success' => false,
            'message' => isset($response['message']) ? $response['message'] : __('Failed to activate license. Please try again.', 'wc-delcampe-integration')
        );
    }
    
    /**
     * Deactivate license
     *
     * @since 1.8.0.0
     * @param string|null $license_key License key to deactivate (null for current)
     * @return array Result array
     */
    public function deactivate_license($license_key = null) {
        global $wpdb;
        
        // Get current license if not provided
        if (!$license_key) {
            $license = $this->get_license_data();
            if (!$license) {
                return array(
                    'success' => false,
                    'message' => __('No active license found.', 'wc-delcampe-integration')
                );
            }
            // SECURITY ENHANCEMENT v1.10.22.0: Decrypt license key for API use
            $license_key = $this->encryption->decrypt($license->license_key);
        }
        
        // Make API request (uses plain text key for API)
        $response = $this->api_request('deactivate', array(
            'license_key' => $license_key,
            'instance_id' => $this->get_instance_id()
        ));
        
        // Remove from database regardless of API response
        // SECURITY ENHANCEMENT v1.10.22.0: Use encrypted key for database operations
        $table = $wpdb->prefix . 'delcampe_license';
        $encrypted_key = $this->encryption->is_encrypted($license_key) ? $license_key : $this->encryption->encrypt($license_key);
        $wpdb->delete($table, array('license_key' => $encrypted_key));
        
        // Clear cache
        delete_transient('delcampe_license_data');
        $this->license_data = null;
        
        // Clear scheduled events
        wp_clear_scheduled_hook('delcampe_daily_license_check');
        
        // Log deactivation
        $this->log_activation($license_key, 'deactivated');
        
        return array(
            'success' => true,
            'message' => __('License deactivated successfully.', 'wc-delcampe-integration')
        );
    }
    
    /**
     * Check license status
     *
     * @since 1.8.0.0
     * @return bool True if valid, false otherwise
     */
    public function check_license_status() {
        $license = $this->get_license_data();
        
        if (!$license || $license->license_status !== 'active') {
            return false;
        }
        
        // Check if we need to revalidate
        if ($this->should_revalidate($license)) {
            // SECURITY ENHANCEMENT v1.10.22.0: Decrypt license key for API validation
            $decrypted_key = $this->encryption->decrypt($license->license_key);
            
            $response = $this->api_request('check', array(
                'license_key' => $decrypted_key,
                'instance_id' => $license->instance_id
            ));
            
            if (!is_wp_error($response)) {
                // Keep using encrypted key for cache update
                $this->update_license_cache($license->license_key, $response);
            } else {
                // Check grace period
                if ($this->is_in_grace_period($license)) {
                    return true;
                }
            }
        }
        
        return $this->is_license_valid();
    }
    
    /**
     * Check if trial is active
     * 
     * @since 1.10.8.0
     * @return bool
     */
    public function is_trial_active() {
        $trial_start = get_option('delcampe_trial_start_date', '');
        
        if (empty($trial_start)) {
            // No trial started yet - auto-start it
            $this->start_trial();
            return true;
        }
        
        $trial_end = strtotime($trial_start . ' +14 days');
        $now = current_time('timestamp');
        
        return $now <= $trial_end;
    }
    
    /**
     * Start trial if not already started
     * 
     * @since 1.10.8.0
     * @param array $user_data Optional user information (added v1.10.25.0)
     * @return bool
     */
    public function start_trial($user_data = array()) {
        $trial_start = get_option('delcampe_trial_start_date', '');
        
        if (!empty($trial_start)) {
            // Trial already started
            return false;
        }
        
        // Capture user information if provided (v1.10.25.0)
        if (!empty($user_data)) {
            $trial_user_info = array(
                'email' => isset($user_data['email']) ? sanitize_email($user_data['email']) : '',
                'name' => isset($user_data['name']) ? sanitize_text_field($user_data['name']) : '',
                'company' => isset($user_data['company']) ? sanitize_text_field($user_data['company']) : '',
                'website' => home_url(),
                'timestamp' => current_time('Y-m-d H:i:s'),
                'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
            );
            
            update_option('delcampe_trial_user_info', $trial_user_info);
            
            // Notify license server for tracking
            $this->notify_license_server_trial($trial_user_info);
        }
        
        update_option('delcampe_trial_start_date', current_time('Y-m-d H:i:s'));
        update_option('delcampe_trial_active', true);
        
        return true;
    }
    
    /**
     * Notify license server about trial activation
     * 
     * @since 1.10.25.0
     * @param array $user_info User information
     */
    private function notify_license_server_trial($user_info) {
        $api_url = 'https://license.sos-tech.ca/api/trial-activation';
        
        $response = wp_remote_post($api_url, array(
            'body' => array(
                'plugin' => 'delcampe-sync',
                'site_url' => home_url(),
                'user_info' => json_encode($user_info)
            ),
            'timeout' => 10
        ));
        
        // Log response for debugging
        if (is_wp_error($response)) {
            delcampe_log('[License] Trial notification failed: ' . $response->get_error_message());
        }
    }
    
    /**
     * Get trial days remaining
     * 
     * @since 1.10.8.0
     * @return int Days remaining or 0 if expired
     */
    public function get_trial_days_remaining() {
        $trial_start = get_option('delcampe_trial_start_date', '');
        
        if (empty($trial_start)) {
            return 14; // Full trial period available
        }
        
        $trial_end = strtotime($trial_start . ' +14 days');
        $now = current_time('timestamp');
        
        if ($now > $trial_end) {
            return 0;
        }
        
        return ceil(($trial_end - $now) / 86400);
    }
    
    /**
     * Get trial listings count
     * 
     * @since 1.10.8.0
     * @return array
     */
    public function get_trial_listings_count() {
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_listings';
        
        // SECURITY FIX v1.10.21.0: Properly escape table name to prevent SQL injection
        // Note: Table name is constructed from wpdb->prefix which is safe, but adding escaping for defense in depth
        $safe_table = esc_sql($table);
        $total = $wpdb->get_var("SELECT COUNT(*) FROM `$safe_table` WHERE status IN ('published', 'prepared', 'changed')");
        $limit = 5;
        
        return array(
            'used' => intval($total),
            'limit' => $limit,
            'remaining' => max(0, $limit - intval($total))
        );
    }

    /**
     * Check if license is valid
     *
     * @since 1.8.0.0
     * @version 1.10.8.0 - Added trial support
     * @return bool
     */
    public function is_license_valid() {
        // Check if we have a valid license first
        $license = $this->get_license_data();
        
        if ($license) {
            // Check status
            if ($license->license_status === 'active') {
                // Check expiration
                if (!$license->expiration_date || strtotime($license->expiration_date) >= time()) {
                    return true;
                }
            }
        }
        
        // If no valid license, check if trial is active
        if ($this->is_trial_active()) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Get license type
     *
     * @since 1.8.0.0
     * @version 1.10.8.0 - Added trial support
     * @return string License type or 'none'
     */
    public function get_license_type() {
        // Check for real license first
        $license = $this->get_license_data();
        if ($license && $license->license_status === 'active') {
            return $license->license_type;
        }
        
        // If no valid license, check if trial is active
        if ($this->is_trial_active()) {
            return 'trial';
        }
        
        return 'none';
    }
    
    /**
     * Check feature access
     *
     * @since 1.8.0.0
     * @version 1.10.8.0 - Added trial support with listing limits
     * @param string $feature Feature to check
     * @return bool|mixed Feature access or limit
     */
    public function has_feature_access($feature) {
        // SECURITY FIX v1.10.21.0: Removed debug bypass vulnerability
        // Debug mode should never bypass license checks in production
        // For development testing, use a proper development license key
        
        if (!$this->is_license_valid()) {
            return false;
        }
        
        $license_type = $this->get_license_type();
        
        // Special handling for trial listings limit
        if ($license_type === 'trial' && $feature === 'listings') {
            // Check current listing count
            $trial_count = $this->get_trial_listings_count();
            if ($trial_count['used'] >= $trial_count['limit']) {
                return false; // Limit reached
            }
            return $trial_count['limit'];
        }
        
        // Check if feature exists for license type
        if (isset($this->feature_matrix[$license_type][$feature])) {
            return $this->feature_matrix[$license_type][$feature];
        }
        
        return false;
    }
    
    /**
     * Get feature limit
     *
     * @since 1.8.0.0
     * @param string $feature Feature name
     * @return mixed Feature limit or false
     */
    public function get_feature_limit($feature) {
        return $this->has_feature_access($feature);
    }
    
    /**
     * Make API request to license server
     *
     * @since 1.8.0.0
     * @param string $action API action
     * @param array $data Request data
     * @return array|WP_Error Response array or error
     */
    private function api_request($action, $data) {
        $url = self::LICENSE_SERVER_URL . $action;
        
        // Add plugin information
        $data['plugin_slug'] = self::PLUGIN_SLUG;
        $data['plugin_version'] = DELCAMPE_PLUGIN_VERSION;
        
        $args = array(
            'timeout' => 30,
            'redirection' => 5,
            'httpversion' => '1.1',
            'blocking' => true,
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'User-Agent' => 'WooCommerce Delcampe Integration/' . DELCAMPE_PLUGIN_VERSION . '; ' . get_bloginfo('url')
            ),
            'body' => $data,
            'cookies' => array(),
            'sslverify' => true
        );
        
        // Make request
        $response = wp_remote_post($url, $args);
        
        // Check for errors
        if (is_wp_error($response)) {
            return $response;
        }
        
        // Check response code
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            return new WP_Error('api_error', sprintf(__('License server returned error code: %d', 'wc-delcampe-integration'), $response_code));
        }
        
        // Parse response
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data) {
            return new WP_Error('invalid_response', __('Invalid response from license server.', 'wc-delcampe-integration'));
        }
        
        return $data;
    }
    
    /**
     * Get license data from database
     *
     * @since 1.8.0.0
     * @return object|null License data object or null
     */
    public function get_license_data() {
        if (!is_null($this->license_data)) {
            return $this->license_data;
        }
        
        // Check cache first
        $cached = get_transient('delcampe_license_data');
        if ($cached !== false) {
            $this->license_data = $cached;
            return $cached;
        }
        
        // Get from database
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_license';
        
        // SECURITY FIX v1.10.21.0: Use prepared statements for all database queries
        // Check if table exists
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)) !== $table) {
            return null;
        }
        
        // Safe query - table name is from wpdb->prefix, but we'll escape for consistency
        $safe_table = esc_sql($table);
        $license = $wpdb->get_row("SELECT * FROM `$safe_table` WHERE license_status = 'active' ORDER BY id DESC LIMIT 1");
        
        if ($license) {
            set_transient('delcampe_license_data', $license, self::CACHE_DURATION);
            $this->license_data = $license;
        }
        
        return $license;
    }
    
    /**
     * Update license cache
     *
     * @since 1.8.0.0
     * @param string $license_key License key
     * @param array $data Response data
     */
    private function update_license_cache($license_key, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_license';
        
        $update_data = array(
            'license_status' => isset($data['status']) ? $data['status'] : 'inactive',
            'last_check' => current_time('mysql'),
            'cached_response' => json_encode($data)
        );
        
        // Update additional fields if provided
        if (isset($data['license_type'])) {
            $update_data['license_type'] = $data['license_type'];
        }
        if (isset($data['expiration_date'])) {
            $update_data['expiration_date'] = $data['expiration_date'];
        }
        if (isset($data['activation_count'])) {
            $update_data['activation_count'] = $data['activation_count'];
        }
        
        $wpdb->update($table, $update_data, array('license_key' => $license_key));
        
        // Clear cache
        delete_transient('delcampe_license_data');
        $this->license_data = null;
    }
    
    /**
     * Should revalidate license
     *
     * @since 1.8.0.0
     * @param object $license License object
     * @return bool
     */
    private function should_revalidate($license) {
        if (!$license->last_check) {
            return true;
        }
        
        $last_check = strtotime($license->last_check);
        $now = time();
        
        // Check every 24 hours
        return ($now - $last_check) > self::CACHE_DURATION;
    }
    
    /**
     * Check if in grace period
     *
     * @since 1.8.0.0
     * @param object $license License object
     * @return bool
     */
    private function is_in_grace_period($license) {
        if (!$license->last_check) {
            return false;
        }
        
        $last_check = strtotime($license->last_check);
        $now = time();
        
        // Allow 7 days grace period
        return ($now - $last_check) < self::GRACE_PERIOD;
    }
    
    /**
     * Maybe check license on admin init
     *
     * @since 1.8.0.0
     */
    private function maybe_check_license() {
        // Only check on admin pages
        if (!is_admin()) {
            return;
        }
        
        // Don't check on AJAX requests
        if (defined('DOING_AJAX') && DOING_AJAX) {
            return;
        }
        
        // Check if we have a license
        $license = $this->get_license_data();
        if ($license) {
            $this->check_license_status();
        }
    }
    
    /**
     * Get instance ID
     *
     * @since 1.8.0.0
     * @return string Unique instance ID
     */
    public function get_instance_id() {
        $instance_id = get_option('delcampe_instance_id');
        
        if (!$instance_id) {
            $instance_id = wp_generate_password(32, false);
            update_option('delcampe_instance_id', $instance_id);
        }
        
        return $instance_id;
    }
    
    /**
     * Get domain
     *
     * @since 1.8.0.0
     * @return string Current domain
     */
    private function get_domain() {
        $url = get_site_url();
        $parsed = parse_url($url);
        return isset($parsed['host']) ? $parsed['host'] : '';
    }
    
    /**
     * Get environment info
     *
     * @since 1.8.0.0
     * @return array Environment information
     */
    private function get_environment_info() {
        global $wp_version;
        
        return array(
            'php_version' => PHP_VERSION,
            'wp_version' => $wp_version,
            'wc_version' => defined('WC_VERSION') ? WC_VERSION : 'N/A',
            'multisite' => is_multisite(),
            'locale' => get_locale(),
            'timezone' => get_option('timezone_string'),
            'memory_limit' => WP_MEMORY_LIMIT,
            'debug_mode' => WP_DEBUG
        );
    }
    
    /**
     * Validate key format
     *
     * @since 1.8.0.0
     * @param string $key License key
     * @return bool
     */
    private function validate_key_format($key) {
        // Format: XXXX-XXXX-XXXX-XXXX (alphanumeric, uppercase)
        return preg_match('/^[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $key);
    }
    
    /**
     * Log activation event
     *
     * @since 1.8.0.0
     * @param string $license_key License key
     * @param string $action Action performed
     */
    private function log_activation($license_key, $action) {
        if (!defined('DELCAMPE_DEBUG') || !DELCAMPE_DEBUG) {
            return;
        }
        
        $log_entry = sprintf(
            '[%s] License %s: %s (Domain: %s, IP: %s)',
            current_time('Y-m-d H:i:s'),
            $action,
            substr($license_key, 0, 8) . '...',
            $this->get_domain(),
            isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : 'Unknown'
        );
        
        error_log($log_entry);
    }
    
    /**
     * Create license table
     *
     * @since 1.8.0.0
     */
    private function create_license_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_license';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
            id int(11) NOT NULL AUTO_INCREMENT,
            license_key varchar(255) NOT NULL,
            license_status varchar(50) DEFAULT 'inactive',
            license_type varchar(50) DEFAULT 'basic',
            activation_date datetime DEFAULT NULL,
            expiration_date datetime DEFAULT NULL,
            activation_limit int(11) DEFAULT 1,
            activation_count int(11) DEFAULT 0,
            instance_id varchar(255) DEFAULT NULL,
            last_check datetime DEFAULT NULL,
            cached_response longtext,
            PRIMARY KEY (id),
            UNIQUE KEY license_key (license_key)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Migrate existing plain text license keys to encrypted format
     * 
     * @since 1.10.22.0
     * @return array Migration results
     */
    public function migrate_license_keys_to_encrypted() {
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_license';
        
        $results = array(
            'total' => 0,
            'migrated' => 0,
            'already_encrypted' => 0,
            'failed' => 0
        );
        
        // Check if table exists
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)) !== $table) {
            return $results;
        }
        
        // Get all license records
        $licenses = $wpdb->get_results("SELECT * FROM `$table`");
        $results['total'] = count($licenses);
        
        foreach ($licenses as $license) {
            // Check if already encrypted
            if ($this->encryption->is_encrypted($license->license_key)) {
                $results['already_encrypted']++;
                continue;
            }
            
            // Encrypt the license key
            $encrypted_key = $this->encryption->encrypt($license->license_key);
            
            if ($encrypted_key !== false) {
                // Update the record with encrypted key
                $updated = $wpdb->update(
                    $table,
                    array('license_key' => $encrypted_key),
                    array('id' => $license->id)
                );
                
                if ($updated !== false) {
                    $results['migrated']++;
                    error_log('[License Migration] Successfully encrypted license ID: ' . $license->id);
                } else {
                    $results['failed']++;
                    error_log('[License Migration] Failed to update license ID: ' . $license->id);
                }
            } else {
                $results['failed']++;
                error_log('[License Migration] Failed to encrypt license ID: ' . $license->id);
            }
        }
        
        // Clear cache after migration
        delete_transient('delcampe_license_data');
        $this->license_data = null;
        
        error_log('[License Migration] Complete - Total: ' . $results['total'] . 
                  ', Migrated: ' . $results['migrated'] . 
                  ', Already Encrypted: ' . $results['already_encrypted'] . 
                  ', Failed: ' . $results['failed']);
        
        return $results;
    }
    
    /**
     * Display license notices
     *
     * @since 1.8.0.0
     * @version 1.10.8.0 - Added trial notices
     */
    public function display_license_notices() {
        // Only show on plugin pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'delcampe') === false) {
            return;
        }
        
        // Don't show on license page
        if ($screen->id === 'delcampe_page_delcampe-license') {
            return;
        }
        
        // Check license type to determine what notice to show
        $license_type = $this->get_license_type();
        
        if ($license_type === 'trial') {
            $days_remaining = $this->get_trial_days_remaining();
            $trial_count = $this->get_trial_listings_count();
            
            if ($days_remaining <= 3) {
                $message = sprintf(
                    __('Your free trial expires in %d days! You have used %d of %d listings. <a href="%s">Upgrade now</a> to continue using all features.', 'wc-delcampe-integration'),
                    $days_remaining,
                    $trial_count['used'],
                    $trial_count['limit'],
                    admin_url('admin.php?page=delcampe-settings&tab=license')
                );
                echo '<div class="notice notice-warning"><p>' . $message . '</p></div>';
            } else {
                $message = sprintf(
                    __('Free trial active: %d days remaining, %d of %d listings used. <a href="%s">View license options</a>', 'wc-delcampe-integration'),
                    $days_remaining,
                    $trial_count['used'],
                    $trial_count['limit'],
                    admin_url('admin.php?page=delcampe-settings&tab=license')
                );
                echo '<div class="notice notice-info"><p>' . $message . '</p></div>';
            }
            return;
        }
        
        if (!$this->is_license_valid()) {
            $message = sprintf(
                __('Your trial has expired. <a href="%s">Purchase a license</a> to continue using WooCommerce Delcampe Integration.', 'wc-delcampe-integration'),
                admin_url('admin.php?page=delcampe-settings&tab=license')
            );
            
            echo '<div class="notice notice-error"><p>' . $message . '</p></div>';
        } else {
            // Check if license is expiring soon
            $license = $this->get_license_data();
            if ($license && $license->expiration_date) {
                $expiration = strtotime($license->expiration_date);
                $days_until_expiration = floor(($expiration - time()) / 86400);
                
                if ($days_until_expiration <= 30 && $days_until_expiration > 0) {
                    $message = sprintf(
                        __('Your WooCommerce Delcampe Integration license expires in %d days. <a href="%s">Renew your license</a>.', 'wc-delcampe-integration'),
                        $days_until_expiration,
                        admin_url('admin.php?page=delcampe-settings&tab=license')
                    );
                    
                    echo '<div class="notice notice-warning"><p>' . $message . '</p></div>';
                }
            }
        }
    }
    
    /**
     * Render license page
     *
     * @since 1.8.0.0
     */
    public function render_license_page() {
        $license = $this->get_license_data();
        $is_valid = $this->is_license_valid();
        
        include DELCAMPE_PATH . 'includes/license/views/license-page-fixed.php';
    }
    
    /**
     * AJAX activate license handler
     *
     * @since 1.8.0.0
     */
    public function ajax_activate_license() {
        check_ajax_referer('delcampe_license_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Unauthorized', 'wc-delcampe-integration'));
        }
        
        $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
        
        $result = $this->activate_license($license_key);
        
        wp_send_json($result);
    }
    
    /**
     * AJAX deactivate license handler
     *
     * @since 1.8.0.0
     */
    public function ajax_deactivate_license() {
        check_ajax_referer('delcampe_license_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Unauthorized', 'wc-delcampe-integration'));
        }
        
        $result = $this->deactivate_license();
        
        wp_send_json($result);
    }
    
    /**
     * AJAX check license handler
     *
     * @since 1.8.0.0
     */
    public function ajax_check_license() {
        check_ajax_referer('delcampe_license_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Unauthorized', 'wc-delcampe-integration'));
        }
        
        $valid = $this->check_license_status();
        $license = $this->get_license_data();
        
        wp_send_json(array(
            'valid' => $valid,
            'license' => $license
        ));
    }
}
