<?php
/**
 * License Helper Functions
 *
 * Global functions for checking license status and feature access
 *
 * @package WC_Delcampe_Integration
 * @version 1.8.0.0
 * @since   1.8.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Check if license is valid
 *
 * @since 1.8.0.0
 * @return bool True if valid license exists
 */
function delcampe_has_valid_license() {
    $license_manager = Delcampe_License_Manager::get_instance();
    return $license_manager->is_license_valid();
}

/**
 * Check feature access
 *
 * @since 1.8.0.0
 * @param string $feature Feature to check
 * @return bool|mixed True/false for access, or numeric/string limit
 */
function delcampe_has_feature($feature) {
    $license_manager = Delcampe_License_Manager::get_instance();
    return $license_manager->has_feature_access($feature);
}

/**
 * Get feature limit
 *
 * @since 1.8.0.0
 * @param string $feature Feature name
 * @return mixed Feature limit (numeric, 'unlimited', or false)
 */
function delcampe_get_feature_limit($feature) {
    $license_manager = Delcampe_License_Manager::get_instance();
    return $license_manager->get_feature_limit($feature);
}

/**
 * Get license type
 *
 * @since 1.8.0.0
 * @return string License type (trial, basic, standard, professional, enterprise, or none)
 */
function delcampe_get_license_type() {
    $license_manager = Delcampe_License_Manager::get_instance();
    return $license_manager->get_license_type();
}

/**
 * Check if can create new listing
 *
 * @since 1.8.0.0
 * @return bool True if under limit or unlimited
 */
function delcampe_can_create_listing() {
    $limit = delcampe_get_feature_limit('listings');
    
    if ($limit === 'unlimited') {
        return true;
    }
    
    if (!is_numeric($limit)) {
        return false;
    }
    
    // Count current active listings
    global $wpdb;
    $table = $wpdb->prefix . 'delcampe_listings';
    $count = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM {$table} WHERE status IN (%s, %s, %s, %s)",
        'prepared', 'verified', 'published', 'changed'
    ));
    
    return $count < $limit;
}

/**
 * Check if can create new profile
 *
 * @since 1.8.0.0
 * @return bool True if under limit or unlimited
 */
function delcampe_can_create_profile() {
    $limit = delcampe_get_feature_limit('profiles');
    
    if ($limit === 'unlimited') {
        return true;
    }
    
    if (!is_numeric($limit)) {
        return false;
    }
    
    // Count current profiles
    global $wpdb;
    $table = $wpdb->prefix . 'delcampe_profiles';
    $count = $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
    
    return $count < $limit;
}

/**
 * Get remaining listings count
 *
 * @since 1.8.0.0
 * @return int|string Remaining count or 'unlimited'
 */
function delcampe_get_remaining_listings() {
    $limit = delcampe_get_feature_limit('listings');
    
    if ($limit === 'unlimited') {
        return 'unlimited';
    }
    
    if (!is_numeric($limit)) {
        return 0;
    }
    
    // Count current active listings
    global $wpdb;
    $table = $wpdb->prefix . 'delcampe_listings';
    $count = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM {$table} WHERE status IN (%s, %s, %s, %s)",
        'prepared', 'verified', 'published', 'changed'
    ));
    
    return max(0, $limit - $count);
}

/**
 * Get remaining profiles count
 *
 * @since 1.8.0.0
 * @return int|string Remaining count or 'unlimited'
 */
function delcampe_get_remaining_profiles() {
    $limit = delcampe_get_feature_limit('profiles');
    
    if ($limit === 'unlimited') {
        return 'unlimited';
    }
    
    if (!is_numeric($limit)) {
        return 0;
    }
    
    // Count current profiles
    global $wpdb;
    $table = $wpdb->prefix . 'delcampe_profiles';
    $count = $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
    
    return max(0, $limit - $count);
}

/**
 * Display feature limit notice
 *
 * @since 1.8.0.0
 * @param string $feature Feature name
 * @param string $action Action being attempted
 */
function delcampe_display_limit_notice($feature, $action = '') {
    $limit = delcampe_get_feature_limit($feature);
    
    if ($limit === 'unlimited') {
        return;
    }
    
    $feature_labels = array(
        'listings' => __('listings', 'wc-delcampe-integration'),
        'profiles' => __('profiles', 'wc-delcampe-integration'),
        'templates' => __('templates', 'wc-delcampe-integration'),
        'multi_account' => __('accounts', 'wc-delcampe-integration')
    );
    
    $feature_label = isset($feature_labels[$feature]) ? $feature_labels[$feature] : $feature;
    
    if (is_numeric($limit)) {
        $message = sprintf(
            __('You have reached the limit of %d %s for your license. Please upgrade to %s more %s.', 'wc-delcampe-integration'),
            $limit,
            $feature_label,
            $action,
            $feature_label
        );
    } else {
        $message = sprintf(
            __('Your license does not include access to %s. Please upgrade to %s.', 'wc-delcampe-integration'),
            $feature_label,
            $action
        );
    }
    
    echo '<div class="notice notice-warning"><p>' . $message . ' ';
    echo '<a href="' . admin_url('admin.php?page=delcampe-license') . '">' . __('Upgrade License', 'wc-delcampe-integration') . '</a>';
    echo '</p></div>';
}

/**
 * Check if feature is available
 *
 * @since 1.8.0.0
 * @param string $feature Feature to check
 * @param bool $die Whether to die with error message
 * @return bool True if available
 */
function delcampe_require_feature($feature, $die = true) {
    if (!delcampe_has_valid_license()) {
        if ($die) {
            wp_die(
                __('A valid license is required to access this feature.', 'wc-delcampe-integration'),
                __('License Required', 'wc-delcampe-integration'),
                array('back_link' => true)
            );
        }
        return false;
    }
    
    if (!delcampe_has_feature($feature)) {
        if ($die) {
            $license_type = delcampe_get_license_type();
            $required_licenses = array(
                'templates' => __('Standard', 'wc-delcampe-integration'),
                'multi_account' => __('Professional', 'wc-delcampe-integration'),
                'automation' => __('Standard', 'wc-delcampe-integration'),
                'api_access' => __('Professional', 'wc-delcampe-integration'),
                'white_label' => __('Enterprise', 'wc-delcampe-integration')
            );
            
            $required = isset($required_licenses[$feature]) ? $required_licenses[$feature] : __('higher', 'wc-delcampe-integration');
            
            wp_die(
                sprintf(
                    __('This feature requires a %s license or higher. Your current license is: %s', 'wc-delcampe-integration'),
                    $required,
                    ucfirst($license_type)
                ),
                __('Upgrade Required', 'wc-delcampe-integration'),
                array('back_link' => true)
            );
        }
        return false;
    }
    
    return true;
}

/**
 * Add license status to admin bar
 *
 * @since 1.8.0.0
 * @param WP_Admin_Bar $wp_admin_bar Admin bar object
 */
function delcampe_add_license_admin_bar($wp_admin_bar) {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    $license_valid = delcampe_has_valid_license();
    $license_type = delcampe_get_license_type();
    
    $title = $license_valid 
        ? sprintf(__('Delcampe: %s', 'wc-delcampe-integration'), ucfirst($license_type))
        : __('Delcampe: No License', 'wc-delcampe-integration');
    
    $wp_admin_bar->add_node(array(
        'id' => 'delcampe-license',
        'title' => '<span class="ab-icon dashicons dashicons-admin-network"></span>' . $title,
        'href' => admin_url('admin.php?page=delcampe-license'),
        'meta' => array(
            'class' => $license_valid ? 'delcampe-license-valid' : 'delcampe-license-invalid'
        )
    ));
}
// Hook registration moved to delcampe-license-hooks.php to avoid early translation loading

/**
 * Add license status body class
 *
 * @since 1.8.0.0
 * @param string $classes Admin body classes
 * @return string Modified classes
 */
function delcampe_add_license_body_class($classes) {
    $license_valid = delcampe_has_valid_license();
    $license_type = delcampe_get_license_type();
    
    $classes .= $license_valid ? ' delcampe-licensed ' : ' delcampe-unlicensed ';
    $classes .= ' delcampe-license-' . $license_type . ' ';
    
    return $classes;
}
// Hook registration moved to delcampe-license-hooks.php to avoid early translation loading

/**
 * Filter capabilities based on license
 *
 * @since 1.8.0.0
 * @param array $allcaps All capabilities
 * @param array $caps Required capabilities
 * @param array $args Arguments
 * @return array Modified capabilities
 */
function delcampe_filter_capabilities($allcaps, $caps, $args) {
    // Check if trying to access premium features
    if (isset($caps[0]) && strpos($caps[0], 'delcampe_use_') === 0) {
        $feature = str_replace('delcampe_use_', '', $caps[0]);
        
        if (!delcampe_has_feature($feature)) {
            $allcaps[$caps[0]] = false;
        }
    }
    
    return $allcaps;
}
// Hook registration moved to delcampe-license-hooks.php to avoid early translation loading

/**
 * Redirect to license page if no valid license
 *
 * @since 1.8.0.0
 */
function delcampe_license_redirect() {
    if (!is_admin()) {
        return;
    }
    
    // Don't redirect on AJAX, cron, or REST requests
    if (wp_doing_ajax() || wp_doing_cron() || (defined('REST_REQUEST') && REST_REQUEST)) {
        return;
    }
    
    // Get current screen
    $screen = get_current_screen();
    if (!$screen) {
        return;
    }
    
    // Check if on plugin page
    if (strpos($screen->id, 'delcampe') === false) {
        return;
    }
    
    // Don't redirect on license page or any license-related pages
    $license_pages = array(
        'delcampe_page_delcampe-license',
        'toplevel_page_delcampe',
        'admin_page_delcampe-license'
    );
    
    if (in_array($screen->id, $license_pages)) {
        return;
    }
    
    // Also check if we're already on the license page by checking the page parameter
    if (isset($_GET['page']) && $_GET['page'] === 'delcampe-license') {
        return;
    }
    
    // Allow access to main Delcampe page without license to prevent loops
    if ($screen->id === 'toplevel_page_delcampe' || (isset($_GET['page']) && $_GET['page'] === 'delcampe')) {
        return;
    }
    
    // Check license
    if (!delcampe_has_valid_license()) {
        wp_redirect(admin_url('admin.php?page=delcampe-license&notice=required'));
        exit;
    }
}
// Hook registration moved to delcampe-license-hooks.php to avoid early translation loading

/**
 * Add license CSS to admin
 *
 * @since 1.8.0.0
 */
function delcampe_license_admin_css() {
    ?>
    <style>
        #wp-admin-bar-delcampe-license .ab-icon:before {
            content: "\f112";
            top: 2px;
        }
        
        #wp-admin-bar-delcampe-license.delcampe-license-invalid > a {
            background: #dc3232 !important;
            color: #fff !important;
        }
        
        #wp-admin-bar-delcampe-license.delcampe-license-valid > a {
            background: #46b450 !important;
            color: #fff !important;
        }
        
        .delcampe-feature-disabled {
            opacity: 0.5;
            pointer-events: none;
            position: relative;
        }
        
        .delcampe-feature-disabled:after {
            content: "Upgrade Required";
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0,0,0,0.8);
            color: #fff;
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 12px;
            white-space: nowrap;
            z-index: 9999;
        }
    </style>
    <?php
}
// Hook registration moved to delcampe-license-hooks.php to avoid early translation loading

// Include the deferred hooks file
require_once dirname(__FILE__) . '/delcampe-license-hooks.php';
