<?php
/**
 * Delcampe Description Builder
 * Version: 1.0.1.0
 *
 * Generates formatted descriptions for Delcampe listings based on product data,
 * profile settings, and stamp-specific attributes.
 *
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Profiles
 * @since 1.0.0.0
 * @version 1.0.1.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Delcampe_Description_Builder
 * 
 * Builds comprehensive descriptions for Delcampe listings
 * 
 * @version 1.0.1.0
 */
class Delcampe_Description_Builder {

    /**
     * Product data
     * @var WC_Product
     */
    private $product;

    /**
     * Profile settings
     * @var array
     */
    private $profile;

    /**
     * Product attributes
     * @var array
     */
    private $attributes;

    /**
     * Constructor
     * 
     * @param WC_Product $product WooCommerce product
     * @param array      $profile Profile settings
     */
    public function __construct( $product, $profile ) {
        $this->product = $product;
        $this->profile = $profile;
        $this->attributes = $this->get_product_attributes();
    }

    /**
     * Build the complete description
     * 
     * @return string Formatted description for Delcampe
     */
    public function build() {
        $template_type = $this->profile['description_settings']['template_type'] ?? 'stamp_standard';
        
        // Build description based on template type
        switch ( $template_type ) {
            case 'stamp_detailed':
                return $this->build_detailed_stamp_description();
            case 'stamp_minimal':
                return $this->build_minimal_stamp_description();
            case 'custom':
                return $this->build_custom_description();
            default:
                return $this->build_standard_stamp_description();
        }
    }

    /**
     * Build standard stamp description
     * 
     * @return string
     */
    private function build_standard_stamp_description() {
        $sections = array();
        
        // Add custom header if set
        if ( ! empty( $this->profile['description_settings']['custom_header'] ) ) {
            $sections[] = $this->profile['description_settings']['custom_header'];
        }
        
        // Main description
        $sections[] = $this->get_main_description();
        
        // Technical details
        if ( $this->should_include( 'include_technical' ) ) {
            $technical = $this->get_technical_details();
            if ( ! empty( $technical ) ) {
                $sections[] = $this->format_section( __( 'Technical Details', 'wc-delcampe-integration' ), $technical );
            }
        }
        
        // Catalog information
        if ( $this->should_include( 'include_catalog_info' ) ) {
            $catalog = $this->get_catalog_information();
            if ( ! empty( $catalog ) ) {
                $sections[] = $this->format_section( __( 'Catalog Information', 'wc-delcampe-integration' ), $catalog );
            }
        }
        
        // Condition details
        if ( $this->should_include( 'include_condition_details' ) ) {
            $condition = $this->get_condition_details();
            if ( ! empty( $condition ) ) {
                $sections[] = $this->format_section( __( 'Condition Details', 'wc-delcampe-integration' ), $condition );
            }
        }
        
        // Shipping information
        if ( $this->should_include( 'include_shipping_info' ) ) {
            $shipping = $this->get_shipping_information();
            if ( ! empty( $shipping ) ) {
                $sections[] = $this->format_section( __( 'Shipping Information', 'wc-delcampe-integration' ), $shipping );
            }
        }
        
        // Payment terms
        if ( $this->should_include( 'include_payment_terms' ) ) {
            $payment = $this->get_payment_terms();
            if ( ! empty( $payment ) ) {
                $sections[] = $this->format_section( __( 'Payment Terms', 'wc-delcampe-integration' ), $payment );
            }
        }
        
        // Guarantee
        if ( $this->should_include( 'include_guarantee' ) ) {
            $sections[] = $this->get_guarantee_text();
        }
        
        // Add custom footer if set
        if ( ! empty( $this->profile['description_settings']['custom_footer'] ) ) {
            $sections[] = $this->profile['description_settings']['custom_footer'];
        }
        
        return implode( "\n\n", array_filter( $sections ) );
    }

    /**
     * Build detailed stamp description
     * 
     * @return string
     */
    private function build_detailed_stamp_description() {
        $sections = array();
        
        // Add custom header
        if ( ! empty( $this->profile['description_settings']['custom_header'] ) ) {
            $sections[] = $this->profile['description_settings']['custom_header'];
        }
        
        // Title section with catalog number
        $sections[] = $this->get_detailed_title_section();
        
        // Main description
        $sections[] = $this->get_main_description();
        
        // Comprehensive technical details
        $sections[] = $this->format_section( 
            __( 'Complete Technical Specifications', 'wc-delcampe-integration' ), 
            $this->get_comprehensive_technical_details() 
        );
        
        // Detailed condition report
        $sections[] = $this->format_section( 
            __( 'Detailed Condition Report', 'wc-delcampe-integration' ), 
            $this->get_detailed_condition_report() 
        );
        
        // Provenance and certificates
        $provenance = $this->get_provenance_information();
        if ( ! empty( $provenance ) ) {
            $sections[] = $this->format_section( __( 'Provenance & Certificates', 'wc-delcampe-integration' ), $provenance );
        }
        
        // Complete shipping and handling
        $sections[] = $this->format_section( 
            __( 'Shipping & Handling', 'wc-delcampe-integration' ), 
            $this->get_detailed_shipping_information() 
        );
        
        // Payment options and policies
        $sections[] = $this->format_section( 
            __( 'Payment Options & Policies', 'wc-delcampe-integration' ), 
            $this->get_detailed_payment_information() 
        );
        
        // Extended guarantee and returns
        $sections[] = $this->get_extended_guarantee_text();
        
        // Add custom footer
        if ( ! empty( $this->profile['description_settings']['custom_footer'] ) ) {
            $sections[] = $this->profile['description_settings']['custom_footer'];
        }
        
        return implode( "\n\n", array_filter( $sections ) );
    }

    /**
     * Build minimal stamp description
     * 
     * @return string
     */
    private function build_minimal_stamp_description() {
        $sections = array();
        
        // Brief description
        $sections[] = $this->get_brief_description();
        
        // Essential details only
        $essential = array();
        
        // Condition
        $condition = $this->attributes['condition'] ?? $this->profile['stamp_settings']['default_condition'] ?? '';
        if ( ! empty( $condition ) ) {
            $essential[] = sprintf( __( 'Condition: %s', 'wc-delcampe-integration' ), $condition );
        }
        
        // Catalog number
        $catalog_number = $this->attributes['catalog_number'] ?? '';
        if ( ! empty( $catalog_number ) ) {
            $catalog_prefix = $this->profile['stamp_settings']['catalog_prefix'] ?? '';
            $essential[] = sprintf( __( 'Catalog: %s %s', 'wc-delcampe-integration' ), $catalog_prefix, $catalog_number );
        }
        
        if ( ! empty( $essential ) ) {
            $sections[] = implode( "\n", $essential );
        }
        
        // Shipping cost
        $shipping_cost = $this->profile['shipping_settings']['shipping_cost'] ?? '';
        if ( ! empty( $shipping_cost ) ) {
            $sections[] = sprintf( __( 'Shipping: %s', 'wc-delcampe-integration' ), $shipping_cost );
        }
        
        // Payment methods
        $payment_methods = $this->profile['payment_settings']['payment_methods'] ?? array();
        if ( ! empty( $payment_methods ) ) {
            $sections[] = __( 'Payment: ', 'wc-delcampe-integration' ) . implode( ', ', $payment_methods );
        }
        
        return implode( "\n\n", array_filter( $sections ) );
    }

    /**
     * Build custom description
     * 
     * @return string
     */
    private function build_custom_description() {
        // Start with WooCommerce description
        $description = $this->product->get_description();
        
        // Process template variables
        $description = $this->process_template_variables( $description );
        
        // Add header and footer
        $sections = array();
        
        if ( ! empty( $this->profile['description_settings']['custom_header'] ) ) {
            $sections[] = $this->profile['description_settings']['custom_header'];
        }
        
        $sections[] = $description;
        
        if ( ! empty( $this->profile['description_settings']['custom_footer'] ) ) {
            $sections[] = $this->profile['description_settings']['custom_footer'];
        }
        
        return implode( "\n\n", array_filter( $sections ) );
    }

    /**
     * Get main description
     * 
     * @return string
     */
    private function get_main_description() {
        $description = $this->product->get_description();
        
        if ( empty( $description ) ) {
            $description = $this->product->get_short_description();
        }
        
        return $this->process_template_variables( $description );
    }

    /**
     * Get brief description
     * 
     * @return string
     */
    private function get_brief_description() {
        $description = $this->product->get_short_description();
        
        if ( empty( $description ) ) {
            $description = wp_trim_words( $this->product->get_description(), 50 );
        }
        
        return $this->process_template_variables( $description );
    }

    /**
     * Get technical details
     * 
     * @return string
     */
    private function get_technical_details() {
        $details = array();
        
        // Country
        if ( ! empty( $this->attributes['country'] ) ) {
            $details[] = sprintf( __( 'Country: %s', 'wc-delcampe-integration' ), $this->attributes['country'] );
        }
        
        // Year/Period
        if ( ! empty( $this->attributes['year'] ) ) {
            $details[] = sprintf( __( 'Year: %s', 'wc-delcampe-integration' ), $this->attributes['year'] );
        }
        
        // Perforation
        if ( $this->profile['stamp_settings']['perforation_details'] && ! empty( $this->attributes['perforation'] ) ) {
            $details[] = sprintf( __( 'Perforation: %s', 'wc-delcampe-integration' ), $this->attributes['perforation'] );
        }
        
        // Watermark
        if ( $this->profile['stamp_settings']['watermark_info'] && ! empty( $this->attributes['watermark'] ) ) {
            $details[] = sprintf( __( 'Watermark: %s', 'wc-delcampe-integration' ), $this->attributes['watermark'] );
        }
        
        // Color
        if ( ! empty( $this->attributes['color'] ) ) {
            $details[] = sprintf( __( 'Color: %s', 'wc-delcampe-integration' ), $this->attributes['color'] );
        }
        
        // Type
        if ( ! empty( $this->attributes['stamp_type'] ) ) {
            $details[] = sprintf( __( 'Type: %s', 'wc-delcampe-integration' ), $this->attributes['stamp_type'] );
        }
        
        return implode( "\n", $details );
    }

    /**
     * Get comprehensive technical details
     * 
     * @return string
     */
    private function get_comprehensive_technical_details() {
        $details = array();
        
        // All available technical attributes
        $technical_attrs = array(
            'country' => __( 'Country of Issue', 'wc-delcampe-integration' ),
            'year' => __( 'Year of Issue', 'wc-delcampe-integration' ),
            'stamp_type' => __( 'Stamp Type', 'wc-delcampe-integration' ),
            'denomination' => __( 'Denomination', 'wc-delcampe-integration' ),
            'color' => __( 'Color(s)', 'wc-delcampe-integration' ),
            'perforation' => __( 'Perforation', 'wc-delcampe-integration' ),
            'watermark' => __( 'Watermark', 'wc-delcampe-integration' ),
            'printing_method' => __( 'Printing Method', 'wc-delcampe-integration' ),
            'paper_type' => __( 'Paper Type', 'wc-delcampe-integration' ),
            'dimensions' => __( 'Dimensions', 'wc-delcampe-integration' ),
            'designer' => __( 'Designer', 'wc-delcampe-integration' ),
            'engraver' => __( 'Engraver', 'wc-delcampe-integration' ),
            'printer' => __( 'Printer', 'wc-delcampe-integration' ),
            'issue_quantity' => __( 'Issue Quantity', 'wc-delcampe-integration' ),
            'theme' => __( 'Theme/Topic', 'wc-delcampe-integration' ),
        );
        
        foreach ( $technical_attrs as $attr => $label ) {
            if ( ! empty( $this->attributes[$attr] ) ) {
                $details[] = sprintf( '%s: %s', $label, $this->attributes[$attr] );
            }
        }
        
        // Add varieties/errors if present
        if ( ! empty( $this->attributes['varieties'] ) ) {
            $details[] = "\n" . __( 'Varieties/Errors:', 'wc-delcampe-integration' );
            $details[] = $this->attributes['varieties'];
        }
        
        return implode( "\n", $details );
    }

    /**
     * Get catalog information
     * 
     * @return string
     */
    private function get_catalog_information() {
        $info = array();
        
        $catalog_number = $this->attributes['catalog_number'] ?? '';
        if ( empty( $catalog_number ) ) {
            return '';
        }
        
        $catalog_prefix = $this->profile['stamp_settings']['catalog_prefix'] ?? 'Michel';
        
        // Main catalog number
        $info[] = sprintf( '%s: %s', $catalog_prefix, $catalog_number );
        
        // Additional catalog references
        $other_catalogs = array(
            'scott_number' => 'Scott',
            'yvert_number' => 'Yvert',
            'sg_number' => 'Stanley Gibbons',
            'sassone_number' => 'Sassone',
            'edifil_number' => 'Edifil',
        );
        
        foreach ( $other_catalogs as $attr => $catalog_name ) {
            if ( ! empty( $this->attributes[$attr] ) ) {
                $info[] = sprintf( '%s: %s', $catalog_name, $this->attributes[$attr] );
            }
        }
        
        // Catalog value if provided
        if ( ! empty( $this->attributes['catalog_value'] ) ) {
            $info[] = sprintf( __( 'Catalog Value: %s', 'wc-delcampe-integration' ), $this->attributes['catalog_value'] );
        }
        
        return implode( "\n", $info );
    }

    /**
     * Get condition details
     * 
     * @return string
     */
    private function get_condition_details() {
        $details = array();
        
        // Main condition
        $condition = $this->attributes['condition'] ?? $this->profile['stamp_settings']['default_condition'] ?? '';
        if ( ! empty( $condition ) ) {
            $details[] = sprintf( __( 'Overall Condition: %s', 'wc-delcampe-integration' ), $this->get_condition_label( $condition ) );
        }
        
        // Gum condition
        $gum = $this->attributes['gum_condition'] ?? $this->profile['stamp_settings']['gum_condition'] ?? '';
        if ( ! empty( $gum ) ) {
            $details[] = sprintf( __( 'Gum: %s', 'wc-delcampe-integration' ), $this->get_gum_label( $gum ) );
        }
        
        // Centering
        $centering = $this->attributes['centering'] ?? $this->profile['stamp_settings']['centering'] ?? '';
        if ( ! empty( $centering ) ) {
            $details[] = sprintf( __( 'Centering: %s', 'wc-delcampe-integration' ), $this->get_centering_label( $centering ) );
        }
        
        // Grade
        $grade = $this->attributes['grade'] ?? $this->profile['stamp_settings']['grade'] ?? '';
        if ( ! empty( $grade ) ) {
            $details[] = sprintf( __( 'Grade: %s', 'wc-delcampe-integration' ), $grade );
        }
        
        // Faults/defects
        if ( ! empty( $this->attributes['faults'] ) ) {
            $details[] = sprintf( __( 'Faults: %s', 'wc-delcampe-integration' ), $this->attributes['faults'] );
        }
        
        // Additional condition notes
        if ( ! empty( $this->attributes['condition_notes'] ) ) {
            $details[] = "\n" . __( 'Additional Notes:', 'wc-delcampe-integration' );
            $details[] = $this->attributes['condition_notes'];
        }
        
        return implode( "\n", $details );
    }

    /**
     * Get detailed condition report
     * 
     * @return string
     */
    private function get_detailed_condition_report() {
        $report = array();
        
        // Overall assessment
        $condition = $this->attributes['condition'] ?? $this->profile['stamp_settings']['default_condition'] ?? '';
        $report[] = '** ' . __( 'OVERALL ASSESSMENT', 'wc-delcampe-integration' ) . ' **';
        $report[] = $this->get_condition_label( $condition );
        $report[] = '';
        
        // Front examination
        $report[] = '** ' . __( 'FRONT EXAMINATION', 'wc-delcampe-integration' ) . ' **';
        
        if ( ! empty( $this->attributes['front_condition'] ) ) {
            $report[] = $this->attributes['front_condition'];
        } else {
            $front_items = array();
            if ( ! empty( $this->attributes['centering'] ) ) {
                $front_items[] = __( 'Centering: ', 'wc-delcampe-integration' ) . $this->get_centering_label( $this->attributes['centering'] );
            }
            if ( ! empty( $this->attributes['margins'] ) ) {
                $front_items[] = __( 'Margins: ', 'wc-delcampe-integration' ) . $this->attributes['margins'];
            }
            if ( ! empty( $this->attributes['color_quality'] ) ) {
                $front_items[] = __( 'Color: ', 'wc-delcampe-integration' ) . $this->attributes['color_quality'];
            }
            $report[] = ! empty( $front_items ) ? implode( "\n", $front_items ) : __( 'No visible defects', 'wc-delcampe-integration' );
        }
        $report[] = '';
        
        // Back examination
        $report[] = '** ' . __( 'BACK EXAMINATION', 'wc-delcampe-integration' ) . ' **';
        
        if ( ! empty( $this->attributes['back_condition'] ) ) {
            $report[] = $this->attributes['back_condition'];
        } else {
            $gum = $this->attributes['gum_condition'] ?? $this->profile['stamp_settings']['gum_condition'] ?? '';
            if ( ! empty( $gum ) ) {
                $report[] = __( 'Gum: ', 'wc-delcampe-integration' ) . $this->get_gum_label( $gum );
            }
            if ( ! empty( $this->attributes['hinge_marks'] ) ) {
                $report[] = __( 'Hinge Marks: ', 'wc-delcampe-integration' ) . $this->attributes['hinge_marks'];
            }
        }
        $report[] = '';
        
        // Perforations
        if ( ! empty( $this->attributes['perforation_condition'] ) ) {
            $report[] = '** ' . __( 'PERFORATIONS', 'wc-delcampe-integration' ) . ' **';
            $report[] = $this->attributes['perforation_condition'];
            $report[] = '';
        }
        
        // Special features
        $special = array();
        if ( ! empty( $this->attributes['plate_number'] ) ) {
            $special[] = __( 'Plate Number: ', 'wc-delcampe-integration' ) . $this->attributes['plate_number'];
        }
        if ( ! empty( $this->attributes['selvage'] ) ) {
            $special[] = __( 'Selvage: ', 'wc-delcampe-integration' ) . $this->attributes['selvage'];
        }
        if ( ! empty( $this->attributes['cancellation'] ) ) {
            $special[] = __( 'Cancellation: ', 'wc-delcampe-integration' ) . $this->attributes['cancellation'];
        }
        
        if ( ! empty( $special ) ) {
            $report[] = '** ' . __( 'SPECIAL FEATURES', 'wc-delcampe-integration' ) . ' **';
            $report[] = implode( "\n", $special );
        }
        
        return implode( "\n", $report );
    }

    /**
     * Get provenance information
     * 
     * @return string
     */
    private function get_provenance_information() {
        $info = array();
        
        // Certificate
        if ( $this->profile['stamp_settings']['certificate'] || ! empty( $this->attributes['certificate'] ) ) {
            $cert_details = $this->attributes['certificate_details'] ?? $this->profile['stamp_settings']['certificate_details'] ?? '';
            if ( ! empty( $cert_details ) ) {
                $info[] = sprintf( __( 'Certificate: %s', 'wc-delcampe-integration' ), $cert_details );
            } else {
                $info[] = __( 'Certificate: Available', 'wc-delcampe-integration' );
            }
        }
        
        // Expertization
        if ( $this->profile['stamp_settings']['expertized'] || ! empty( $this->attributes['expertized'] ) ) {
            $expert_name = $this->attributes['expertizer_name'] ?? $this->profile['stamp_settings']['expertizer_name'] ?? '';
            if ( ! empty( $expert_name ) ) {
                $info[] = sprintf( __( 'Expertized by: %s', 'wc-delcampe-integration' ), $expert_name );
            } else {
                $info[] = __( 'Expertized: Yes', 'wc-delcampe-integration' );
            }
        }
        
        // Previous collections
        if ( ! empty( $this->attributes['provenance'] ) ) {
            $info[] = sprintf( __( 'Provenance: %s', 'wc-delcampe-integration' ), $this->attributes['provenance'] );
        }
        
        // Exhibition history
        if ( ! empty( $this->attributes['exhibitions'] ) ) {
            $info[] = sprintf( __( 'Exhibitions: %s', 'wc-delcampe-integration' ), $this->attributes['exhibitions'] );
        }
        
        return implode( "\n", $info );
    }

    /**
     * Get shipping information
     * 
     * @return string
     */
    private function get_shipping_information() {
        $info = array();
        
        $settings = $this->profile['shipping_settings'] ?? array();
        
        // Shipping type and cost
        $shipping_type = $settings['shipping_type'] ?? 'Standard';
        $shipping_cost = $settings['shipping_cost'] ?? '';
        
        if ( ! empty( $shipping_cost ) ) {
            $info[] = sprintf( 
                __( '%s Mail: %s', 'wc-delcampe-integration' ), 
                $this->get_shipping_type_label( $shipping_type ),
                $shipping_cost 
            );
        }
        
        // Combined shipping
        if ( ! empty( $settings['combined_shipping'] ) ) {
            $additional_cost = $settings['additional_item_cost'] ?? '0.50';
            $info[] = sprintf( __( 'Additional items: %s each', 'wc-delcampe-integration' ), $additional_cost );
        }
        
        // Ships to
        $ships_to = $settings['ships_to'] ?? 'Worldwide';
        $info[] = __( 'Ships to: ', 'wc-delcampe-integration' ) . $this->get_shipping_destination_label( $ships_to );
        
        // Handling time
        $handling_time = $settings['handling_time'] ?? '2';
        $info[] = sprintf( __( 'Handling time: %s business days', 'wc-delcampe-integration' ), $handling_time );
        
        // Insurance
        if ( ! empty( $settings['insurance_optional'] ) ) {
            $insurance_cost = $settings['insurance_cost'] ?? '5.00';
            $info[] = sprintf( __( 'Optional insurance available: %s', 'wc-delcampe-integration' ), $insurance_cost );
        }
        
        return implode( "\n", $info );
    }

    /**
     * Get detailed shipping information
     * 
     * @return string
     */
    private function get_detailed_shipping_information() {
        $info = array();
        $settings = $this->profile['shipping_settings'] ?? array();
        
        // Shipping methods and costs
        $info[] = '** ' . __( 'SHIPPING METHODS & COSTS', 'wc-delcampe-integration' ) . ' **';
        
        $shipping_type = $settings['shipping_type'] ?? 'Standard';
        $shipping_cost = $settings['shipping_cost'] ?? '';
        
        $info[] = sprintf( 
            '%s: %s', 
            $this->get_shipping_type_label( $shipping_type ),
            $shipping_cost 
        );
        
        // Additional shipping options if available
        if ( ! empty( $this->attributes['shipping_options'] ) ) {
            $info[] = $this->attributes['shipping_options'];
        }
        
        $info[] = '';
        
        // Combined shipping policy
        if ( ! empty( $settings['combined_shipping'] ) ) {
            $info[] = '** ' . __( 'COMBINED SHIPPING', 'wc-delcampe-integration' ) . ' **';
            $additional_cost = $settings['additional_item_cost'] ?? '0.50';
            $info[] = __( 'We offer combined shipping to save on postage costs.', 'wc-delcampe-integration' );
            $info[] = sprintf( __( 'Additional items: %s each', 'wc-delcampe-integration' ), $additional_cost );
            $info[] = __( 'Please wait for combined invoice before paying for multiple items.', 'wc-delcampe-integration' );
            $info[] = '';
        }
        
        // Destinations
        $info[] = '** ' . __( 'SHIPPING DESTINATIONS', 'wc-delcampe-integration' ) . ' **';
        $ships_to = $settings['ships_to'] ?? 'Worldwide';
        $info[] = $this->get_shipping_destination_details( $ships_to );
        
        if ( ! empty( $settings['excluded_countries'] ) ) {
            $info[] = sprintf( __( 'Excluded countries: %s', 'wc-delcampe-integration' ), $settings['excluded_countries'] );
        }
        $info[] = '';
        
        // Handling and dispatch
        $info[] = '** ' . __( 'HANDLING & DISPATCH', 'wc-delcampe-integration' ) . ' **';
        $handling_time = $settings['handling_time'] ?? '2';
        $info[] = sprintf( __( 'Items are carefully packaged and dispatched within %s business days after payment is received.', 'wc-delcampe-integration' ), $handling_time );
        $info[] = __( 'All stamps are protected in glassine envelopes and sent in rigid mailers.', 'wc-delcampe-integration' );
        $info[] = '';
        
        // Insurance and tracking
        $info[] = '** ' . __( 'INSURANCE & TRACKING', 'wc-delcampe-integration' ) . ' **';
        if ( ! empty( $settings['insurance_optional'] ) ) {
            $insurance_cost = $settings['insurance_cost'] ?? '5.00';
            $info[] = sprintf( __( 'Optional insurance is available for %s', 'wc-delcampe-integration' ), $insurance_cost );
            $info[] = __( 'We recommend insurance for valuable items.', 'wc-delcampe-integration' );
        }
        $info[] = __( 'Tracking information will be provided when available.', 'wc-delcampe-integration' );
        
        return implode( "\n", $info );
    }

    /**
     * Get payment terms
     * 
     * @return string
     */
    private function get_payment_terms() {
        $terms = array();
        $settings = $this->profile['payment_settings'] ?? array();
        
        // Payment methods
        $methods = $settings['payment_methods'] ?? array();
        if ( ! empty( $methods ) ) {
            $terms[] = __( 'Accepted payment methods: ', 'wc-delcampe-integration' ) . implode( ', ', array_map( array( $this, 'get_payment_method_label' ), $methods ) );
        }
        
        // Payment deadline
        $deadline = $settings['payment_deadline'] ?? '7';
        $terms[] = sprintf( __( 'Payment expected within %s days', 'wc-delcampe-integration' ), $deadline );
        
        // Payment instructions
        if ( ! empty( $settings['payment_instructions'] ) ) {
            $terms[] = $settings['payment_instructions'];
        }
        
        return implode( "\n", $terms );
    }

    /**
     * Get detailed payment information
     * 
     * @return string
     */
    private function get_detailed_payment_information() {
        $info = array();
        $settings = $this->profile['payment_settings'] ?? array();
        
        // Payment methods
        $info[] = '** ' . __( 'ACCEPTED PAYMENT METHODS', 'wc-delcampe-integration' ) . ' **';
        $methods = $settings['payment_methods'] ?? array();
        
        foreach ( $methods as $method ) {
            $method_info = $this->get_payment_method_details( $method );
            if ( ! empty( $method_info ) ) {
                $info[] = $method_info;
            }
        }
        $info[] = '';
        
        // Payment terms
        $info[] = '** ' . __( 'PAYMENT TERMS', 'wc-delcampe-integration' ) . ' **';
        $deadline = $settings['payment_deadline'] ?? '7';
        $info[] = sprintf( __( 'Payment is expected within %s days after purchase.', 'wc-delcampe-integration' ), $deadline );
        $info[] = __( 'Items will be dispatched promptly upon receipt of payment.', 'wc-delcampe-integration' );
        
        if ( ! empty( $settings['payment_instructions'] ) ) {
            $info[] = '';
            $info[] = $settings['payment_instructions'];
        }
        $info[] = '';
        
        // Currency and conversion
        $currency = $this->profile['listing_details']['currency'] ?? get_woocommerce_currency();
        $info[] = '** ' . __( 'CURRENCY', 'wc-delcampe-integration' ) . ' **';
        $info[] = sprintf( __( 'All prices are in %s', 'wc-delcampe-integration' ), $currency );
        $info[] = __( 'Currency conversion will be handled by your payment provider.', 'wc-delcampe-integration' );
        
        return implode( "\n", $info );
    }

    /**
     * Get guarantee text
     * 
     * @return string
     */
    private function get_guarantee_text() {
        return __( '** SATISFACTION GUARANTEE **', 'wc-delcampe-integration' ) . "\n" .
               __( 'All items are guaranteed to be genuine and as described. If you are not satisfied with your purchase, please contact us within 14 days for a full refund.', 'wc-delcampe-integration' );
    }

    /**
     * Get extended guarantee text
     * 
     * @return string
     */
    private function get_extended_guarantee_text() {
        $text = array();
        
        $text[] = '** ' . __( 'OUR GUARANTEE TO YOU', 'wc-delcampe-integration' ) . ' **';
        $text[] = '';
        $text[] = __( 'AUTHENTICITY:', 'wc-delcampe-integration' );
        $text[] = __( 'All stamps are guaranteed to be 100% genuine. We stand behind every item we sell.', 'wc-delcampe-integration' );
        $text[] = '';
        $text[] = __( 'ACCURATE DESCRIPTION:', 'wc-delcampe-integration' );
        $text[] = __( 'Items are described to the best of our ability. High-resolution scans show actual items.', 'wc-delcampe-integration' );
        $text[] = '';
        $text[] = __( 'RETURN POLICY:', 'wc-delcampe-integration' );
        $return_days = $this->profile['shipping_settings']['return_policy'] ?? '14';
        $text[] = sprintf( __( '%s-day return policy for any reason. Return shipping at buyer\'s expense.', 'wc-delcampe-integration' ), $return_days );
        $text[] = '';
        $text[] = __( 'PROFESSIONAL PACKAGING:', 'wc-delcampe-integration' );
        $text[] = __( 'All items are carefully packaged to ensure safe delivery.', 'wc-delcampe-integration' );
        $text[] = '';
        $text[] = __( 'MEMBER:', 'wc-delcampe-integration' );
        
        // Add membership info if available
        if ( ! empty( $this->attributes['memberships'] ) ) {
            $text[] = $this->attributes['memberships'];
        } else {
            $text[] = __( 'Established seller with excellent feedback.', 'wc-delcampe-integration' );
        }
        
        return implode( "\n", $text );
    }

    /**
     * Get product attributes
     * 
     * @return array
     */
    private function get_product_attributes() {
        $attributes = array();
        
        // Get WooCommerce attributes
        $product_attributes = $this->product->get_attributes();
        
        foreach ( $product_attributes as $attribute ) {
            if ( $attribute->is_taxonomy() ) {
                $terms = wp_get_post_terms( $this->product->get_id(), $attribute->get_name(), array( 'fields' => 'names' ) );
                $attributes[ $attribute->get_name() ] = implode( ', ', $terms );
            } else {
                $attributes[ $attribute->get_name() ] = implode( ', ', $attribute->get_options() );
            }
        }
        
        // Get meta fields
        $meta_fields = array(
            'condition',
            'catalog_number',
            'country',
            'year',
            'perforation',
            'watermark',
            'gum_condition',
            'centering',
            'grade',
            'certificate',
            'certificate_details',
            'expertized',
            'expertizer_name',
            'color',
            'stamp_type',
            'faults',
            'condition_notes',
        );
        
        foreach ( $meta_fields as $field ) {
            $value = get_post_meta( $this->product->get_id(), '_' . $field, true );
            if ( ! empty( $value ) ) {
                $attributes[ $field ] = $value;
            }
        }
        
        return $attributes;
    }

    /**
     * Process template variables
     * 
     * @param string $text Text with variables
     * @return string Processed text
     */
    private function process_template_variables( $text ) {
        $variables = array(
            '{title}' => $this->product->get_name(),
            '{sku}' => $this->product->get_sku(),
            '{price}' => $this->product->get_price(),
            '{catalog_number}' => $this->attributes['catalog_number'] ?? '',
            '{condition}' => $this->attributes['condition'] ?? '',
            '{country}' => $this->attributes['country'] ?? '',
            '{year}' => $this->attributes['year'] ?? '',
        );
        
        return str_replace( array_keys( $variables ), array_values( $variables ), $text );
    }

    /**
     * Format section
     * 
     * @param string $title   Section title
     * @param string $content Section content
     * @return string Formatted section
     */
    private function format_section( $title, $content ) {
        if ( empty( $content ) ) {
            return '';
        }
        
        return "=== " . strtoupper( $title ) . " ===\n" . $content;
    }

    /**
     * Check if section should be included
     * 
     * @param string $setting Setting name
     * @return bool
     */
    private function should_include( $setting ) {
        return ! empty( $this->profile['description_settings'][ $setting ] );
    }

    /**
     * Get condition label
     * 
     * @param string $condition Condition code
     * @return string Label
     */
    private function get_condition_label( $condition ) {
        $conditions = delcampe_get_stamp_conditions_translated();
        return $conditions[ $condition ] ?? $condition;
    }

    /**
     * Get gum label
     * 
     * @param string $gum Gum code
     * @return string Label
     */
    private function get_gum_label( $gum ) {
        $conditions = delcampe_get_gum_conditions();
        return $conditions[ $gum ] ?? $gum;
    }

    /**
     * Get centering label
     * 
     * @param string $centering Centering code
     * @return string Label
     */
    private function get_centering_label( $centering ) {
        $options = delcampe_get_centering_options();
        return $options[ $centering ] ?? $centering;
    }

    /**
     * Get shipping type label
     * 
     * @param string $type Type code
     * @return string Label
     */
    private function get_shipping_type_label( $type ) {
        $types = delcampe_get_shipping_types();
        return $types[ $type ] ?? $type;
    }

    /**
     * Get shipping destination label
     * 
     * @param string $destination Destination code
     * @return string Label
     */
    private function get_shipping_destination_label( $destination ) {
        $destinations = delcampe_get_shipping_countries();
        return $destinations[ $destination ] ?? $destination;
    }

    /**
     * Get shipping destination details
     * 
     * @param string $destination Destination code
     * @return string Details
     */
    private function get_shipping_destination_details( $destination ) {
        switch ( $destination ) {
            case 'Worldwide':
                return __( 'We ship to all countries worldwide.', 'wc-delcampe-integration' );
            case 'Europe':
                return __( 'We ship to all European countries.', 'wc-delcampe-integration' );
            case 'EU':
                return __( 'We ship to European Union member countries only.', 'wc-delcampe-integration' );
            case 'North_America':
                return __( 'We ship to USA, Canada, and Mexico.', 'wc-delcampe-integration' );
            case 'Custom':
                return __( 'Please check our shipping policy for eligible countries.', 'wc-delcampe-integration' );
            default:
                return __( 'Please contact us for shipping availability.', 'wc-delcampe-integration' );
        }
    }

    /**
     * Get payment method label
     * 
     * @param string $method Method code
     * @return string Label
     */
    private function get_payment_method_label( $method ) {
        $methods = delcampe_get_payment_methods();
        return $methods[ $method ] ?? $method;
    }

    /**
     * Get payment method details
     * 
     * @param string $method Method code
     * @return string Details
     */
    private function get_payment_method_details( $method ) {
        $settings = $this->profile['payment_settings'] ?? array();
        
        switch ( $method ) {
            case 'PayPal':
                $email = $settings['paypal_email'] ?? '';
                if ( ! empty( $email ) ) {
                    return sprintf( __( 'PayPal: Send payment to %s', 'wc-delcampe-integration' ), $email );
                }
                return __( 'PayPal: Fast and secure payment', 'wc-delcampe-integration' );
                
            case 'BankTransfer':
                $details = $settings['bank_details'] ?? '';
                if ( ! empty( $details ) ) {
                    return __( 'Bank Transfer: ', 'wc-delcampe-integration' ) . "\n" . $details;
                }
                return __( 'Bank Transfer: Details provided after purchase', 'wc-delcampe-integration' );
                
            case 'Check':
                return __( 'Check/Cheque: Contact for mailing address', 'wc-delcampe-integration' );
                
            case 'MoneyOrder':
                return __( 'Money Order: International money orders accepted', 'wc-delcampe-integration' );
                
            case 'Cash':
                return __( 'Cash: At buyer\'s risk, well-concealed only', 'wc-delcampe-integration' );
                
            case 'Skrill':
                return __( 'Skrill: International payment service', 'wc-delcampe-integration' );
                
            default:
                return sprintf( __( '%s: Contact for details', 'wc-delcampe-integration' ), $method );
        }
    }

    /**
     * Get detailed title section
     * 
     * @return string
     */
    private function get_detailed_title_section() {
        $parts = array();
        
        // Country and year
        if ( ! empty( $this->attributes['country'] ) ) {
            $parts[] = strtoupper( $this->attributes['country'] );
        }
        if ( ! empty( $this->attributes['year'] ) ) {
            $parts[] = $this->attributes['year'];
        }
        
        // Main title
        $parts[] = $this->product->get_name();
        
        // Catalog number
        if ( $this->profile['stamp_settings']['add_catalog_number'] && ! empty( $this->attributes['catalog_number'] ) ) {
            $catalog_prefix = $this->profile['stamp_settings']['catalog_prefix'] ?? '';
            $parts[] = sprintf( '(%s %s)', $catalog_prefix, $this->attributes['catalog_number'] );
        }
        
        // Condition
        $condition = $this->attributes['condition'] ?? $this->profile['stamp_settings']['default_condition'] ?? '';
        if ( ! empty( $condition ) ) {
            $parts[] = '- ' . $condition;
        }
        
        return implode( ' ', $parts );
    }
}
